/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* FILE, fgets, fprintf, ... */
#include <string.h>	/* strchr, strcmp, strlen, ... */
#include <unistd.h>	/* unlink */

#include <gtk/gtk.h>

#include "source.h"
#include "utils.h"
#include "server.h"
#include "pref.h"


GSList *qw_masters = NULL;
struct master *unbound_qw = NULL;
struct master *quake2 = NULL;
struct master *favorites = NULL;
struct master *allsources = NULL;


static char *sp2under (char *str) {
  char *ptr;

  if (str) {
    for (ptr = str; *ptr; ptr++) {
      if (*ptr == ' ') 
	*ptr = '_';
    }
  }

  return str;
}


static void read_saved_server_list (struct master *m) {
  FILE *f;
  char *fn;
  char *nl;
  int n;
  struct server *s;
  char *addr;
  int port;
  enum server_type type;
  char buf[1024];
  char *s_addr, *s_name;

  m->servers = NULL;

  fn = file_in_dir (user.rcdir, (m->name) ? m->name : m->address);
  sp2under (fn);
  f = fopen (fn, "r");
  g_free (fn);

  if (!f)
    return;

  while (!feof (f)) {
    fgets (buf, 1024, f);

    nl = buf + strlen (buf) - 1;
    if (*nl == '\n')
      *nl = '\0';

    s_addr = strchr (buf, '|');
    if (!s_addr)
      continue;
    *s_addr++ = '\0';
    
    s_name = strchr (s_addr, '|');
    if (!s_name)
      continue;
    *s_name++ = '\0';

    if (strcmp (buf, "QW") == 0)
      type = QW_SERVER;
    else if (strcmp (buf, "Q2") == 0)
      type = Q2_SERVER;
    else
      continue;

    if (!parse_address (s_addr, &addr, &port))
      continue;

    if (port == -1)
      port = (type == Q2_SERVER)? Q2_DEFAULT_PORT : QW_DEFAULT_PORT;

    s = server_lookup (addr, port);
    if (!s) {
      s = server_new (addr, port, type);

      if (s_name[0])            /* if name is not empty */
	s->name = g_strdup (s_name);
    }

    add_server_to_master (m, s);
    g_free (addr);
  }
}


void save_server_list (struct master *m) {
  FILE *f;
  char *fn;
  GSList *list;
  struct server *s;

  fn = file_in_dir (user.rcdir, (m->name) ? m->name : m->address);
  sp2under (fn);

  if (m->servers) {
    f = fopen (fn, "w");
    g_free (fn);
    if (!f)
      return;
  }
  else {
    unlink (fn);
    g_free (fn);
    return;
  }

  for (list = m->servers; list; list = list->next) {
    s = (struct server *) list->data;
    fprintf (f, "%s|%s:%d|%s\n", (s->type == QW_SERVER) ? "QW" : "Q2",
	                         s->address, s->port, (s->name)? s->name : "");
  }

  fclose (f);
}


static GSList *read_master_file (char *filename) {
  FILE *f;
  char buf[1024];
  char *str;
  GSList *list = NULL;
  GSList *ptr;
  struct master *m;

  f = fopen (filename, "r");
  if (!f)
    return NULL;

  while (!feof (f)) {
    fgets (buf, 1024, f);

    str = strchr (buf, '\n');
    if (str != NULL)
      *str = '\0';

    str = strchr (buf, '|');
    if (str == NULL)
      break; 		/* error */
    *str++ = '\0';

    m = g_malloc (sizeof (struct master));
    m->address = strdup_strip (buf);
    m->name = strdup_strip (str);
    m->servers = NULL;
    if (default_save_lists)
      read_saved_server_list (m);
    list = g_slist_prepend (list, m);
  }
  
  fclose (f);

  return list;
}


static char *master_dat[] = {

#include "master-dat.h"

  NULL, NULL
};


static GSList *builtin_master_file (void) {
  GSList *list = NULL;
  struct master *m;
  int i;

  for (i = 0; master_dat[i]; i += 2) {
    m = g_malloc (sizeof (struct master));
    m->address = strdup_strip (master_dat[i]);
    m->name = strdup_strip (master_dat[i + 1]);
    m->servers = NULL;
    if (default_save_lists)
      read_saved_server_list (m);
    list = g_slist_prepend (list, m);
  }

  return list;
}


static GSList *read_master_dat (char *dir) {
  const char master_dat[] = "master.dat";
  GSList *list;
  char *fn;

  fn = file_in_dir (dir, master_dat);
  list = read_master_file (fn);
  g_free (fn);
  return list;
}


void init_masters (void) {
  qw_masters = NULL;

  /* search for master.dat in ${rcdir}, ${quakedir}, /usr/local/lib, /usr/lib */

  (qw_masters = read_master_dat (user.rcdir)) || 
  (qw_masters = read_master_dat (default_quake_dir)) || 
  (qw_masters = read_master_dat ("/usr/local/lib/")) ||
  (qw_masters = read_master_dat ("/usr/lib/"));

  if (!qw_masters)
    qw_masters = builtin_master_file ();

  qw_masters = g_slist_reverse (qw_masters);

  unbound_qw = g_malloc (sizeof (struct master));
  unbound_qw->name = g_strdup ("Unbound QW");
  unbound_qw->address = NULL;
  unbound_qw->servers = NULL;
  read_saved_server_list (unbound_qw);

  quake2 = g_malloc (sizeof (struct master));
  quake2->name = g_strdup ("Quake II");
  quake2->address = NULL;
  quake2->servers = NULL;
  read_saved_server_list (quake2);

  favorites = g_malloc (sizeof (struct master));
  favorites->name = g_strdup ("Favorites");
  favorites->address = NULL;
  favorites->servers = NULL;
  read_saved_server_list (favorites);

  allsources = g_malloc (sizeof (struct master));
  allsources->name = g_strdup ("All Sources");
  allsources->address = NULL;
  allsources->servers = NULL;
}


static void free_master (struct master *m, int save) {
  if (m) {
    if (save)
      save_server_list (m);

    if (m->address) g_free (m->address);
    if (m->name) g_free (m->name);
    free_servers (m->servers);
    g_free (m);
  }
}


void free_masters (void) {
  GSList *ptr;

  save_server_list (favorites);
  free_master (favorites, TRUE);
  favorites = NULL;

  save_server_list (unbound_qw);
  free_master (unbound_qw, TRUE);
  unbound_qw = NULL;

  save_server_list (quake2);
  free_master (quake2, TRUE);
  quake2 = NULL;

  free_master (allsources, FALSE);
  allsources = NULL;

  for (ptr = qw_masters; ptr; ptr = ptr->next)
    free_master ((struct master *) ptr->data, default_save_lists);
  g_slist_free (qw_masters);
  qw_masters = NULL;
}


void add_server_to_master (struct master *m, struct server *s) {
  if (g_slist_find (m->servers, s))
    return;

  /* "prepend" is faster, but "append" is more natural way */

  m->servers = g_slist_append (m->servers, s);
  s->ref_count++;
}


void remove_server_from_master (struct master *m, struct server *s) {
  if (g_slist_find (m->servers, s)) {
    m->servers = g_slist_remove (m->servers, s);
    server_unref (s);
  }
}



