/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* sprintf, vsnprintf */
#include <stdlib.h>	/* qsort */
#include <stdarg.h>	/* vsnprintf */
#include <arpa/inet.h>	/* inet_aton */

#include <gtk/gtk.h>

#include "xqf.h"
#include "qstat.h"
#include "source.h"
#include "pref.h"
#include "filter.h"
#include "skin.h"
#include "dialogs.h"
#include "utils.h"
#include "server.h"
#include "qrun.h"
#include "psearch.h"


GtkWidget *main_window = NULL;
GtkWidget *server_clist = NULL;
GtkWidget *player_clist = NULL;
GtkWidget *serverinfo_clist = NULL;

static GtkWidget *server_menu = NULL;
static GtkWidget *connect_menu_item = NULL;
static GtkWidget *observe_menu_item = NULL;
static GtkWidget *add_to_favs_menu_item = NULL;
static GtkWidget *add_menu_item = NULL;
static GtkWidget *delete_menu_item = NULL;
static GtkWidget *refresh_menu_item = NULL;
static GtkWidget *refrsel_menu_item = NULL;
static GtkWidget *find_player_menu_item = NULL;
static GtkWidget *refresh_button = NULL;
static GtkWidget *update_button = NULL;

static GtkWidget *status_bar_label = NULL;

static GSList *xqf_windows = NULL;
static GtkWidget *target_window = NULL;

static struct master *cur_master = NULL;
static struct server *cur_server = NULL;

static struct condef *con = NULL;


struct clist_coldef server_clist_columns[7] = {
  { "Name", 	250, 	GTK_JUSTIFY_LEFT,	SORT_SERVER_NAME,	NULL },
  { "Address",	150, 	GTK_JUSTIFY_LEFT,	SORT_SERVER_ADDRESS,	NULL },
  { "Ping",	50,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_PING,	NULL },
  { "TO",	40,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_TO,		NULL },
  { "Players",	70,  	GTK_JUSTIFY_RIGHT,	SORT_SERVER_PLAYERS,	NULL },
  { "Map",	60,  	GTK_JUSTIFY_LEFT,	SORT_SERVER_MAP,	NULL },
  { "Game",	60,	GTK_JUSTIFY_LEFT,	SORT_SERVER_GAME,	NULL }
};


struct clist_coldef player_clist_columns[6] = {
  { "Name",	120,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_NAME,	NULL },
  { "Frags",	60,	GTK_JUSTIFY_RIGHT,      -SORT_PLAYER_FRAGS,	NULL },
  { "Colors",	60,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_COLOR,      NULL },
  { "Skin",	70,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_SKIN,       NULL },
  { "Ping",	50,	GTK_JUSTIFY_RIGHT,      SORT_PLAYER_PING,       NULL },
  { "Time",	50,	GTK_JUSTIFY_LEFT,       SORT_PLAYER_TIME,       NULL }
};


struct clist_coldef serverinfo_clist_columns[2] = {
  { "Rule",	70,	GTK_JUSTIFY_LEFT,       0,	NULL },
  { "Value",	160,	GTK_JUSTIFY_LEFT,       0,	NULL }
};


int psort_column = 1;
int ssort_column = 2;


#ifdef USEPIXS

#include "xpm/q.xpm"
#include "xpm/q2.xpm"

static GdkPixmap *q_pix = NULL;
static GdkBitmap *q_pix_mask = NULL;
static GdkPixmap *q2_pix = NULL;
static GdkBitmap *q2_pix_mask = NULL;


static void free_pixmaps (void) {
  if (q_pix) {
    gdk_pixmap_unref (q_pix);
    q_pix = NULL;
  }
  if (q_pix_mask) {
    gdk_bitmap_unref (q_pix_mask);
    q_pix_mask = NULL;
  }
  if (q2_pix) {
    gdk_pixmap_unref (q2_pix);
    q2_pix = NULL;
  }
  if (q2_pix_mask) {
    gdk_bitmap_unref (q2_pix_mask);
    q2_pix_mask = NULL;
  }
}

#endif


void print_status (char *fmt, ...) {
  char buf[1024];
  va_list ap;

  va_start (ap, fmt);
  vsnprintf (buf, 1024, fmt, ap);
  va_end (ap);

  gtk_label_set (GTK_LABEL (status_bar_label), buf);
}


int window_delete_event_callback (GtkWidget *widget, gpointer data) {
  target_window = widget;
  gtk_widget_destroy ((GtkWidget *) (xqf_windows->data));
  return FALSE;
}


void register_window (GtkWidget *window) {
  xqf_windows = g_slist_prepend (xqf_windows, window);
}


void unregister_window (GtkWidget *window) {
  GSList *first;

  first = xqf_windows;
  xqf_windows = g_slist_next (xqf_windows);
  g_slist_free_1 (first);

  if (target_window && target_window != window)
    gtk_widget_destroy ((GtkWidget *) (xqf_windows->data));
  else
    target_window = NULL;
}


static void server_menu_set_sensitive (void) {
  if (cur_server) {
    gtk_widget_set_sensitive (GTK_WIDGET (connect_menu_item), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (observe_menu_item), 
                                (cur_server->type == QW_SERVER)? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (add_to_favs_menu_item), 
                                      (cur_master != favorites)? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (delete_menu_item),
                                  (cur_master->address == NULL)? TRUE : FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (refrsel_menu_item), TRUE);
  }
  else {
    gtk_widget_set_sensitive (GTK_WIDGET (connect_menu_item), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (observe_menu_item), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (add_to_favs_menu_item), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (delete_menu_item),  FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (refrsel_menu_item), FALSE);
  }

  if (cur_master->servers) {
    gtk_widget_set_sensitive (GTK_WIDGET (find_player_menu_item), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (refresh_menu_item), TRUE);
    gtk_widget_set_sensitive (GTK_WIDGET (refresh_button), TRUE);
  }
  else {
    gtk_widget_set_sensitive (GTK_WIDGET (find_player_menu_item), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (refresh_menu_item), FALSE);
    gtk_widget_set_sensitive (GTK_WIDGET (refresh_button), FALSE);
  }

  if (cur_master->address || cur_master == allsources)
    gtk_widget_set_sensitive (GTK_WIDGET (update_button), TRUE);
  else
    gtk_widget_set_sensitive (GTK_WIDGET (update_button), FALSE);

  if (cur_master == unbound_qw || cur_master == quake2)
    gtk_widget_set_sensitive (GTK_WIDGET (add_menu_item), TRUE);
  else
    gtk_widget_set_sensitive (GTK_WIDGET (add_menu_item), FALSE);
}


static void populate_server_clist (struct server **servers, int num) {
  char *row[7];
  char buf1[64], buf2[32], buf3[32], buf4[32];
  int i;

  if (num == 0) {
    gtk_clist_clear (GTK_CLIST (server_clist));
    return;
  }

  qsort (servers, num, sizeof (gpointer), qsort_servers);

  gtk_clist_freeze (GTK_CLIST (server_clist));
  gtk_clist_clear (GTK_CLIST (server_clist));

  for (i = 0; i < num; i++) {
#ifdef USEPIXS
    row[0] = ((*servers)->name)? (*servers)->name : "";
#else
    char bufx[128];

    sprintf (bufx, "[%s] %s", ((*servers)->type == QW_SERVER)? "QW" : "Q2",
	     ((*servers)->name)? (*servers)->name : "");
    row[0] = bufx;
#endif

    sprintf (buf1, "%s:%d", (*servers)->address, (*servers)->port);
    row[1] = buf1;

    if ((*servers)->ping >= 0) {
      sprintf (buf2, "%d", ((*servers)->ping > MAX_PING)? 
                                                  MAX_PING : (*servers)->ping);
      row[2] = buf2;
    }
    else {
      row[2] = "n/a";
    }

    if ((*servers)->retries >= 0) {
      if ((*servers)->ping == MAX_PING + 1) {
	row[3] = "D";	/* DOWN */
      }
      else if ((*servers)->ping == MAX_PING) {
	row[3] = "T";	/* TIMEOUT */
      }
      else {
	sprintf (buf3, "%d", (*servers)->retries);
	row[3] = buf3;
      }
    }
    else {
      row[3] = "n/a";
    }

    sprintf (buf4, "%d/%d", (*servers)->curplayers, (*servers)->maxplayers);
    row[4] = buf4;

    row[5] = ((*servers)->map)? (*servers)->map : "";
    row[6] = ((*servers)->game)? (*servers)->game : "";

    gtk_clist_append (GTK_CLIST (server_clist), row);

#ifdef USEPIXS
    switch ((*servers)->type) {
    case Q2_SERVER:
      if (q2_pix == NULL) {
	q2_pix = gdk_pixmap_create_from_xpm_d (
                          GTK_CLIST (server_clist)->clist_window, &q2_pix_mask,
                          &main_window->style->white, q2_xpm);
      }
      gtk_clist_set_pixtext (GTK_CLIST (server_clist), i, 0, row[0], 2,
                                                          q2_pix, q2_pix_mask);
      break;

    case QW_SERVER:
    default:
      if (q_pix == NULL) {
	q_pix = gdk_pixmap_create_from_xpm_d (
                          GTK_CLIST (server_clist)->clist_window, &q_pix_mask, 
                          &main_window->style->white, q_xpm);
      }
      gtk_clist_set_pixtext (GTK_CLIST (server_clist), i, 0, row[0], 2,
                                                            q_pix, q_pix_mask);
      break;
    }
#endif

    servers++;
  }

  gtk_clist_thaw (GTK_CLIST (server_clist));
}


static void populate_player_clist (struct server *s) {
  GdkBitmap *bm = NULL;
  GdkPixmap *pm = NULL;
  GSList *pm_cache = NULL;
  char *row[6];
  char buf1[32], buf2[32], buf3[32], buf4[32];
  int i;
  struct player *p;

  if (s->type == QW_SERVER) {
    allocate_quake_player_colors (main_window->window);
    bm = create_dummy_mask (main_window);
  }

  if (s->players == NULL) {
    gtk_clist_clear (GTK_CLIST (player_clist));
    return;
  }

  gtk_clist_freeze (GTK_CLIST (player_clist));
  gtk_clist_clear (GTK_CLIST (player_clist));

  qsort (s->players, s->curplayers, sizeof (gpointer), qsort_players);

  for (i = 0; i < s->curplayers; i++) {
    p = s->players[i];

    row[0] = (p->name)? p->name : "";

    sprintf (buf1, "%d", p->frags);
    row[1] = buf1;

    sprintf (buf3, "%d", p->ping);
    row[4] = buf3;

    if (s->type == QW_SERVER) {
      sprintf (buf2, "%d:%d", p->shirt, p->pants);
      row[2] = buf2;

      row[3] = (p->skin)? p->skin : "";

      sprintf (buf4, "%02d:%02d", p->time / 60 / 60, p->time / 60 % 60);
      row[5] = buf4;
    }
    else {
      row[2] = "";
      row[3] = "";
      row[5] = "";
    }

    gtk_clist_append (GTK_CLIST (player_clist), row);

    if (s->type == QW_SERVER) {
      pm = create_pm (main_window, p->shirt, p->pants, &pm_cache);
      gtk_clist_set_pixtext (GTK_CLIST (player_clist), i, 2, row[2], 2, pm, bm);
    }

    if (cur_filter == &filters[3] && test_player (p)) {
      gtk_clist_set_background (GTK_CLIST (player_clist), i,
                    &GTK_WIDGET (player_clist)->style->bg[GTK_STATE_SELECTED]);
      gtk_clist_set_foreground (GTK_CLIST (player_clist), i,
                    &GTK_WIDGET (player_clist)->style->fg[GTK_STATE_SELECTED]);
    }
  }

  gtk_clist_thaw (GTK_CLIST (player_clist));

  if (pm_cache)
    clear_pm_cache (pm_cache);
  if (bm)
    gdk_bitmap_unref (bm);
}


static void populate_serverinfo_clist (struct server *s) {
  char *row[2];
  char **info = s->info;

  if (info == NULL) {
    gtk_clist_clear (GTK_CLIST (serverinfo_clist));
    return;
  }

  gtk_clist_freeze (GTK_CLIST (serverinfo_clist));
  gtk_clist_clear (GTK_CLIST (serverinfo_clist));

  while (*info) {
    row[0] = (*info)? *info : "";
    info++;

    row[1] = (*info)? *info : "";
    info++;

    gtk_clist_append (GTK_CLIST (serverinfo_clist), row);
  }

  gtk_clist_thaw (GTK_CLIST (serverinfo_clist));
}


static void set_filter_callback (GtkWidget *widget, struct filter *filter) {

  /* it's called "by hand" if widget == NULL */

  if (filter == cur_filter && widget != NULL)
    return;

  if (cur_filter->widget) {
    gtk_arrow_set (GTK_ARROW (cur_filter->widget), GTK_ARROW_RIGHT, 
		                                               GTK_SHADOW_OUT);
  }

  if (filter->widget) {
    gtk_arrow_set (GTK_ARROW (filter->widget), GTK_ARROW_RIGHT, GTK_SHADOW_IN);
  }

  cur_filter = filter;

  if (cur_master == NULL)
    return;

  if (filter->num < 0)
    build_filtered_list (filter, cur_master);

  if (filter->num > 0) {
    populate_server_clist (filter->servers, filter->num);
  }
  else {
    gtk_clist_clear (GTK_CLIST (server_clist));
    if (cur_server) {
      gtk_clist_clear (GTK_CLIST (player_clist));
      gtk_clist_clear (GTK_CLIST (serverinfo_clist));
      cur_server = NULL;
    }
    server_menu_set_sensitive ();
  }
}


static void find_player_callback (GtkWidget *widget, gpointer data) {
  switch (psearch_dialog ()) {
    
  case PSEARCH_OK:
    free_filter_data (&filters[3]);

  case PSEARCH_USELAST:
    set_filter_callback (NULL, &filters[3]);
    break;

  case PSEARCH_CANCEL:
  default:
    break;

  }
}


static void start_preferences_dialog (GtkWidget *widget, int page_num) {
  preferences_dialog (page_num);
  rc_save ();

  if (filter_prefs_changed) {
    free_filters_data ();
    set_filter_callback (NULL, cur_filter);
  }
}


static void stat_one_server (struct server *s) {
  GSList *list = NULL;
  int i;

  if (!s)
    return;

  list = g_slist_prepend (list, s);
  stat_server_list (list);
  g_slist_free (list);

  /* brute force update -- that should be rewritten */

  free_filters_data ();
  set_filter_callback (NULL, cur_filter);

  for (i = 0; i < cur_filter->num; i++) {
    if (cur_filter->servers[i] == s) {
      gtk_clist_select_row (GTK_CLIST (server_clist), i, 0);
      break;
    }
  }

  if (s->ping >= MAX_PING) {
    print_status ("%s:%d is %s", s->address, s->port,
                            (s->ping == MAX_PING + 1)? "down" : "unreachable");
  }
  else {
    print_status ("%s:%d is Ok", s->address, s->port);
  }
}


static void server_menu_launch_callback (GtkWidget *widget, int observe) {
  struct server *s = cur_server;

  if (!s || (s->type != QW_SERVER && s->type != Q2_SERVER))
      return;

  /*TODO: resolve host name if needed */

  stat_one_server (s);

  if (s->ping >= MAX_PING) {
    dialog_ok (NULL, "Server %s:%d is unreachable", s->address, s->port);
    return;
  }

  if (!quake_config_is_valid (s->type)) {
    start_preferences_dialog (NULL, PREF_PAGE_OPTIONS);
    return;
  }

  if (observe && s->type == Q2_SERVER) {
    dialog_ok (NULL, "Quake2 doesn't support spectators");
    return;
  }

  if (s->curplayers == s->maxplayers && !observe) {
    dialog_ok (NULL, "Server %s:%d is full", s->address, s->port);
    return;
  }

  con = condef_new (s);

  if (observe) {
    if ((s->flags & SERVER_SP_PASSWORD) == 0) {
      con->observe = g_strdup ("1");
    }
    else {
      con->observe = enter_string_dialog ("Spectator Password:");
      if (!con->observe)
	return;
    }
  }
  else {
    if (s->flags & SERVER_PASSWORD) {
      con->password = enter_string_dialog ("Server Password:");
      if (!con->password)
	return;
    }
  }

  con->server = g_malloc (strlen (s->address) + 1 + 5 + 1);
  sprintf (con->server, "%s:%5d", s->address, s->port);

  con->gamedir = g_strdup (s->game);

  if (default_terminate) {
    con->s->ref_count++;
    gtk_widget_destroy (main_window);
  }
  else {
    launch_quake_fork (con);
    condef_free (con);
    con = NULL;
  }
}


static void server_selected_callback (GtkWidget *widget, int row, int column, 
                                                       GdkEventButton *event) {
  struct server *s = cur_filter->servers[row];
  GdkEventButton *bevent = (GdkEventButton *) event; 

  cur_server = s;
  populate_player_clist (s);
  populate_serverinfo_clist (s);
  server_menu_set_sensitive ();

  if (event) {
    if (event->type == GDK_BUTTON_PRESS) {
      if (bevent->button == 2) {
	stat_one_server (cur_server);
      }
    }

    if (event->type == GDK_2BUTTON_PRESS) {
      if (bevent->button == 1) {
	server_menu_launch_callback (NULL, 0);
      }
    }
  }
}


static void server_clist_column_set_title (int col, int append) {
  char tmp[64];
  char *ptr;
  int sortmode = server_clist_columns[col].sortmode;

  if (append) {
    sprintf (tmp, "%s %c", server_clist_columns[col].name, 
                                                    (sortmode > 0)? '<' : '>');
    ptr = tmp;
  }
  else {
    ptr = server_clist_columns[col].name;
  }
  gtk_label_set (GTK_LABEL (server_clist_columns[col].widget), ptr);
}


static void server_clist_click_column_callback (GtkWidget *widget, int col) {
  int sortmode = server_clist_columns[col].sortmode;

  server_clist_column_set_title (ssort_column, FALSE);

  if (ssort_column == col)
    server_clist_columns[col].sortmode = -sortmode;
  else {
    ssort_column = col;
  }

  server_clist_column_set_title (ssort_column, TRUE);

  if (cur_filter->servers && sortmode != SORT_SERVER_NONE) {
    populate_server_clist (cur_filter->servers, cur_filter->num);
  }
}


static void player_clist_column_set_title (int col, int append) {
  char tmp[64];
  char *ptr;
  int sortmode = player_clist_columns[col].sortmode;

  if (append) {
    sprintf (tmp, "%s %c", player_clist_columns[col].name, 
	                                            (sortmode > 0)? '<' : '>');
    ptr = tmp;
  }
  else {
    ptr = player_clist_columns[col].name;
  }
  gtk_label_set (GTK_LABEL (player_clist_columns[col].widget), ptr);
}


static void player_clist_click_column_callback (GtkWidget *widget, int col) {
  int sortmode = player_clist_columns[col].sortmode;

  player_clist_column_set_title (psort_column, FALSE);

  if (psort_column == col)
    player_clist_columns[col].sortmode = -sortmode;
  else 
    psort_column = col;

  player_clist_column_set_title (psort_column, TRUE);

  if (cur_server && sortmode != SORT_PLAYER_NONE) {
    populate_player_clist (cur_server);
  }
}


static void select_source_callback (GtkWidget *widget, struct master *master) {
  if (master->address)
    print_status ("Source: %s [%s]", master->name, master->address);
  else
    print_status ("Source: %s", master->name);

  if (cur_master != master) {
    cur_master = master;
    if (master == allsources) {
      free_servers (cur_master->servers);
      cur_master->servers = all_servers ();
    }
    free_filters_data ();
    set_filter_callback (NULL, cur_filter);
  }
}


static void update_source (struct master *m) {
  if (!m || m == allsources)
    return;

  if (m->address) {
    if (!stat_master (m)) {
      print_status ("Master is not responding");
      return;
    }
  }
  else {
    stat_server_list (m->servers);
  }
}


static void update_master_callback (GtkWidget *widget, gpointer data) {
  if (!cur_master)
    return;

  if (cur_master != allsources) {
    update_source (cur_master);
  }
  else {
    g_slist_foreach (qw_masters, (GFunc) update_source, NULL);
    update_source (unbound_qw);
    update_source (quake2);
    update_source (favorites);

    free_servers (cur_master->servers);
    cur_master->servers = all_servers ();
  }

  free_filters_data ();
  set_filter_callback (NULL, cur_filter);

  if (cur_master->servers)
    print_status ("%d servers", g_slist_length (cur_master->servers));

#ifdef DEBUG
    check_global_list ();
#endif
}


static void refresh_selected_callback (GtkWidget *widget, gpointer data) {
  /*
   *  Only one server can be selected because of GTK_SELECTION_BROWSE
   */

  stat_one_server (cur_server);
}


static void refresh_callback (GtkWidget *widget, gpointer data) {
  GSList *list = NULL;
  int n;

  if (!cur_master)
    return;

  if (cur_master == allsources) {
    free_servers (cur_master->servers);
    cur_master->servers = all_servers ();
  }

  stat_server_list (cur_master->servers);
  free_filters_data ();
  set_filter_callback (NULL, cur_filter);
}


static void add_to_favorites_callback (GtkWidget *widget, gpointer data) {
  if (cur_server) {
    add_server_to_master (favorites, cur_server);
    save_server_list (favorites);
  }
}


static void add_server_callback (GtkWidget *widget, gpointer data) {
  struct server *s;
  char *str, *addr;
  int port;
  struct in_addr tmp;
  enum server_type type;

  if (cur_master == unbound_qw)
    type = QW_SERVER;
  else if (cur_master == quake2)
    type = Q2_SERVER;
  else
    return;

  str = enter_string_dialog ("Server:");
  if (!str)
    return;

  if (!parse_address (str, &addr, &port) || !inet_aton (addr, &tmp)) {
    dialog_ok (NULL, "\"%s\" is not valid IP:PORT combination", str);
    g_free (str);
    return;
  }
  g_free (str);
  
  if (port == -1)
    port = (type == QW_SERVER)? QW_DEFAULT_PORT : Q2_DEFAULT_PORT;

  s = server_lookup (addr, port);
  if (!s)
    s = server_new (addr, port, type);

  add_server_to_master (cur_master, s);
  stat_one_server (s);
  save_server_list (cur_master);

  g_free (addr);
}


static void del_server_callback (GtkWidget *widget, gpointer data) {
  if (cur_server && cur_master->address == NULL) {
    remove_server_from_master (cur_master, cur_server);
    save_server_list (cur_master);

    /* removing server from filters should be done in less destructive way

       gtk_clist_remove (GTK_CLIST (server_clist), row)
    */

    free_filters_data ();
    set_filter_callback (NULL, cur_filter);
  }
}


static int server_clist_event_callback (GtkWidget *widget, GdkEvent *event) {
  GdkEventButton *bevent = (GdkEventButton *) event; 

  if (event->type == GDK_BUTTON_PRESS) {
    if (bevent->button == 3) {
      gtk_menu_popup (GTK_MENU (server_menu), NULL, NULL, NULL, NULL,
		                                 bevent->button, bevent->time);
    }
  }

  return FALSE;
}


static GtkWidget *create_server_menu (void) {
  GtkWidget *menu;
  GtkWidget *menu_item;

  menu = gtk_menu_new();
  
  connect_menu_item = gtk_menu_item_new_with_label ("Connect");
  gtk_menu_append (GTK_MENU (menu), connect_menu_item);
  gtk_signal_connect (GTK_OBJECT (connect_menu_item), "activate",
             GTK_SIGNAL_FUNC (server_menu_launch_callback), (gpointer) FALSE);
  gtk_widget_show (connect_menu_item);

  observe_menu_item = gtk_menu_item_new_with_label ("Observe");
  gtk_menu_append (GTK_MENU (menu), observe_menu_item);
  gtk_signal_connect (GTK_OBJECT (observe_menu_item), "activate",
              GTK_SIGNAL_FUNC (server_menu_launch_callback), (gpointer) TRUE);
  gtk_widget_show (observe_menu_item);

  menu_item = gtk_menu_item_new ();
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  add_to_favs_menu_item = gtk_menu_item_new_with_label ("Add to Favorites");
  gtk_menu_append (GTK_MENU (menu), add_to_favs_menu_item);
  gtk_signal_connect (GTK_OBJECT (add_to_favs_menu_item), "activate",
		      GTK_SIGNAL_FUNC (add_to_favorites_callback), NULL);
  gtk_widget_show (add_to_favs_menu_item);

  add_menu_item = gtk_menu_item_new_with_label ("Add...");
  gtk_menu_append (GTK_MENU (menu), add_menu_item);
  gtk_signal_connect (GTK_OBJECT (add_menu_item), "activate",
		      GTK_SIGNAL_FUNC (add_server_callback), NULL);
  gtk_widget_show (add_menu_item);

  delete_menu_item = gtk_menu_item_new_with_label ("Delete");
  gtk_menu_append (GTK_MENU (menu), delete_menu_item);
  gtk_signal_connect (GTK_OBJECT (delete_menu_item), "activate",
		      GTK_SIGNAL_FUNC (del_server_callback), NULL);
  gtk_widget_show (delete_menu_item);

  menu_item = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_widget_show (menu_item);

  refresh_menu_item = gtk_menu_item_new_with_label ("Refresh");
  gtk_menu_append (GTK_MENU (menu), refresh_menu_item);
  gtk_signal_connect (GTK_OBJECT (refresh_menu_item), "activate",
		      GTK_SIGNAL_FUNC (refresh_callback), NULL);
  gtk_widget_show (refresh_menu_item);

  refrsel_menu_item = gtk_menu_item_new_with_label ("Refresh Selected");
  gtk_menu_append (GTK_MENU (menu), refrsel_menu_item);
  gtk_signal_connect (GTK_OBJECT (refrsel_menu_item), "activate",
		      GTK_SIGNAL_FUNC (refresh_selected_callback), NULL);
  gtk_widget_show (refrsel_menu_item);

  menu_item = gtk_menu_item_new ();
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_widget_show (menu_item);

  find_player_menu_item = gtk_menu_item_new_with_label ("Find Player...");
  gtk_menu_append (GTK_MENU (menu), find_player_menu_item);
  gtk_signal_connect (GTK_OBJECT (find_player_menu_item), "activate",
		      GTK_SIGNAL_FUNC (find_player_callback), NULL);
  gtk_widget_show (find_player_menu_item);

  return menu;
}


static GtkWidget *create_source_menu_item (struct master *m) {
  GtkWidget *menu_item;

  menu_item = gtk_menu_item_new_with_label (m->name);
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
                       GTK_SIGNAL_FUNC (select_source_callback), (gpointer) m);
  gtk_widget_show (menu_item);

  return menu_item;
}


static GtkWidget *create_source_menu (void) {
  GtkWidget *menu;
  GtkWidget *menu_item;
  GSList *list;
  struct master *m;

  menu = gtk_menu_new ();

  menu_item = create_source_menu_item (favorites);
  gtk_menu_append (GTK_MENU (menu), menu_item);

  menu_item = gtk_menu_item_new ();		/* separator */
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  for (list = qw_masters; list; list = list->next) {
    menu_item = create_source_menu_item ((struct master *) list->data);
    gtk_menu_append (GTK_MENU (menu), menu_item);
  }

  menu_item = create_source_menu_item (unbound_qw);
  gtk_menu_append (GTK_MENU (menu), menu_item);

  menu_item = gtk_menu_item_new ();		/* separator */
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = create_source_menu_item (quake2);
  gtk_menu_append (GTK_MENU (menu), menu_item);

  menu_item = gtk_menu_item_new ();		/* separator */
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = create_source_menu_item (allsources);
  gtk_menu_append (GTK_MENU (menu), menu_item);

  return menu;
}


static GtkWidget *create_clist_coldef (int num, struct clist_coldef *cols) {
  GtkWidget *clist;
  GtkWidget *alignment;
  GtkWidget *label;
  int i;

  clist = gtk_clist_new (num);
  GTK_CLIST_SET_FLAGS (clist, CLIST_SHOW_TITLES);

  for (i = 0; i < num; i++) {
    gtk_clist_set_column_width (GTK_CLIST (clist), i, cols[i].width);
    if (cols[i].justify != GTK_JUSTIFY_LEFT) {
      gtk_clist_set_column_justification (GTK_CLIST (clist), i, cols[i].justify);
    }

    alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);

    label = gtk_label_new (cols[i].name);
    gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
    gtk_container_add (GTK_CONTAINER (alignment), label);
    gtk_widget_show (label);

    cols[i].widget = label;

    gtk_clist_set_column_widget (GTK_CLIST (clist), i, alignment);
    gtk_widget_show (alignment);
  }

  return clist;
}


static void about_dialog (GtkWidget *widget, gpointer data) {
  dialog_ok ("About XQF", 
	     "X11 QuakeWorld/Quake2 FrontEnd v0.5\n"
	     "Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>\n\n"
	     "http://www.botik.ru/~roma/quake/");
}


static GtkWidget *create_main_window_menu_bar (void) {
  GtkWidget *menu;
  GtkWidget *root_menu;
  GtkWidget *menu_item;
  GtkWidget *menu_bar;

  menu_bar = gtk_menu_bar_new ();

  /*
   *  File
   */

  root_menu = gtk_menu_item_new_with_label ("File");

  menu = gtk_menu_new();

  menu_item = gtk_menu_item_new_with_label ("About...");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
 		      GTK_SIGNAL_FUNC (about_dialog), (gpointer) NULL);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new ();
  gtk_widget_set_sensitive (GTK_WIDGET (menu_item), FALSE);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Exit");
  gtk_signal_connect_object (GTK_OBJECT (menu_item), "activate",
	       GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (main_window));
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  /*
   *  Preferences
   */

  root_menu = gtk_menu_item_new_with_label ("Preferences");

  menu = gtk_menu_new();

  menu_item = gtk_menu_item_new_with_label ("Player Profile...");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
      GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_PLAYER);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Gameplay...");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
    GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_GAMEPLAY);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Filtering...");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
      GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_FILTER);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new_with_label ("Options...");
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
     GTK_SIGNAL_FUNC (start_preferences_dialog), (gpointer) PREF_PAGE_OPTIONS);
  gtk_menu_append (GTK_MENU (menu), menu_item);
  gtk_widget_show (menu_item);

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (root_menu), menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menu_bar), root_menu);
  gtk_widget_show (root_menu);

  return menu_bar;
}


void create_main_window (void) {
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *hpaned;
  GtkWidget *vpaned;
  GtkWidget *button;
  GtkWidget *option_menu;
  GtkWidget *menu_bar;
  GtkWidget *label;
  GtkWidget *hbox2;
  GtkWidget *vseparator;
  GtkWidget *event_box;
  GtkWidget *alignment;
  GtkWidget *arrow;
  GtkWidget *hbox3;
  GtkTooltips *tooltips;
  struct filter *filter;

  server_menu = create_server_menu ();

  main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_signal_connect (GTK_OBJECT (main_window), "delete_event",
		      GTK_SIGNAL_FUNC (window_delete_event_callback), NULL);
  gtk_signal_connect (GTK_OBJECT (main_window), "destroy",
		      GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_window_set_title (GTK_WINDOW (main_window), "XQF");

  register_window (main_window);

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (main_window), main_vbox);

  menu_bar = create_main_window_menu_bar ();
  gtk_box_pack_start (GTK_BOX (main_vbox), menu_bar, FALSE, FALSE, 0);
  gtk_widget_show (menu_bar);

  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_border_width (GTK_CONTAINER (vbox), 4);
  gtk_box_pack_start (GTK_BOX (main_vbox), vbox, TRUE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 16);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

  /*
   *  Source
   */

  hbox2 = gtk_hbox_new (FALSE, 2);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);

  label = gtk_label_new ("Source");
  gtk_box_pack_start (GTK_BOX (hbox2), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  option_menu = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (option_menu), 
                                                        create_source_menu ());
  gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), 0);
  gtk_box_pack_start (GTK_BOX (hbox2), option_menu, FALSE, FALSE, 2);
  gtk_widget_show (option_menu);   

  tooltips = gtk_tooltips_new ();

  update_button = gtk_button_new_with_label (" U ");
  gtk_box_pack_start (GTK_BOX (hbox2), update_button, FALSE, FALSE, 2);
  gtk_signal_connect (GTK_OBJECT (update_button), "clicked",
		      GTK_SIGNAL_FUNC (update_master_callback), NULL);
  gtk_widget_show (update_button);

  gtk_tooltips_set_tips (tooltips, update_button, "Update server list from master");

  refresh_button = gtk_button_new_with_label (" R ");
  gtk_box_pack_start (GTK_BOX (hbox2), refresh_button, FALSE, FALSE, 2);
  gtk_signal_connect (GTK_OBJECT (refresh_button), "clicked",
		      GTK_SIGNAL_FUNC (refresh_callback), NULL);
  gtk_widget_show (refresh_button);

  gtk_tooltips_set_tips (tooltips, refresh_button, "Refresh current list");

  gtk_widget_show (hbox2);

  /*
   * Separator 
   */

  hbox2 = gtk_hbox_new (FALSE, 1);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);

  vseparator = gtk_vseparator_new (); 
  gtk_box_pack_start (GTK_BOX (hbox2), vseparator, FALSE, FALSE, 0);
  gtk_widget_show (vseparator);

  vseparator = gtk_vseparator_new (); 
  gtk_box_pack_start (GTK_BOX (hbox2), vseparator, FALSE, FALSE, 0);
  gtk_widget_show (vseparator);

  gtk_widget_show (hbox2);

  /*
   *  Filter buttons
   */

  hbox2 = gtk_hbox_new (FALSE, 4);
  gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);

  for (filter = filters; filter->name; filter++) {
    button = gtk_button_new ();
    gtk_box_pack_start (GTK_BOX (hbox2), button, FALSE, FALSE, 2);

    hbox3 = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (button), hbox3);

    arrow = gtk_arrow_new (GTK_ARROW_RIGHT, 
		       (filter == cur_filter)? GTK_SHADOW_IN : GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (hbox3), arrow, FALSE, FALSE, 2);
    gtk_widget_show (arrow);

    filter->widget = arrow;

    label = gtk_label_new (filter->name);
    gtk_box_pack_start (GTK_BOX (hbox3), label, FALSE, FALSE, 2);
    gtk_widget_show (label);

    gtk_widget_show (hbox3);

    if (filter != &filters[3]) {
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
                    GTK_SIGNAL_FUNC (set_filter_callback), (gpointer) filter);
    }
    else {
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (find_player_callback), NULL);
    }
    gtk_widget_show (button);
  }

  gtk_widget_show (hbox2);

  gtk_widget_show (hbox);

  vpaned = gtk_vpaned_new ();
  gtk_box_pack_start (GTK_BOX (vbox), vpaned, TRUE, TRUE, 0);

  /*
   *  Server CList
   */

  server_clist = create_clist_coldef (7, server_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (server_clist), GTK_SELECTION_BROWSE);
  gtk_widget_set_usize (GTK_WIDGET (server_clist), 750, 300);
  gtk_clist_set_policy (GTK_CLIST (server_clist), 
			          GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (server_clist), "click_column",
                   GTK_SIGNAL_FUNC (server_clist_click_column_callback), NULL);
  gtk_signal_connect_after (GTK_OBJECT (server_clist), "select_row",
                             GTK_SIGNAL_FUNC (server_selected_callback), NULL);
  gtk_signal_connect (GTK_OBJECT(server_clist), "event",
                          GTK_SIGNAL_FUNC (server_clist_event_callback), NULL);
  gtk_paned_add1 (GTK_PANED (vpaned), server_clist);
  gtk_widget_show (server_clist);

  server_clist_column_set_title (ssort_column, TRUE);

  hpaned = gtk_hpaned_new ();
  gtk_paned_add2 (GTK_PANED (vpaned), hpaned);

  /*
   *  Player CList
   */

  player_clist = create_clist_coldef (6, player_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (player_clist), 
                                                       GTK_SELECTION_EXTENDED);
  gtk_widget_set_usize (GTK_WIDGET (player_clist), 480, 180);
  gtk_clist_set_policy (GTK_CLIST (player_clist),
			          GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_signal_connect (GTK_OBJECT (player_clist), "click_column",
                   GTK_SIGNAL_FUNC (player_clist_click_column_callback), NULL);
  gtk_paned_add1 (GTK_PANED (hpaned), player_clist);
  gtk_widget_show (player_clist);

  player_clist_column_set_title (psort_column, TRUE);

  /*
   *  Server Info CList
   */

  serverinfo_clist = create_clist_coldef (2, serverinfo_clist_columns);
  gtk_clist_set_selection_mode (GTK_CLIST (serverinfo_clist), 
				                       GTK_SELECTION_EXTENDED);
  gtk_widget_set_usize (GTK_WIDGET (serverinfo_clist), 270, 180);
  gtk_clist_set_policy (GTK_CLIST (serverinfo_clist),
			          GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_clist_column_titles_passive (GTK_CLIST (serverinfo_clist));
  gtk_paned_add2 (GTK_PANED (hpaned), serverinfo_clist);
  gtk_widget_show (serverinfo_clist);

  gtk_widget_show (hpaned);
  gtk_widget_show (vpaned);

  /*
   *  Status Bar
   */

  event_box = gtk_event_box_new ();
  gtk_widget_set_usize (event_box, 700, -1);
  gtk_box_pack_end (GTK_BOX (vbox), event_box, FALSE, FALSE, 0);

  alignment = gtk_alignment_new (0.0, 0.5, 0.0, 0.0);
  gtk_container_add (GTK_CONTAINER (event_box), alignment);

  status_bar_label = gtk_label_new ("Status Bar");
  gtk_label_set_justify (GTK_LABEL (status_bar_label), GTK_JUSTIFY_LEFT);
  gtk_container_add (GTK_CONTAINER (alignment), status_bar_label);
  gtk_widget_show (status_bar_label);

  gtk_widget_show (alignment);

  gtk_widget_show (event_box);

  gtk_widget_show (vbox);
  gtk_widget_show (main_vbox);

  gtk_widget_show (main_window);
}


static int init_user_info (void) {
  user.name = get_user_name ();

  user.home = get_user_home ();
  if (user.home == NULL) {
    fprintf (stderr, "You are homeless!\n");
    return 0;
  }

  user.rcdir  = file_in_dir (user.home, RC_DIR);
  return 1;
}


static void free_user_info (void) {
  if (user.name)    { g_free (user.name); user.name = NULL; }
  if (user.home)    { g_free (user.home); user.home = NULL; }
  if (user.rcdir)   { g_free (user.rcdir); user.rcdir = NULL; }
}


int main (int argc, char *argv[]) {
  gtk_init (&argc, &argv);
  gtk_preview_set_gamma (1.2);
  gtk_rc_parse ("/etc/gtkrc");

  set_sigchild ();

  if (!init_user_info ())
    return 1;

  rc_check_dir ();
  rc_parse ();
  user_fix_defaults ();

  init_masters ();
  init_filters ();

  create_main_window ();
  if (qw_masters) {
    select_source_callback (NULL, favorites);
  }

  gtk_main ();
  unregister_window (main_window);

  if (server_menu) {
    gtk_widget_destroy (server_menu);
    server_menu = NULL;
  }

#ifdef USEPIXS
  free_pixmaps ();
#endif

  free_filters_data ();
  free_masters ();
  free_user_info ();

#ifdef DEBUG
  check_global_list ();
#endif

  if (con && con->server) {
    launch_quake (con);
    server_unref (con->s);
    condef_free (con);
    con = NULL;
  }

  return 0;
}

