// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  PTEColored.C

  Stephane Rehel

  October 25 1997
*/

#include <stdlib.h>

#include "tools/file/MLWFileBlock.h"
#include "tools/file/MLRFileBlock.h"

#include "graphics/mlgl.h"
#include "graphics/MLEvent.h"
#include "graphics/SystemWindow.h"
#include "graphics/MLPicture.h"

#include "interface/MLXORRectAction.h"
#include "interface/MLMotionAction.h"
#include "interface/MLClickedMotionAction.h"
#include "interface/MLMode.h"
#include "interface/MLCanvasPopup.h"
#include "interface/MLPopup.h"
#include "interface/MLCanvasButton.h"
#include "interface/MLCanvasActionsPool.h"

#include "PTEColored.h"

#include "photo/PhotoModule.h"
#include "photo/PhotoModes.h"

#include "photo/modes/PhotoMSelect.h"
#include "photo/modes/PhotoMDrag.h"
#include "photo/modes/PhotoMPan.h"
#include "photo/modes/PhotoMZoom.h"

#include "photo/photo/Photo.h"

/////////////////////////////////////////////////////////////////////////////

PTEColored::PTEColored( PhotoCanvas* _canvas ):
  PhotoEngine(_canvas)
{
  x= y= 0;

  scale.red= 1.;
  scale.green= 1.;
  scale.blue= 1.;
  scale.alpha= 1.;
  bias.red= 0.;
  bias.green= 0.;
  bias.blue= 0.;
  bias.alpha= 0;
  pixelZoom= 1.;

  name= "Colored";

  displayTypeButton= new MLCanvasButton(canvas);
  displayTypeButton->create("Preview");
  PhotoEngine::actions->addRightAction(displayTypeButton);

  optionsPopup= new MLCanvasPopup(canvas);
  optionsPopup->create("Options");
  MLPopup* p= optionsPopup->getPopup();
  options_popup_wire_front= p->addCheck("Wire front");
  PhotoEngine::actions->addRightAction(optionsPopup);
}

/////////////////////////////////////////////////////////////////////////////

PTEColored::~PTEColored()
{
}

/////////////////////////////////////////////////////////////////////////////

// convert screen coordinates to image coordinates
IPoint PTEColored::screen_to_image( const IPoint& p )
{
  IPoint r= p;

  if( photoModule == 0 )
    return r;

  Photo* photo= photoModule->getPhoto();
  if( photo == 0 )
    return r;

  MLImage* image= photo->getImage();
  if( image == 0 )
    return r;

  if( image->getData() == 0 )
    return r;

  int width=  image->getWidth();
  int height= image->getHeight();

  SystemWindow* window= MLEngine::getGfxWindow();
  IVector windowSize= window->getSize();

  float x1= int(double((p.x()-windowSize.x()/2) - x)/pixelZoom)+width /2;
  float y1= int(double((p.y()-windowSize.y()/2) - y)/pixelZoom)+height/2;

  // Position of the image in window coordinates
//  float x1= windowSize.x()/2 + int(double(-width /2+X)*pixelZoom+x);
//  float y1= windowSize.y()/2 + int(double(-height/2+Y)*pixelZoom+y);
//  float x2= windowSize.x()/2 + int(double(-width /2+width -1)*pixelZoom+x);
//  float y2= windowSize.y()/2 + int(double(-height/2+height-1)*pixelZoom+y);

  return IPoint( int(x1), int(y1) );
}

/////////////////////////////////////////////////////////////////////////////

// protected
// return IFALSE if window is unmapped, etc.
void PTEColored::draw_photo( Photo* photo )
{
  if( photo == 0 )
    return;

  MLImage* image= photo->getImage();
  if( image == 0 )
    return;

  unsigned char* data= image->getData();
  int width=  image->getWidth();
  int height= image->getHeight();

  if( data == 0 )
    return;

  SystemWindow* window= MLEngine::getGfxWindow();
  IVector windowSize= window->getSize();

  // Position of the image in window coordinates
  float x1= windowSize.x()/2 + int(double(-width/2  )*pixelZoom+x);
  float y1= windowSize.y()/2 + int(double(-height/2 )*pixelZoom+y);
  float x2= windowSize.x()/2 + int(double(-width /2+width -1)*pixelZoom+x);
  float y2= windowSize.y()/2 + int(double(-height/2+height-1)*pixelZoom+y);
//  float x2= x1 + int(double(width  - 1)*pixelZoom);
//  float y2= y1 + int(double(height - 1)*pixelZoom);

  if( int(x1) >= windowSize.x() || int(x2) < 0 ||
      int(y1) >= windowSize.y() || int(y2) < 0 )
    return;

  int block_row_length= image->getWidth();

  int block_skip_pixels= 0;
  if( x1 < 0. )
    {
    block_skip_pixels= -int(x1/pixelZoom);
    x1= 0.;
    }
  if( x2 >= float(windowSize.x()) )
    x2= float(windowSize.x() - 1);
  int block_width= int( (x2 - x1 + 1)/pixelZoom );

  block_skip_pixels= max( 0, block_skip_pixels );
  block_skip_pixels= min( width-1, block_skip_pixels );
  block_width= min( 1, block_width );
  block_width= max( width - block_skip_pixels, block_width );

  int block_skip_rows= 0;
  if( y1 < 0. )
    {
    block_skip_rows= -int( y1/pixelZoom );
    y1= 0.;
    }
  if( y2 >= float(windowSize.y()) )
    y2= float(windowSize.y() - 1);
  int block_height= int( (y2 - y1 + 1)/pixelZoom );

  block_skip_rows= max( 0, block_skip_rows );
  block_skip_rows= min( height-1, block_skip_rows );
  block_height= min( 1, block_height );
  block_height= max( height - block_skip_rows, block_height );

  glRasterPos2f( x1, y1 );

  GLint swapbytes, lsbfirst, rowlength;
  GLint skiprows, skippixels, alignment;

  // Save the current packing mode for bitmaps.
  glGetIntegerv( GL_UNPACK_SWAP_BYTES, &swapbytes );
  glGetIntegerv( GL_UNPACK_LSB_FIRST, &lsbfirst );
  glGetIntegerv( GL_UNPACK_ROW_LENGTH, &rowlength );
  glGetIntegerv( GL_UNPACK_SKIP_ROWS, &skiprows );
  glGetIntegerv( GL_UNPACK_SKIP_PIXELS, &skippixels );
  glGetIntegerv( GL_UNPACK_ALIGNMENT, &alignment );

  // Enforce a standard packing mode
  glPixelStorei( GL_UNPACK_SWAP_BYTES, GL_FALSE );
  glPixelStorei( GL_UNPACK_LSB_FIRST, GL_FALSE );
  glPixelStorei( GL_UNPACK_ROW_LENGTH,  block_row_length );
  glPixelStorei( GL_UNPACK_SKIP_ROWS,   block_skip_rows );
  glPixelStorei( GL_UNPACK_SKIP_PIXELS, block_skip_pixels );

  glPushAttrib(GL_ENABLE_BIT);
  glPushAttrib(GL_PIXEL_MODE_BIT);
  glPixelZoom( pixelZoom, pixelZoom );

  glPixelTransferf( GL_RED_SCALE,   scale.red );
  glPixelTransferf( GL_GREEN_SCALE, scale.green );
  glPixelTransferf( GL_BLUE_SCALE,  scale.blue );
  glPixelTransferf( GL_ALPHA_SCALE, scale.alpha );
  glPixelTransferf( GL_RED_BIAS,    bias.red );
  glPixelTransferf( GL_GREEN_BIAS,  bias.green );
  glPixelTransferf( GL_BLUE_BIAS,   bias.blue );
  glPixelTransferf( GL_ALPHA_BIAS,  bias.alpha );

  if( image->hasAlpha() )
    {
    glEnable(GL_ALPHA_TEST);
    glAlphaFunc( GL_GEQUAL, 0.5 );
    glPixelStorei( GL_UNPACK_ALIGNMENT, 4 );
    glDrawPixels( block_width,
                  block_height,
                  GL_RGBA, GL_UNSIGNED_BYTE, (void*) data );
    }
   else
    {
    glDisable(GL_ALPHA_TEST);
    glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
    glDrawPixels( block_width,
                  block_height,
                  GL_RGB, GL_UNSIGNED_BYTE, (void*) data );
    }

  glPopAttrib();
  glPopAttrib();

  // Restore saved packing modes.
  glPixelStorei( GL_UNPACK_SWAP_BYTES, swapbytes );
  glPixelStorei( GL_UNPACK_LSB_FIRST, lsbfirst );
  glPixelStorei( GL_UNPACK_ROW_LENGTH, rowlength );
  glPixelStorei( GL_UNPACK_SKIP_ROWS, skiprows );
  glPixelStorei( GL_UNPACK_SKIP_PIXELS, skippixels );
  glPixelStorei( GL_UNPACK_ALIGNMENT, alignment );

  return;
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::draw()
{
  if( ! MLEngine::mapped() || photoModule->isIconic() )
    return;

  SystemWindow* window= MLEngine::getGfxWindow();
  if( window == 0 )
    return;

  window->currentPixel();

  mlBack();

  glClearColor( 0., 0., 0., 0. );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  Photo* the_photo= photoModule->getPhoto();
  if( the_photo != 0 )
    draw_photo(the_photo);

  window->swapBuffers();
}

/////////////////////////////////////////////////////////////////////////////

IBOOL PTEColored::handleAction( MLAction* action )
{
  if( action == displayTypeButton )
    {
    /* ... */

    updatePopupsFlags();

    return ITRUE;
    }

  if( action == optionsPopup )
    {
    int label= optionsPopup->getLabel();

    if( label == options_popup_wire_front )
      {
      /* ... */

      updatePopupsFlags();

      return ITRUE;
      }
    }

  if( ! MLEngine::validPointerAction(action) )
    return PhotoEngine::handleAction(action);

  switch( MLEngine::getMode()->getID() )
    {
    case PhotoModes::SELECT:
      {
      PhotoMSelect* s= (PhotoMSelect*) getMode();

      MLXORRectAction* a= (MLXORRectAction*) action;

      /* ... */
      s->button= a->button;

      photoModule->validateFunction();

      return ITRUE;
      }

    case PhotoModes::PAN:
      {
//      PhotoMDrag* d= (PhotoMDrag*) getMode();

      MLMotionAction* a= (MLMotionAction*) action;

      int new_x= x;
      int new_y= y;

      if( a->button == 2 )
        {
        if( a->type == MLMotionAction::START )
          {
          new_x= 0;
          new_y= 0;
          }
        }
       else
        {
        if( a->type != MLMotionAction::MOVING )
          return ITRUE;

        new_x+= a->delta_size.x();
        new_y+= a->delta_size.y();
        }

      if( (x-new_x)!=0 || (y-new_y)!=0 )
        {
        x= new_x;
        y= new_y;
        refreshScreen(PhotoModule::REFRESH_PHOTO);
        }

      return ITRUE;
      }

    case PhotoModes::ZOOM:
      {
//      PhotoMDrag* d= (PhotoMDrag*) getMode();

      MLMotionAction* a= (MLMotionAction*) action;

      double zoom= pixelZoom;

      if( a->button == 2 )
        {
        if( a->type == MLMotionAction::START )
          zoom= 1.;
        }
       else
        {
        if( a->type != MLMotionAction::MOVING )
          return ITRUE;

        const IVector& size= a->window->getSize();

        double dx= double(a->delta_size.x()) / double(size.x()) * 2.;
        double dy= double(a->delta_size.y()) / double(size.y()) * 2.;

        double speed= 2.;
        double dz= speed * sqrt(dx*dx+dy*dy);
        if( dy > 0. )
          dz= -dz;
        zoom += zoom * dz;

        zoom= max( zoom, 1e-4 );
        zoom= min( zoom, 1e+4 );
        }

      if( fabs(pixelZoom-zoom) > 1e-5 )
        {
        pixelZoom= zoom;

        refreshScreen(PhotoModule::REFRESH_PHOTO);
        }

      return ITRUE;
      }

    case PhotoModes::RECTZOOM:
      {
      MLXORRectAction* a= (MLXORRectAction*) action;

      if( a->button == 2 )
        {
        pixelZoom= 1.;
        refreshScreen(PhotoModule::REFRESH_PHOTO);
        return ITRUE;
        }

      IVector size( abs(a->p2.x() - a->p1.x()),
                    abs(a->p2.y() - a->p1.y()) );
      if( size.x() == 0 || size.y() == 0 )
        return ITRUE;

      IVector windowSize= MLEngine::getGfxWindow()->getSize();

      double zoom= double(windowSize.x()) / double(size.x());
      if( double(size.y()) * zoom > double(windowSize.y()) )
        zoom= double(windowSize.y()) / double(size.y());

      IPoint center= (a->p1 + a->p2) / 2;

      x= (center.x() - windowSize.x()/2);
      y= (center.y() - windowSize.y()/2);

      pixelZoom= zoom;

      refreshScreen(PhotoModule::REFRESH_PHOTO);

      return ITRUE;
      }

    default:
      break;
    }

  return PhotoEngine::handleAction(action);
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::refreshScreen( unsigned what )
{
  if( (what & PhotoModule::REFRESH_PHOTO) != 0 )
    MLEngine::postRefresh();
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::leaveMode( MLMode* mm )
{
  if( mm != 0 && photoModule != 0 )
    photoModule->postCmdStop(); // close the undo list for non-lonely commands

  PhotoEngine::leaveMode(mm);
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::write( MLWFileBlock& block )
{
  PhotoEngine::write(block);

//  block << double(r) << double(g) << double(b);

  block << int(0);
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::read( MLRFileBlock& block )
{
  PhotoEngine::read(block);

//  r= block.getDouble();
//  g= block.getDouble();
//  b= block.getDouble();

  if( block.getInteger() == 0 )
    goto end;

end:
  updatePopupsFlags();
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::resetAll()
{
  x= y= 0;

  PhotoEngine::resetAll();

  optionsPopup->setChecked(options_popup_wire_front,IFALSE);

  updatePopupsFlags();
}

/////////////////////////////////////////////////////////////////////////////

void PTEColored::updatePopupsFlags()
{
  if( optionsPopup == 0 )
    return;

  optionsPopup->setAvailable(options_popup_wire_front,IFALSE);

  displayTypeButton->setTitle(" Wire  ");
}

/////////////////////////////////////////////////////////////////////////////
