// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLImporter.C

  Stephane Rehel

  April 27 1997
*/

#include <stdio.h>

#include "tools/OString.h"
#include "tools/Command.h"
#include "tools/ListFiles.h"
#include "tools/Path.h"
#include "tools/Filename.h"

#include "MLImporter.h"

#include "scene/MeshRep.h"
#include "scene/MLPolygon.h"
#include "scene/MLVertex.h"
#include "scene/MLScene.h"
#include "scene/MLAbstractMesh.h"
#include "scene/MLAbstractMeshesIterator.h"

#include "scene/file/MLDatabase.h"
#include "scene/file/MLDatabasesList.h"

/////////////////////////////////////////////////////////////////////////////

MLImporter::MLImporter()
{
  compressed= IFALSE;
  verbose= IFALSE;
  reverseFaces= IFALSE;
  in_filename= "";
  filename_ext= "";
  scene_name= "";
  scene= 0;
  layers2objects= IFALSE;
  no_write= IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

MLImporter::~MLImporter()
{}

/////////////////////////////////////////////////////////////////////////////

void MLImporter::addPolygon( MeshRep* mesh,
                             int nVertices, const Point* points )
{
  int i1= mesh->vertices.getSize() + 1;

  MLPolygon& polygon= mesh->addPolygon();
  polygon.alloc(nVertices);
  polygon.flags |= MLPolygon::ALLOCATED | MLPolygon::BASE_POLYGON;
  for( int i= 0; i < nVertices; ++i )
    {
    MLVertex& vertex= mesh->vertices.append();
    vertex.flags |= MLVertex::ALLOCATED | MLVertex::BASE_VERTEX;
    vertex.point= points[i];

    if( reverseFaces )
      polygon.elements[i].vertex= i1 + nVertices - 1 - i;
     else
      polygon.elements[i].vertex= i1 + i;
    }

  mesh->nVertices += nVertices;
  mesh->nPolygons++;
}

/////////////////////////////////////////////////////////////////////////////

void MLImporter::finalize()
{
  scene->getRootObject()->updateTransformations();

  MLAbstractMeshesIterator aoi(scene);
  while( ! aoi.eol() )
    {
    MLAbstractMesh* ao= aoi++;

    MeshRep* mesh= ao->getMeshRep();
    if( mesh == 0 )
      continue;

    fprintf( stderr, "finalizing mesh '%s' (%d vertices, %d polygons)\n",
             ao->getName().name.get(),
             mesh->nVertices, mesh->nPolygons );

    if( verbose ) fprintf( stderr, "  Edges pass 1/2\n" );
    mesh->findEdges();
    if( verbose ) fprintf( stderr, "  Vertices pass\n" );
    mesh->mergeVertices();
    if( verbose ) fprintf( stderr, "  Edges pass 2/2\n" );
    mesh->findEdges();
    if( verbose ) fprintf( stderr, "  Global pass\n" );
    mesh->finalizeRep();

    if( verbose ) fprintf( stderr, "  Reindexing vertices 1/2\n" );
    mesh->reindexVertices();
    if( verbose ) fprintf( stderr, "  Reindexing edges\n" );
    mesh->reindexEdges();
    if( verbose ) fprintf( stderr, "  Reindexing vertices 2/2\n" );
    mesh->reindexSVertices();
    if( verbose ) fprintf( stderr, "  Reindexing polygons\n" );
    mesh->reindexPolygons();

    ao->generateMesh(); // update bbox

    fprintf( stderr, "finalized '%s': %d vertices, %d polygons\n",
             ao->getName().name.get(),
             mesh->nVertices, mesh->nPolygons );
    }
}

/////////////////////////////////////////////////////////////////////////////

void MLImporter::convertName( OString& name )
{
  for( int i= 0; i < int(name.length()); ++i )
    {
    if( name[i] == '.' || name[i] == ',' || name[i] == '/' )
      name.put(i,'-');
    }

  name= name.trim();
}

/////////////////////////////////////////////////////////////////////////////

// static
int MLImporter::parseOptions( Command& cmd )
{
  OString databasePath= cmd.getStringAfter("-db");
  if( databasePath.length() == 0 )
    {
    MLDatabasesList dblist;
    dblist.build();
    if( ! dblist.get().empty() )
      {
      fprintf( stderr, "Getting default database location from ~/.mlrc\n" );
      databasePath= dblist.get().get(1)->getBasePath();
      }
    }

  if( databasePath.length() == 0 )
    {
    fprintf( stderr, "Error: Can't find any database.\n" );
    return -1;
    }

  if( ! ListFiles::isAccessibleDirectory(databasePath) )
    {
    fprintf( stderr, "Error: Unable to access directory '%s'\n",
             databasePath.get() );
    return 1;
    }

  database= new MLDatabase(Path(databasePath));

  reverseFaces= IFALSE;
  if( cmd.findOption("-rv") )
    reverseFaces= ITRUE;

  verbose= IFALSE;
  if( cmd.findOption("-v") )
    verbose= ITRUE;

  layers2objects= IFALSE;
  if( cmd.findOption("-dxfl2o") )
    layers2objects= ITRUE;

  no_write= IFALSE;
  if( cmd.findOption("-nw") )
    no_write= ITRUE;

  scene_name= cmd.getStringAfter("-outname");

  const char* tmp_strg= cmd.getFirstUnusedOption();
  if( tmp_strg != 0 )
    {
    fprintf( stderr, "Error: Unknown option '%s'\n", tmp_strg );
    return -1;
    }

  in_filename= cmd.getFirst();
  if( in_filename.length() == 0 )
    {
    fprintf( stderr, "Error: no filename specified\n" );
    return -1;
    }

  Filename in_f= Filename(in_filename);
  filename_ext= in_f.getExtension();

  filename_ext.toLower();

  if( filename_ext == ".gz" || filename_ext == ".z" )
    {
    compressed= ITRUE;
    OString f= in_filename.getFirst( in_filename.length()
                                    -filename_ext.length());
    filename_ext= Filename(f).getExtension();
    }

  if( scene_name.length() == 0 )
    scene_name= in_f.getFilename();

  int i;
  for( i= 0; i < scene_name.length(); ++i )
    if( scene_name[i] == '.' ||
        scene_name[i] == ',' ||
        scene_name[i] == '/' )
      scene_name.put(i,'-');

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

FILE* MLImporter::open_file() const
{
  if( ! compressed )
    return fopen( in_filename.get(), "r" );

  return popen( (OString("gzip -cd ") + in_filename).get(), "r" );
}

/////////////////////////////////////////////////////////////////////////////

void MLImporter::close_file( FILE* file ) const
{
  if( file == 0 || file == stdin )
    return;

  if( compressed )
    pclose(file);
   else
    fclose(file);
}

/////////////////////////////////////////////////////////////////////////////
