// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  Command.h

  Unix-like command line: one command + some arguments

  Creation: SR, January 30th, 1996
  Revisions:

*/

#ifndef __Command_h
#define __Command_h

#ifndef __IList_h
#include "IList.h"
#endif

#ifndef __OString_h
#include "OString.h"
#endif

/////////////////////////////////////////////////////////////////////////////

class Command
{
protected:
  struct Argument
    {
    Argument( const OString& _arg )
      {
      arg= _arg;
      used= IFALSE;
      option= (arg.length()>0) ? (arg[0]=='-') : IFALSE;
      }
    Argument( const char* _arg )
      {
      arg= (_arg==0) ? "" : _arg;
      used= IFALSE;
      option= (arg.length()>0) ? (arg[0]=='-') : IFALSE;
      }
    OString arg;
    IBOOL used, option;
    };

  OString command;
  IList<Argument> arguments;
  IListIterator<Argument> li;
  int error;

public:
  Command( const char* fcl ): li(arguments)
    {
    init(fcl);
    }
  Command( const OString& fcl ): li(arguments)
    {
    init(fcl.get());
    }
  Command( int argc, const char* argv[] ): li(arguments)
    {
    init( argc, argv );
    }

  virtual ~Command()
    {
    destroy();
    }

  void destroy();

  // fullCommandLine may be 0
  void init( const char* fullCommandLine );
  void init( int argc, const char* argv[] );

  void addArgument( const OString& arg );
  void addArgument( const char* arg );

  void reset();

// parsing
  const OString& get() const
    {
    assert( command.assigned() );
    return command;
    }

  int getNArguments() const
    {
    return 1 + arguments.getNElements();
    }
  int argc() const
    {
    return getNArguments();
    }
  const OString& argv( int i ) const;

  IBOOL getError() const
    {
    return error != 0;
    }

  int getInteger();
  double getDouble();
  const OString& getString();
  int getIntegerAfter( const char* option );
  double getDoubleAfter( const char* option );
  const OString& getStringAfter( const char* option );

private:
  Argument* getFirstArgument();
  Argument* getNextArgument();

public:
  const OString& getFirst();
  const OString& getNext();

  IBOOL findOption( const OString& option );
  IBOOL findOption( const char* s );
  IBOOL findOption( char ch );

  //
  // Example: -I/foo/tools
  // -> then getOptionString( 'I' ) returns "/foo/tools"
  // return 0 if not found
  //
  const char* findOptionString( char ch );

  const char* getFirstUnusedOption();

private:
  const char* skipSpaces( const char* ) const;
};

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __Command_h

