// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  Path.c

  Stephane Rehel
  April 15 1997
*/

#include "Path.h"

#include "ListFiles.h"

/////////////////////////////////////////////////////////////////////////////

#define unixSeparator '/'
#define  dosSeparator '\\'

// static
const char Path::separator= unixSeparator;

static const char* currentPath= "./";

/////////////////////////////////////////////////////////////////////////////

Path::Path()
{
  path= currentPath;
}

/////////////////////////////////////////////////////////////////////////////

Path::Path( const Path& p )
{
  path= p.path;
}

/////////////////////////////////////////////////////////////////////////////

Path::Path( const OString& fullname )
{
  operator = ( fullname );
}

/////////////////////////////////////////////////////////////////////////////

Path& Path::operator = ( const OString& fullname )
{
  path= fullname.trim();
  path= Path::toUnixPathSeparators(path);

  if( path.length() == 0 )
    path= currentPath;

  if( path == currentPath )
    return *this;

  // remove multiple '/'
  int i;
  for( i= 0; i < int(path.length()-1); ++i )
    {
    if( path[i] == separator && path[i+1] == separator )
#ifdef __WINDOWS__
      if( i > 0 ) // allow "//D/ml/database" for cygwin
#endif
        path.remove(i);

    }

  if( path[ path.length() - 1 ] != separator )
    path += separator;

  // replace internal "./" by "/"
  for( i= 1; i < int(path.length()-1); ++i )
    {
    if( path[i] == '.' && path[i+1] == separator )
      path.remove(i);
    }

  return *this;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL Path::isAbsolute() const
{
  if( path[0] == separator )
    return ITRUE;

  if( path.length() < 2 )
    return IFALSE;

#ifdef __WINDOWS__
  if( path[1] == ':' )
    return ITRUE;
#endif

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL Path::isCurrent() const
{
  return path == currentPath;
}

/////////////////////////////////////////////////////////////////////////////

// if error, return FALSE
IBOOL Path::operator == ( const Path& path2 ) const
{
  if( &path2 == this )
    return ITRUE;
  if( get() == path2.get() )
    return ITRUE;

  return ListFiles::sameFile( get(), path2.get() );
}

/////////////////////////////////////////////////////////////////////////////

Path& Path::operator += ( const OString& p )
{
  if( p == "" || p == "." || p == currentPath )
    return *this;

  OString pp= p;
  if( pp[0] == '.' )
    pp.remove(0);

  if( pp[0] == Path::separator )
    pp.remove(0);

  return operator = ( path + p );
}

/////////////////////////////////////////////////////////////////////////////

// static
OString Path::checkPathSeparators( const OString& filename )
{
  OString f= filename;

  char replace=(separator==unixSeparator) ? dosSeparator : unixSeparator;

  for( int i= 0; i < f.length(); ++i )
    if( f[i] == replace )
      f.put( i, separator );

  return f;
}

/////////////////////////////////////////////////////////////////////////////

// static
OString Path::toUnixPathSeparators( const OString& filename )
{
  OString f= filename;

  for( int i= 0; i < f.length(); ++i )
    if( f[i] == dosSeparator )
      f.put( i, unixSeparator );

  return f;
}

/////////////////////////////////////////////////////////////////////////////
