/* filesel.c */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <fnmatch.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <ctype.h>
#include <string.h>
#include <dirent.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SmeLine.h>
#include <X11/Xaw/List.h>

#include "../common/cmalloc.h"
#include "dialogs.h"

Widget fsel_pshell, fsel_form,
	fsel_filelabel, fsel_filetext, fsel_fileviewport, fsel_filelist,
	fsel_formatlabel, fsel_formatbutton, fsel_formatmenu,
	fsel_dirlabel, fsel_dirtext, fsel_dirviewport, fsel_dirlist,
	fsel_okbutton, fsel_cancelbutton;

/* collect directory contents */

int compar(const void *p, const void *q)
{
	return strcmp(*(const char **)p, *(const char **)q);
}

int
getdirent(char ***ldirp, int *ndirp, char ***lfilep, int *nfilep,
	char *path, char *pattern)
{
	DIR *dird;
	struct dirent *dire;
	char fn[1024];
	struct stat buf;
	char **ldir = NULL, **lfile = NULL;
	int ndir = 0, mdir = 0, nfile = 0, mfile = 0;

	if ((dird = opendir(path)) == NULL) {
		fprintf(stderr, "Can't open %s\n", path);
		return 1;
	}
	while ((dire = readdir(dird))) {
		sprintf(fn, "%s/%s", path, dire->d_name);
		if ((stat(fn, &buf))) {
			fprintf(stderr, "Can't stat %s\n", fn);
			continue;
		}
		if (S_IFDIR & buf.st_mode) {
			if (ndir >= mdir)
				ldir = crealloc(ldir,
					(mdir += 256)*sizeof(char *));
			ldir[ndir++] = cstrdup(dire->d_name);
		}
		else if (S_IFREG & buf.st_mode && 
				!fnmatch(pattern, dire->d_name, 0)) {
			if (nfile >= mfile)
				lfile = crealloc(lfile,
					(mfile += 256)*sizeof(char *));
			lfile[nfile++] = cstrdup(dire->d_name);
		}
	}
	closedir(dird);

	qsort(ldir, ndir, sizeof(char *), compar);
	qsort(lfile, nfile, sizeof(char *), compar);

	*ndirp = ndir;
	*ldirp = ldir;
	*nfilep = nfile;
	*lfilep = lfile;

	return 0;
}

static void
freedirent(char **ldir, int ndir, char **lfile, int nfile)
{
	int i;

	for (i = 0; i < ndir; i++)
		cfree(ldir[i]);
	cfree(ldir);
	for (i = 0; i < nfile; i++)
		cfree(lfile[i]);
	cfree(lfile);
}

/* and that's all there is to it */


/* Foolproof default format */
static char *fileformats[] = {
	"All (*)",
	NULL};

char **files, **dirs;
int nfiles, ndirs;

static void make_files(String files[], Cardinal nfiles)
{
	static String foo[1] = {"<none>"};

	if (nfiles == 0)
		XawListChange(fsel_filelist,
			foo, 1, 0, True);
	else
		XawListChange(fsel_filelist,
			files, nfiles, 0, True);
}

static void make_dirs(String dirs[], Cardinal ndirs)
{
	XawListChange(fsel_dirlist,
		dirs, ndirs, 0, True);
}

static void fsel_scan()
{
	String string;
	char dir[1024];
	char pattern[1024];
	char *pst;

	if (files != NULL)
		freedirent(dirs, ndirs, files, nfiles);
	XtVaGetValues(fsel_dirtext,
		XtNstring, &string, NULL);
	strcpy(dir, string);
	XtVaGetValues(fsel_formatbutton,
		XtNlabel, &string, NULL);
	pst = strchr(string, '(');
	if (pst == NULL) {
		/* apparently no pattern here; use foolproof default */
		strcpy(pattern, "*");
	}
	else {
		strcpy(pattern, pst+1);
		if ((pst = strchr(pattern, ')')))
			*pst = '\0';
	}
	if (getdirent(&dirs, &ndirs, &files, &nfiles, dir, pattern)) {
		fprintf(stderr, "Grmbl. getdirent() failed\n");
	}
	make_files(files, nfiles);
	make_dirs(dirs, ndirs);
}

static void format_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	String fmt = (String)client_data;
	XtVaSetValues(fsel_formatbutton,
		XtNlabel, fmt, NULL);
	fsel_scan();
}

static void file_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;

	XtVaSetValues(fsel_filetext,
		XtNstring, string, NULL);
	XawListUnhighlight(fsel_filelist);
	XawListUnhighlight(fsel_dirlist);
}

static void dir_select(Widget w, XtPointer client_data, XtPointer call_data)
{
	String oldpath;
	char path[1024], newpath[1024];

	XawListReturnStruct *list_struct = (XawListReturnStruct *)call_data;
	String string = list_struct->string;

	XtVaGetValues(fsel_dirtext,
		XtNstring, &oldpath, NULL);
	sprintf(path, "%s/%s", oldpath, string);


	if (!realpath(path, newpath))
		fprintf(stderr, "Couldn't realpath %s\n", path);

	XtVaSetValues(fsel_dirtext,
		XtNstring, newpath, NULL);

	XawListUnhighlight(fsel_filelist);
	XawListUnhighlight(fsel_dirlist);
	fsel_scan();
}

static void make_menu(char **menu_entry_names, Widget menu)
{
	int i;
	Widget entry;

	for (i = 0; menu_entry_names[i]; i++) {
		String item = menu_entry_names[i];

		if (item[0] == '-')	/* line pane */
			entry = XtCreateManagedWidget(item,
				smeLineObjectClass, menu, NULL, 0);
		else {
			entry = XtCreateManagedWidget(item,
				smeBSBObjectClass, menu, NULL, 0);
			XtAddCallback(entry,
				XtNcallback, format_select,
				menu_entry_names[i]);
		}
	}
}

#define ABORT 0
#define DONE 1
#define WAITING 2

int status;

static void fsel_done(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(fsel_pshell);
	status = DONE;
}

static void fsel_abort(Widget w, XtPointer client_data, XtPointer call_data)
{
	XtPopdown(fsel_pshell);
	status = ABORT;
}

static void fsel_done_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	XtPopdown(fsel_pshell);
	status = DONE;
}

static void fsel_cancel_action(Widget w, XEvent *event, String *params, Cardinal *n)
{
	XtPopdown(fsel_pshell);
	status = ABORT;
}

static XtActionsRec actions[] =
{
	{"fsel-done", fsel_done_action},
	{"fsel-cancel", fsel_cancel_action}
};

void fsel_init(Widget topLevel)
{
	XtAppContext app_context = XtWidgetToApplicationContext(topLevel);
	XtAppAddActions(app_context, actions, XtNumber(actions));

	fsel_pshell = XtVaCreatePopupShell("fsel_pshell",
		transientShellWidgetClass, topLevel, NULL);

	fsel_form = XtCreateManagedWidget("fsel_form",
		formWidgetClass, fsel_pshell, NULL, 0);

	fsel_filelabel = XtVaCreateManagedWidget("fsel_filelabel",
		labelWidgetClass, fsel_form, NULL);

	fsel_filetext = XtVaCreateManagedWidget("fsel_filetext",
		asciiTextWidgetClass, fsel_form,
		XtNeditType, XawtextEdit, NULL);
	XtVaSetValues(fsel_filetext,
		XtNstring, "", NULL);

	fsel_fileviewport = XtVaCreateManagedWidget("fsel_fileviewport",
		viewportWidgetClass, fsel_form, NULL);

	fsel_filelist = XtVaCreateManagedWidget("fsel_filelist",
		listWidgetClass, fsel_fileviewport, NULL);

	XtAddCallback(fsel_filelist, XtNcallback, file_select, NULL);

	fsel_formatlabel = XtVaCreateManagedWidget("fsel_formatlabel",
		labelWidgetClass, fsel_form, NULL);

	fsel_formatbutton = XtVaCreateManagedWidget("fsel_formatbutton",
		menuButtonWidgetClass, fsel_form,
		XtNlabel, fileformats[0],
		XtNwidth, 200, NULL);

	fsel_dirlabel = XtVaCreateManagedWidget("fsel_dirlabel",
		labelWidgetClass, fsel_form, NULL);

	fsel_dirtext = XtVaCreateManagedWidget("fsel_dirtext",
		asciiTextWidgetClass, fsel_form, NULL);

	fsel_dirviewport = XtVaCreateManagedWidget("fsel_dirviewport",
		viewportWidgetClass, fsel_form, NULL);

	fsel_dirlist = XtVaCreateManagedWidget("fsel_dirlist",
		listWidgetClass, fsel_dirviewport, NULL);

	XtAddCallback(fsel_dirlist, XtNcallback, dir_select, NULL);

	fsel_okbutton = XtVaCreateManagedWidget("fsel_okbutton",
		commandWidgetClass, fsel_form, NULL);
	XtAddCallback(fsel_okbutton, XtNcallback, fsel_done, NULL);

	fsel_cancelbutton = XtVaCreateManagedWidget("fsel_cancelbutton",
		commandWidgetClass, fsel_form, NULL);
	XtAddCallback(fsel_cancelbutton, XtNcallback, fsel_abort, NULL);
}

int fsel_input(Widget pw, char *path, char *name, char *patterns[], char *fmt)
{
	String string;
	XtAppContext app_context = XtWidgetToApplicationContext(pw);

	if (patterns == NULL)
		patterns = fileformats;

	/* First, create all the formats on the menu */
	XtVaSetValues(fsel_formatbutton,
		XtNlabel, patterns[0],
		XtNwidth, 200, NULL);
	fsel_formatmenu = XtVaCreatePopupShell("menu",
		simpleMenuWidgetClass, fsel_formatbutton, NULL);

	make_menu(patterns, fsel_formatmenu);

	XtVaSetValues(fsel_dirtext, XtNstring, path, NULL);
	XtVaSetValues(fsel_filetext, XtNstring, name, NULL);
	fsel_scan();
	status = WAITING;

	center(pw, fsel_pshell);
	XtPopup(fsel_pshell, XtGrabNonexclusive);

	XtSetKeyboardFocus(fsel_pshell, fsel_filetext);

	while (status == WAITING) {
		XEvent event_return;
		XtAppNextEvent(app_context, &event_return);
		XtDispatchEvent(&event_return);
	}

	/* And get rid of the formats */
	XtDestroyWidget(fsel_formatmenu);

	XtVaGetValues(fsel_filetext, XtNstring, &string, NULL);
	strcpy(name, string);
	XtVaGetValues(fsel_dirtext, XtNstring, &string, NULL);
	strcpy(path, string);
	XtVaGetValues(fsel_formatbutton, XtNlabel, &string, NULL);
	strcpy(fmt, string);
	return status;
}

