/*
 * gzip.c - handle compress, packed, gziped, or lzh (compress -H) files.
 *
 * copyright (c) 1994 matthew green.
 *
 * portions of this file are derived from gzip by jean-loup gailly, and
 * thus this entire file is under the gnu gpl.
 */

/*
 * The unzip code was written and put in the public domain by Mark Adler.
 * Portions of the lzw code are derived from the public domain 'compress'
 * written by Spencer Thomas, Joe Orost, James Woods, Jim McKie, Steve Davies,
 * Ken Turkowski, Dave Mack and Peter Jannesen.
 *
 * See the license_msg below and the file COPYING for the software license.
 */

/*
 * Copyright (C) 1992-1993 Jean-loup Gailly
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef lint
static char rcsid[] = "$Id: gzip.c,v 1.5 1994/12/07 06:17:43 mrg Exp $";
#endif

#include <ctype.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/stat.h>
#include <errno.h>

#include "gzip.h"
#include "lzw.h"

/* configuration */

#if !defined(S_ISDIR) && defined(S_IFDIR)
#  define S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif
#if !defined(S_ISREG) && defined(S_IFREG)
#  define S_ISREG(m) (((m) & S_IFMT) == S_IFREG)
#endif

/* global buffers */

unsigned char inbuf[INBUFSIZ +INBUF_EXTRA];
unsigned char outbuf[OUTBUFSIZ+OUTBUF_EXTRA];
unsigned short d_buf[DIST_BUFSIZE];
unsigned char window[2L * WSIZE];
unsigned short tab_prefix[1L<<BITS];

/* local variables */

int method = DEFLATED;/* compression method */
int gzip_return_code = OK;   /* program exit code */
int part_nb;          /* number of parts in .gz file */

long bytes_in;             /* number of input bytes */
long bytes_out;            /* number of output bytes */
long total_in = 0;         /* input bytes for all files */
long total_out = 0;        /* output bytes for all files */
struct stat istat;         /* status for input file */
unsigned insize;           /* valid bytes in inbuf */
unsigned inptr;            /* index of next byte to be processed in inbuf */
unsigned outcnt;           /* bytes in output buffer */

/* local functions */

static int	get_method __P((int));
int		(*work) __P((int infile, int outfile)) = zip; /* function to call */

int
zclose(fd)
	int fd;
{
	return close(fd);
}

int
zread(fd, buf, n)
	int fd;
	char *buf;
	int n;
{
	int count = 0;

	if ((*work)(fd, ofd) != OK)
		gzip_return_code = ITZ_WORK_FAILED;
	else
		gzip_return_code = OK;
	return count;
}

int
zopen(path)
	const char *path;
{
	int fd;

	if (stat(path, &istat) != OK)
	{
		gzip_return_code = errno | IGZ_ERRNO;
		return -1;
	}

	if (S_ISDIR(istat.st_mode))
	{
		gzip_return_code = IGZ_ISDIR;
		return -1;
	}
	
	if (!S_ISREG(istat.st_mode))
	{
		gzip_return_code = IGZ_ISNOTREG;
		return -1;
	}

	fd = open(path, O_RDONLY);

	if (fd == -1)
	{
		gzip_return_code = errno | IGZ_ERRNO;
		close(fd);
		return;
	}

	clear_bufs(); /* clear input and output buffers */
	part_nb = 0;

	method = get_method(fd);	/* updates ofname if original given */
	if (method < 0)
	{
		close(fd);
		return -1;		/* error message already emitted */
	}

	return fd;
}


/*
 * Check the magic number of the input file and update ofname if an
 * original name was given and to_stdout is not set.
 * Return the compression method, -1 for error, -2 for warning.
 * Set inptr to the offset of the next byte to be processed.
 * This function may be called repeatedly for an input file consisting
 * of several contiguous gzip'ed members.
 * IN assertions: there is at least one remaining compressed member.
 *   If the member is a zip file, it must be the only one.
 */
static int
get_method(in)
	int in;        /* input file descriptor */
{
	unsigned char flags;
	char magic[2]; /* magic header */

	magic[0] = (char) get_byte();
	magic[1] = (char) get_byte();
	method = -1;                 /* unknown yet */
	part_nb++;                   /* number of parts in gzip file */
	header_bytes = 0;
	/* assume multiple members in gzip file except for record oriented I/O */

	if (memcmp(magic, GZIP_MAGIC, 2) == 0 || memcmp(magic, OLD_GZIP_MAGIC, 2) == 0)
	{

		method = (int) get_byte();
		if (method != DEFLATED)
		{
			gzip_return_code = IGZ_NEED_NEW | IGZ_DEFLATE;
			return -1;
		}
		work = unzip;
		flags = (unsigned char) get_byte();

		if (flags & ENCRYPTED)
		{
			gzip_return_code = IGZ_NEED_NEW | IGZ_ENCRYPT;
			return -1;
		}
		if (flags & CONTINUATION)
		{
			gzip_return_code = IGZ_NEED_NEW | IGZ_CONTIN;
			return -1;
		}
		if (flags & RESERVED)
		{
			gzip_return_code = IGZ_NEED_NEW | IGZ_RESERVED;
			return -1;
		}
		(void) get_byte();	/* ignore timestamp information */
		(void) get_byte();
		(void) get_byte();
		(void) get_byte();

		(void) get_byte();  /* Ignore extra flags for the moment */
		(void) get_byte();  /* Ignore OS type for the moment */

		if (flags & CONTINUATION)
		{
			unsigned part = (unsigned) get_byte();
			part |= ((unsigned) get_byte()) << 8;
		}
		if (flags & EXTRA_FIELD)
		{
			unsigned len = (unsigned) get_byte();
			len |= ((unsigned) get_byte()) << 8;
			while (len--)
				(void) get_byte();
		}

		/* Discard file comment if any */
		if (flags & COMMENT)
			while (get_char())
				;
		if (part_nb == 1)
			header_bytes = inptr + 2 * sizeof(long); /* include crc and size */

	}
	else if (memcmp(magic, PKZIP_MAGIC, 2) == 0 && inptr == 2 && memcmp((char *) inbuf, PKZIP_MAGIC, 4) == 0)
	{
		/* To simplify the code, we support a zip file when alone only.
		 * We are thus guaranteed that the entire local header fits in inbuf.
		 */
		inptr = 0;
		work = unzip;
		if (check_zipfile(in) != OK)
			return -1;
		/* check_zipfile may get ofname from the local header */
	}
	else if (memcmp(magic, PACK_MAGIC, 2) == 0)
	{
		work = unpack;
		method = PACKED;
	}
	else if (memcmp(magic, LZW_MAGIC, 2) == 0)
	{
		work = unlzw;
		method = COMPRESSED;
	}
	else if (memcmp(magic, LZH_MAGIC, 2) == 0)
	{
		work = unlzh;
		method = LZHED;
	}
	else
	{
		method = STORED;
		work = copy;
		inptr = 0;
	}
	if (method >= 0)
		return method;

	if (part_nb == 1)
	{
		gzip_return_code = ERROR;
		return -1;
	}
	else
		return -2;
}
