/*
 * Copyright (c) 1990,1993 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/syslog.h>
#include <sys/time.h>
#include <netatalk/endian.h>
#include <dirent.h>
#include <limits.h>
#include <atalk/afp.h>
#include <string.h>
#include <fcntl.h>
#include "auth.h"
#include "directory.h"
#include "volume.h"

/* setgid directories */
#ifndef DIRBITS
#define DIRBITS S_ISGID
#endif

#ifdef ultrix
#include <sys/mount.h>
#include <sys/quota.h>
#endif ultrix

#ifdef _IBMR2
#include <sys/statfs.h>
#endif _IBMR2

#if defined( sun ) || defined( ibm032 ) || defined( linux )
#include <sys/vfs.h>
#endif sun ibm032 linux

#if defined( __svr4__ )
#include <sys/fs/ufs_quota.h>
#include <sys/statvfs.h>
#include <sys/mnttab.h>
#endif __svr4__

#ifdef BSD4_4
#include <ufs/ufs/quota.h>
#include <sys/mount.h>
#define dqb_btimelimit	dqb_btime
#endif BSD4_4

#ifdef linux
#ifdef NEED_QUOTACTL_WRAPPER
#include <asm/types.h>
#include <asm/unistd.h>
#include <linux/quota.h>
int quotactl(int cmd, const char *special, int id, caddr_t addr)
{
  return syscall(__NR_quotactl, cmd, special, id, addr);
}
#else
#include <sys/quota.h>
#endif

#include <sys/mount.h>
#include <mntent.h>
#define dqb_btimelimit  dqb_btime
#endif

#if !defined( linux ) && !defined( ultrix ) && !defined( __svr4__ ) && \
	!defined( BSD4_4 )
#include <ufs/quota.h>
#include <mntent.h>
#endif linux ultrix __svr4__

#ifdef ibm032
typedef unsigned short	mode_t;
#endif ibm032

#if defined( sun ) && !defined( __svr4__ )
#ifdef i386
typedef int	mode_t;
#endif i386
#endif sun __svr4__

#ifdef __svr4__
#define statfs statvfs
#else __svr4__
#define	f_frsize f_bsize
#endif __svr4__

/*
 * Get the free space on a partition.
 */
ustatfs_getvolspace( vol, bfree, btotal, bsize )
    struct vol	*vol;
    VolSpace    *bfree, *btotal;
    u_int32_t   *bsize;
{
#ifdef ultrix
    struct fs_data	sfs;
#else ultrix
    struct statfs	sfs;
#endif ultrix

    if ( statfs( vol->v_path, &sfs ) < 0 ) {
	return( AFPERR_PARAM );
    }

#ifdef ultrix
    *bfree = (VolSpace) sfs.fd_req.bfreen * 1024;
    *bsize = 1024;
#else
    *bfree = (VolSpace) sfs.f_bavail * sfs.f_frsize;
    *bsize = sfs.f_frsize;
#endif ultrix

#ifdef ultrix
    *btotal = (VolSpace) 
      ( sfs.fd_req.btot - ( sfs.fd_req.bfree - sfs.fd_req.bfreen )) * 1024;
#else ultrix
    *btotal = (VolSpace) 
      ( sfs.f_blocks - ( sfs.f_bfree - sfs.f_bavail )) * sfs.f_frsize;
#endif ultrix

    return( AFP_OK );
}

#ifdef __svr4__
/*
 * Return the mount point associated with the filesystem
 * on which "file" resides.  Returns NULL on failure.
 */
    char *
mountp( file )
    char	*file;
{
    struct stat			sb;
    FILE 			*mtab;
    dev_t			devno;
    static struct mnttab	mnt;

    if ( stat( file, &sb ) < 0 ) {
	return( NULL );
    }
    devno = sb.st_dev;

    if (( mtab = fopen( "/etc/mnttab", "r" )) == NULL ) {
	return( NULL );
    }

    while ( getmntent( mtab, &mnt ) == 0 ) {
	if ( stat( mnt.mnt_special, &sb ) == 0 && devno == sb.st_rdev ) {
	    fclose( mtab );
	    return( mnt.mnt_mountp );
	}
    }

    fclose( mtab );
    return( NULL );
}

#else __svr4__
#ifdef ultrix
/*
 * Return the block-special device name associated with the filesystem
 * on which "file" resides.  Returns NULL on failure.
 */

    char *
special( file )
    char *file;
{
    static struct fs_data	fsd;

    if ( getmnt(0, &fsd, 0, STAT_ONE, file ) < 0 ) {
	syslog( "special: getmnt %s: %m", file );
	return( NULL );
    }

    return( fsd.fd_req.devname );
}

#else ultrix
#ifdef BSD4_4

    char *
special( file )
    char	*file;
{
    static struct statfs	sfs;

    if ( statfs( file, &sfs ) < 0 ) {
	return( NULL );
    }
    return( sfs.f_mntfromname );
}

#else /* BSD4_4 */

    char *
special( file )
    char *file;
{
    struct stat		sb;
    FILE 		*mtab;
    dev_t		devno;
    struct mntent	*mnt;

    if ( stat( file, &sb ) < 0 ) {
	return( NULL );
    }
    devno = sb.st_dev;

    if (( mtab = setmntent( "/etc/mtab", "r" )) == NULL ) {
	return( NULL );
    }

    while (( mnt = getmntent( mtab )) != NULL ) {
	if ( stat( mnt->mnt_fsname, &sb ) == 0 && devno == sb.st_rdev ) {
	    endmntent( mtab );
	    return( mnt->mnt_fsname );
	}
    }

    endmntent( mtab );
    return( NULL );
}

#endif BSD4_4
#endif ultrix
#endif __svr4__

getquota( vol, dq )
    struct vol		*vol;
    struct dqblk	*dq;
{
    char		*p;
#ifdef __svr4__
    char		buf[ MAXNAMLEN ];
    struct quotctl	qc;

    if ( vol->v_qfd == -1 ) {
	if (( p = mountp( vol->v_path )) == NULL ) {
	    syslog( LOG_ERR, "getquota: mountp %s fails", vol->v_path );
	    return( AFPERR_PARAM );
	}

	sprintf( buf, "%s/quotas", p );
	if (( vol->v_qfd = open( buf, O_RDONLY, 0 )) < 0 ) {
	    syslog( LOG_INFO, "open %s: %m", buf );
	    return( AFPERR_PARAM );
	}
    }

    qc.op = Q_GETQUOTA;
    qc.uid = uuid;
    qc.addr = (caddr_t)dq;
    if ( ioctl( vol->v_qfd, Q_QUOTACTL, &qc ) < 0 ) {
	return( AFPERR_PARAM );
    }

#else __svr4__
    if ( vol->v_gvs == NULL ) {
	if (( p = special( vol->v_path )) == NULL ) {
	    syslog( LOG_ERR, "getquota: special %s fails", vol->v_path );
	    return( AFPERR_PARAM );
	}

	if (( vol->v_gvs = (char *)malloc( strlen( p ) + 1 )) == NULL ) {
	    syslog( LOG_ERR, "getquota: malloc: %m" );
	    return AFPERR_MISC;
	}
	strcpy( vol->v_gvs, p );
    }

#ifdef ultrix
    if ( quota( Q_GETDLIM, uuid, vol->v_gvs, dq ) != 0 ) {
	return( AFPERR_PARAM );
    }
#else ultrix

#ifndef USRQUOTA
#define USRQUOTA   0
#endif

#ifndef QCMD
#define QCMD(a,b)  (a)
#endif

#ifdef BSD4_4
    if ( quotactl( vol->v_gvs, QCMD(Q_GETQUOTA,USRQUOTA), 
		   uuid, (char *)dq ) != 0 ) {
	return( AFPERR_PARAM );
    }
#else /* BSD4_4 */
    if ( quotactl(QCMD(Q_GETQUOTA, USRQUOTA), vol->v_gvs, uuid,
		  (caddr_t) dq ) != 0 ) {
	return( AFPERR_PARAM );
    }
#endif  /* BSD4_4 */
#endif ultrix
#endif __svr4__

    return( 0 );
}

uquota_getvolspace( vol, bfree, btotal )
    struct vol	*vol;
    VolSpace	*bfree, *btotal;
{
    struct dqblk	dqblk;

    if ( getquota( vol, &dqblk ) != 0 ) {
	return( AFPERR_PARAM );
    }

    /* no limit set for this user. it might be set in the future. */
    if (dqblk.dqb_bsoftlimit == 0 && dqblk.dqb_bhardlimit == 0) {
       *btotal = *bfree = ~((VolSpace) 0);
    } else if ( overquota( &dqblk )) {
	*btotal = (VolSpace) dbtob( dqblk.dqb_bhardlimit );
	*bfree = (VolSpace) 
	  dbtob( dqblk.dqb_bhardlimit ) - dbtob( dqblk.dqb_curblocks );
    } else {
	*btotal = (VolSpace) dbtob( dqblk.dqb_bsoftlimit );
	*bfree = (VolSpace) 
	  dbtob( dqblk.dqb_bsoftlimit ) - dbtob( dqblk.dqb_curblocks );
    }

    return( AFP_OK );
}

overquota( dqblk )
    struct dqblk	*dqblk;
{
    struct timeval	tv;

    if ( dqblk->dqb_curblocks < dqblk->dqb_bsoftlimit ) {
	return( 0 );
    }
#ifdef ultrix
    if ( dqblk->dqb_bwarn ) {
	return( 0 );
    }
#else ultrix
    if ( gettimeofday( &tv, 0 ) < 0 ) {
	syslog( LOG_ERR, "overquota: gettimeofday: %m" );
	return( AFPERR_PARAM );
    }
    if ( !dqblk->dqb_btimelimit || dqblk->dqb_btimelimit > tv.tv_sec ) {
	return( 0 );
    }
#endif /* ultrix */
    return( 1 );
}

int utombits( bits )
    mode_t	bits;
{
    int		mbits;

    mbits = 0;

    mbits |= ( bits & ( S_IREAD >> 6 )) ? AR_UREAD : 0;
    mbits |= ( bits & ( S_IWRITE >> 6 )) ? AR_UWRITE : 0;
    mbits |= ( bits & ( S_IEXEC >> 6) ) ? AR_USEARCH : 0;

    return( mbits );
}

void utommode( stat, ma )
    struct stat		*stat;
    struct maccess	*ma;
{
    mode_t		mode;

    mode = stat->st_mode;

    ma->ma_world = utombits( mode );
    mode = mode >> 3;

    ma->ma_group = utombits( mode );
    mode = mode >> 3;

    ma->ma_owner = utombits( mode );

    if ( uuid == stat->st_uid ) {
	ma->ma_user = ma->ma_owner | AR_UOWN;
    } else if ( gmem( stat->st_gid )) {
	ma->ma_user = ma->ma_group;
    } else {
	ma->ma_user = ma->ma_world;
    }

    /*
     * There are certain things the mac won't try if you don't have
     * the "owner" bit set, even tho you can do these things on unix wiht
     * only write permission.  What were the things?
     */
    if ( ma->ma_user & AR_UWRITE ) {
	ma->ma_user |= AR_UOWN;
    }
}


gmem( gid )
    int	gid;
{
    int		i;

    for ( i = 0; i < ngroups; i++ ) {
	if ( groups[ i ] == gid ) {
	    return( 1 );
	}
    }
    return( 0 );
}

mode_t mtoubits( bits )
    u_char	bits;
{
    mode_t	mode;

    mode = 0;

    mode |= ( bits & AR_UREAD ) ? ( S_IREAD >> 6 ) : 0;
    mode |= ( bits & AR_UWRITE ) ? ( S_IWRITE >> 6 ) : 0;
    mode |= ( bits & AR_USEARCH ) ? ( S_IEXEC >> 6 ) : 0;

    return( mode );
}

mode_t mtoumode( ma )
    struct maccess	*ma;
{
    mode_t		mode;

    mode = 0;
    mode |= mtoubits( ma->ma_owner );
    mode = mode << 3;

    mode |= mtoubits( ma->ma_group );
    mode = mode << 3;

    mode |= mtoubits( ma->ma_world );

    return( mode );
}


setdeskmode( mode )
    mode_t	mode;
{
    static char		wd[ MAXPATHLEN + 1];
    char		modbuf[ 12 ], *m;
    struct dirent	*deskp, *subp;
    DIR			*desk, *sub;

    if ( getcwd( wd , MAXPATHLEN) == NULL ) {
	return( -1 );
    }
    if ( chdir( ".AppleDesktop" ) < 0 ) {
	return( -1 );
    }
    if (( desk = opendir( "." )) == NULL ) {
	if ( chdir( wd ) < 0 ) {
	    syslog( LOG_ERR, "setdeskmode: chdir %s: %m", wd );
	}
	return( -1 );
    }
    for ( deskp = readdir( desk ); deskp != NULL; deskp = readdir( desk )) {
	if ( strcmp( deskp->d_name, "." ) == 0 ||
		strcmp( deskp->d_name, ".." ) == 0 || strlen( deskp->d_name ) > 2 ) {
	    continue;
	}
	strcpy( modbuf, deskp->d_name );
	strcat( modbuf, "/" );
	m = strchr( modbuf, '\0' );
	if (( sub = opendir( deskp->d_name )) == NULL ) {
	    continue;
	}
	for ( subp = readdir( sub ); subp != NULL; subp = readdir( sub )) {
	    if ( strcmp( subp->d_name, "." ) == 0 ||
		    strcmp( subp->d_name, ".." ) == 0 ) {
		continue;
	    }
	    *m = '\0';
	    strcat( modbuf, subp->d_name );
	    /* XXX: need to preserve special modes */
	    if ( chmod( modbuf,  DIRBITS | mode ) < 0 ) {
		syslog( LOG_DEBUG, "setdeskmode: chmod %s: %m", modbuf );
	    }
	}
	closedir( sub );
	/* XXX: need to preserve special modes */
	if ( chmod( deskp->d_name,  DIRBITS | mode ) < 0 ) {
	    syslog( LOG_DEBUG, "setdeskmode: chmod %s: %m", deskp->d_name );
	}
    }
    closedir( desk );
    if ( chdir( wd ) < 0 ) {
	syslog( LOG_ERR, "setdeskmode: chdir %s: %m", wd );
	return -1;
    }
    /* XXX: need to preserve special modes */
    if ( chmod( ".AppleDesktop",  DIRBITS | mode ) < 0 ) {
	syslog( LOG_DEBUG, "setdeskmode: chmod .AppleDesktop: %m" );
    }
    return( 0 );
}

setdirmode( mode )
    mode_t	mode;
{
    static char		buf[ MAXPATHLEN + 1];
    struct stat		st;
    char		*m;
    struct dirent	*dirp;
    DIR			*dir;

    if (( dir = opendir( "." )) == NULL ) {
	syslog( LOG_ERR, "setdirmode: opendir .: %m" );
	return( -1 );
    }

    for ( dirp = readdir( dir ); dirp != NULL; dirp = readdir( dir )) {
	if ( *dirp->d_name == '.' ) {
	    continue;
	}
	if ( stat( dirp->d_name, &st ) < 0 ) {
	    syslog( LOG_DEBUG, "setdirmode: stat %s: %m", dirp->d_name );
	    continue;
	}
	if (( st.st_mode & S_IFMT ) == S_IFREG ) {
	    /* XXX: need to preserve special modes */
	    if ( chmod( dirp->d_name, DIRBITS | mode ) < 0 ) {
		syslog( LOG_DEBUG, "setdirmode: chmod %s: %m", dirp->d_name );
	    }
	}
    }
    closedir( dir );
    if (( dir = opendir( ".AppleDouble" )) == NULL ) {
	syslog( LOG_ERR, "setdirmode: opendir .AppleDouble: %m" );
	return( -1 );
    }
    strcpy( buf, ".AppleDouble" );
    strcat( buf, "/" );
    m = strchr( buf, '\0' );
    for ( dirp = readdir( dir ); dirp != NULL; dirp = readdir( dir )) {
	if ( strcmp( dirp->d_name, "." ) == 0 ||
		strcmp( dirp->d_name, ".." ) == 0 ) {
	    continue;
	}
	*m = '\0';
	strcat( buf, dirp->d_name );
	if ( chmod(buf,  DIRBITS | mode) < 0 ) {
	    syslog( LOG_DEBUG, "setdirmode: chmod %s: %m", buf );
	}
    }
    closedir( dir );
    /* XXX: need to preserve special modes */
    if ( chmod( ".AppleDouble",  DIRBITS | mode ) < 0 ) {
	syslog( LOG_ERR, "setdirmode: chmod .AppleDouble: %m" );
	return( -1 );
    }
    /* XXX: need to preserve special modes */
    if ( chmod( ".",  DIRBITS | mode ) < 0 ) {
	syslog( LOG_ERR, "setdirmode: chmod .: %m" );
	return( -1 );
    }
    return( 0 );
}

setdeskowner( uid, gid )
    int		uid;
    int		gid;
{
    static char		wd[ MAXPATHLEN + 1];
    char		modbuf[ 12 ], *m;
    struct dirent	*deskp, *subp;
    DIR			*desk, *sub;

    if ( getcwd( wd, MAXPATHLEN ) == NULL ) {
	return( -1 );
    }
    if ( chdir( ".AppleDesktop" ) < 0 ) {
	return( -1 );
    }
    if (( desk = opendir( "." )) == NULL ) {
	if ( chdir( wd ) < 0 ) {
	    syslog( LOG_ERR, "setdeskowner: chdir %s: %m", wd );
	}
	return( -1 );
    }
    for ( deskp = readdir( desk ); deskp != NULL; deskp = readdir( desk )) {
	if ( strcmp( deskp->d_name, "." ) == 0 ||
		strcmp( deskp->d_name, ".." ) == 0 || strlen( deskp->d_name ) > 2 ) {
	    continue;
	}
	strcpy( modbuf, deskp->d_name );
	strcat( modbuf, "/" );
	m = strchr( modbuf, '\0' );
	if (( sub = opendir( deskp->d_name )) == NULL ) {
	    continue;
	}
	for ( subp = readdir( sub ); subp != NULL; subp = readdir( sub )) {
	    if ( strcmp( subp->d_name, "." ) == 0 ||
		    strcmp( subp->d_name, ".." ) == 0 ) {
		continue;
	    }
	    *m = '\0';
	    strcat( modbuf, subp->d_name );
	    if ( chown( modbuf, uid, gid ) < 0 ) {
		syslog( LOG_DEBUG, "setdeskown: chown %s: %m", modbuf );
	    }
	}
	closedir( sub );
	if ( chown( deskp->d_name, uid, gid ) < 0 ) {
	    syslog( LOG_DEBUG, "setdeskowner: chown %s: %m", deskp->d_name );
	}
    }
    closedir( desk );
    if ( chdir( wd ) < 0 ) {
	syslog( LOG_ERR, "setdeskowner: chdir %s: %m", wd );
	return -1;
    }
    if ( chown( ".AppleDesktop", uid, gid ) < 0 ) {
	syslog( LOG_ERR, "setdeskowner: chown .AppleDesktop: %m" );
    }
    return( 0 );
}

setdirowner( uid, gid )
    int		uid;
    int		gid;
{
    static char		buf[ MAXPATHLEN + 1];
    struct stat		st;
    char		*m;
    struct dirent	*dirp;
    DIR			*dir;

    if (( dir = opendir( "." )) == NULL ) {
	return( -1 );
    }
    for ( dirp = readdir( dir ); dirp != NULL; dirp = readdir( dir )) {
	if ( *dirp->d_name == '.' ) {
	    continue;
	};
	if ( stat( dirp->d_name, &st ) < 0 ) {
	    syslog( LOG_DEBUG, "setdirowner: stat %s: %m", dirp->d_name );
	    continue;
	}
	if (( st.st_mode & S_IFMT ) == S_IFREG ) {
	    if ( chown( dirp->d_name, uid, gid ) < 0 ) {
		syslog( LOG_DEBUG, "setdirowner: chown %s: %m", dirp->d_name );
	    }
	}
    }
    closedir( dir );
    if (( dir = opendir( ".AppleDouble" )) == NULL ) {
	return( -1 );
    }
    strcpy( buf, ".AppleDouble" );
    strcat( buf, "/" );
    m = strchr( buf, '\0' );
    for ( dirp = readdir( dir ); dirp != NULL; dirp = readdir( dir )) {
	if ( strcmp( dirp->d_name, "." ) == 0 ||
		strcmp( dirp->d_name, ".." ) == 0 ) {
	    continue;
	}
	*m = '\0';
	strcat( buf, dirp->d_name );
	if ( chown( buf, uid, gid ) < 0 ) {
	    syslog( LOG_DEBUG, "setdirowner: chown %d/%d %s: %m",
		    uid, gid, buf );
	}
    }
    closedir( dir );

    /*
     * We cheat: we know that chown doesn't do anything.
     */
    if ( stat( ".AppleDouble", &st ) < 0 ) {
	syslog( LOG_ERR, "setdirowner: stat .AppleDouble: %m" );
	return( -1 );
    }
    if ( gid && gid != st.st_gid && chown( ".AppleDouble", uid, gid ) < 0 ) {
	return( -1 );
    }

    if ( stat( ".", &st ) < 0 ) {
	return( -1 );
    }
    if ( gid && gid != st.st_gid && chown( ".", uid, gid ) < 0 ) {
	return( -1 );
    }

    return( 0 );
}
