/*
 * tnmUtil.c --
 *
 *	Utility functions used by various modules of the Tnm extension.
 *
 * Copyright (c) 1993-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"


/*
 *----------------------------------------------------------------------
 *
 * TnmGetTableValue --
 *
 *	This procedure searches for a key in a given table and 
 *	returns the corresponding value.
 *
 * Results:
 *	Returns a pointer to the static value or NULL if the key
 *	is not contained in the table.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char *
TnmGetTableValue(table, key)
    TnmTable *table;
    unsigned key;
{
    TnmTable *elemPtr;

    if (table) {
	for (elemPtr = table; elemPtr->value; elemPtr++) {
	    if (elemPtr->key == key) {
		return elemPtr->value;
	    }
	}
    }

    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetTableKey --
 *
 *	This procedure searches for a value for a given table and
 *	returns the corresponding key.
 *
 * Results:
 *	Returns the key code or -1 if the string does not contain
 *	the given value.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmGetTableKey(table, value)
    TnmTable *table;
    char *value;
{
    TnmTable *elemPtr;

    if (table) {
	for (elemPtr = table; elemPtr->value; elemPtr++) {
	    if (strcmp(value, elemPtr->value) == 0) {
		return elemPtr->key;
	    }
	}
    }
    
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetTableValues --
 *
 *	This procedure returns a list of all values in a TnmTable.
 *	The list is contained in a string where each element is
 *	seperated by a ", " substring.
 *
 * Results:
 *	Returns a pointer to the static string which is overwritten
 *	by the next call to this procedure.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char *
TnmGetTableValues(table)
    TnmTable *table;
{
    static char *buffer = NULL;
    static int length = 0;
    TnmTable *elemPtr;
    int cnt = 8;
    char *p;

    if (buffer == NULL) {
	length = 256;
	buffer = ckalloc(length);
    }

    /*
     * First count the number of bytes that we need to build 
     * the result string and make sure that the buffer is long
     * enough to hold the result.
     */

    if (table) {
	for (elemPtr = table; elemPtr->value; elemPtr++) {
	    cnt += strlen(elemPtr->value) + 2;
	}
    }

    if (cnt > length) {
	length = cnt;
	buffer = ckrealloc(buffer, length);
    }

    /*
     * Build the result string.
     */
    
    p = buffer;
    if (table) {
	for (elemPtr = table; elemPtr->value; elemPtr++) {
	    char *s = elemPtr->value;
	    if (p != buffer) {
		TnmTable *nextPtr = elemPtr + 1;
		*p++ = ',';
		*p++ = ' ';
		if (nextPtr->value == NULL) {
		    *p++ = 'o';
		    *p++ = 'r';
		    *p++ = ' ';
		}
	    }
	    while (*s) {
		*p++ = *s++;
	    }
	}
    }
    *p = '\0';

    return buffer;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmSetConfig --
 *
 *	This procedure allows to configure various aspects of an
 *	object. It is usually used to process the configure object
 *	command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmSetConfig(interp, config, object, argc, argv)
    Tcl_Interp *interp;
    TnmConfig *config;
    ClientData object;
    int argc;
    char **argv;
{
    int i, option, code;
    TnmTable *elemPtr;
    Tcl_DString dst;

    if (argc % 2) {
        Tcl_AppendResult(interp, "wrong # of args: should be \"", argv[0],
			 " ", argv[1], " ?option value? ?option value? ...\"",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * First scan through the list of options to make sure that
     * we don't run on an unknown option later when we already
     * modified the object.
     */

    for (i = 2; i < argc; i += 2) {
	option = TnmGetTableKey(config->optionTable, argv[i]);
	if (option < 0) {
	    Tcl_AppendResult(interp, "unknown option \"", argv[i],
			     "\": should be ", 
			     TnmGetTableValues(config->optionTable),
			     (char *) NULL);
	    return TCL_ERROR;
	}
    }

    /*
     * Now call the function to actually modify the object. Note,
     * this version does not rollback changes so an object might
     * end up in a half modified state.
     */
	
    for (i = 2; i < argc; i += 2) {
	option = TnmGetTableKey(config->optionTable, argv[i]);
	code = (config->setOption)(interp, object, option, argv[i+1]);
	if (code != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    /*
     * Use a Tcl_DString to assemble the list of all options since
     * the result buffer of the interp might get modified in the
     * callback.
     */
    
    Tcl_DStringInit(&dst);
    for (elemPtr = config->optionTable; elemPtr->value; elemPtr++) {
	char *value = (config->getOption)(interp, object, elemPtr->key);
	if (value) {
	    Tcl_DStringAppendElement(&dst, elemPtr->value);
	    Tcl_DStringAppendElement(&dst, value);
	}
    }
    Tcl_DStringResult(interp, &dst);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetConfig --
 *
 *	This procedure retrieves the value of a configuration option
 *	of an object. It is usually used to process the cget object
 *	command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmGetConfig(interp, config, object, argc, argv)
    Tcl_Interp *interp;
    TnmConfig *config;
    ClientData object;
    int argc;
    char **argv;
{
    int option;
    char *value;

    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " cget option\"", (char *) NULL);
	return TCL_ERROR;
    }

    option = TnmGetTableKey(config->optionTable, argv[2]);
    if (option < 0) {
	Tcl_AppendResult(interp, "unknown option \"", argv[2], 
			 "\": should be ", 
			 TnmGetTableValues(config->optionTable),
			 (char *) NULL);
	return TCL_ERROR;
    }

    value = (config->getOption)(interp, object, option);
    if (value) {
	Tcl_SetResult(interp, value, TCL_STATIC);
	return TCL_OK;
    }

    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, "invalid option \"", argv[2], "\"", 
		     (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetUnsigned --
 *
 *	This procedure converts a string into an unsigned integer 
 *	value. This is simply Tcl_GetInt() with an additional check.
 *
 * Results:
 *	A standard TCL result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmGetUnsigned(interp, string, intPtr)
    Tcl_Interp *interp;
    char *string;
    int *intPtr;
{
    if (Tcl_GetInt(interp, string, intPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (*intPtr < 0) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "expected unsigned integer but got \"",
			 string, "\"", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetPositive --
 *
 *	This procedure converts a string into a positive integer 
 *	value. This is simply Tcl_GetInt() with an additional check.
 *
 * Results:
 *	A standard TCL result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmGetPositive(interp, string, intPtr)
    Tcl_Interp *interp;
    char *string;
    int *intPtr;
{
    if (Tcl_GetInt(interp, string, intPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (*intPtr < 1) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "expected positive integer but got \"",
			 string, "\"", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmSetIPAddress --
 *
 *	This procedure retrieves the network address for the given
 *	host name or address. The argument is validated to ensure that
 *	only legal IP address and host names are accepted. This
 *	procedure maintains a cache of successful name lookups to
 *	reduce the overall DNS overhead.
 *
 * Results:
 *	A standard TCL result. This procedure leaves an error message 
 *	in interp->result if interp is not NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmSetIPAddress(interp, host, addr)
    Tcl_Interp *interp;
    char *host;
    struct sockaddr_in *addr;
{
    static Tcl_HashTable *hostTable = NULL;
    Tcl_HashEntry *hostEntry;
    struct hostent *hp = NULL;
    int code, type;

#define TNM_IP_HOST_NAME 1
#define TNM_IP_HOST_ADDRESS 2

    if (hostTable == NULL) {
	hostTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(hostTable, TCL_STRING_KEYS);
    }
    addr->sin_family = AF_INET;

    /*
     * First check whether we got a host name, an IP address or
     * something else completely different.
     */

    type = TNM_IP_HOST_NAME;
    code = TnmValidateIpHostName(NULL, host);
    if (code != TCL_OK) {
	code = TnmValidateIpAddress(NULL, host);
	if (code != TCL_OK) {
	    if (interp) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "illegal IP address or name \"",
				 host, "\"", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
	type = TNM_IP_HOST_ADDRESS;
    }

    /*
     * Convert the IP address into the internal format.
     */

    if (type == TNM_IP_HOST_ADDRESS) {
	int hostaddr = inet_addr(host);
	if (hostaddr == -1) {
	    if (interp) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "invalid IP address \"", 
				 host, "\"", (char *) NULL);
	    }
	    return TCL_ERROR;
	}
	memcpy((char *) &addr->sin_addr, (char *) &hostaddr, 4);
	return TCL_OK;
    }

    /*
     * Try to convert the name into an IP address. First check
     * whether this name is already known in our address cache.
     * If not, try to resolve the name and add an entry to the
     * cache if successful. Otherwise return an error.
     */

    if (type == TNM_IP_HOST_NAME) {
	struct sockaddr_in *caddr;
	int isnew;

	hostEntry = Tcl_FindHashEntry(hostTable, host);
	if (hostEntry) {
	    struct sockaddr_in *caddr;
	    caddr = (struct sockaddr_in *) Tcl_GetHashValue(hostEntry);
	    addr->sin_addr.s_addr = caddr->sin_addr.s_addr;
	    return TCL_OK;
	}

	hp = gethostbyname(host);
	if (! hp) {
	    if (interp) {
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "unknown IP host name \"", 
				 host, "\"", (char *) NULL);
	    }
	    return TCL_ERROR;
	}

	memcpy((char *) &addr->sin_addr, (char *) hp->h_addr, hp->h_length);
	caddr = (struct sockaddr_in *) ckalloc(sizeof(struct sockaddr_in));
	*caddr = *addr;
	hostEntry = Tcl_CreateHashEntry(hostTable, host, &isnew);
	Tcl_SetHashValue(hostEntry, (ClientData) caddr);
	return TCL_OK;
    }

    /*
     * We should not reach here.
     */

    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetIPName --
 *
 *	This procedure retrieves the network name for the given
 *	network address. It maintains a cache of the last name lookups
 *	to reduce overhead.
 *
 * Results:
 *	A pointer to a static string containing the name or NULL
 *	if the name could not be found. An error message is left
 *	in the interpreter if interp is not NULL
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char *
TnmGetIPName(interp, addr)
    Tcl_Interp *interp;
    struct sockaddr_in *addr;
{
    static Tcl_HashTable *hostTable = NULL;
    Tcl_HashEntry *hostEntry;
    struct hostent *host;

    if (hostTable == NULL) {
	hostTable = (Tcl_HashTable *) ckalloc(sizeof(Tcl_HashTable));
	Tcl_InitHashTable(hostTable, TCL_ONE_WORD_KEYS);
    }

    hostEntry = Tcl_FindHashEntry(hostTable, (char *) addr->sin_addr.s_addr);
    if (hostEntry) {
	return (char *) Tcl_GetHashValue(hostEntry);
    }
    
    host = gethostbyaddr((char *) &addr->sin_addr, 4, AF_INET);
    if (host) {
	int isnew;
	char *name = ckstrdup(host->h_name);
	hostEntry = Tcl_CreateHashEntry(hostTable,
					(char *) addr->sin_addr.s_addr,
					&isnew);
	Tcl_SetHashValue(hostEntry, (ClientData) name);
	return name;
    }

    if (interp) {
	char buffer[20];
	sprintf(buffer, "%u.%u.%u.%u",
		(addr->sin_addr.s_addr >> 24) & 0xff, 
		(addr->sin_addr.s_addr >> 16) & 0xff,
		(addr->sin_addr.s_addr >> 8) & 0xff, 
		addr->sin_addr.s_addr & 0xff);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "unknown IP address \"", buffer, "\"", 
			 (char *) NULL);
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmSetIPPort --
 *
 *	This procedure interprets a string value as an IP port number
 *	and writes the port into the socket address structure.
 *
 * Results:
 *	A standard TCL result. This procedure leaves an error message 
 *	in interp->result if interp is not NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmSetIPPort(interp, protocol, port, addr)
    Tcl_Interp *interp;
    char *protocol;
    char *port;
    struct sockaddr_in* addr;
{
    if (strcmp(protocol, "udp") != 0 && strcmp(protocol, "tcp") != 0) {
	if (interp) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "unknown IP protocol \"", 
			     protocol, "\"", (char *) NULL);
	}
	return TCL_ERROR;
    }

    if (isdigit(*port)) {
	int number = atoi(port);
	if (number >= 0) {
	    addr->sin_port = htons(number);
	    return TCL_OK;
	}
    } else {
	struct servent *servent = getservbyname(port, protocol);
	if (servent) {
	    addr->sin_port = servent->s_port;
	    return TCL_OK;
	}
    }

    if (interp) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "unknown ", protocol, " port \"", 
			 port, "\"", (char *) NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmGetIPPort --
 *
 *	This procedure retrieves the service name for the given
 *	IP port.
 *
 * Results:
 *	A pointer to a static string containing the name or NULL
 *	if the name could not be found. An error message is left
 *	in the interpreter if interp is not NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

char *
TnmGetIPPort(interp, protocol, addr)
    Tcl_Interp *interp;
    char *protocol;
    struct sockaddr_in *addr;
{
    struct servent *serv;

    if (strcmp(protocol, "udp") != 0 && strcmp(protocol, "tcp") != 0) {
	if (interp) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "unknown IP protocol \"", 
			     protocol, "\"", (char *) NULL);
	}
	return NULL;
    }

    serv = getservbyport(addr->sin_port, protocol);
    if (! serv) {
	char buffer[20];
	sprintf(buffer, "%d", ntohs(addr->sin_port));
	if (interp) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "unknown ", protocol, " port \"", 
			     buffer, "\"", (char *) NULL);
	}
	return NULL;
    }

    return serv->s_name;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmValidateIpHostName --
 *
 *	This procedure should be called to validate IP host names.
 *	Some resolver libraries do not check host names very well
 *	which might yield security problems. Some of these resolver
 *	libraries even return random junk if you call gethostbyname()
 *	with an empty string. This procedure checks the rules defined
 *	in RFC 952 and 1123 since we can't rely on the operating
 *	system.
 *
 * Results:
 *	A standard Tcl result. An error message is left in the Tcl
 *	interpreter if it is not NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmValidateIpHostName(interp, name)
    Tcl_Interp *interp;
    char *name;
{
    char *p = name, last = ' ';
    int dots = 0, alpha = 0;

    /*
     * A host name must start with one of the characters [a-zA-Z0-9]
     * and continue with characters from the set [-.a-zA-Z0-9] and
     * must not end with a '.' or a '-'. Names that only contain
     * digits and three dots are also not allowed.
     */

    if (! isalnum(*p)) {
	goto error;
    }

    while (isalnum(*p) || *p == '-' || *p == '.') {
	if (*p == '.') dots++;
	if (isalpha(*p)) alpha++;
	last = *p++;
    }

    if (*p == '\0' && isalnum(last) && (alpha || dots != 3)) {
	return TCL_OK;
    }

 error:
    if (interp) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "illegal IP host name \"",
			 name, "\"", (char *) NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmValidateIpAddress --
 *
 *	This procedure should be called to validate IP addresses.  An
 *	IP address is accepted as valid if and only if it consists of
 *	a string with the format [0-9]+.[0-9]+.[0-9]+.[0-9]+ where
 *	each number is in the range [0-255].
 *
 *	(Note, this check might change if we start to support IPv6.)
 *
 * Results:
 *	A standard Tcl result. An error message is left in the Tcl
 *	interpreter if it is not NULL.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmValidateIpAddress(interp, address)
    Tcl_Interp *interp;
    char *address;
{
    char *p = address;
    unsigned dots = 0, a;

    dots = 0;
    a = 0;
    while (isdigit(*p) || *p == '.') {
	if (isdigit(*p) && dots < 4) {
	    a = 10 * a + *p - '0';
	} else {
	    dots++;
	    a = 0;
	}
	if (dots > 3 || a > 255) {
	    goto error;
	}
	p++;
    }

    if (*p == '\0' && dots == 3) {
	return TCL_OK;
    }

 error:
    if (interp) {
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "illegal IP address \"",
			 address, "\"", (char *) NULL);
    }
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmWriteMessage --
 *
 *	This procedure writes a message to the error channel. This 
 *	should only be used in situations where there is not better
 *	way to handle the run-time error in Tcl.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

void
TnmWriteMessage(interp, msg)
    Tcl_Interp *interp;
    char *msg;
{
    Tcl_DString buffer;
    Tcl_Channel channel;

    /* 
     * Make a copy of the message to make sure that it is safe to call
     * this function with msg pointing to interp->result.
     */
    
    Tcl_DStringInit(&buffer);
    Tcl_DStringAppend(&buffer, msg, -1);
    channel = Tcl_GetChannel(interp, "stderr", NULL);
    if (channel) {
	Tcl_Write(channel, Tcl_DStringValue(&buffer), -1);
    }
    Tcl_DStringFree(&buffer);
}
