/*
 * snmp_utils.c --
 *
 * This file contains helper routines for the agent.
 *
 * Copyright (c) 1995-1997
 *
 * Erik Schoenfelder		TU Braunschweig, Germany
 * Juergen Schoenwaelder	University of Twente, The Netherlands
 * Olivier Montanuy 
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

/*
 * void snmp_alarm ( int ival, void (*handler)(void) )
 *
 * provides an mechanism to call routines at regular intervals.
 */

/*
 * register a filedescriptor with  
 *	void fd_add (int fd, void (*function)(int fd));
 * with function
 *	void worker (int fd);
 * the function is called every time a select reports it readable.
 */

#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>


#include "snmp_util.h"


#ifdef HANDLER_DEBUG
# define dprintf	fprintf
#else
# define dprintf	if (0) fprintf
#endif

/*
 * function calls in intervals hadling:
 */

typedef struct _snmp_alarm_handler {
  int ival;
  struct timeval next_event;
  void (*handler) ();
  struct _snmp_alarm_handler *next;
} snmp_alarm_handler;

static snmp_alarm_handler *all_handler = 0;


#define tv_le(t1, t2)	(((t1)->tv_sec < (t2)->tv_sec) \
	|| ((t1)->tv_sec == (t2)->tv_sec && (t1)->tv_usec < (t2)->tv_usec))

#define tv_msdiff(t1, t2)	(((t1)->tv_sec - (t2)->tv_sec) * 1000 \
	+ ((t1)->tv_usec - (t2)->tv_usec) / 1000)

static void
snmp_alarm_do_handler ()
{
  struct timeval now;
  unsigned int timeo = 99999 * 1000;		/* next timeout in ms */
  snmp_alarm_handler *h = all_handler;
  struct itimerval itv;
  static int in_handler = 0;
  struct sigaction action;

  dprintf (stderr, "do_handler... (%d)\n", in_handler);
  
  if (! in_handler) {
    in_handler = 1;
  } else {
    return;
  }

#if 0
  /* XXX: does not work correct: */
  /* (re)initialize signal handler: */
  if (signal (SIGALRM, snmp_alarm_do_handler) < 0) {
    perror ("snmpd: failed to set SIGALRM handler; reason");
    in_handler = 0;
    return;
  }
#else
  action.sa_handler = snmp_alarm_do_handler;
  sigemptyset(&action.sa_mask);
  action.sa_flags = 0;
  if (sigaction(SIGALRM, &action, NULL) != 0) {
    perror("snmpd: sigaction failed; reason");
    in_handler = 0;
    return;
  }
#endif

  /* shut off the timer: */
  memset (&itv, 0, sizeof(itv));
  if (setitimer (ITIMER_REAL, &itv, (struct itimerval *) 0) < 0) {
    perror ("snmpd: failed to clear interval timer; reason");
  }

  gettimeofday (&now, (struct timezone *) 0);

  while (h) {
    if (tv_le (&h->next_event, &now)) {
      dprintf (stderr, "** working for handler 0x%lx  (%d)...\n", 
	       (long) h->handler, h->ival);
      (*h->handler) ();
      memcpy (&h->next_event, &now, sizeof(now));
      h->next_event.tv_sec += h->ival;
    }

  dprintf (stderr, "now: %u.%06u\n", (unsigned) now.tv_sec,
	   (unsigned) now.tv_usec);
  dprintf (stderr, "nev: %u.%06u\n", (unsigned) h->next_event.tv_sec, 
	   (unsigned) h->next_event.tv_usec);

  dprintf (stderr, "diff: %u\n", (unsigned) tv_msdiff(&h->next_event, &now));

    if (tv_msdiff(&h->next_event, &now) < timeo) {
      timeo = tv_msdiff(&h->next_event, &now);
      if (timeo <= 0) {
	timeo = 1; 
      }
    }
    h = h->next;
  }

  dprintf (stderr, "next event: %u.%06u  (now %u.%06u)  alarm: %u\n",
	   (unsigned) (timeo + now.tv_sec), (unsigned) now.tv_usec,
	   (unsigned) now.tv_sec, (unsigned) now.tv_usec, (unsigned) timeo);

  dprintf (stderr, "setting alarm (%u)\n\n", timeo);

  itv.it_interval.tv_sec = timeo / 1000;
  itv.it_interval.tv_usec = (timeo % 1000) * 1000;
  itv.it_value.tv_sec = itv.it_interval.tv_sec;
  itv.it_value.tv_usec = itv.it_interval.tv_usec;

  if (setitimer (ITIMER_REAL, &itv, (struct itimerval *) 0) < 0) {
    perror ("snmpd: failed to set interval timer; reason");
  }

  in_handler = 0;
}

void
snmp_alarm (ival, handler)
     int ival;
     void (* handler) ();
{
    snmp_alarm_handler *nnew;

    if (! (nnew = (snmp_alarm_handler *) 
	   calloc (1, sizeof (snmp_alarm_handler)))) {
      fprintf (stderr, "snmpd: snmp_alarm: out of memory...\n");
      return;
    }

    /* create new time event handler control structure and chain in: */
    nnew->ival = ival;
    gettimeofday (&nnew->next_event, (struct timezone *) 0);
    nnew->handler = handler;
    nnew->next = all_handler;
    all_handler = nnew;

    dprintf (stderr, "new customer: ival %d\n", ival);

    /* call to work for and set timer: */
    snmp_alarm_do_handler ();
}



/*
 * filedescriptor helper functions:
 */

struct f_handler {
  int fd;
  void (*handler) ();
  struct f_handler *next;
};

static struct f_handler *fd_allhandler = 0;
static fd_set fd_allmask;
static int fd_maxfds = 0;

/*
 * register a function to work for:
 */
void
fd_add (fd, func)
     int fd;
     void (*func)();
{
  struct f_handler *nnew = (struct f_handler *) 
    malloc (sizeof(struct f_handler));

  if (! nnew) {
    fprintf (stderr, "snmpd: out of memory... - ignored\n");
    return;
  }

  nnew->fd = fd;
  nnew->handler = func;
  nnew->next = fd_allhandler;
  fd_allhandler = nnew;

  FD_SET(fd, &fd_allmask);
  fd_maxfds = fd_maxfds < fd ? fd : fd_maxfds;
}


/*
 * service loop; run by the agent.
 */
void
fd_service ()
{
  fd_set fdset;
  int count;

  while (1) {

    memcpy (&fdset, &fd_allmask, sizeof(fd_allmask));
    count = select (fd_maxfds + 1, &fdset, 0, 0, 0);

    if (count > 0){
      struct f_handler *fhp = fd_allhandler;
      while (fhp) {
	if (FD_ISSET(fhp->fd, &fdset)) {
	  (*fhp->handler) (fhp->fd);
	}
	fhp = fhp->next;
      }

    } else {
      switch (count) {
      case 0:
	break;
      case -1:
	if (errno == EINTR){
	  continue;
	} else {
	  perror ("snmpd: select failed; reason");
	}
	return;
      default:
	printf ("snmpd: select returned %d - aborting.\n", count);
	return;
      }
    }
  }
}






/* ---------------------------------------------------------------------- */


/*
** Various utilities, by O.Montanuy.
** this code is public domain, except for the bits borrowed from CMU-SNMP.
*/
#include <sys/types.h>

#include <sys/socket.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifndef linux
# include <nlist.h>
#else
# include <netinet/in.h>
# include <arpa/inet.h>
#endif
#include "asn1.h"
#include "snmp_api.h"
#include "snmp.h"
#include "snmp_impl.h"
#include "mib.h"
#include "snmp_client.h"
#include "snmp_vars.h"
#include "snmp_config.h"

/*
** Function to safely copy a string, and ensure the last
** character is always '\0'.
*/
void
strcpy_safe(str, str_len, val)
     char *str; int str_len; char *val;
{
  int val_len = 0;
  if((str == NULL)||(str_len < 0)) return ; 
  if(val != NULL) /* Andy sez: only paranoiacs survive. */
    {
      val_len = strlen(val);
      if(val_len >= str_len) val_len--;
      strncpy( str, val, val_len);
    }
  str[val_len]='\0'; /* terminate with a zero */
}


/*
** This function is copied from snmptrap.c (Copyright from CMU)
*/
#ifndef IFF_LOOPBACK
#define IFF_LOOPBACK 0
#endif
#define LOOPBACK    0x7f000001
#define NUM_NETWORKS	16   /* max number of interfaces to check */
u_long 
Util_local_ip_address()
{
  int sd;
  struct ifconf ifc;
  struct ifreq conf[NUM_NETWORKS], *ifrp, ifreq;
  struct sockaddr_in *in_addr;
  int count,interfaces;		/* number of interfaces returned by ioctl */
  if ((sd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) return 0;
  ifc.ifc_len = sizeof(conf);
  ifc.ifc_buf = (caddr_t)conf;
  if (ioctl(sd, SIOCGIFCONF, (char *)&ifc) < 0) { close(sd); return 0; }
  ifrp = ifc.ifc_req;
  interfaces = ifc.ifc_len / sizeof(struct ifreq);
  for(count = 0; count < interfaces; count++, ifrp++)
    {
      ifreq = *ifrp;
      if (ioctl(sd, SIOCGIFFLAGS, (char *)&ifreq) < 0) continue;
      in_addr = (struct sockaddr_in *)&ifrp->ifr_addr;
      if ((ifreq.ifr_flags & IFF_UP)
	  && (ifreq.ifr_flags & IFF_RUNNING)
	  && !(ifreq.ifr_flags & IFF_LOOPBACK)
	  && in_addr->sin_addr.s_addr != LOOPBACK)
	{ close(sd); return in_addr->sin_addr.s_addr;}
    }
  close(sd);
  return 0;
}
/*
** Get time in seconds
*/
long 
Util_time_now()
{
  struct timeval now;
  gettimeofday(&now, NULL);
  return now.tv_sec;
}
/*
** This function copied from snmptrap.c (Copyright from CMU)
** Returns uptime in centiseconds(!).
*/
#ifndef linux
static struct nlist nl[] = {
    { "_boottime" },
    { "" }
};
#endif

long 
Util_time_running()
{
#ifndef linux
    struct timeval boottime, now, diff;
    int kmem;
    if ((kmem = open("/dev/kmem", 0)) < 0) return 0;
    nlist("/vmunix", nl);
    if (nl[0].n_type == 0){ close(kmem);return 0;}
    lseek(kmem, (long)nl[0].n_value, L_SET);
    read(kmem, &boottime, sizeof(boottime));
    close(kmem);
    gettimeofday(&now, 0);
    now.tv_sec--;
    now.tv_usec += 1000000L;
    diff.tv_sec = now.tv_sec - boottime.tv_sec;
    diff.tv_usec = now.tv_usec - boottime.tv_usec;
    if (diff.tv_usec > 1000000L){diff.tv_usec -= 1000000L;diff.tv_sec++;}
    return ((diff.tv_sec * 100) + (diff.tv_usec / 10000));
#else /* linux */
    long uptim = 0, a, b;
    FILE *in = fopen ("/proc/uptime", "r");
    if (in)
      {
	if (2 == fscanf (in, "%ld.%ld", &a, &b)) uptim = a * 100 + b;
	fclose (in);
      }
    return uptim;
#endif /* linux */
}
/*
** Size of minimum file chunk
*/
#define FILE_CHUNK  (0x1000)
/*
** Read binary data from file
**   file         = file name
**   offset       = offset in file (zero = beginning of file)
**   data[dataSz] = buffer where to store data
**  returns number of bytes read
*/
int
Util_file_read(file, offset, data, dataSz)
     char *file; int offset; char *data; int dataSz;
{
  int fd;
  if((file==NULL)||(data==NULL)||(dataSz<=0))
    { return 0; }
  fd = open(file, O_RDONLY);
  if(fd < 0)
    {
      printf("Cannot open file %s\n", file);
      return -1; 
    }
  if(offset > 0)
    { lseek(fd, offset, SEEK_SET); }
  {
    int pos, sz;
    struct stat file_stat;
    if(fstat(fd, &file_stat) < 0)
    {
      printf("Cannot read size of %s\n", file);
      return -1;
    }
    if(file_stat.st_size< dataSz) 
      { dataSz= file_stat.st_size; }
    for(pos=0; pos< dataSz; pos+=sz)
      {
	sz = dataSz - pos;
	if(sz > FILE_CHUNK) 
	  sz = FILE_CHUNK;
	if( read(fd, &data[pos], sz)!=sz) 
	  break;
      }
    close(fd);
    if(pos < dataSz)
      {
	printf("Cannot read file %s\n",file);
	return -1;
      }
    return dataSz;
  }
}
/*
** Write binary data into file
**   file         = file name
**   offset       = offset in file (zero = beginning of file)
**   data[dataSz] = buffer where to get data from
**  return 1 if write was ok, -1 else.
*/
int
Util_file_write(file, offset, data, dataSz)
     char *file; int offset; char *data; int dataSz;
{
  int fd;
  if((file==NULL)||(data==NULL)||(dataSz<=0))
    { return 0; }
  fd = open(file, O_WRONLY|O_CREAT, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
  if(fd<0)
    {
      printf("Cannot open file %s\n", file);
      return -1; 
    }
  if(offset > 0)
    { lseek(fd, offset, SEEK_SET); }
  {
    int pos, sz;
    for(pos=0; pos< dataSz; pos+=sz)
      {
	sz = dataSz - pos;
	if(sz > FILE_CHUNK) 
	  sz = FILE_CHUNK;
	if( write(fd, &data[pos], sz)!=sz) 
	  break;
      }
    close(fd);
    if(pos < dataSz)
      {
	printf("Cannot write file %s\n",file);
	return -1;
      }
  }
  return 1;
}


/* ---------------------------------------------------------------------- */
