/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of GF1 chip (PCM things)
 */

#include "driver.h"
#include "pcm.h"
#include "tables.h"
#include "ulaw.h"

#ifdef GUSCFG_GF1PCM

/* playback flags */

#define PFLG_NONE       	0x0000
#define PFLG_USED       	0x0001
#define PFLG_DMA		0x0002	/* DMA in progress */
#define PFLG_ACTIVE		0x0004	/* voice is playing */
#define PFLG_FLUSH		0x0008

/* record flags */

#define RFLG_NONE		0x0000
#define RFLG_USED		0x0001
#define RFLG_STOP		0x0002

/* some macros */

#define GF1_PCM_VOICES()	( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].voices )
#define GF1_PCM_VOICE_MIN()	( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].min )
#define GF1_PCM_VOICE_MAX()	( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].max )
#define GF1_PCM_RVOICES()	( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices )

#ifdef __i386__

#if 0

extern inline void translate_bytes( unsigned char *table, 
                                    unsigned char *buff, 
                                    unsigned int count )
{
  __asm__ ("cld\n"
           "1:\tlodsb\n"
           "\txlatb\n"
           "\tstosb\n"
           "\tloop 1b\n":
           :"b" ((long) table), "c" (count), "D" ((long) buff), "S" ((long) buff)
           :"bx", "cx", "di", "si", "ax", "memory");
}

#endif

extern inline void divide_bytes( unsigned char *dest1,
			         unsigned char *dest2,
			         unsigned char *src,
			         unsigned int count )
{
  __asm__( "cld\n"
  	   "1:\tlodsw\n"
  	   "\tstosb\n"
  	   "\txchgb %%ah,%%al\n"
  	   "\txchgl %%edi,%%ebx\n"
  	   "\tstosb\n"
  	   "\txchgl %%edi,%%ebx\n"
  	   "\tloop 1b\n":
  	   :"c" (count), "S" ((long)src), "D" ((long)dest1), "b" (dest2)
  	   :"bx", "cx", "di", "si", "ax", "memory");
}

extern inline void divide_words( unsigned short *dest1,
			         unsigned short *dest2,
			         unsigned short *src,
			         unsigned int count )
{
  __asm__( "cld\n"
  	   "1:\tlodsw\n"
  	   "\tstosw\n"
  	   "\tlodsw\n"
  	   "\txchgl %%edi,%%ebx\n"
  	   "\tstosw\n"
  	   "\txchgl %%edi,%%ebx\n"
  	   "\tloop 1b\n":
  	   :"c" (count), "S" ((long)src), "D" ((long)dest1), "b" (dest2)
  	   :"bx", "cx", "di", "si", "ax", "memory");
}

#else

#if 0

static void translate_bytes( unsigned char *table,
			     unsigned char *buff,
			     unsigned int count )
{
  while ( count-- > 0 )
    {
      *buff = table[ *buff ];
      buff++;
    }
}

#endif

static void divide_bytes( unsigned char *dest1,
			  unsigned char *dest2,
		   	  unsigned char *src,
		  	  unsigned int count )
{
  while ( count-- > 0 )
    {
      *dest1++ = *src++;
      *dest2++ = *src++;
    }
}

static void divide_words( unsigned short *dest1,
		  	  unsigned short *dest2,
			  unsigned short *src,
			  unsigned int count )
{
  while ( count-- > 0 )
    {
      *(((unsigned short *)dest1)++) = *(((unsigned short *)src)++);
      *(((unsigned short *)dest2)++) = *(((unsigned short *)src)++);
    }
}

#endif

/*
 *  Playback routines 
 */

static void gus_gf1_pcm_translate( gus_card_t *card,
				   unsigned char *src,
				   unsigned char *dest,
				   unsigned int pos,
				   unsigned int count,
				   unsigned int total,
				   int w_16,
				   int uLaw,
				   int space )
{
  register int voice;
  register unsigned int pos1;
  int i, voices;
  unsigned int count1;
  unsigned char buffer[ 512 ];
  unsigned char *bpos, *dest1, *dest2;

#if 0
  PRINTK( "pcm: src=0x%lx, dest=0x%lx, pos=0x%x, count=0x%x, total=0x%x\n", (long)src, (long)dest, pos, count, total );
#endif
  voices = GF1_PCM_RVOICES();
  i = voices * ( w_16 ? 2 : 1 );
  if ( ( count % i ) != 0 )
    {
      PRINTK( "gus: GF1 PCM problem - unaligned count (voices = %i, count = %i)\n", GF1_PCM_VOICES(), count );
      return;
    }
  if ( ( pos % i ) != 0 )
    {
      PRINTK( "gus: GF1 PCM problem - unaligned copy (voices = %i, pos = %i)\n", GF1_PCM_VOICES(), pos );
      return;
    }
  pos /= voices;
  total /= voices;
#if 0
  printk( "pos = 0x%x, total = 0x%x\n", pos, total );
#endif
  if ( !src )
    MEMSET( buffer, space, sizeof( buffer ) );
  while ( count > 0 )
    {
      if ( !src || space == SP_USER )
        {
          count1 = count > sizeof( buffer ) ? sizeof( buffer ) : count;
          if ( src )
            {
              MEMCPY_FROMFS( buffer, src, count1 );
              src += count1;
            }
          bpos = buffer;
          count -= count1;
        }
       else
        {
          count1 = count;
          bpos = src;
          count = 0;
        }

      /*
       *  Next part strongly needs optimalization...
       *  Optimalization for one and two voices is done (not for uLaw)...
       *  I don't have time for this... If you want - try rewrite this part...
       */
      if ( w_16 )
        {
          if ( voices == 1 )
            {
              MEMCPY( dest + pos, bpos, count1 );
              pos += count1;
            }
           else
          if ( voices == 2 )
            {
              divide_words( (unsigned short *)(dest + pos),
                            (unsigned short *)(dest + pos + total),
                            (unsigned short *)bpos, count1 >> 2 );
              pos += count1 >> 1;
            }
           else
            {
              count1 >>= 1;
              while ( count1 > 0 )
                {
                  for ( voice = 0, pos1 = pos; voice < voices; voice++, pos1 += total )
                    *(unsigned short *)&dest[ pos1 ] = *(((unsigned short *)bpos)++);
                  pos += sizeof( unsigned short );
                  count1 -= voices;
                }
            }
        }
       else 
        {
          if ( uLaw )
            {
              if ( voices == 1 )
                {
                  dest1 = dest + pos;
                  pos += count1;
                  while ( count1-- > 0 )
                    *dest1++ = gus_ulaw_dsp[ *bpos++ ];
                }
               else
              if ( voices == 2 )
                {
                  dest1 = dest + pos;
                  dest2 = dest1 + total;
                  pos += count1;
                  count1 >>= 1;
                  while ( count1-- > 0 )
                    {
                      *dest1++ = gus_ulaw_dsp[ *bpos++ ];
                      *dest2++ = gus_ulaw_dsp[ *bpos++ ];
                    }
                }
               else
                {
                  while ( count1 > 0 )
                    {
                      for ( voice = 0, pos1 = pos; voice < voices; voice++, pos1 += total )
                        dest[ pos1 ] = gus_ulaw_dsp[ *bpos++ ];
                      pos++;
                      count1 -= voices;
                    }
                }
            }
           else
            {
              if ( voices == 1 )
                {
                  MEMCPY( dest + pos, bpos, count1 );
                  pos += count1;
                }
               else
              if ( voices == 2 )
                {
                  pos += count1;
                  divide_bytes( dest + pos, dest + pos + total, bpos, count1 >> 1 );
                }
               else
                {
                  while ( count1 > 0 )
                    {
                      for ( voice = 0, pos1 = pos; voice < voices; voice++, pos1 += total )
                        dest[ pos1 ] = *bpos++;
                      pos++;
                      count1 -= voices;
                    }
                }
            }
        }
    }
#if 0
  PRINTK( "translate end\n" );
#endif
}

static void gus_gf1_pcm_dma_transfer( gus_card_t *card )
{
  gus_pcm_channel_t *pchn;
  int i, j;

  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
  __next_block:
  if ( !(card -> gf1.pcm_pflags & PFLG_DMA) )
    {
      if ( !(pchn -> flags & PCM_FLG_MMAP) )
        {
          if ( !pchn -> used ) return;					/* no new data */
          if ( card -> gf1.pcm_used == card -> gf1.pcm_blocks ) return;	/* GUS's RAM full */
        }
       else
        {
          if ( card -> gf1.pcm_used >= 2 ) return;			/* DRAM full */
        }
      card -> gf1.pcm_pflags |= PFLG_DMA;
      card -> gf1.pcm_dma_voice = 0;
      if ( pchn -> flags & PCM_FLG_MMAP )
        gus_gf1_pcm_translate( card,
      			       pchn -> buffer + ( card -> gf1.pcm_head * card -> gf1.pcm_block_size ),
      			       pchn -> hidden_buffer,
      			       0,
      			       card -> gf1.pcm_block_size,
      			       PAGE_SIZE,
      			       pchn -> mode & PCM_MODE_16,
      			       pchn -> mode & PCM_MODE_ULAW,
      			       SP_KERNEL );
       else
        pchn -> locks = 1 << pchn -> tail;
    }
   else
    {
      if ( ++card -> gf1.pcm_dma_voice >= GF1_PCM_RVOICES() )
        {
          card -> gf1.pcm_pflags &= ~PFLG_DMA;
          card -> gf1.pcm_used++;
          card -> gf1.pcm_head++;
          card -> gf1.pcm_head %= card -> gf1.pcm_blocks;
          if ( !(pchn -> flags & PCM_FLG_MMAP) )
            {
              pchn -> used--;
              pchn -> sizes[ pchn -> tail ] = 0;
              pchn -> tail++;
              pchn -> tail %= pchn -> blocks;
              pchn -> locks = 0;
#if 0
              printk( "dma transfer done: used = %i, head = %i, tail = %i\n", pchn -> used, pchn -> head, pchn -> tail );
#endif
              if ( pchn -> flags & PCM_FLG_SLEEP )
                {
                  pchn -> flags &= ~PCM_FLG_SLEEP;
                  WAKEUP( card -> pcm_gf1, playback );
                }
            }
          /* Can I now transfer next block? */
          goto __next_block;
        }
    }
#if 0
  printk( "dma start!!!\n" );
#endif
  if ( !(pchn -> flags & PCM_FLG_MMAP) )
    {
      i = j = card -> gf1.pcm_dma_voice * ( pchn -> used_size / GF1_PCM_RVOICES() );
      j += ( pchn -> tail * pchn -> block_size ) / GF1_PCM_RVOICES();
      i += ( card -> gf1.pcm_head * card -> gf1.pcm_block_size ) / GF1_PCM_RVOICES();
      gus_gf1_init_dma_transfer( card,
                                 card -> gf1.pcm_mem + i,
                                 pchn -> buffer + j,
                                 pchn -> block_size / GF1_PCM_RVOICES(),
			         pchn -> mode & PCM_MODE_U,
                                 pchn -> mode & PCM_MODE_16 );
    }
   else
    {
      i = card -> gf1.pcm_dma_voice * ( pchn -> used_size / GF1_PCM_RVOICES() );
      j = card -> gf1.pcm_dma_voice * ( PAGE_SIZE / GF1_PCM_RVOICES() );
      i += ( card -> gf1.pcm_head * card -> gf1.pcm_block_size ) / GF1_PCM_RVOICES();
      gus_gf1_init_dma_transfer( card,
                                 card -> gf1.pcm_mem + i,
                                 pchn -> hidden_buffer + j,
                                 card -> gf1.pcm_block_size / GF1_PCM_RVOICES(),
			         pchn -> mode & PCM_MODE_U,
                                 pchn -> mode & PCM_MODE_16 );
    }
}

static unsigned short gus_gf1_pcm_volume( gus_card_t *card, int vol, int pan )
{
  if ( vol > 127 ) vol = 127;
  vol = VOLUME( vol );
  if ( pan < 64 )		/* left */
    vol += VOLUME( card -> mixer.pcm_volume_level_left );
   else
    vol += VOLUME( card -> mixer.pcm_volume_level_right );
  if ( vol > MAX_VOLUME ) vol = MAX_VOLUME;
  vol = MAX_VOLUME - vol;
  return vol;
}

static void gus_gf1_pcm_trigger_up( gus_card_t *card )
{
  unsigned long flags; 
  unsigned char voice_ctrl, ramp_ctrl;
  unsigned short rate;
  unsigned int curr, begin, end;
  unsigned short vol;
  unsigned char pan;
  int i;
  gus_pcm_channel_t *pchn;

  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
  rate = gus_translate_freq( card, pchn -> rate << 1 );
  voice_ctrl = pchn -> mode & PCM_MODE_16 ? 0x24 : 0x20;	/* enable WAVE IRQ */
  ramp_ctrl = 0x24;
  if ( card -> gf1.pcm_tail + 1 == card -> gf1.pcm_blocks )
    {
      ramp_ctrl &= ~0x04;			/* disable rollover */
      voice_ctrl |= 0x08;			/* enable loop */
    }
       
  for ( i = 0; i < GF1_PCM_VOICES(); i++ )
    {
      begin = card -> gf1.pcm_mem + i * ( pchn -> used_size / GF1_PCM_RVOICES() );
      curr = begin + ( card -> gf1.pcm_tail * card -> gf1.pcm_block_size ) / GF1_PCM_RVOICES();
      end = curr + ( card -> gf1.pcm_block_size / GF1_PCM_RVOICES() );
      if ( card -> gf1.pcm_pos > 0 )		/* stored position */
        curr = begin + card -> gf1.pcm_pos;
      end -= pchn -> mode & PCM_MODE_16 ? 2 : 1;
#if 0
      PRINTK( "init: curr=0x%x, begin=0x%x, end=0x%x, ctrl=0x%x, ramp=0x%x, rate=0x%x\n", curr, begin, end, voice_ctrl, ramp_ctrl, rate );
#endif
      if ( pchn -> mode & PCM_MODE_MULTI )
        vol = gus_gf1_pcm_volume( card, card -> gf1.pcm_volume[ i ], pan = card -> gf1.pcm_pan[ i ] );
       else
        {
          if ( pchn -> voices == 2 ) pan = !i ? 8 : 128 - 9; else pan = 64;
          vol = gus_gf1_pcm_volume( card, 128, pan );
        }
      pan >>= 3;
      if ( pan > 15 ) pan = 15;
      CLI( &flags );
      gf1_select_voice( card, i + GF1_PCM_VOICE_MIN() );
      gus_write8( card, GF1_VB_PAN, pan );
      gus_write16( card, GF1_VW_FREQUENCY, rate );
      gus_write_addr( card, GF1_VA_START, begin << 4, voice_ctrl & 4 );
      gus_write_addr( card, GF1_VA_END, end << 4, voice_ctrl & 4 );
      gus_write_addr( card, GF1_VA_CURRENT, curr << 4, voice_ctrl & 4 );
      gus_write16( card, GF1_VW_VOLUME, MIN_VOLUME << 4 );
      gus_write8( card, GF1_VB_VOLUME_RATE, 0x2f );
      gus_write8( card, GF1_VB_VOLUME_START, MIN_OFFSET );
      gus_write8( card, GF1_VB_VOLUME_END, vol >> 4 ); /* ramp end */
      gus_write8( card, GF1_VB_VOLUME_CONTROL, ramp_ctrl );
      if ( !card -> gf1.enh_mode )
        {
          gus_delay( card );
          gus_write8( card, GF1_VB_VOLUME_CONTROL, ramp_ctrl );
        }
      STI( &flags );
    }
  CLI( &flags );
  for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ )
    {
      gf1_select_voice( card, i );
      if ( card -> gf1.enh_mode )
        gus_write8( card, GF1_VB_MODE, 0x00 );	/* deactivate voice */
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice_ctrl );
      voice_ctrl &= ~0x20;
    }
  voice_ctrl |= 0x20;
  if ( !card -> gf1.enh_mode )
    {
      gus_delay( card );
      for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ ) 
        {
          gf1_select_voice( card, i );
          gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice_ctrl );
          voice_ctrl &= ~0x20;	/* disable IRQ for next voice */
        }
    }
  STI( &flags );
  card -> gf1.pcm_pflags |= PFLG_ACTIVE;
  if ( !( pchn -> flags & PCM_FLG_MMAP ) ) {
    pchn -> interrupts++;
    pchn -> processed_bytes += card -> gf1.pcm_block_size;
  }
}

/*
 *
 */

static void gus_interrupt_pcm_wave( gus_card_t *card, int voice )
{
  unsigned long flags;
  gus_pcm_channel_t *pchn;
  unsigned char voice_ctrl, ramp_ctrl;
  int i;
  unsigned int end, step;

  if ( voice != GF1_PCM_VOICE_MIN() || !(card -> gf1.pcm_pflags & PFLG_ACTIVE) )
    {
#if 0
      PRINTD( "gus_gf1_pcm: unknown wave irq?\n" );
#endif
      gus_gf1_smart_stop_voice( card, voice );
      return;
    }
  CLI( &flags );
  voice_ctrl = gus_read8( card, GF1_VB_ADDRESS_CONTROL ) & ~0x8b; 
  ramp_ctrl = gus_read8( card, GF1_VB_VOLUME_CONTROL ) & ~0x84;
  STI( &flags );
#if 0
  CLI( &flags );
  for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ )
    {
      gf1_select_voice( card, i );
      PRINTK( "** curr=0x%x, begin=0x%x, end=0x%x, ctrl=0x%x, ramp=0x%x\n", gus_read_addr( card, GF1_VA_CURRENT, voice_ctrl & 4 ), gus_read_addr( card, GF1_VA_START, voice_ctrl & 4 ), gus_read_addr( card, GF1_VA_END, voice_ctrl & 4 ), gus_read8( card, GF1_VB_ADDRESS_CONTROL ), gus_read8( card, GF1_VB_VOLUME_CONTROL ) );
    }
  gf1_select_voice( card, voice );
  STI( &flags );
#endif
  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
#if 0
  printk( "irq: used = %i, head = %i, tail = %i\n", card -> gf1.pcm_used, card -> gf1.pcm_head, card -> gf1.pcm_tail );
#endif
  card -> gf1.pcm_tail++;
  card -> gf1.pcm_tail %= card -> gf1.pcm_blocks;
  pchn -> processed_bytes += card -> gf1.pcm_block_size;
  if ( !(--card -> gf1.pcm_used) )
    {
      if ( !(pchn -> flags & PCM_FLG_SYNC) )
        pchn -> discarded++;
      CLI( &flags );
      for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ )
        gus_gf1_smart_stop_voice( card, i );
      gf1_select_voice( card, voice );		/* return back voice */
      STI( &flags );
#if 0
      printk( "irq: pcm voices stopped...\n" );
#endif
      card -> gf1.pcm_pflags &= ~PFLG_ACTIVE;
      if ( !(pchn -> flags & PCM_FLG_MMAP) )
        {
          if ( !(card -> gf1.pcm_pflags & PFLG_DMA) && pchn -> used > 0 )
            gus_gf1_pcm_dma_transfer( card );	/* some data is waiting */
           else
            {
              if ( card -> gf1.pcm_pflags & PFLG_FLUSH )
                WAKEUP( card -> pcm_gf1, playback );
              card -> gf1.pcm_pflags = 0;	/* stopped */
            }
        }
       else
        if ( !(card -> gf1.pcm_pflags & PFLG_DMA) )
          gus_gf1_pcm_dma_transfer( card );
      return;
    }
  if ( card -> gf1.pcm_tail + 1 == card -> gf1.pcm_blocks )	/* last block? */
    voice_ctrl |= 0x08;			/* enable loop */
   else
    ramp_ctrl |= 0x04;			/* enable rollover */
  end = card -> gf1.pcm_mem + ( ( ( card -> gf1.pcm_tail + 1 ) * card -> gf1.pcm_block_size ) / GF1_PCM_RVOICES() );
  end -= voice_ctrl & 4 ? 2 : 1;
  step = pchn -> used_size / GF1_PCM_RVOICES();
  CLI( &flags );
  voice_ctrl |= 0x20;
  for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++, end += step )
    {
      gf1_select_voice( card, i );
      gus_write_addr( card, GF1_VA_END, end << 4, voice_ctrl & 4 );
      gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice_ctrl );
      gus_write8( card, GF1_VB_VOLUME_CONTROL, ramp_ctrl );
#if 0
      printk( "++ end=0x%x ctrl=0x%x ramp=0x%x\n", end, voice_ctrl, ramp_ctrl );
#endif
      voice_ctrl &= ~0x20;
    }
  if ( !card -> gf1.enh_mode )
    {
      gus_delay( card );
      voice_ctrl |= 0x20;
      for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ )
        {
          gf1_select_voice( card, i );
          gus_write8( card, GF1_VB_ADDRESS_CONTROL, voice_ctrl );
          gus_write8( card, GF1_VB_VOLUME_CONTROL, ramp_ctrl );
          voice_ctrl &= ~0x20;
        }
    }
  gf1_select_voice( card, voice );	/* return back voice */
  STI( &flags );
  if ( !(card -> gf1.pcm_pflags & PFLG_DMA) )
    gus_gf1_pcm_dma_transfer( card );
  if ( pchn -> flags & PCM_FLG_MMAP )
    {
      if ( ( card -> gf1.pcm_tail % card -> gf1.pcm_mmap_div ) == 0 )
        {
          pchn -> interrupts++;
          if ( pchn -> flags & PCM_FLG_SLEEP )
            {
              pchn -> flags &= ~PCM_FLG_SLEEP;
              WAKEUP( card -> pcm_gf1, playback );
            }
        }
    }
   else
    pchn -> interrupts++;
}

static void gus_interrupt_pcm_volume( gus_card_t *card, int voice )
{
  gus_pcm_channel_t *pchn;
  unsigned short vol;
  unsigned char pan;
  int i;

  /* stop ramp, but leave rollover bit untouched */
  gus_i_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
  /* are we active? */
  if ( !(card -> gf1.pcm_pflags & PFLG_ACTIVE) ) return;
  /* load real volume - better precision */
  i = voice - GF1_PCM_VOICE_MIN();
  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
  if ( pchn -> mode & PCM_MODE_MULTI )
    vol = gus_gf1_pcm_volume( card, card -> gf1.pcm_volume[ i ], pan = card -> gf1.pcm_pan[ i ] );
   else
    {
      if ( pchn -> voices == 2 ) pan = !i ? 16 : 128 - 16; else pan = 64;
      vol = gus_gf1_pcm_volume( card, 128, pan );
    }
  gus_i_write16( card, GF1_VW_VOLUME, vol << 4 );
}

static void gus_interrupt_pcm_dma_write( gus_card_t *card )
{
  gus_pcm_channel_t *pchn;

  gus_gf1_done_dma_transfer( card );		/* ACK */
#if 0
  {
    int i;
    unsigned long flags;
    unsigned char voice_ctrl;

    printk( "dma done!!!\n" );
    CLI( &flags );
    gf1_select_voice( card, GF1_PCM_VOICE_MIN() );
    voice_ctrl = gus_read8( card, GF1_VB_ADDRESS_CONTROL ) & ~0x8b; 
    for ( i = GF1_PCM_VOICE_MIN(); i <= GF1_PCM_VOICE_MAX(); i++ )
      {
        gf1_select_voice( card, i );
        PRINTK( "** curr=0x%x, begin=0x%x, end=0x%x, ctrl=0x%x, ramp=0x%x\n", gus_read_addr( card, GF1_VA_CURRENT, voice_ctrl & 4 ), gus_read_addr( card, GF1_VA_START, voice_ctrl & 4 ), gus_read_addr( card, GF1_VA_END, voice_ctrl & 4 ), gus_read8( card, GF1_VB_ADDRESS_CONTROL ), gus_read8( card, GF1_VB_VOLUME_CONTROL ) );
      }
    STI( &flags );
  }
#endif
  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
  if ( !(card -> gf1.pcm_pflags & PFLG_DMA) )
    {
#if 0
      PRINTD( "gus_gf1_pcm: unknown dma write interrupt!!!\n" );
#endif
      return;
    }
  gus_gf1_pcm_dma_transfer( card );
  if ( !(card -> gf1.pcm_pflags & (PFLG_DMA|PFLG_ACTIVE)) )
    gus_gf1_pcm_trigger_up( card );
}

static void gus_interrupt_pcm_dma_read( gus_card_t *card )
{
  gus_pcm_channel_t *pchn;

  pchn = &card -> pcm_gf1 -> chn[ PCM_RECORD ];
  gus_gf1_done_rdma_transfer( card );		/* ACK */
  if ( !( pchn -> flags & PCM_FLG_MMAP ) )
    {
      if ( pchn -> used < pchn -> blocks )
        pchn -> used++;
       else
        card -> gf1.pcm_record_overflow++;
      pchn -> sizes[ pchn -> head++ ] = pchn -> block_size;
      pchn -> head %= pchn -> blocks;
      pchn -> locks = 1 << pchn -> head;
    }
   else
    {
      pchn -> head++;
      pchn -> head %= pchn -> blocks;
    }
  pchn -> processed_bytes += pchn -> block_size;
  pchn -> interrupts++;
  gus_gf1_init_rdma_transfer(
    card,
    pchn -> buffer + ( pchn -> head * pchn -> block_size ),
    pchn -> block_size,
    card -> gf1.pcm_sampling_ctrl_reg );
  if ( pchn -> flags & PCM_FLG_SLEEP )
    {
      pchn -> flags &= ~PCM_FLG_SLEEP;
      WAKEUP( card -> pcm_gf1, record );
    }
}

static void gus_gf1_pcm_voices_change_start( gus_card_t *card )
{
  unsigned char voice_ctrl;
  
  if ( card -> gf1.pcm_pflags & PFLG_ACTIVE )
    {
      gf1_select_voice( card, GF1_PCM_VOICE_MIN() );
      voice_ctrl = gus_i_read8( card, GF1_VB_ADDRESS_CONTROL );
      card -> gf1.pcm_pos = ( gus_i_read_addr( card, GF1_VA_CURRENT, voice_ctrl & 4 ) >> 4 ) - card -> gf1.pcm_mem;
    }
  gus_gf1_stop_voices( card, GF1_PCM_VOICE_MIN(), GF1_PCM_VOICE_MAX() );
}

static void gus_gf1_pcm_voices_change_stop( gus_card_t *card )
{
  gus_gf1_clear_voices( card, GF1_PCM_VOICE_MIN(), GF1_PCM_VOICE_MAX() );
  if ( (card -> gf1.pcm_pflags & PFLG_ACTIVE) && card -> gf1.pcm_pos > 0 )
    {
      gus_gf1_pcm_trigger_up( card );
      card -> gf1.pcm_pos = 0;
    }
}

static void gus_gf1_pcm_volume_change( gus_card_t *card )
{
}

void gus_gf1_pcm_volume_update( gus_card_t *card )
{
  gus_pcm_channel_t *pchn;
  unsigned long flags;
  unsigned short vol;
  unsigned char pan;
  int i;

  /* are we active? */
  if ( !(card -> gf1.mode & GF1_MODE_PCM_PLAY) ) return;
  if ( !(card -> gf1.pcm_pflags & PFLG_ACTIVE) ) return;
  /* load real volume - better precision */
  pchn = &card -> pcm_gf1 -> chn[ PCM_PLAYBACK ];
  for ( i = 0; i < GF1_PCM_VOICES(); i++ )
    {
      CLI( &flags );
      gf1_select_voice( card, i + GF1_PCM_VOICE_MIN() );
      gus_i_ctrl_stop( card, GF1_VB_VOLUME_CONTROL );
      if ( pchn -> mode & PCM_MODE_MULTI )
        vol = gus_gf1_pcm_volume( card, card -> gf1.pcm_volume[ i ], pan = card -> gf1.pcm_pan[ i ] );
       else
        {
          if ( pchn -> voices == 2 ) pan = !i ? 8 : 128 - 9; else pan = 64;
          vol = gus_gf1_pcm_volume( card, 100, pan );
        }
      gus_i_write16( card, GF1_VW_VOLUME, vol << 4 );
      STI( &flags );
    }
}

/*
 *
 */

static int gf1_open_playback( gus_pcm_t *pcm )
{
  unsigned long flags;
  int pg;
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  if ( card -> gf1.mode & GF1_MODE_ENGINE )
    {
      int tmp;
      
      if ( !card -> gf1.pcm_memory ) return -EIO;
      CLI( &flags );
      tmp = card -> dmas[ GUS_DMA_GPLAY ] -> lock & WK_LOCK;
      GETLOCK( card, dma1_lock ) |= WK_SLEEP;
      while ( tmp && ( GETLOCK( card, dma1_lock ) & WK_SLEEP ) )
        {
          STI( &flags );
          SLEEP( card, dma1_lock, 60 * HZ );
          CLI( &flags );
        }
      GETLOCK( card, dma1_lock ) &= ~WK_SLEEP;
      card -> dmas[ GUS_DMA_GPLAY ] -> lock = 0;
    }
   else
    {
      if ( card -> dmas[ GUS_DMA_GPLAY ] -> lock & WK_LOCK )
        return -EBUSY;
    }
  if ( gus_dma_malloc( card, GUS_DMA_GPLAY, "GF1 PCM - playback", 1 ) < 0 )
    return -ENOMEM;
  if ( gus_memory_manager_pcm_alloc( card, 1 ) < 0 ) {
    gus_dma_free( card, GUS_DMA_GPLAY, 1 );
    return -ENOMEM;
  }
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  gus_pcm_set_dma_struct( card, GUS_DMA_GPLAY, pchn );
  pchn -> hidden_buffer = gus_malloc_pages( PAGE_SIZE, &pg, 1 );
  if ( !pchn -> hidden_buffer )
    {
      gus_dma_free( card, GUS_DMA_GPLAY, 1 );
      gus_memory_manager_pcm_free( card, 1 );
      return -ENOMEM;
    }
  gus_gf1_open( card, GF1_MODE_PCM_PLAY );
  card -> gf1.pcm_pflags = PFLG_NONE;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].interrupt_handler_wave = gus_interrupt_pcm_wave;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].interrupt_handler_volume = gus_interrupt_pcm_volume;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].voices_change_start = gus_gf1_pcm_voices_change_start;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].voices_change_stop = gus_gf1_pcm_voices_change_stop;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].volume_change = gus_gf1_pcm_volume_change;
  card -> gf1.interrupt_handler_pcm_dma_write = gus_interrupt_pcm_dma_write;  
  card -> gf1.pcm_pos = 0;		/* nothing */
  return 0;
}

static int gf1_open_record( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( gus_dma_malloc( card, GUS_DMA_GRECORD, "GF1 PCM - record", 1 ) < 0 )
    return -ENOMEM;
  gus_pcm_set_dma_struct( card, GUS_DMA_GRECORD, &pcm -> chn[ PCM_RECORD ] );
  gus_gf1_open( card, GF1_MODE_PCM_RECORD );
  card -> gf1.pcm_rflags = RFLG_NONE;
  card -> gf1.interrupt_handler_pcm_dma_read = gus_interrupt_pcm_dma_read;
  return 0;
}

static void gf1_close_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  gus_gf1_clear_voices( card, GF1_PCM_VOICE_MIN(), GF1_PCM_VOICE_MAX() );
  card -> gf1.pcm_pflags = PFLG_NONE;
  if ( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices > 0 )
    {
      card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices = 0;
      gus_reselect_active_voices( card );
    }
  gus_free_pages( pcm -> chn[ PCM_PLAYBACK ].hidden_buffer, PAGE_SIZE );
  gus_dma_free( card, GUS_DMA_GPLAY, 1 );
  gus_gf1_close( card, GF1_MODE_PCM_PLAY );
  gus_set_default_handlers( card, GF1_HANDLER_PCM_DMA_WRITE | GF1_HANDLER_RANGE | GF1_VOICE_RANGE_PCM );
  gus_memory_manager_pcm_free( card, 1 );
}

static void gf1_close_record( gus_pcm_t *pcm )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( card -> gf1.pcm_rflags & RFLG_USED )
    gus_gf1_done_rdma_transfer( card );		/* for sure */    
  card -> gf1.pcm_rflags = RFLG_NONE;
  gus_dma_free( card, GUS_DMA_GRECORD, 1 );
  gus_gf1_close( card, GF1_MODE_PCM_RECORD );
  gus_set_default_handlers( card, GF1_HANDLER_PCM_DMA_READ );
}

static void gf1_init_playback( gus_pcm_t *pcm )
{
  int i, j, k;
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( !( pchn -> flags & PCM_FLG_MMAP ) && !pchn -> used ) return;
  if ( !( card -> gf1.pcm_pflags & PFLG_USED ) )
    {
      if ( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices != pchn -> voices )	/* ok.. some change */
        {
          card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices = pchn -> voices;
          gus_reselect_active_voices( card );
        }
      card -> gf1.pcm_head = card -> gf1.pcm_tail = card -> gf1.pcm_used = 0;
      if ( !( pchn -> flags & PCM_FLG_MMAP ) )
        {
          card -> gf1.pcm_blocks = pchn -> blocks;
          card -> gf1.pcm_block_size = pchn -> block_size;
        }
       else
        {
          if ( pchn -> rate <= 11025 ) j = 128; else
          if ( pchn -> rate <= 22050 ) j = 256; else
                                       j = 512;
          if ( pchn -> mode & PCM_MODE_16 ) j <<= 1;
          j *= pchn -> voices;
          i = pchn -> block_size;
          k = 1;
          while ( j <= i ) { k <<= 1; i >>= 1; }
          k >>= 1; i <<= 1;
          if ( i > PAGE_SIZE )
            {
              PRINTK( "gus: gf1 pcm - MMAP block size overflow\n" );
              return;
            }
          card -> gf1.pcm_mmap_div = k;
          card -> gf1.pcm_block_size = i;
#if 0
          card -> gf1.pcm_block_size = 0x1000;
#endif
          card -> gf1.pcm_blocks = pchn -> used_size / card -> gf1.pcm_block_size;
#if 0
          printk( "used_size = 0x%x, size = 0x%x, block_size = 0x%x, blocks = %i, div = %i\n", pchn -> used_size, pchn -> size, card -> gf1.pcm_block_size, card -> gf1.pcm_blocks, card -> gf1.pcm_mmap_div );
#endif
        }
      card -> gf1.pcm_pflags = PFLG_USED;
      gus_gf1_pcm_dma_transfer( card );
      pchn -> flags |= PCM_FLG_TRIGGER;
    }
}

static void gf1_done_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( card -> gf1.pcm_pflags & PFLG_USED )
    {
      if ( pchn -> flags & PCM_FLG_MMAP )
        gus_gf1_stop_voices( card, GF1_PCM_VOICE_MIN(), GF1_PCM_VOICE_MAX() );
       else
        {
          card -> gf1.pcm_pflags |= PFLG_FLUSH;
          while ( card -> gf1.pcm_pflags & (PFLG_DMA|PFLG_ACTIVE) )
            {
              SLEEP( card -> pcm_gf1, playback, HZ * 120 );
              if ( card -> gf1.pcm_pflags & (PFLG_DMA|PFLG_ACTIVE) )
                { 
                  if ( TABORT( card -> pcm_gf1, playback ) ) 
                    pchn -> flags |= PCM_FLG_ABORT;
                   else
                  if ( TIMEOUT( card -> pcm_gf1, playback ) )
                    PRINTK( "pcm: flush failed (playback)\n" );
                   else
                    continue;
                  gus_gf1_stop_voices( card, GF1_PCM_VOICE_MIN(), GF1_PCM_VOICE_MAX() );
                  card -> gf1.pcm_pflags = PFLG_NONE;
                }
            }
        }
      pchn -> flags &= ~PCM_FLG_TRIGGER;
      card -> gf1.pcm_pflags = PFLG_NONE;
    }
}

static void gf1_init_record( gus_pcm_t *pcm )
{
  unsigned int tmp;
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  if ( ( card -> gf1.pcm_rflags & RFLG_USED ) == 0 )
    {
      pchn = &pcm -> chn[ PCM_RECORD ];
      pchn -> head = pchn -> tail = pchn -> used = 0;
      if ( card -> version == GUS_CARD_VERSION_ACE )
        {
          PRINTK( "gus: GUS ACE can't record PCM data\n" );
          return;
        }
#ifdef GUSCFG_PNP
      if ( card -> pnp_flag )
        {
          PRINTK( "gus: InterWave doesn't have original GUS sampling emulation - aborting...\n" );
          return;
        }
#endif
      pchn -> flags |= PCM_FLG_TRIGGER;
      card -> gf1.pcm_rflags = RFLG_USED;
      tmp = ((9878400L>>4)/pchn -> rate)-2;
      gus_i_write8( card, 0x48, tmp );	/* set sampling frequency */
      card -> gf1.pcm_sampling_ctrl_reg = 0x21;	/* IRQ at end, enable & start */
      if ( pchn -> voices == 2 )
        card -> gf1.pcm_sampling_ctrl_reg |= 2;
      if ( card -> dmas[ GUS_DMA_GRECORD ] -> dma > 3 )
        card -> gf1.pcm_sampling_ctrl_reg |= 4;
      if ( pchn -> mode & PCM_MODE_U )
        card -> gf1.pcm_sampling_ctrl_reg |= 0x80;
      pchn -> locks = 1;
      gus_gf1_init_rdma_transfer( card,
      			      pchn -> buffer + ( pchn -> head * pchn -> block_size ),
                              pchn -> block_size,
                              card -> gf1.pcm_sampling_ctrl_reg );
    }
}

static void gf1_done_record( gus_pcm_t *pcm )
{
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_RECORD ];
  if ( card -> gf1.pcm_rflags & RFLG_USED )
    {
      gus_gf1_done_rdma_transfer( card );	/* for sure */
      card -> gf1.pcm_rflags = RFLG_NONE;
      pchn -> flags &= ~PCM_FLG_TRIGGER;
    }
}

/*
 * ------------------------------------------------------------------------
 */
 
static void gf1_dma_playback( gus_pcm_t *pcm, gus_pcm_channel_t *pchn, char *buffer, unsigned int count )
{
  gus_card_t *card;

  card = pcm -> card;
  if ( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices != pchn -> voices )	/* ok.. some change */
    {
      card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices = pchn -> voices;
      gus_reselect_active_voices( card );
    }
  gus_gf1_pcm_translate( card, 
  			 buffer,
  			 pchn -> buffer,
  			 ( pchn -> head * pchn -> block_size ) + pchn -> sizes[ pchn -> head ],
  			 count,
  			 pchn -> used_size,
  			 pchn -> mode & PCM_MODE_16,
  			 pchn -> mode & PCM_MODE_ULAW,
  			 SP_USER );
}

static void gf1_dma_playback_neutral( gus_pcm_t *pcm, gus_pcm_channel_t *pchn )
{
  gus_card_t *card;
  unsigned int size, count;

  card = pcm -> card;
  if ( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices != pchn -> voices )	/* ok.. some change */
    {
      card -> gf1.voice_ranges[ GF1_VOICE_RANGE_PCM ].rvoices = pchn -> voices;
      gus_reselect_active_voices( card );
    }
  size = pchn -> sizes[ pchn -> head ];
  count = pchn -> block_size - size;
  if ( !count ) return;
  gus_gf1_pcm_translate( card, 
  			 NULL,
  			 pchn -> buffer,
  			 ( pchn -> head * pchn -> block_size ) + size,
  			 count,
  			 pchn -> used_size,
  			 pchn -> mode & PCM_MODE_16,
  			 pchn -> mode & PCM_MODE_ULAW,
  			 pchn -> neutral_byte );  
}

static void gf1_dma_record( gus_pcm_t *pcm, gus_pcm_channel_t *pchn, char *buffer, unsigned int count )
{
  char *src_buf;

  src_buf = pchn -> buffer + ( pchn -> tail * pchn -> block_size ) + ( pchn -> block_size - pchn -> sizes[ pchn -> tail ] );
  if ( pchn -> mode & PCM_MODE_ULAW ) {
    gus_translate_memcpy_tofs( gus_dsp_ulaw, buffer, src_buf, count );
  } else {
    MEMCPY_TOFS( buffer, src_buf, count );
  }
}

static void gf1_sync_playback( gus_pcm_t *pcm )
{
  gf1_init_playback( pcm );		/* for sure */
  gf1_done_playback( pcm );
}

static void gf1_sync_record( gus_pcm_t *pcm )
{
  gf1_done_record( pcm );
}

static unsigned int gf1_pointer_playback( gus_pcm_t *pcm )
{
  gus_card_t *card;
  unsigned int res = 0;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  if ( ( pchn -> flags & PCM_FLG_TRIGGER ) == PCM_FLG_TRIGGER )
    res = card -> gf1.pcm_head * card -> gf1.pcm_block_size;
  return res;
}

static unsigned int gf1_pointer_record( gus_pcm_t *pcm )
{
  gus_card_t *card;
  unsigned int res = 0;
  gus_pcm_channel_t *pchn;
  
  card = pcm -> card;
  pchn = &pcm -> chn[ PCM_RECORD ];
  if ( ( pchn -> flags & PCM_FLG_TRIGGER ) == PCM_FLG_TRIGGER )
    res = ( pchn -> head * pchn -> block_size ) +
          ( pchn -> block_size - get_dma_residue( card -> dmas[ GUS_DMA_GRECORD ] -> dma ) );
  return res;
}

/*
 *  Init routine
 */
  
void gus_init_gf1_pcm( gus_pcm_t *pcm )
{
  int pnp_flag;
  gus_card_t *card;
  gus_pcm_channel_t *pchn;

  card = pcm -> card;
  pnp_flag = card -> pnp_flag;
  card -> pcm_gf1 = pcm;

  card -> gf1.pcm_pflags = PFLG_NONE;
  card -> gf1.pcm_rflags = RFLG_NONE;

  pcm -> flags |= PCM_LFLG_USED;
  pcm -> info_flags = GUS_PCM_INFO_PLAYBACK | GUS_PCM_INFO_MMAP |
                      GUS_PCM_INFO_PLAYBACK_BATCH;
  if ( !pnp_flag && !card -> ess_flag )	/* plain GF1 chip - we can record */
    pcm -> info_flags |= GUS_PCM_INFO_RECORD | GUS_PCM_INFO_DUPLEX;
  pcm -> info_name = pnp_flag ? "GFA1" : "GF1";

  pchn = &pcm -> chn[ PCM_PLAYBACK ];
  pchn -> formats = AFMT_MU_LAW | AFMT_U8 | AFMT_S8 | AFMT_S16_LE | AFMT_U16_LE;
  pchn -> min_fragment = 9;
  pchn -> max_rate = 44100;
  pchn -> max_voices = 32;
  pchn -> hw_open = gf1_open_playback;
  pchn -> hw_close = gf1_close_playback;
  pchn -> hw_init = gf1_init_playback;
  pchn -> hw_done = gf1_done_playback;
  pchn -> hw_dma = gf1_dma_playback;
  pchn -> hw_dma_neutral = gf1_dma_playback_neutral;
  pchn -> hw_sync = gf1_sync_playback;
  pchn -> hw_pointer = gf1_pointer_playback;

  pchn++;
  pchn -> formats = !pnp_flag ? (AFMT_MU_LAW | AFMT_U8 | AFMT_S8) : 0;
  pchn -> min_fragment = 4;
  pchn -> max_rate = !pnp_flag ? 44100 : 0;
  pchn -> max_voices = !pnp_flag ? 2 : 0;
  pchn -> hw_open = gf1_open_record;
  pchn -> hw_close = gf1_close_record;
  pchn -> hw_init = gf1_init_record;
  pchn -> hw_done = gf1_done_record;
  pchn -> hw_dma = gf1_dma_record;
  pchn -> hw_dma_neutral = NULL;
  pchn -> hw_sync = gf1_sync_record;
  pchn -> hw_pointer = gf1_pointer_record;
}

#endif /* GUSCFG_GF1PCM */
