/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Sequencer routines (/dev/sequencer and /dev/music devices)
 *  OSS lite 3.XX compatible - interface is copyrighted by Hannu Savolainen
 */

#include "driver.h"
#include "midi.h"

#ifdef GUSCFG_OSS

#if 0
#define OSS_SEQUENCER_DEBUG
#endif
#if 0
#define V_DEBUG
#endif

#ifdef GUSCFG_DEBUG
#if 0
#define DEBUG_EVENT_ALL( n, e ) seq_print_event( n, e, 0, -1 )
#undef USE_PRINT_EVENT
#define USE_PRINT_EVENT
#endif
#if 0
#define DEBUG_EVENT( n, e, p ) seq_print_event( n, e, p, 1 )
#undef USE_PRINT_EVENT
#define USE_PRINT_EVENT
#endif
#endif /* GUSCFG_DEBUG */

#ifndef DEBUG_EVENT_ALL
#define DEBUG_EVENT_ALL( n, e ) /* nothing */
#endif
#ifndef DEBUG_EVENT
#define DEBUG_EVENT( n, e, p ) /* nothing */
#endif

#define OSS_VERSION	0x0307f0
#ifndef OSS_GETVERSION
#define OSS_GETVERSION _IOR('M', 118, int)
#endif
#ifndef SNDCTL_SEQ_GETTIME
#define SNDCTL_SEQ_GETTIME _IOR('Q',19,int)
#endif

/*
 *
 */

#define SEQ_MODE_NONE	0
#define SEQ_MODE_1	1
#define SEQ_MODE_2	2

#define SEQ_DEV_CNT	3
#define SEQ_DEV_MIDI	0		/* MIDI port */
#define SEQ_DEV_GF1	1		/* GF1 chip */
#define SEQ_DEV_EMUL	2		/* buildin GF1 MIDI emulation */

#define SEQ_OQUEUE_LEN	1024
#define SEQ_IQUEUE_LEN	1024

#define SEQ_MAX_PATCH	256
#define SEQ_MAX_PGM	256

#define EVENT_SIZE	8

#define GF1_VOL_MODE_NONE	0
#define GF1_VOL_MODE_ADAGIO	1
#define GF1_VOL_MODE_LINEAR	2

#define SEQ_LFLG_WRITE	0x0001
#define SEQ_LFLG_READ	0x0002
#define SEQ_LFLG_BOTH	(SEQ_LFLG_WRITE|SEQ_LFLG_READ)

#define GUS_TIMEBASE( base )	( ( base * 5 ) / 3 )

/*
 *
 */

static struct synth_info gus_uss_info[ SEQ_DEV_CNT ] = 
{
  {
    "GUS MIDI interface", 
    0, 
    SYNTH_TYPE_MIDI,
    0,
    0,
    128,
    0,
    128,
    SYNTH_CAP_INPUT
  },
  {
    "Gravis UltraSound",
    0,
    SYNTH_TYPE_SAMPLE,
    SAMPLE_TYPE_GUS,
    0,
    16,
    0,
    SEQ_MAX_PATCH
  },
  {
    "GUS MIDI emulation", 
    0, 
    SYNTH_TYPE_MIDI,
    0,
    0,
    128,
    0,
    128,
    0
  }
};

static struct midi_info gus_uss_midi_info[ SEQ_DEV_CNT ] = 
{
  {
    "GUS UART", 
    0, 
    0,
    SNDCARD_UART6850,
  },
  {
    "- Nothing -",
    0,
    0,
    -1
  },
  {
    "GUS MIDI emulation",
    0, 
    0,
    SNDCARD_GUS,
  }
};

static int seq_mode = SEQ_MODE_NONE;
static unsigned short seq_flags = 0;
static int seq_abort;
static int seq_curr_tempo;
static int seq_curr_timebase;
static unsigned int seq_abs_tick;
static unsigned int seq_abs_tick_input;
static unsigned int seq_abs_tick_input_old;
static unsigned int seq_time;
static int seq_playing;
static int seq_flush;
static int seq_stop_flag;
static int seq_out_of_band;

static unsigned char *seq_i_queue = NULL;
static unsigned int seq_i_used = 0;
static unsigned int seq_i_head = 0;
static unsigned int seq_i_tail = 0;
static unsigned char *seq_o_queue = NULL;
static unsigned int seq_o_used = 0;
static unsigned int seq_o_head = 0;
static unsigned int seq_o_tail = 0;
static unsigned int seq_o_threshold = 0;

/*
 *
 */

void gus_sequencer_init( gus_card_t *card )
{
  card -> seq.vol_mode = GF1_VOL_MODE_NONE;
  card -> seq.iw_effect[ 0 ] = 4;
  card -> seq.iw_effect[ 1 ] = 2;
}

/*
 *
 */

#ifdef USE_PRINT_EVENT
static void seq_print_event( char *name, unsigned char *event, int pos, int voice )
{
  if ( voice < 0 || ( event[ pos ] % 32 ) == voice )
    PRINTK( "[%s] = %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
    			name,
  			event[ 0 ], event[ 1 ], event[ 2 ], event[ 3 ],
  			event[ 4 ], event[ 5 ], event[ 6 ], event[ 7 ] );
}
#endif

static unsigned short seq_file_flags( unsigned short f_flags )
{
  switch ( f_flags & O_ACCMODE ) {
    case O_WRONLY:	return SEQ_LFLG_WRITE;
    case O_RDONLY:	return SEQ_LFLG_READ;
    default:		return SEQ_LFLG_BOTH;
  }
}

static void gus_reset_seq_voices( gus_card_t *card )
{
  short i;
  gus_seq_voice_t *pvoice;
  
  pvoice = card -> seq.voices;
  for ( i = 0; i < 32; i++, pvoice++ )
    {
      pvoice -> flags = 0;
      pvoice -> pgm = -1;
      pvoice -> bender = 0;
      pvoice -> bender_range = 2 * 128;
      pvoice -> midi_main_vol = 127;
      pvoice -> midi_volume = 127;
      pvoice -> midi_expression_vol = 127;
      pvoice -> midi_patch_vol = 127;
      pvoice -> panning = 0;
      pvoice -> volume = 0;
      pvoice -> instrument = NULL;
    }
}

static gus_voice_t *get_voice( gus_card_t *card, int voice )
{
  return card -> gf1.syn_voices + ( voice % 32 );
}

static gus_seq_voice_t *get_seq_voice( gus_card_t *card, int voice )
{
  return card -> seq.voices + ( voice % 32 );
}

static void gus_preload_general_midi_bank( gus_card_t *card )
{
  if ( !(card -> seq.flags & GUS_SEQ_F_MIDI_EMUL) )
    {
      if ( !gus_intr_count )
        {
          if ( seq_playing ) gus_timer_ioctl( card, GUS_IOCTL_TIMER_STOP, 0 );
          if ( card -> gf1.default_midi_emul != card -> gf1.midi_emul )
            {
              card -> gf1.midi_emul = card -> gf1.default_midi_emul;
              gus_gf1_daemon_midi_emul_change( card );
            }
          gus_gf1_daemon_preload( card, "ALL" );
          if ( card -> gf1.enh_mode && ( card -> seq.iw_effect[ 0 ] >= 0 || card -> seq.iw_effect[ 1 ] >= 0 ) )
            gus_gf1_daemon_iw_effect( card );
          if ( seq_playing ) gus_timer_ioctl( card, GUS_IOCTL_TIMER_CONTINUE, 0 );
        }
       else
        PRINTK( "gus: sequencer - preload in interrupt - aborting...\n" );
      card -> seq.flags |= GUS_SEQ_F_MIDI_EMUL;
    }
}

static void gus_put_seq_input_event( unsigned char *event, int count )
{
  if ( seq_i_used < SEQ_IQUEUE_LEN )
    {
#if 0
      {
        unsigned char *ptr;
        int count1 = count;
        
        ptr = event;
        PRINTK( "in event: " );
        while ( count1-- )
          PRINTK( "%02x:", *ptr++ );
        PRINTK( "\n" );
      }
#endif
      MEMCPY( seq_i_queue + ( seq_i_head++ * 8 ), event, count );
      seq_i_head %= SEQ_IQUEUE_LEN;
      seq_i_used++;
      if ( GETLOCK( gus_cards[ 0 ], seq_in ) & WK_SLEEP )
        {
          GETLOCK( gus_cards[ 0 ], seq_in ) &= ~WK_SLEEP;
          WAKEUP( gus_cards[ 0 ], seq_in );
        }
    }
}

/*
 *  patch load/convert routine - grrr...
 */

static int gus_patch_load( struct patch_info *fs_patch )
{
  struct patch_info patch;
  gus_instrument_t instr;
  gus_layer_t layer;
  gus_wave_t wave;
  gus_card_t *card;

  if ( VERIFY_AREA( VERIFY_READ, fs_patch, sizeof( *fs_patch ) ) ) 
    return -ENXIO;
  MEMCPY_FROMFS( &patch, fs_patch, sizeof( patch ) );
  if ( patch.device_no >= SEQ_DEV_CNT * gus_cards_count ) return -ENXIO;
  if ( patch.key != GUS_PATCH || ( patch.device_no % SEQ_DEV_CNT ) != SEQ_DEV_GF1 )
    return -ENXIO;
  card = gus_cards[ patch.device_no / SEQ_DEV_CNT ];
  MEMSET( &instr, 0, sizeof( instr ) );
  MEMSET( &layer, 0, sizeof( layer ) );
  MEMSET( &wave, 0, sizeof( wave ) );
  instr.name = "OSS GUS PATCH";
  instr.info.layer = &layer;
  layer.wave = &wave;
  instr.mode = layer.mode = wave.mode = GUS_INSTR_PATCH;
  instr.number.instrument = patch.instr_no;
  wave.begin.ptr = fs_patch -> data;
  if ( patch.mode & WAVE_16_BITS ) wave.format |= GUS_WAVE_16BIT;
  if ( patch.mode & WAVE_UNSIGNED ) wave.format |= GUS_WAVE_INVERT;
  if ( patch.mode & WAVE_LOOPING ) wave.format |= GUS_WAVE_LOOP;
  if ( patch.mode & WAVE_BIDIR_LOOP ) wave.format |= GUS_WAVE_BIDIR;
  if ( patch.mode & WAVE_LOOP_BACK ) wave.format |= GUS_WAVE_BACKWARD;
  if ( patch.mode & WAVE_ENVELOPES )
    {
      int i, same, old;
    
      wave.data.patch.flags |= GUS_WAVE_PATCH_ENVELOPE;
      if ( patch.mode & WAVE_SUSTAIN_ON )
        wave.data.patch.flags |= GUS_WAVE_PATCH_SUSTAIN;
      same = 0;
      old = patch.env_offset[ 0 ];
      for ( i = 0; i < 6; i++ )
        {
          wave.data.patch.envelope_offset[ i ] = patch.env_offset[ i ];
          wave.data.patch.envelope_rate[ i ] = patch.env_rate[ i ];
          if ( old == patch.env_offset[ i ] ) same++;
        }
      if ( same == 6 )		/* ok.. ProPatches set have this ugly patches.. */
        {
          /* attack */
          wave.data.patch.envelope_rate[ 0 ] = 20;
          /* final release */
          wave.data.patch.envelope_rate[ 5 ] = 40;
          wave.data.patch.flags |= GUS_WAVE_PATCH_SUSTAIN;
          wave.format &= ~GUS_WAVE_LOOP;
        }
    }
  if ( patch.loop_start > patch.len ) patch.loop_start = 0;
  if ( patch.loop_end > patch.len ) patch.loop_end = patch.len - 1;
  wave.size = patch.len;
  wave.loop_start = patch.loop_start << 4;
#if 0
  printk( "patch - loop start = 0x%x, loop end = 0x%x\n", patch.loop_start, patch.loop_end );
#endif
  if ( patch.loop_end < 2 )
    {
      /* ok.. this come from some .mod player */
      wave.loop_end = patch.len << 4;
      wave.loop_end -= wave.format & GUS_WAVE_16BIT ? 40 : 24;
    }
   else
    wave.loop_end = patch.loop_end << 4;
  wave.data.patch.sample_rate = patch.base_freq;
  wave.data.patch.root_frequency = patch.base_note;
  wave.data.patch.high_frequency = patch.high_note;
  wave.data.patch.low_frequency = patch.low_note;
  wave.data.patch.balance = patch.panning + 128;
#if 0
  printk( "patch.panning = %i, balance = %i\n", patch.panning, wave.data.patch.balance );
#endif
  if ( patch.mode & WAVE_VIBRATO )
    {
      wave.data.patch.vibrato_sweep = patch.vibrato_sweep;
      wave.data.patch.vibrato_depth = patch.vibrato_depth;
      wave.data.patch.vibrato_rate = patch.vibrato_rate;
    }
   else
    {
      wave.data.patch.vibrato_sweep =
      wave.data.patch.vibrato_rate =
      wave.data.patch.vibrato_depth = 0;
    }
  if ( patch.mode & WAVE_TREMOLO )
    {
      wave.data.patch.tremolo_sweep = patch.tremolo_sweep;
      wave.data.patch.tremolo_depth = patch.tremolo_depth;
      wave.data.patch.tremolo_rate = patch.tremolo_rate;
    }
   else
    wave.data.patch.tremolo_sweep =
    wave.data.patch.tremolo_rate =
    wave.data.patch.tremolo_depth = 0;
  if ( patch.mode & WAVE_SCALE )
    {
      wave.data.patch.scale_frequency = patch.scale_frequency;
      wave.data.patch.scale_factor = patch.scale_factor;
    }
   else
    {
      wave.data.patch.scale_frequency = 0;
      wave.data.patch.scale_factor = 1024;
    }
  return gus_memory_alloc( card, &card -> gf1.mem_alloc, &instr, MTST_NONE, SP_KERNEL_WAVE_USER );
}

static int gus_sysex_load( struct sysex_info *fs_sysex )
{
  struct sysex_info sysex;
  unsigned char *data;
  gus_card_t *card;
  
  if ( seq_mode != SEQ_MODE_2 ) return -ENODEV;
  if ( VERIFY_AREA( VERIFY_READ, fs_sysex, 2 + 2 + 4 ) ) 
    return -ENXIO;
  MEMCPY_FROMFS( &sysex, fs_sysex, 2 + 2 + 4 );
  if ( sysex.device_no >= SEQ_DEV_CNT * gus_cards_count ) return -ENXIO;
  if ( sysex.key != SYSEX_PATCH ||
       ( ( ( sysex.device_no % SEQ_DEV_CNT ) != SEQ_DEV_MIDI ) &&
         ( ( sysex.device_no % SEQ_DEV_CNT ) != SEQ_DEV_EMUL ) ) )
    return -ENXIO;
  card = gus_cards[ sysex.device_no / SEQ_DEV_CNT ];
  data = gus_malloc( sysex.len );
  if ( !data ) return -ENOMEM;
  MEMCPY_FROMFS( data, (char *)fs_sysex + 2 + 2 + 4, sysex.len );
#if 0
  {
    int i = sysex.len;
    unsigned char *ptr = data;
    
    while ( i-- > 0 )
      printk( "%02x:", *ptr++ );
    printk( "\n" );
  }
#endif
  if ( data[ 0 ] != 0xf0 || data[ sysex.len - 1 ] != 0xf7 )
    {
      gus_free( data, sysex.len );
      return -EINVAL;
    }
  card -> midi[ ( sysex.device_no % SEQ_DEV_CNT ) == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, data, sysex.len );
  gus_free( data, sysex.len );
  return 0;
}

static unsigned short get_pitchbend( gus_seq_voice_t *pvoice )
{
  int bender, sensitivity;

  if ( !pvoice -> bender ) return 1024;
  if ( pvoice -> bender < -8192 ) pvoice -> bender = 8192;
  if ( pvoice -> bender > 8191 ) pvoice -> bender = 8191;
  bender = pvoice -> bender + 8192;
  sensitivity = ( pvoice -> bender_range * 128 ) / 100;
#if 0
  printk( "bender = %i, range = %i, gus bender = %i, sensitivity = %i\n", bender, pvoice -> bender_range, gus_compute_pitchbend( bender, sensitivity ), sensitivity );
#endif  
  return gus_compute_pitchbend( bender, sensitivity );
}

static inline unsigned short gus_adagio_vol( gus_card_t *card, int vel, int mainv, int xpn, int voicev )
{
  int x;
  
  x = 256 + 6 * ( voicev - 64 );
  if ( voicev > 65 ) xpn += voicev - 64;
  xpn += ( voicev - 64 ) / 2;
  x = vel * xpn * 6 + ( voicev / 4 ) * x;
#if 1
  x /= 10;	/* perex */
#endif
#if 0
  PRINTK( "x = %d\n", x );
#endif
  return x >> 2;
}

static unsigned short get_volume( gus_card_t *card, short voice )
{
  unsigned short volume = 0;
  gus_seq_voice_t *pvoice;

  pvoice = get_seq_voice( card, voice );
  if ( pvoice -> midi_volume > 127 ) pvoice -> midi_volume = 127;
  switch ( card -> seq.vol_mode ) {
    case GF1_VOL_MODE_NONE:
      volume = card -> seq.volume_base + ( pvoice -> midi_volume * card -> seq.volume_scale );
      break;
    case GF1_VOL_MODE_ADAGIO:
      volume = gus_adagio_vol( card,
      			       pvoice -> midi_volume,
                               pvoice -> midi_main_vol,
                               pvoice -> midi_expression_vol,
                               pvoice -> midi_patch_vol );
      break;
    case GF1_VOL_MODE_LINEAR:
#if 0
      printk( "expression = %i, midi = %i, main = %i\n",
      			pvoice -> midi_expression_vol,
      			pvoice -> midi_volume,
      			pvoice -> midi_main_vol );
#endif
      volume = ( pvoice -> midi_expression_vol * pvoice -> midi_volume * pvoice -> midi_main_vol ) / 127;
      break;
    default:
      PRINTK( "get_volume: unknown volume mode\n" );
      return 0;
  }
  return volume;
}

static void gus_sequencer_volume_change( gus_card_t *card, short voice )
{
  gus_voice_t *cvoice;
  gus_seq_voice_t *pvoice;
  
  cvoice = get_voice( card, voice );
  pvoice = get_seq_voice( card, voice );
  pvoice -> volume = get_volume( card, voice );
  if ( cvoice -> commands )
    cvoice -> commands -> voice_volume( card, cvoice, pvoice -> volume > GUS_VOLUME_DATA ? GUS_VOLUME_DATA : pvoice -> volume );
#if 0
  printk( "volume change - voice = %i, new_vol = %i\n", voice, pvoice -> volume );
#endif
}

static void gus_sequencer_start_voices( void )
{
  int i, j;
  gus_card_t *card;
  gus_voice_t *cvoice;
  gus_seq_voice_t *pvoice;
  
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      pvoice = card -> seq.voices;
      for ( j = 0; j < 32; j++, pvoice++ )
        if ( pvoice -> flags & SEQFLG_GO )
          {
            cvoice = &card -> gf1.syn_voices[ j ];
            if ( cvoice -> commands )
              {
                cvoice -> commands -> voice_go( card, cvoice );
#if 0
                gus_print_voice_registers( card );
#endif
              }
            pvoice -> flags &= ~SEQFLG_GO;
          }
    }
}

static void gus_sequencer_note_on( gus_card_t *card, short voice, short note )
{
  gus_voice_t *cvoice;
  gus_seq_voice_t *pvoice;
  
#if 0
  if ( voice != 1 ) return;
#endif
  if ( note == 255 )
    {
      gus_sequencer_volume_change( card, voice );
      return;
    }
  cvoice = get_voice( card, voice );
  pvoice = get_seq_voice( card, voice );
#if 0
  printk( "note on - voice = %i, note = %i, instrument = %i\n", voice, note, cvoice -> instrument -> number.instrument );
#endif
  if ( cvoice -> instrument && cvoice -> commands )
    {
      cvoice -> flags &= ~VFLG_DYNAMIC;
      cvoice -> flags |= VFLG_OSS_PAN;
      cvoice -> commands -> voice_start( card, cvoice );
      cvoice -> instrument = pvoice -> instrument;
      cvoice -> commands -> voice_freq( card, cvoice, GUS_FREQ_OSSNOTE | note );
      cvoice -> commands -> voice_freq( card, cvoice, GUS_FREQ_PITCHBEND | get_pitchbend( pvoice ) );
      gus_sequencer_volume_change( card, voice );
      cvoice -> commands -> voice_pan( card, cvoice, pvoice -> panning );
      if ( seq_out_of_band )
        {
          cvoice -> commands -> voice_go( card, cvoice );
          pvoice -> flags &= ~SEQFLG_GO;
        }
       else
        pvoice -> flags |= SEQFLG_GO;
    }
}

static void gus_sequencer_note_off( gus_card_t *card, short voice )
{
  gus_voice_t *cvoice;
  gus_seq_voice_t *pvoice;
  
#if 0
  if ( voice != 1 ) return;
#endif
#if 0
  printk( "note off - voice = %i\n", voice );
#endif
  pvoice = get_seq_voice( card, voice );
  cvoice = get_voice( card, voice );
  if ( pvoice -> flags & SEQFLG_GO )
    pvoice -> flags &= ~SEQFLG_GO;
  if ( cvoice -> commands )
    cvoice -> commands -> voice_stop( card, cvoice, 0x02 );
}

static void gus_sequencer_pgm_change( gus_card_t *card, short voice, int pgm )
{
  gus_voice_t *cvoice;
  gus_seq_voice_t *pvoice;
        
  cvoice = get_voice( card, voice );
  pvoice = get_seq_voice( card, voice );
  gus_engine_voice_program( card, cvoice, pgm );
  pvoice -> instrument = cvoice -> instrument;
#if 0
  printk( "pgm change - voice = %i, pgm = %i, instrument = 0x%x\n", voice, pgm, cvoice -> instrument );
#endif  
}

/*
 *
 */

static void gus_do_controller_event( gus_card_t *card, short dev, short voice, short cmd, short value )
{
  if ( seq_mode != SEQ_MODE_1 ) return;		/* this routine is for mode1 only */
  if ( dev != SEQ_DEV_GF1 ) return;		/* this routine is for native GF1 interface only */
  switch ( cmd ) {
    case CTRL_PITCH_BENDER:
      {
        gus_voice_t *cvoice;
        gus_seq_voice_t *pvoice;

        cvoice = get_voice( card, voice );
        pvoice = get_seq_voice( card, voice );
        pvoice -> bender = value;
#if 0
        printk( "bender (%i) - value = %i, range = %i\n", voice, value, pvoice -> bender_range );
#endif
        if ( cvoice -> commands )
          cvoice -> commands -> voice_freq( card, cvoice, GUS_FREQ_PITCHBEND | get_pitchbend( pvoice ) );
      }
      break;
    case CTRL_PITCH_BENDER_RANGE:
      {
        gus_seq_voice_t *pvoice;
        
#if 0
        printk( "bender range (%i) = %i\n", voice, value );
#endif
        pvoice = get_seq_voice( card, voice );
        pvoice -> bender_range = value;
      }
      break;
    case CTL_EXPRESSION:
      if ( card -> seq.vol_mode == GF1_VOL_MODE_ADAGIO ) value /= 128;
    case CTRL_EXPRESSION:
      {
        gus_seq_voice_t *pvoice;
        
        pvoice = get_seq_voice( card, voice );
        pvoice -> midi_expression_vol = value > 127 ? 127 : value;
        gus_sequencer_volume_change( card, voice );
      }
      break;
    case CTL_PAN:
      {
        gus_voice_t *cvoice;
        gus_seq_voice_t *pvoice;

        cvoice = get_voice( card, voice );
        pvoice = get_seq_voice( card, voice );
        pvoice -> panning = value << 7;
        if ( pvoice -> panning >= 8192 ) pvoice -> panning += 0x3f;
#if 0
        if ( cvoice -> commands )
          cvoice -> commands -> voice_pan( card, cvoice, pvoice -> panning );
#endif
      }
      break;
    case CTL_MAIN_VOLUME:
      value = ( value * 100 ) / 16383;
    case CTRL_MAIN_VOLUME:
      {
        gus_seq_voice_t *pvoice;
           
        pvoice = get_seq_voice( card, voice );
        pvoice -> midi_main_vol = value;
        gus_sequencer_volume_change( card, voice );
      }
      break;
    case SEQ_VOLMODE:
      switch ( value ) {
        case VOL_METHOD_ADAGIO:
          card -> seq.vol_mode = GF1_VOL_MODE_ADAGIO;
          break;
        case VOL_METHOD_LINEAR:
          card -> seq.vol_mode = GF1_VOL_MODE_LINEAR;
          break;
        default:
          card -> seq.vol_mode = GF1_VOL_MODE_NONE;
          break;
      }
      break;
#ifdef V_DEBUG
    default:
      PRINTK( "gus_do_controller_event: cmd = 0x%x ???\n", cmd );
#endif
  }
}

static void gus_do_seq_voice_event( gus_card_t *card, int device, unsigned char *event )
{
  unsigned char cmd = event[ 2 ];
  unsigned char chn = event[ 3 ];
  unsigned char note = event[ 4 ];
  unsigned char parm = event[ 5 ];
  gus_seq_voice_t *pvoice;
  unsigned char midi_cmd[ 3 ];
  
  pvoice = get_seq_voice( card, chn );
  switch ( cmd ) {
    case MIDI_NOTEON:
#if 0
      PRINTK( "note on: device = %i, note = %i\n", device, note );
#endif
      DEBUG_EVENT( "NOTEON", event, 3 );
      if ( note > 127 && note != 255 ) break;
      if ( seq_mode == SEQ_MODE_1 && device == SEQ_DEV_GF1 )
        {
          pvoice -> midi_volume = parm;
          gus_sequencer_note_on( card, chn, note );
        }
       else
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
          midi_cmd[ 0 ] = GUS_MCMD_NOTE_ON | ( chn & 0x0f );
          midi_cmd[ 1 ] = note & 0x7f;
          midi_cmd[ 2 ] = parm & 0x7f;
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 3 );
        }
      break;
    case MIDI_NOTEOFF:
      DEBUG_EVENT( "NOTEOFF", event, 3 );
      if ( seq_mode == SEQ_MODE_1 && device == SEQ_DEV_GF1 )
        gus_sequencer_note_off( card, chn );
       else
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
          midi_cmd[ 0 ] = GUS_MCMD_NOTE_OFF | ( chn & 0x0f );
          midi_cmd[ 1 ] = note & 0x7f;
          midi_cmd[ 2 ] = parm & 0x7f;
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 3 );
        }
      break;
    case MIDI_KEY_PRESSURE:
      DEBUG_EVENT( "KEY_PRESSURE", event, 3 );
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
          midi_cmd[ 0 ] = GUS_MCMD_NOTE_PRESSURE | ( chn & 0x0f );
          midi_cmd[ 1 ] = note & 0x7f;
          midi_cmd[ 2 ] = parm & 0x7f;
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 3 );
        }
      break;
#ifdef V_DEBUG
    default:
      PRINTK( "gus_do_seq_voice_event: cmd = 0x%x\n", event[ 1 ] );
      return -EINVAL;
#endif
  }
}

static void gus_do_seq_common_event( gus_card_t *card, int device, unsigned char *event )
{
  unsigned char cmd = event[ 2 ];
  unsigned char chn = event[ 3 ];
  unsigned char p1 = event[ 4 ];
  /* unsigned char p2 = event[ 5 ]; */
  unsigned short w14 = *(short *)&event[ 6 ];
  unsigned char midi_cmd[ 3 ];

#if 0
  PRINTK( "common: cmd = 0x%x\n", cmd );
#endif
  switch ( cmd ) {
    case MIDI_PGM_CHANGE:
#if 0
      PRINTK( "MIDI_PGM_CHANGE: chn %d = instr %d\n", chn, p1 );
#endif
      DEBUG_EVENT( "PGM_CHANGE", event, 3 );
      if ( seq_mode == SEQ_MODE_1 )
        {
          if ( device == SEQ_DEV_GF1 )
            gus_sequencer_pgm_change( card, chn, p1 );
        }
       else
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
          midi_cmd[ 0 ] = GUS_MCMD_PGM_CHANGE | ( chn & 0x0f );
          midi_cmd[ 1 ] = p1;
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 2 );
        }
      break;
    case MIDI_CTL_CHANGE:
      DEBUG_EVENT( "CTL_CHANGE", event, 3 );
      if ( seq_mode == SEQ_MODE_1 )
        {
          if ( device == SEQ_DEV_GF1 )
            {
              if ( p1 == CTRL_MAIN_VOLUME )
                {
                  w14 = (unsigned short)( ( (int)w14 * 16383 ) / 100 );
                  p1 = CTL_MAIN_VOLUME;
                }
               else
              if ( p1 == CTRL_EXPRESSION )
                {
                  w14 *= 128;
                  p1 = CTL_EXPRESSION;
                } 
              gus_do_controller_event( card, device, chn, p1, w14 );
            }
        }
       else
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
#if 0
          PRINTK( "MIDI_CTL_CHANGE: chn %d, parm 0x%x = %d\n", chn, p1, w14 & 0x7f );
#endif
          midi_cmd[ 0 ] = GUS_MCMD_CONTROL | ( chn & 0x0f );
          midi_cmd[ 1 ] = p1;
          midi_cmd[ 2 ] = w14 & 0x7f;
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 3 );
        }
      break;
    case MIDI_PITCH_BEND:
      DEBUG_EVENT( "PITCH_BEND", event, 3 );
      if ( seq_mode == SEQ_MODE_1 )
        {
          if ( device == SEQ_DEV_GF1 )
            gus_do_controller_event( card, device, chn, CTRL_PITCH_BENDER, (short)w14 - 8192 );
        }
       else
      if ( seq_mode == SEQ_MODE_2 && device != SEQ_DEV_GF1 )
        {
          midi_cmd[ 0 ] = GUS_MCMD_BENDER | ( chn & 0x0f );
          midi_cmd[ 1 ] = ( w14 & 0x7f );
          midi_cmd[ 2 ] = ( ( w14 >> 7 ) & 0x7f );
          card -> midi[ device == SEQ_DEV_MIDI ? GUS_MIDID_UART : GUS_MIDID_SYNTH ].putcmd( card, midi_cmd, 3 );
        }
      break;
#ifdef V_DEBUG
    default:
      PRINTK( "gus_do_seq_common_event: event = 0x%x\n", cmd );
#endif
  }
}

static int gus_do_seq_timing_event( unsigned char *event )
{
  unsigned int arg = *(unsigned int *)&event[ 4 ];
  int i;

  switch ( event[ 1 ] ) {
    case TMR_WAIT_REL:
      arg += seq_abs_tick;
    case TMR_WAIT_ABS:
      i = arg - seq_abs_tick;
      seq_abs_tick = arg;
      if ( i > 0 )
        {
          if ( !seq_playing ) seq_stop_flag = 1;
          gus_cards[ 0 ] -> gf1.timer_wait_ticks = i - 1;
          return 1;
        }
      break;
    case TMR_START:
      gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_START, 0 );
      seq_playing = ( gus_cards[ 0 ] -> gf1.timer_enabled & ( 4 | 8 ) ) != 0;
      seq_stop_flag = 0;
      break;
    case TMR_STOP:
      gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_STOP, 0 );
      seq_playing = ( gus_cards[ 0 ] -> gf1.timer_enabled & ( 4 | 8 ) ) != 0;
      break;
    case TMR_CONTINUE:
      gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_CONTINUE, 0 );
      seq_playing = ( gus_cards[ 0 ] -> gf1.timer_enabled & ( 4 | 8 ) ) != 0;
      seq_stop_flag = 0;
      break;
    case TMR_TEMPO:
      gus_timer_tempo( gus_cards[ 0 ], arg );
      break;
    case TMR_ECHO:
      if ( seq_i_used < SEQ_IQUEUE_LEN )
        {
          if ( seq_mode == SEQ_MODE_2 )
            gus_put_seq_input_event( event, 8 );
           else
            {
              event[ 0 ] = SEQ_ECHO;
              event[ 1 ] = (unsigned char)arg;
              event[ 2 ] = (unsigned char)(arg >> 8);
              event[ 3 ] = (unsigned char)(arg >> 16);
              gus_put_seq_input_event( event, 4 );
            }
        }
      break;
#ifdef V_DEBUG
    default:
      PRINTK( "gus_do_seq_timing_event: cmd = 0x%x\n", event[ 0 ] );
#endif
  }
  return 0;
}

static void gus_do_seq_ext_event( gus_card_t *card, int device, unsigned char *event )
{
  switch ( event[ 1 ] ) {
    case SEQ_PGMCHANGE:
#if 0
      PRINTK( "MIDI_PGM_CHANGE: chn %d = instr %d\n", event[ 3 ], event[ 4 ] );
#endif
      if ( seq_mode == SEQ_MODE_1 && device == SEQ_DEV_GF1 )
        {
          DEBUG_EVENT( "PGMCHANGE", event, 3 );
          gus_sequencer_pgm_change( card, event[ 3 ], event[ 4 ] );
        }
    case SEQ_CONTROLLER:
      if ( seq_mode == SEQ_MODE_1 && device == SEQ_DEV_GF1 )
        {
          DEBUG_EVENT( "CONTROLLER", event, 3 );
          gus_do_controller_event( card,
                                   device,
                                   event[ 3 ], 
                                   event[ 4 ], 
                                   *(short *)&event[ 5 ] );
        }
      break;
    case SEQ_VOLMODE:
      if ( seq_mode == SEQ_MODE_1 && device == SEQ_DEV_GF1 )
        {
          if ( event[ 3 ] == VOL_METHOD_LINEAR )
            card -> seq.vol_mode = GF1_VOL_MODE_LINEAR;
           else
          if ( event[ 3 ] == VOL_METHOD_ADAGIO )
            card -> seq.vol_mode = GF1_VOL_MODE_ADAGIO;
           else
            card -> seq.vol_mode = GF1_VOL_MODE_NONE; 
        }
#ifdef V_DEBUG
    default:
      PRINTK( "gus_do_seq_ext_event: cmd = 0x%x ???\n", event[ 0 ] );
#endif
  }
}

static void gus_sequencer_midi_get_command( unsigned int device, unsigned char *buffer, unsigned int count )
{
  unsigned char uss_dev;
  unsigned char tmr_event[ 8 ];
  unsigned char event[ 8 ];
  gus_card_t *card;

  if ( seq_mode == SEQ_MODE_1 )
    {
      unsigned int tstamp;
      
      uss_dev = SEQ_DEV_CNT * ( device >> 16 ) + SEQ_DEV_MIDI;      
      tstamp = jiffies - seq_time;
      if ( tstamp != seq_abs_tick_input_old )
        {
          *(unsigned int *)&tmr_event[ 0 ] = ( tstamp << 8 ) | SEQ_WAIT;
          gus_put_seq_input_event( event, 4 );
          seq_abs_tick_input_old = tstamp;
        }
      while ( count-- )
        {
          event[ 0 ] = SEQ_MIDIPUTC;
          event[ 1 ] = *buffer++;
          event[ 2 ] = uss_dev;
          event[ 3 ] = 0;
          gus_put_seq_input_event( event, 4 );
        }
      return;
    }
  switch ( device & 0x0f ) {
    case GUS_MIDID_UART:	uss_dev = SEQ_DEV_MIDI; break;
    case GUS_MIDID_SYNTH:
    default:			return;
  }
  card = gus_cards[ device >> 16 ];
  if ( (card -> seq.flags & GUS_SEQ_F_D0SWAP) && uss_dev == SEQ_DEV_MIDI )
    uss_dev = SEQ_DEV_EMUL;
  uss_dev += SEQ_DEV_CNT * ( device >> 16 );
  MEMSET( &event, 0, 8 );
  switch ( buffer[ 0 ] & 0xf0 ) {
    case GUS_MCMD_NOTE_OFF:
    case GUS_MCMD_NOTE_ON:
    case GUS_MCMD_NOTE_PRESSURE:
      event[ 0 ] = EV_CHN_VOICE;
      event[ 1 ] = uss_dev;
      event[ 3 ] = buffer[ 0 ] & 0x0f;	/* channel */
      event[ 4 ] = buffer[ 1 ];		/* key */
      event[ 5 ] = buffer[ 2 ];		/* velocity */
      switch ( buffer[ 0 ] & 0xf0 ) {
        case GUS_MCMD_NOTE_OFF:	
          event[ 2 ] = MIDI_NOTEOFF;
          break;
	case GUS_MCMD_NOTE_ON:		
	  event[ 2 ] = MIDI_NOTEON;
	  if ( !event[ 5 ] ) buffer[ 2 ] = MIDI_NOTEOFF;
	  break;
	case GUS_MCMD_NOTE_PRESSURE:
	  event[ 2 ] = MIDI_KEY_PRESSURE;
	  break;
      }
      break;
    case GUS_MCMD_CONTROL:
    case GUS_MCMD_PGM_CHANGE:
    case GUS_MCMD_CHANNEL_PRESSURE:
    case GUS_MCMD_BENDER:
      event[ 0 ] = EV_CHN_COMMON;
      event[ 1 ] = uss_dev;
      event[ 3 ] = buffer[ 0 ] & 0x0f;	/* channel */
      switch ( buffer[ 0 ] & 0xf0 ) {
        case GUS_MCMD_CONTROL:
          event[ 2 ] = MIDI_CTL_CHANGE;
          event[ 4 ] = buffer[ 1 ];
          event[ 5 ] = *(unsigned short *)&event[ 6 ] = buffer[ 2 ];
          break;
        case GUS_MCMD_PGM_CHANGE:
          event[ 2 ] = MIDI_PGM_CHANGE;
          event[ 4 ] = *(unsigned short *)&event[ 6 ] = buffer[ 1 ];
          break;
        case GUS_MCMD_CHANNEL_PRESSURE:
          event[ 2 ] = MIDI_CHN_PRESSURE;
          event[ 4 ] = buffer[ 1 ];
          event[ 5 ] = *(unsigned short *)&event[ 6 ] = buffer[ 2 ];
          break;
        case GUS_MCMD_BENDER:
          event[ 2 ] = MIDI_PITCH_BEND;
          *(unsigned short *)&event[ 6 ] = ( buffer[ 2 ] << 7 ) | buffer[ 1 ];
          break;
      }
      break;
    default:
      return;			/* ignore all SysEx and etc. */
  }
#if 0
  printk( "in event = %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x\n",
  		event[ 0 ], event[ 1 ], event[ 2 ], event[ 3 ],
  		event[ 4 ], event[ 5 ], event[ 6 ], event[ 7 ] );
#endif
  if ( seq_abs_tick_input != seq_abs_tick_input_old )
    {
      tmr_event[ 0 ] = EV_TIMING;
      tmr_event[ 1 ] = TMR_WAIT_ABS;
      tmr_event[ 2 ] = tmr_event[ 3 ] = 0;
      *(unsigned int *)&tmr_event[ 4 ] = seq_abs_tick_input_old = seq_abs_tick_input;
      gus_put_seq_input_event( tmr_event, 8 );
    }
  gus_put_seq_input_event( event, 8 );
}

static void gus_sequencer_midi_put_command( unsigned int device, unsigned char *buffer, unsigned int count )
{
  gus_card_t *card;
  
  card = gus_cards[ device >> 16 ];
#if 0
  {
    int i;
    printk( "put command (%i): ", count );
    for ( i = 0; i < count; i++ )
      printk( "%02x:", buffer[ i ] );
    printk( "\n" );
  }
#endif
  switch ( device & 0xffff ) {
    case SEQ_DEV_MIDI:
      card -> midi[ GUS_MIDID_UART ].putcmd( card, buffer, count );
      break;
    case SEQ_DEV_EMUL:
      card -> midi[ GUS_MIDID_SYNTH ].putcmd( card, buffer, count );
      break;
  }
}

static void gus_do_seq_midiputc_event( gus_card_t *card, int device, unsigned char *event )
{
  if ( seq_mode != SEQ_MODE_1 ) return;
  switch ( device ) {
    case SEQ_DEV_GF1:
#if 0
      PRINTK( "gus_do_seq_midiputc_event: to native GF1?\n" );
#endif
      break;
    case SEQ_DEV_MIDI:
      gus_midi_cmd_byte( &card -> seq.ext_dev, event[ 1 ] );
      break;
    case SEQ_DEV_EMUL:
      if ( !(card -> seq.flags & GUS_SEQ_F_MIDI_EMUL) )
        gus_preload_general_midi_bank( card );
      gus_midi_cmd_byte( &card -> seq.gf1_emul, event[ 1 ] );
      break;
  }
}

static void gus_do_seq_private_event( gus_card_t *card, int device, unsigned char *event )
{
  if ( seq_mode != SEQ_MODE_1 ) return;
  if ( device == SEQ_DEV_GF1 )
    {
      unsigned short p1 = *(unsigned short *)&event[ 4 ];
      unsigned short p2 = *(unsigned short *)&event[ 6 ];
      unsigned int pl = *(unsigned int *)&event[ 4 ];
      gus_voice_t *cvoice;
      gus_seq_voice_t *pvoice;
      
      DEBUG_EVENT( "PRIVATE", event, 3 );
      switch ( event[ 2 ] ) {
        case _GUS_NUMVOICES:
          card -> seq.gf1_voices = p1 > 32 ? 32 : ( p1 < 1 ? 1 : p1 );
          gus_engine_reset_voices( card, card -> seq.gf1_voices, 0, 0 );
          return;
        case _GUS_VOICESAMPLE:
          PRINTK( "_GUS_VOICESAMPLE: obsolete\n" );
          return;
        case _GUS_VOLUME_SCALE:
          card -> seq.volume_base = p1;
          card -> seq.volume_scale = p2;
          return;
      }
      cvoice = get_voice( card, event[ 3 ] % 32 );
      pvoice = get_seq_voice( card, event[ 3 ] % 32 );
#if 0
      PRINTK( "gus_do_seq_private_event: %i - cmd = 0x%x, p1 = 0x%x, p2 = 0x%x, pl = 0x%x\n", event[ 3 ] % 32, event[ 2 ], p1, p2, pl );
#endif
      switch ( event[ 2 ] ) {
        case _GUS_VOICEON:
        case _GUS_VOICEMODE:
        case _GUS_VOICEOFF:
          if ( cvoice -> commands )
            cvoice -> commands -> voice_stop( card, cvoice, 0x00 );
          break;
        case _GUS_VOICEFADE:
          if ( cvoice -> commands )
            cvoice -> commands -> voice_stop( card, cvoice, 0x01 );
          break;
        case _GUS_VOICEFREQ:
        case _GUS_VOICEVOL:
        case _GUS_VOICEVOL2:
        case _GUS_RAMPRANGE:
        case _GUS_RAMPRATE:
        case _GUS_RAMPON:
        case _GUS_RAMPMODE:
        case _GUS_RAMPOFF:
#if 0
          printk( "sequencer: obsolete GUS COMMAND = 0x%x\n", event[ 2 ] );
#endif
          break;
        case _GUS_VOICEBALA:
          p1 &= 0x0f;
          p1 = p1 > 7 ? ( p1 << 10 ) | 0x3ff : ( p1 << 10 );
          if ( cvoice -> commands )
            {
              cvoice -> flags |= VFLG_OSS_BALANCE;
              cvoice -> commands -> voice_pan( card, cvoice, p1 );
              cvoice -> flags &= ~VFLG_OSS_BALANCE;
            }
          break;
        case _GUS_VOICE_POS:
          if ( cvoice -> commands )
            cvoice -> commands -> voice_pos( card, cvoice, pl << 4 );
          break;
        default:
          PRINTK( "gus_do_seq_private_event: cmd = 0x%x ???\n", event[ 2 ] );
      }
    }
#ifdef V_DEBUG
   else
    PRINTK( "gus_do_seq_private_event: MIDI\n" );
#endif
}

static int gus_do_seq_local( unsigned char *event )
{
  extern int gus_pcm_trigger_sequencer( unsigned int );
  unsigned char cmd = event[ 1 ];
  unsigned int parm = *((unsigned int *)&event[ 4 ]);
  
  switch ( cmd ) {
    case LOCL_STARTAUDIO:
      return gus_pcm_trigger_sequencer( parm );
    default:
      PRINTK( "gus_seq_local: unknown command 0x%x...\n", cmd );
      return -1;
  }
  return 0;
}

static int gus_sequencer_play_event( unsigned char *event )
{
  int device;
  gus_card_t *card;

  DEBUG_EVENT_ALL( "ALL-PLAY", event );
  switch ( event[ 0 ] ) {
    case EV_SEQ_LOCAL:  return gus_do_seq_local( event );
    case EV_TIMING:	return gus_do_seq_timing_event( event );
    case EV_CHN_VOICE:
    case EV_CHN_COMMON:
    case SEQ_PRIVATE:   device = event[ 1 ]; break;
    case SEQ_MIDIPUTC:
    case SEQ_EXTENDED:	device = event[ 2 ]; break;
    default:
      PRINTK( "gus_sequencer_play_event: bad event code 0x%x\n", event[ 0 ] );
      return -ENXIO;
  }
  if ( ( device / SEQ_DEV_CNT ) >= gus_cards_count ) return -ENODEV;
  card = gus_cards[ device / SEQ_DEV_CNT ];
  device %= SEQ_DEV_CNT;
  if ( seq_mode == SEQ_MODE_2 )
    gus_preload_general_midi_bank( card );
  if ( card -> gf1.syn_voices_change ) return 0;
  switch ( event[ 0 ] ) {
    case EV_CHN_VOICE:	gus_do_seq_voice_event( card, device, event ); break;
    case EV_CHN_COMMON:	gus_do_seq_common_event( card, device, event ); break;
    case SEQ_EXTENDED:	gus_do_seq_ext_event( card, device, event ); break;
    case SEQ_MIDIPUTC:	gus_do_seq_midiputc_event( card, device, event ); break; 
    case SEQ_PRIVATE:	gus_do_seq_private_event( card, device, event ); break;
  }
  return 0;
}

void gus_sequencer_tick( void )
{
  seq_abs_tick_input++;
}

void gus_sequencer_process_events( int timer )
{
  int res;
  char *buffer;
 
#if 0
  printk( "sequencer process events - start\n" ); 
#endif
  while ( seq_o_used > 0 )
    {
      buffer = seq_o_queue + 8 * seq_o_tail++;
      res = !seq_abort ? gus_sequencer_play_event( buffer ) : 0;
      seq_o_tail %= SEQ_OQUEUE_LEN;
      seq_o_used--;
      if ( res == 1 ) break;
    }
  gus_sequencer_start_voices();
  if ( ( SEQ_OQUEUE_LEN - seq_o_used ) >= seq_o_threshold &&
       ( GETLOCK( gus_cards[ 0 ], seq_out ) & WK_SLEEP ) )
    {
      GETLOCK( gus_cards[ 0 ], seq_out ) &= ~WK_SLEEP;
      WAKEUP( gus_cards[ 0 ], seq_out );
    }
#if 0
  printk( "sequencer process events - stop\n" );
#endif
}

/*
 *
 */

int gus_lseek_sequencer( struct file *file, off_t offset, int orig )
{
  return -EIO;
}

int gus_read_sequencer( struct file *file, char *buf, int count )
{
  unsigned char event_size;
  int result;

  if ( seq_abort ) return -EINTR;
  result = 0;
  event_size = seq_mode == SEQ_MODE_1 ? 4 : 8;
  if ( !seq_i_used )
    {
      if ( file -> f_flags & O_NONBLOCK ) return result;
      while ( !seq_abort && !seq_i_used )
        {
          GETLOCK( gus_cards[ 0 ], seq_in ) |= WK_SLEEP;
          SLEEP( gus_cards[ 0 ], seq_in, HZ * 60 );
          GETLOCK( gus_cards[ 0 ], seq_in ) &= ~WK_SLEEP;
          if ( TABORT( gus_cards[ 0 ], seq_in ) )
            {
              seq_abort = 1;
              return -EINTR;
            }
          if ( TIMEOUT( gus_cards[ 0 ], seq_in ) )
            return -EIO;
        }
    }
  while ( seq_i_used && count >= event_size )
    {
      MEMCPY_TOFS( buf, seq_i_queue + seq_i_tail++ * 8, event_size );
      seq_i_tail %= SEQ_IQUEUE_LEN;
      seq_i_used--;
      buf += event_size;
      count -= event_size;
      result += event_size;
    }
  return result;
}

int gus_write_sequencer( struct file *file, char *buf, int count )
{
  unsigned char event[ EVENT_SIZE ];
  unsigned char event_code, event_size;
  unsigned short dev;
  int result;
  
  if ( seq_abort ) return -EINTR;
  result = 0;
  while ( count >= 4 )
    {
      MEMCPY_FROMFS( event, buf, 4 );
#if 0
      printk( "write event = %02x:%02x:%02x:%02x\n", event[ 0 ], event[ 1 ], event[ 2 ], event[ 3 ] );
#endif
      event_code = event[ 0 ];
      if ( event_code == SEQ_FULLSIZE )
        {
          dev = *(unsigned short *)&event[ 2 ];
          if ( dev >= gus_cards_count * SEQ_DEV_CNT ) return -ENXIO;
          result = -EIO;
          if ( seq_mode == SEQ_MODE_1 )
            {
              if ( ( dev % SEQ_DEV_CNT ) == SEQ_DEV_GF1 )
                result = gus_patch_load( (struct patch_info *)buf );
            }
           else
          if ( seq_mode == SEQ_MODE_2 )
            result = gus_sysex_load( (struct sysex_info *)buf );
          return result < 0 ? result : 0;
        }
      if ( event_code >= 128 )
        {
          if ( seq_mode == SEQ_MODE_2 && event_code == SEQ_EXTENDED ) return -EINVAL;
          event_size = 8;
          if ( count < event_size )
            {
              if ( !seq_playing && !seq_stop_flag )
                gus_sequencer_process_events( 0 );
              return result;
            }
          MEMCPY_FROMFS( &event[ 4 ], buf + 4, 4 );
        }
       else
        {
          if ( seq_mode == SEQ_MODE_2 ) return -EINVAL;
          event_size = 4;
        }
      if ( !seq_abort && seq_o_used >= SEQ_OQUEUE_LEN )
        {
          if ( file -> f_flags & O_NONBLOCK ) return result ? result : -EAGAIN;
          while ( seq_o_used >= SEQ_OQUEUE_LEN )
            {
              GETLOCK( gus_cards[ 0 ], seq_out ) |= WK_SLEEP;
              SLEEP( gus_cards[ 0 ], seq_out, HZ * 60 );
              GETLOCK( gus_cards[ 0 ], seq_out ) &= ~WK_SLEEP;
              if ( TABORT( gus_cards[ 0 ], seq_out ) )
                {
                  seq_abort = 1;
                  return -EINTR;
                } 
              if ( TIMEOUT( gus_cards[ 0 ], seq_out ) )
                return -EIO;
            }
        }
      if ( seq_abort ) break;
      MEMCPY( seq_o_queue + 8 * seq_o_head++, event, event_size );
      seq_o_head %= SEQ_OQUEUE_LEN;
      seq_o_used++;
      if ( !seq_playing && !seq_stop_flag )
        gus_sequencer_process_events( 0 );
      count -= event_size;
      buf += event_size;
      result += event_size;
    }
  return result;
}

static int gus_sequencer_reset( void )
{  
  int i;
  gus_card_t *card;
  
  gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_STOP, 0 );
  seq_abort = 0;
  seq_playing = 0;
  seq_flush = 0;
  if ( seq_flags & SEQ_LFLG_WRITE )
    for ( i = 0; i < gus_cards_count; i++ )
      {
        card = gus_cards[ i ];
        if ( seq_mode == SEQ_MODE_1 )
          gus_engine_voices_init( card, 0 );
        card -> gf1.syn_abort_flag = 0;
        gus_reset_seq_voices( card );
        if ( seq_mode == SEQ_MODE_1 )
          {
            gus_midi_cmd_init( &card -> seq.gf1_emul, 1 );
            gus_midi_cmd_init( &card -> seq.ext_dev, 1 );
          }
      }
  seq_abs_tick = seq_abs_tick_input = 0;
  seq_abs_tick_input_old = ~0;
  seq_time = jiffies;
  seq_o_used = seq_i_used = 0;
  seq_o_head = seq_o_tail = seq_i_head = seq_i_tail = 0;
  seq_stop_flag = 0;
  seq_out_of_band = 0;
  return 0;
}

static int gus_sequencer_flush( void )
{
  int res;

  res = 0;
  seq_flush = 1;
#if 0
  printk( "seq_abort = %i, seq_playing = %i, seq_o_used = %i\n",
  				seq_abort, seq_playing, seq_o_used );
#endif
  while ( !seq_abort && seq_playing && seq_o_used > 0 )
    {
      GETLOCK( gus_cards[ 0 ], seq_out ) |= WK_SLEEP;
      SLEEP( gus_cards[ 0 ], seq_out, HZ * 60 );
      GETLOCK( gus_cards[ 0 ], seq_out ) &= ~WK_SLEEP;
      if ( TABORT( gus_cards[ 0 ], seq_out ) )
        {
          seq_abort = 1;
          break;
        }
      if ( TIMEOUT( gus_cards[ 0 ], seq_out ) )
        {
          res = -EIO;
          break;
        }
    }
  seq_flush = 0;
  return res;
}

static void gus_sequencer_free( void )
{
  int i, j, flags;
  gus_card_t *card;

  if ( seq_i_queue )
    {
      gus_free( seq_i_queue, SEQ_IQUEUE_LEN * EVENT_SIZE );
      seq_i_queue = NULL;
    }
  if ( seq_o_queue )
    {
      gus_free( seq_o_queue, SEQ_OQUEUE_LEN * EVENT_SIZE );
      seq_o_queue = NULL;
    }
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      if ( card -> seq.voices )
        {
          gus_free( card -> seq.voices, sizeof( struct GUS_STRU_SEQ_VOICE ) * 32 );
          card -> seq.voices = NULL;
        }
      for ( j = 0; j <= GUS_MIDID_LAST; j++ )
        {
          flags = card -> midi[ j ].flags;
          if ( !(flags & GUS_MIDIF_SEQUENCER) ) continue;
          if ( flags & GUS_MIDIF_USED_OUT ) card -> midi[ j ].done_write( card );
          if ( flags & GUS_MIDIF_USED_IN ) card -> midi[ j ].done_read( card );
          card -> midi[ j ].flags &= ~( GUS_MIDIF_USED | GUS_MIDIF_SEQUENCER );
        }
      if ( seq_mode == SEQ_MODE_1 )
        {
          if ( card -> seq.gf1_emul.buf )
            {
              gus_free( card -> seq.gf1_emul.buf, card -> seq.gf1_emul.size );
              card -> seq.gf1_emul.buf = NULL;
            }
          if ( card -> seq.ext_dev.buf )
            {
              gus_free( card -> seq.ext_dev.buf, card -> seq.ext_dev.size );
              card -> seq.ext_dev.buf = NULL;
            }
        }
    }
  gus_cards[ 0 ] -> gf1.timer_sequencer = 0;
}

int gus_open_sequencer( unsigned short minor, struct file *file )
{
  int i, j, res;
  gus_card_t *card;
  unsigned short flags;
  unsigned short mode;

  if ( seq_mode != SEQ_MODE_NONE ) return -EBUSY;
  mode = SEQ_MODE_1;
  if ( minor == GUS_MINOR_MUSIC ) mode = SEQ_MODE_2;
  flags = seq_file_flags( file -> f_flags );
  if ( ( seq_i_queue = (unsigned char *)gus_malloc( SEQ_IQUEUE_LEN * EVENT_SIZE ) ) == NULL )
    {
      gus_sequencer_free();
      return -ENOMEM;
    }
  if ( ( seq_o_queue = (unsigned char *)gus_malloc( SEQ_OQUEUE_LEN * EVENT_SIZE ) ) == NULL )
    {
      gus_sequencer_free();
      return -ENOMEM;
    }
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      if ( ( card -> seq.voices = (struct GUS_STRU_SEQ_VOICE *)gus_malloc( sizeof( struct GUS_STRU_SEQ_VOICE ) * 32 ) ) == NULL )
        {
          gus_sequencer_free();
          return -ENOMEM;
        }
    }
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      card -> seq.flags = 0;
      for ( j = 0; j <= GUS_MIDID_LAST; j++ )
        {
          struct GUS_STRU_MIDI *midi;
        
          midi = &card -> midi[ j ];
          if ( flags & SEQ_LFLG_WRITE )
            {
              if ( midi -> flags & GUS_MIDIF_USED_OUT )
                {
                  gus_sequencer_free();
                  return -EBUSY;
                }
              midi -> tx_size = GUS_MIDI_BUF_SIZE;
              midi -> tx_used = midi -> tx_head = midi -> tx_tail = 0;
#if 0
              printk( "- %i - %i - start\n", i, j );
#endif
              if ( ( res = midi -> init_write( card ) ) < 0 )
                { 
                  gus_sequencer_free();
                  return res;
                }
#if 0
              printk( "- %i - %i - end\n", i, j );
#endif
              midi -> flags |= GUS_MIDIF_USED_OUT | GUS_MIDIF_SEQUENCER;
            }
          if ( ( flags & SEQ_LFLG_READ ) && j != GUS_MIDID_SYNTH )
            {
              if ( midi -> flags & GUS_MIDIF_USED_IN )
                {
                  gus_sequencer_free();
                  return -EBUSY;
                }
              midi -> rx.size = GUS_MIDI_BUF_SIZE;
              midi -> rx.dev = ( i << 16 ) | j;
              midi -> rx.ack = gus_sequencer_midi_get_command;
              midi -> rx_size = 0;
              if ( ( res = midi -> init_read( card ) ) < 0 )
                {
                  gus_sequencer_free();
                  return res;
                }
              midi -> flags |= GUS_MIDIF_USED_IN | GUS_MIDIF_SEQUENCER;
            }
        }
      if ( mode == SEQ_MODE_1 )
        {
          gus_midi_cmd_init( &card -> seq.gf1_emul, 1 );
          card -> seq.gf1_emul.size = GUS_MIDI_BUF_SIZE;
          card -> seq.gf1_emul.buf = gus_malloc( GUS_MIDI_BUF_SIZE );
          if ( !card -> seq.gf1_emul.buf )
            {
              gus_sequencer_free();
              return -ENOMEM;
            }
          card -> seq.gf1_emul.dev = ( i << 16 ) | SEQ_DEV_EMUL;
          card -> seq.gf1_emul.ack = gus_sequencer_midi_put_command;
          gus_midi_cmd_init( &card -> seq.ext_dev, 1 );
          card -> seq.ext_dev.size = GUS_MIDI_BUF_SIZE;
          card -> seq.ext_dev.buf = gus_malloc( GUS_MIDI_BUF_SIZE );
          if ( !card -> seq.ext_dev.buf )
            {
              gus_sequencer_free();
              return -ENOMEM;
            }
          card -> seq.ext_dev.dev = ( i << 16 ) | SEQ_DEV_MIDI;
          card -> seq.ext_dev.ack = gus_sequencer_midi_put_command;
        }
      card -> seq.volume_base = 3071;
      card -> seq.volume_scale = 4;
      card -> seq.vol_mode = GF1_VOL_MODE_ADAGIO;
      card -> seq.gf1_voices = 0;
    }
  seq_mode = mode;
  seq_flags = flags;
  if ( ( res = gus_timer_open( gus_cards[ 0 ], "OSS sequencer" ) ) < 0 )
    {
      gus_sequencer_free();
      return res;
    }
  seq_curr_timebase = 100;
  gus_timer_base( gus_cards[ 0 ], GUS_TIMEBASE( seq_curr_timebase ) );
  gus_timer_tempo( gus_cards[ 0 ], seq_curr_tempo = 60 );
  gus_timer_set_master( gus_cards[ 0 ] );
  for ( i = 1; i < gus_cards_count; i++ )
    gus_timer_set_slave( gus_cards[ 0 ], gus_cards[ i ] );
  gus_cards[ 0 ] -> gf1.timer_sequencer = 1;
  seq_o_threshold = SEQ_OQUEUE_LEN / 2;
  gus_sequencer_reset();
  MOD_INC_USE_COUNT;
  return 0;		/* success */
}

void gus_release_sequencer( struct file *file )
{
#if 1
  gus_sequencer_flush();
#endif
  gus_timer_close( gus_cards[ 0 ] );
  gus_sequencer_free();
  seq_mode = SEQ_MODE_NONE;
  MOD_DEC_USE_COUNT;
}

int gus_ioctl_sequencer( struct file *file, unsigned int cmd, unsigned long arg )
{
#if 0
  printk( "ioctl cmd = 0x%x\n", cmd );
#endif
  switch ( cmd ) {
    case OSS_GETVERSION:
      return IOCTL_OUT( arg, OSS_VERSION );
    case SNDCTL_SEQ_NRSYNTHS:	
    case SNDCTL_SEQ_NRMIDIS:
      return IOCTL_OUT( arg, gus_cards_count * SEQ_DEV_CNT );
    case SNDCTL_SYNTH_INFO:
      {
        struct synth_info inf;
        
        MEMCPY_FROMFS( &inf, (void *)arg, sizeof( inf ) );
        if ( inf.device >= 0 && inf.device < gus_cards_count * SEQ_DEV_CNT )
          {
            MEMCPY_TOFS( (void *)arg,
            		 &gus_uss_info[ inf.device % SEQ_DEV_CNT ], 
	         	 sizeof( struct synth_info ) );
          }
         else
          return -ENXIO;
        return 0;
      }
    case SNDCTL_MIDI_INFO:
      {
        struct midi_info inf;

        MEMCPY_FROMFS( &inf, (void *)arg, sizeof( inf ) );
        if ( inf.device >= 0 && inf.device < gus_cards_count * SEQ_DEV_CNT )
          {
            MEMCPY_TOFS( (void *)arg,
            		 &gus_uss_midi_info[ inf.device % SEQ_DEV_CNT ], 
	         	 sizeof( struct midi_info ) );
          }
         else
          return -ENXIO;
        return 0;
      }
    case SNDCTL_SEQ_RESET:
      return gus_sequencer_reset();
    case SNDCTL_SEQ_SYNC:
      return gus_sequencer_flush();
    case SNDCTL_SEQ_CTRLRATE:
      if ( IOCTL_IN( arg ) != 0 )
        return -EINVAL;
      return IOCTL_OUT( arg, ( ( seq_curr_tempo * seq_curr_timebase ) + 30 ) / 60 );
    case SNDCTL_SEQ_GETINCOUNT:
      return IOCTL_OUT( arg, seq_i_used );
    case SNDCTL_SEQ_GETOUTCOUNT:
      return IOCTL_OUT( arg, SEQ_OQUEUE_LEN - seq_o_used );
    case SNDCTL_SEQ_RESETSAMPLES: 
      {
        int dev;
        gus_card_t *card;
        
        dev = IOCTL_IN( arg );
        if ( dev < 0 || dev >= gus_cards_count * SEQ_DEV_CNT ) return -ENXIO;
        if ( ( dev % SEQ_DEV_CNT ) != SEQ_DEV_GF1 ) return -EIO;
        card = gus_cards[ dev / SEQ_DEV_CNT ];
        if ( card -> seq.gf1_voices > 0 )
          gus_engine_reset( card, card -> seq.gf1_voices, 0 );
         else
          gus_gf1_midi_engine_reset( gus_cards[ dev / SEQ_DEV_CNT ] );
        return 0;
      }
#ifdef SNDCTL_SEQ_THRESHOLD
    case SNDCTL_SEQ_THRESHOLD:
#else
    case SNDCTL_SEQ_TRESHOLD:		/* typo from author of VoxWare */
#endif
      {
        int tmp;
        
        tmp = IOCTL_IN( arg );
        if ( tmp < 1 )
          tmp = 1;
        if ( tmp >= SEQ_OQUEUE_LEN )
          tmp = SEQ_OQUEUE_LEN;
        seq_o_threshold = tmp;
      }
      return 0;
    case SNDCTL_SEQ_OUTOFBAND:
      {
        unsigned char event[ 8 ];
        
        MEMCPY_FROMFS( event, &((char *)arg)[ 0 ], 8 );
        gus_int_disable();
        seq_out_of_band = 1;
        gus_sequencer_play_event( event );
        seq_out_of_band = 0;
        gus_int_enable();
      }
      return 0;

    case SNDCTL_SYNTH_MEMAVL:
      {
        int dev;

        dev = IOCTL_IN( arg );
        if ( dev >= 0 && dev < gus_cards_count * SEQ_DEV_CNT )
          return IOCTL_OUT( arg, ( dev % SEQ_DEV_CNT ) == SEQ_DEV_GF1 ? gus_memory_free_size( gus_cards[ dev / SEQ_DEV_CNT ], &gus_cards[ dev / SEQ_DEV_CNT ] -> gf1.mem_alloc ) : 0x7fffffff );
      }
      return -ENXIO;
    case SNDCTL_TMR_TIMEBASE:
      {
        int val = IOCTL_IN( arg );
        
        if ( val )
          {
            if ( val < 1 ) val = 1;
            if ( val > 1000 ) val = 1000;
            seq_curr_timebase = val;
            gus_timer_base( gus_cards[ 0 ], GUS_TIMEBASE( seq_curr_timebase ) );
          }
        return IOCTL_OUT( arg, seq_curr_timebase );
      }
    case SNDCTL_TMR_START:
      seq_time = jiffies;
      return gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_START, 0 );
    case SNDCTL_TMR_STOP:
      return gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_STOP, 0 );
    case SNDCTL_TMR_CONTINUE:
      return gus_timer_ioctl( gus_cards[ 0 ], GUS_IOCTL_TIMER_CONTINUE, 0 );
    case SNDCTL_TMR_TEMPO:
      {
        int val = IOCTL_IN( arg );
        
        if ( val )
          {
            if ( val < 8 ) val = 8;
            if ( val > 250 ) val = 250;
            seq_curr_tempo = val;
            gus_timer_tempo( gus_cards[ 0 ], seq_curr_tempo );
          }
        return IOCTL_OUT( arg, seq_curr_tempo );
      }
    case SNDCTL_TMR_SOURCE:
      return IOCTL_OUT( arg, TMR_INTERNAL );
    case SNDCTL_TMR_METRONOME:
      return -EINVAL;
    case SNDCTL_SEQ_GETTIME:
      if ( seq_mode == SEQ_MODE_2 ) {
        return IOCTL_OUT( arg, seq_abs_tick );
      }
      return IOCTL_OUT( arg, jiffies - seq_time ); 
  }
#if 1
  PRINTK( "gus_ioctl_sequencer: unimplemented ioctl? (0x%x/0x%x)\n", cmd, (int)arg );
#endif
  return -EIO;
}

#ifdef GUS_POLL
unsigned int gus_poll_sequencer( struct file *file, poll_table *wait )
{
  unsigned long flags;
  unsigned int mask;
  
  CLI( &flags );
  GETLOCK( gus_cards[ 0 ], seq_in ) |= WK_SLEEP;
  SLEEP_POLL( gus_cards[ 0 ], seq_in, wait );
  GETLOCK( gus_cards[ 0 ], seq_out ) |= WK_SLEEP;
  SLEEP_POLL( gus_cards[ 0 ], seq_out, wait );
  STI( &flags );
  
  mask = 0;
  if ( seq_i_used )
    mask |= POLLIN | POLLRDNORM;
  if ( ( SEQ_OQUEUE_LEN - seq_o_used ) >= seq_o_threshold )
    mask |= POLLOUT | POLLWRNORM;

  return mask;
}
#else
int gus_select_sequencer( struct file *file, int sel_type, select_table *wait )
{
  unsigned long flags;

  switch ( sel_type ) {
    case SEL_IN:
      CLI( &flags );
      if ( !seq_i_used )
        {
          GETLOCK( gus_cards[ 0 ], seq_in ) |= WK_SLEEP;
          SLEEPS( gus_cards[ 0 ], seq_in, wait );
          STI( &flags );
          return 0;
        }
      STI( &flags );
      return 1;
    case SEL_OUT:
      CLI( &flags );
      if ( ( SEQ_OQUEUE_LEN - seq_o_used ) < seq_o_threshold )
        {
          GETLOCK( gus_cards[ 0 ], seq_out ) |= WK_SLEEP;
          SLEEPS( gus_cards[ 0 ], seq_out, wait );
          STI( &flags );
          return 0;
        }
      STI( &flags );
      return 1;
    case SEL_EX:
      return 0;
  }
  return 0;
}
#endif

void gus_sequencer_info( gus_info_buffer_t *buffer )
{
  int i, j;
  gus_card_t *card;
  char id[ 9 ];

  gus_iprintf( buffer, "\n\nOSS SEQUENCER:\n" );
  gus_iprintf( buffer, "  Mode          : %s\n",
  		 seq_mode == SEQ_MODE_1 ? "1" :
  		 seq_mode == SEQ_MODE_2 ? "2" : "none" );
  for ( i = 0; i < gus_cards_count; i++ )
    {
      card = gus_cards[ i ];
      strncpy( id, card -> id, 8 );
      id[ 8 ] = 0;
      gus_iprintf( buffer, "  Card #%i (%s):\n", card -> number, id );
      for ( j = 0; j < SEQ_DEV_CNT; j++ )
        gus_iprintf( buffer, "    %02i: %s\n", i * SEQ_DEV_CNT + j, gus_uss_info[ j ].name );
      if ( card -> gf1.enh_mode )
        {
          gus_iprintf( buffer, "    >>: Effect 1/3    : " );
          if ( card -> seq.iw_effect[ 0 ] < 0 ) gus_iprintf( buffer, "disable" ); else gus_iprintf( buffer, "%i", card -> seq.iw_effect[ 0 ] );
          if ( card -> seq.iw_effect[ 1 ] < 0 ) gus_iprintf( buffer, ",disable\n" ); else gus_iprintf( buffer, ",%i\n", card -> seq.iw_effect[ 1 ] );
        }
      gus_iprintf( buffer, "    >>: D0 -> D2 swap : %s\n",
         	card -> seq.flags & GUS_SEQ_F_D0SWAP ? "enable" : "disable" );
    }
}

#endif /* GUSCFG_OSS */
