/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: chash.c 1.3.1.2.1.1 Sun, 12 Oct 1997 20:29:34 -0700 jmacd $
 */

#include "xdelta.h"

static inline CKList*
cklist_prepend (ChecksumTable* table, CKList* rest, Checksum* cksum, ChecksumArray* base)
{
  CKList *ckl = g_chunk_new (CKList, table->cklist_chunk);

  ckl->cksum = cksum;
  ckl->array = base;
  ckl->next = rest;

  return ckl;
}

ChecksumTable*
c_hash_table_new (gint bits)
{
  ChecksumTable* table = g_new0 (ChecksumTable, 1);
  gint i = bits;

  table->buckets = g_new0 (CKList*, 1 << bits);
  table->mask = 0xffffffffU;
  table->bits = bits;
  table->cklist_chunk = g_mem_chunk_new ("cklist",
					 sizeof (CKList),
					 sizeof (CKList) * (1 << (bits - 1)),
					 G_ALLOC_ONLY);

  for (; i < 32; i += 1)
    table->mask ^= 1 << i;

  return table;
}

void
c_hash_table_free (ChecksumTable* table)
{
  g_mem_chunk_destroy (table->cklist_chunk);

  g_free (table);
}

gint
c_hash (const ChecksumTable* table, gint high, gint low)
{
  gint it = (high >> 2) + (low << 3) + (high << 16);

  return (it ^ high ^ low) & table->mask;
}

void
c_hash_table_insert (ChecksumTable *table,
		     Checksum      *checksum,
		     ChecksumArray *array)
{
  gint hash_val = c_hash (table, checksum->high, checksum->low);

#ifdef DEBUG_HASH
  table->high_checksums[checksum->high] += 1;
  table->low_checksums[checksum->low] += 1;
  table->hash_items += 1;

  if (table->buckets[hash_val])
    table->hash_coll += 1;
#endif

  table->buckets[hash_val] = cklist_prepend (table, table->buckets[hash_val], checksum, array);
}

CKList*
c_hash_table_lookup (const ChecksumTable* table,
		     const Checksum *checksum)
{
  return table->buckets[c_hash (table, checksum->high, checksum->low)];
}

#ifdef DEBUG_HASH
static gint
cklist_length (CKList *ckl)
{
  if (!ckl)
    return 0;

  return 1 + cklist_length (ckl);
}

void
c_hash_stats(ChecksumTable* table)
{
  gint high_checksum_counts [1<<16];
  gint low_checksum_counts  [1<<16];
  gint hash_collision_counts[1<<16];
  gint i, coll_metric=0;

  memset (high_checksum_counts, 0, sizeof (high_checksum_counts));
  memset (low_checksum_counts, 0, sizeof (low_checksum_counts));
  memset (hash_collision_counts, 0, sizeof (hash_collision_counts));

  for(i=0; i<(1<<16); i+=1)
    {
      low_checksum_counts[table->low_checksums[i]] += 1;
      high_checksum_counts[table->high_checksums[i]] += 1;
    }

  for(i=0; i<(1<<table->bits); i += 1)
    hash_collision_counts[cklist_length (table->buckets[i])] += 1;

  for (i=255;i>=0;i-=1)
    {
      coll_metric += hash_collision_counts[i] * i;
#if 0
      if (high_checksum_counts[i] != 0 ||
	  low_checksum_counts[i] != 0 ||
	  hash_collision_counts[i] != 0)
	g_print ("count=%d: high=%d low=%d coll=%d\n",
		 i,
		 high_checksum_counts[i],
		 low_checksum_counts[i],
		 hash_collision_counts[i]);
#endif
    }

  g_print ("Hash items:       %d\n", table->hash_items);
  g_print ("Hash buckets:     %d\n", ipow2(table->bits));
  g_print ("Hash collisions:  %d\n", table->hash_coll);
  g_print ("Collision metric: %d\n", coll_metric);
}
#endif
