/***************************************
  $Header: /home/amb/wwwoffle/RCS/control.c 2.21 1998/03/02 15:05:58 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.1.
  The HTML interactive control pages.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "config.h"
#include "sockets.h"
#include "errors.h"


/*+ The action to perform. +*/
typedef enum _Action
{
 None,                          /*+ Undecided. +*/
 Online,                        /*+ Tell the server that we are online. +*/
 Autodial,                      /*+ Tell the server that we are in autodial mode. +*/
 Offline,                       /*+ Tell the server that we are offline. +*/
 Fetch,                         /*+ Tell the server to fetch the requested pages. +*/
 Config,                        /*+ Tell the server to re-read the configuration file. +*/
 Purge,                         /*+ Tell the server to purge pages. +*/
 Delete,                        /*+ Delete a page from the cache or a request from the outgoing directory. +*/
 ConfigEdit,                    /*+ Edit the config file. +*/
}
Action;


static void MainControlPage(int fd);
static void ActionControlPage(int fd,Action action,char *command);
static void DeleteControlPage(int fd,char *file);
static void IllegalControlPage(int fd,char *path,char *args);
static void ControlAuthFail(int fd,char *path);
static int MatchPassword(char *try,char *actual);


/*++++++++++++++++++++++++++++++++++++++
  Send to the client one of the pages to control WWWOFFLE using HTML.

  int fd The file descriptor of the client.

  char *path The path that was specified.

  char *args The arguments that were passed.

  char *request_head The head of the HTTP request.

  char *request_body The body of the HTTP request.
  ++++++++++++++++++++++++++++++++++++++*/

void ControlPage(int fd,char *path,char *args,char *request_head,char *request_body)
{
 Action action=None;
 char *newpath=(char*)malloc(strlen(path)+1);
 char *command="";

 strcpy(newpath,path);

 if(*newpath && newpath[strlen(newpath)-1]=='/')
    newpath[strlen(newpath)-1]=0;

 if(!strcmp(newpath,"online"))
   {action=Online;command="-online";}
 else if(!strcmp(newpath,"autodial"))
   {action=Autodial;command="-autodial";}
 else if(!strcmp(newpath,"offline"))
   {action=Offline;command="-offline";}
 else if(!strcmp(newpath,"fetch"))
   {action=Fetch;command="-fetch";}
 else if(!strcmp(newpath,"config"))
   {action=Config;command="-config";}
 else if(!strcmp(newpath,"purge"))
   {action=Purge;command="-purge";}
 else if(!strcmp(newpath,"delete"))
    action=Delete;
 else if(!strcmp(newpath,"edit"))
    action=ConfigEdit;

 if(PassWord)
   {
    char *auth=strstr(request_head,"\nAuthorization:");

    if(!auth && (action==Delete && !strncmp(args,"password=",9)))
       ;
    else if(!auth)
      {
       ControlAuthFail(fd,newpath);
       return;
      }
    else
      {
       char *eol,*type,*pswd;
       int n;

       auth+=15;
       eol=strchr(auth,'\n');
       type=(char*)malloc(eol-auth);
       pswd=(char*)malloc(eol-auth);
       *eol=0;
       n=sscanf(auth,"%s %s",type,pswd);
       *eol='\n';

       if(n!=2 || strcasecmp(type,"Basic") || !MatchPassword(pswd,PassWord))
         {
          ControlAuthFail(fd,newpath);
          PrintMessage(Important,"Interactive control webpage authorisation failed.");
          return;
         }
      }
   }

 if(action==None && *newpath)
   {
    IllegalControlPage(fd,newpath,NULL);
    return;
   }

 if(action==None)
    MainControlPage(fd);
 else if(action==Delete)
    DeleteControlPage(fd,args);
 else if(action==ConfigEdit)
    ConfigEditPage(fd,args,request_body);
 else
    ActionControlPage(fd,action,command);
}


/*++++++++++++++++++++++++++++++++++++++
  The main control page with all of the buttons.

  int fd The file descriptor to write to.
  ++++++++++++++++++++++++++++++++++++++*/

static void MainControlPage(int fd)
{
 char *page=
 "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Interactive Control Page</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Interactive Control Page</H1>\n"
 "The <b>wwwoffled</b> demon program can be controlled from here.\n"
 "<h2>Command Line Replacements</h2>\n"
 "All of the control actions available in the <b>wwwoffle</b> program are here.\n"
 "<p>\n"
 "<form action=\"/control/online\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"online\">\n"
 "<input type=\"submit\" value=\"Online\"> Put the <b>wwwoffled</b> program online.\n"
 "</form>\n"
 "<form action=\"/control/autodial\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"autodial\">\n"
 "<input type=\"submit\" value=\"Autodial\"> Put the <b>wwwoffled</b> program in the autodial mode.\n"
 "</form>\n"
 "<form action=\"/control/offline\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"offline\">\n"
 "<input type=\"submit\" value=\"Offline\"> Put the <b>wwwoffled</b> program offline.\n"
 "</form>\n"
 "<form action=\"/control/fetch\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"fetch\">\n"
 "<input type=\"submit\" value=\"Fetch\"> Fetch the pages that <b>wwwoffled</b> has pending.\n"
 "</form>\n"
 "<form action=\"/control/purge\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"purge\">\n"
 "<input type=\"submit\" value=\"Purge\"> Force the <b>wwwoffled</b> cache to be purged.\n"
 "</form>\n"
 "<h2>Edit The Configuration File</h2>\n"
 "The configuration file (wwwoffle.conf) can be edited on the following page.\n"
 "<p>\n"
 "<a href=\"/control/edit\">Edit the configuration file</a>.\n"
 "<p>\n"
 "<form action=\"/control/config\" method=post>\n"
 "<input type=\"hidden\" name=action value=\"config\">\n"
 "<input type=\"submit\" value=\"Config\"> Force <b>wwwoffled</b> to re-read the configuration file.\n"
 "</form>\n"
 "<h2>Delete Pages or Outgoing Requests</h2>\n"
 "You can also delete a specified cached URL or a request from the outgoing directory.\n"
 "<form action=\"/control/delete\" method=get>\n"
 "<input type=\"submit\" value=\"Delete URL From Cache\">\n"
 "<input type=\"text\" size=40 name=url value=\"\">\n"
 "<br><input type=\"checkbox\" name=\"all\" value=\"yes\"> Delete all files from a particular host\n"
 "</form>\n"
 "<form action=\"/control/delete\" method=get>\n"
 "<input type=\"submit\" value=\"Delete URL From Monitored list\">\n"
 "<input type=\"text\" size=40 name=mon value=\"\">\n"
 "</form>\n"
 "<form action=\"/control/delete\" method=get>\n"
 "<input type=\"submit\" value=\"Delete URL From Outgoing\">\n"
 "<input type=\"text\" size=40 name=req value=\"\">\n"
 "<br><input type=\"checkbox\" name=\"all\" value=\"yes\"> Delete all outgoing files\n"
 "</form>\n"
 "<p align=center>[<a href=\"/\">Back to the Welcome page</a>]</p>\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,page);
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that performs an action.

  int fd The file descriptor to write to.

  Action action The action to perform.

  char *command The command line argument that would be used with wwwoffle.
  ++++++++++++++++++++++++++++++++++++++*/

static void ActionControlPage(int fd,Action action,char *command)
{
 int socket=OpenClientSocket("localhost",WWWOFFLE_Port);
 init_buffer(socket);

 if(socket==-1)
   {
    ServerError(fd,"Cannot open connection to wwwoffle server on localhost");
    PrintMessage(Warning,"Cannot open connection to wwwoffle server localhost port %d.",WWWOFFLE_Port);
   }
 else
   {
    char *buffer=NULL;
    char *string=(char*)malloc(32+strlen(command)+strlen(ConfigFile));
    char *head=
    "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
    "Content-type: text/html\r\n"
    "\r\n"
    "<HTML>\n"
    "<HEAD>\n"
    "<TITLE>WWWOFFLE - Interactive Control Page</TITLE>\n"
    "</HEAD>\n"
    "<BODY>\n"
    "<H1 align=center>WWWOFFLE Interactive Control Page</H1>\n"
    "This page is equivalent to using <b>wwwoffle</b> on the command line\n"
    "<pre>\n";
    char *middle=
    "</pre>\n"
    "The output of this command is:\n"
    "<pre>\n";
    char *tail=
    "</pre>\n"
    "<p align=center>[<a href=\"/control/\">Back to the Control page</a>]</p>"
    "</BODY>\n"
    "</HTML>\n";

    write_string(fd,head);

    sprintf(string,"wwwoffle %s -c %s\n",command,ConfigFile);
    write_string(fd,string);

    /* Send the message. */

    if(PassWord)
       write_formatted(socket,"WWWOFFLE PASSWORD %s\r\n",PassWord);

    if(action==Online)
       write_string(socket,"WWWOFFLE ONLINE\r\n");
    else if(action==Autodial)
       write_string(socket,"WWWOFFLE AUTODIAL\r\n");
    else if(action==Offline)
       write_string(socket,"WWWOFFLE OFFLINE\r\n");
    else if(action==Fetch)
       write_string(socket,"WWWOFFLE FETCH\r\n");
    else if(action==Config)
       write_string(socket,"WWWOFFLE CONFIG\r\n");
    else if(action==Purge)
       write_string(socket,"WWWOFFLE PURGE\r\n");

    write_string(fd,middle);

    while((buffer=read_line_or_timeout(socket,buffer)))
       write_string(fd,buffer);

    write_string(fd,tail);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  The control page that deletes a cached page or a request.

  int fd The file descriptor to write to.

  char *args The arguments specified.
  ++++++++++++++++++++++++++++++++++++++*/

static void DeleteControlPage(int fd,char *args)
{
 char *head=
 "HTTP/1.0 200 WWWOFFLE Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Delete Page</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Delete Page</H1>\n";
 char *tail=
 "<p align=center>[<a href=\"/control/\">Go to the Control page</a>]</p>"
 "</BODY>\n"
 "</HTML>\n";
 char *url=NULL,*req=NULL,*mon=NULL,*password=NULL,*copy;
 int i,all=0;

 copy=(char*)malloc(strlen(args)+1);
 strcpy(copy,args);

 for(i=0;copy[i];i++)
   {
    if(i!=0 && copy[i-1]=='&')
       copy[i-1]=0;
    if(i==0 || copy[i-1]==0)
      {
       if(!strncmp("password=",&copy[i],9))
          password=copy+i+9;
       if(!strncmp("url=",&copy[i],4))
          url=copy+i+4;
       if(!strncmp("mon=",&copy[i],4))
          mon=copy+i+4;
       if(!strncmp("req=",&copy[i],4))
          req=copy+i+4;
       if(!strncmp("all=yes",&copy[i],7))
          all=1;
      }
   }

 if((!url && !mon && !req) || (url && mon) || (url && req) || (mon && req) || (mon && all) || (password && (all || !req)))
   {
    IllegalControlPage(fd,"delete",args);
    PrintMessage(Important,"Invalid interactive delete page requested; args='%s'.",args);
   }
 else
   {
    if(password)
      {
       char *decreq=UrlDecode(req,0);
       URL *Url=SplitURL(decreq);
       char *hash=HashOutgoingSpoolFile(Url);

       FreeURL(Url);
       free(decreq);

       if(!hash || strcmp(password,hash))
         {
          char *controlurl=(char*)malloc(16+strlen(args));
          sprintf(controlurl,"delete?%s",args);
          ControlAuthFail(fd,controlurl);
          free(controlurl);
          PrintMessage(Important,"Interactive control webpage delete with password failed.");
          return;
         }
      }

    if(req && all)
      {
       char *err=DeleteOutgoingSpoolFile(NULL);

       write_string(fd,head);
       if(err)
          write_formatted(fd,"There was an error while processing the delete request for all outgoing requests.\n"
                             "<br>The error message was\n"
                             "<br><b>%s</b>\n",
                             err);
       else
          write_string(fd,"All the requested files have been removed from the outgoing directory.\n");
       write_string(fd,tail);
      }
    else if(req)
      {
       char *decreq=UrlDecode(req,0);
       URL *Url=SplitURL(decreq);

       if(Url->Protocol)
         {
          char *err=DeleteOutgoingSpoolFile(Url);

          write_string(fd,head);
          if(err)
             write_formatted(fd,"There was an error while processing the delete request for the page\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the outgoing directory.  The error message was\n"
                                "<br><b>%s</b>\n",
                                Url->name,err);
          else
             write_formatted(fd,"The URL\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>has been removed from the outgoing directory.\n",
                                Url->name);
          write_string(fd,tail);
         }
       else
         {
          char *controlurl=(char*)malloc(24+strlen(args));
          sprintf(controlurl,"/control/delete?%s",args);
          IllegalProto(fd,controlurl,Url->proto);
          free(controlurl);
         }

       FreeURL(Url);
       free(decreq);
      }
    else if(url)
      {
       char *decurl=UrlDecode(url,0);
       URL *Url=SplitURL(decurl);

       if(Url->Protocol)
         {
          char *err=DeleteWebpageSpoolFile(Url,all);

          write_string(fd,head);
          if(err && all)
             write_formatted(fd,"There was an error while processing the delete request for all the pages from host\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the cache.  The error message was\n"
                                "<br><b>%s</b>\n",
                                Url->name,err);
          else if(err)
             write_formatted(fd,"There was an error while processing the delete request for the page\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the cache.  The error message was\n"
                                "<br><b>%s</b>\n",
                                Url->name,err);
          else if(all)
             write_formatted(fd,"All of the pages from host\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the cache have been deleted.\n",
                                Url->name);
          else
             write_formatted(fd,"The page\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the cache has been deleted.\n",
                                Url->name);
          write_string(fd,tail);

          DeleteLastTimeSpoolFile(Url);
         }
       else
         {
          char *controlurl=(char*)malloc(24+strlen(args));
          sprintf(controlurl,"/control/delete?%s",args);
          IllegalProto(fd,controlurl,Url->proto);
          free(controlurl);
         }

       FreeURL(Url);
       free(decurl);
      }
    else /* mon */
      {
       char *decmon=UrlDecode(mon,0);
       URL *Url=SplitURL(decmon);

       if(Url->Protocol)
         {
          char *err=DeleteMonitorSpoolFile(Url);

          write_string(fd,head);
          if(err)
             write_formatted(fd,"There was an error while processing the delete request for the URL\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the monitored list.  The error message was\n"
                                "<br><b>%s</b>\n",
                                Url->name,err);
          else
             write_formatted(fd,"The page\n"
                                "<br><b><tt>%s</tt></b>\n"
                                "<br>in the monitored list has been deleted.\n",
                                Url->name);
          write_string(fd,tail);
         }
       else
         {
          char *controlurl=(char*)malloc(24+strlen(args));
          sprintf(controlurl,"/control/delete?%s",args);
          IllegalProto(fd,controlurl,Url->proto);
          free(controlurl);
         }

       FreeURL(Url);
       free(decmon);
      }
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the specified control page is illegal.

  int fd The file descriptor to write to.

  char *path The specified path.

  char *args The specified arguments.
  ++++++++++++++++++++++++++++++++++++++*/

static void IllegalControlPage(int fd,char *path,char *args)
{
 char *head=
 "HTTP/1.0 404 WWWOFFLE Illegal Control Page\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Illegal Interactive Control Page</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Illegal Interactive Control Page</H1>\n"
 "<p align=center>\n"
 "Your request for the control URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "is illegal, select the link below for the main interactive control page.\n"
 "<br>\n"
 "<a href=\"/control/\">/control/</a>\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 if(args)
    write_formatted(fd,"/control/%s?%s",path,args);
 else
    write_formatted(fd,"/control/%s",path);
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Inform the user that the authorisation failed.

  int fd The file descriptor to write to.

  char *path The specified path.
  ++++++++++++++++++++++++++++++++++++++*/

static void ControlAuthFail(int fd,char *path)
{
 char *head=
 "HTTP/1.0 401 WWWOFFLE Authorisation Failed\r\n"
 "WWW-Authenticate: Basic realm=\"control\"\r\n"
 "Content-type: text/html\r\n"
 "\r\n"
 "<HTML>\n"
 "<HEAD>\n"
 "<TITLE>WWWOFFLE - Authorisation Failed</TITLE>\n"
 "</HEAD>\n"
 "<BODY>\n"
 "<H1 align=center>WWWOFFLE Authorisation Failed</H1>\n"
 "<p align=center>\n"
 "Your request for the interactive control URL\n"
 "<br><b><tt>\n";
 char *tail=
 "\n"
 "</tt></b><br>\n"
 "requires a password and you have failed to be authorised.\n"
 "<p align=center>[<a href=\"/\">Back to the Welcome page</a>]</p>\n"
 "</BODY>\n"
 "</HTML>\n";

 write_string(fd,head);
 write_formatted(fd,"/control/%s",path);
 write_string(fd,tail);
}


/*++++++++++++++++++++++++++++++++++++++
  Try to match the authorisation password against the actual one.

  int MatchPassword Returns true if the password is OK.

  char *try The attempted password.

  char *actual The actual password.
  ++++++++++++++++++++++++++++++++++++++*/

static int MatchPassword(char *try,char *actual)
{
 int ok=0,l;
 char *decoded=Base64Decode(try,&l);
 char *colon;

 colon=strchr(decoded,':');

 if(colon && !strcmp(colon+1,actual))
    ok=1;

 free(decoded);

 return(ok);
}
