/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#define SLOPPY_FOCUS

#include "icewm.h"

YFrameWindow::YFrameWindow(YWindow *parent, YFrameClient *client): YFocusWindow(parent) {
    fClient = 0;
    fFocused = false;
    fNextFrame = fPrevFrame = 0;
    fPopupActive = 0;

    style = 0;
    normalX = 0;
    normalY = 0;
    normalWidth = 1;
    normalHeight = 1;
    fSavedFrameState = fState = WithdrawnState;
    movingWindow = 0;
    sizingWindow = 0;
    indicatorsVisible = 0;
    fFrameFunctions = 0;
    fFrameDecors = 0;
    fFrameOptions = 0;
    fFrameIcon = 0;
    fWindowLayer = 0;
    fTaskBarApp = 0;
    fWinListItem = 0;
    fDesktopIcon = 0;
    fTransient = 0;
    fNextTransient = 0;
    fOwner = 0;

    createPointerWindows();

    fClientContainer = new YClientContainer(this, this);
    fClientContainer->show();
 
    fTitleBar = new YFrameTitleBar(this, this);
    fTitleBar->show();
    fMaximizeButton = new YFrameButton(fTitleBar, this, cmdMaximize, cmdMaximizeVert);
    fMaximizeButton->show();
    fMaximizeButton->setToolTip("Maximize");
    fMinimizeButton = new YFrameButton(fTitleBar, this, cmdMinimize, cmdHide);
    fMinimizeButton->setToolTip("Minimize");
    fMinimizeButton->show();
    fCloseButton = new YFrameButton(fTitleBar, this, cmdClose, cmdClose);
    fCloseButton->setToolTip("Close");
    if (useXButton)
        fCloseButton->show();
    fMenuButton = new YFrameButton(fTitleBar, this, cmdSysMenu);
    fMenuButton->show();

    assert(client != 0);
    manage(client);

    getFrameHints();
    getDefaultOptions();
    if (!(frameOptions() & foFullKeys))
        grabKeys();

    if (frameOptions() & foOnTop)
        fWindowLayer = 1;

    insertFrame();
    addAsTransient();
    addTransients();
    fClientContainer->grabButtons();
    if (taskBar && !(frameOptions() & foIgnoreTaskBar)) {
        fTaskBarApp = taskBar->addApp(this);
    }
            
    if (minimizeToDesktop)
        fDesktopIcon = new DesktopIcon(this, app->root());
    if (windowList && !(frameOptions() & foIgnoreWinList))
        fWinListItem = windowList->addWindowListApp(this);
    app->restackWindows(this, 1);
    if (frameOptions() & foAllWorkspaces)
        wmOccupyAll();
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geWindowOpened);
#endif
}

YFrameWindow::~YFrameWindow() {
#ifdef CONFIG_GUIEVENTS
    app->signalGuiEvent(geWindowClosed);
#endif
    if (movingWindow || sizingWindow)
        endMoveSize();
    if (fPopupActive)
        fPopupActive->cancelPopup();
    if (app->colormapWindow() == this)
        app->setColormapWindow(0);

    if (fTaskBarApp) {
        taskBar->removeApp(this);
        fTaskBarApp = 0;
    }
    if (fWinListItem) {
        if (windowList)
            windowList->removeWindowListApp(fWinListItem);
        fWinListItem = 0;
    }
    if (fDesktopIcon) {
        delete fDesktopIcon;
        fDesktopIcon = 0;
    }
    fClientContainer->releaseButtons();
    // perhaps should be done another way
    switchWindow->destroyedFrame(this);
    
    if (fClient != 0) {
        XRemoveFromSaveSet(app->display(), client()->clientWindow());
        XDeleteContext(app->display(), client()->clientWindow(), frameContext);
    }

    deactivate();
    removeTransients();
    removeAsTransient();
    removeFrame();

    delete fClient; fClient = 0;
    delete fClientContainer; fClientContainer = 0;
    delete fMenuButton; fMenuButton = 0;
    delete fCloseButton; fCloseButton = 0;
    delete fMaximizeButton; fMaximizeButton = 0;
    delete fMinimizeButton; fMinimizeButton = 0;
    delete fTitleBar; fTitleBar = 0;

    XDestroyWindow(app->display(), topSide);
    XDestroyWindow(app->display(), leftSide);
    XDestroyWindow(app->display(), rightSide);
    XDestroyWindow(app->display(), bottomSide);
    XDestroyWindow(app->display(), topLeftCorner);
    XDestroyWindow(app->display(), topRightCorner);
    XDestroyWindow(app->display(), bottomLeftCorner);
    XDestroyWindow(app->display(), bottomRightCorner);
}


// create 8 windows that are used to show the proper pointer
// on frame (for resize)
void YFrameWindow::createPointerWindows() {
    XSetWindowAttributes attributes;
    unsigned int klass = InputOnly;

    attributes.event_mask = 0;

    attributes.cursor = sizeTopPointer;
    topSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeLeftPointer;
    leftSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeRightPointer;
    rightSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeBottomPointer;
    bottomSide = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                            CopyFromParent, klass, CopyFromParent,
                            CWCursor | CWEventMask, &attributes);
    
    attributes.cursor = sizeTopLeftPointer;
    topLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                  CopyFromParent, klass, CopyFromParent,
                                  CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeTopRightPointer;
    topRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                   CopyFromParent, klass, CopyFromParent,
                                   CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomLeftPointer;
    bottomLeftCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                     CopyFromParent, klass, CopyFromParent,
                                     CWCursor | CWEventMask, &attributes);

    attributes.cursor = sizeBottomRightPointer;
    bottomRightCorner = XCreateWindow(app->display(), handle(), 0, 0, 1, 1, 0,
                                      CopyFromParent, klass, CopyFromParent,
                                      CWCursor | CWEventMask, &attributes);

    XMapSubwindows(app->display(), handle());
    indicatorsVisible = 1;
}

void YFrameWindow::grabKeys() {
    grabKey(XK_Tab, Mod1Mask);
    grabKey(XK_Tab, Mod1Mask | ShiftMask);
    grabKey(XK_F1, Mod1Mask);
    grabKey(XK_F2, Mod1Mask);
    grabKey(XK_F3, Mod1Mask);
    grabKey(XK_F4, Mod1Mask);
    grabKey(XK_F5, Mod1Mask);
    grabKey(XK_F6, Mod1Mask);
    grabKey(XK_F7, Mod1Mask);
    grabKey(XK_F8, Mod1Mask);
    grabKey(XK_F9, Mod1Mask);
    grabKey(XK_F10, Mod1Mask);
    grabKey(XK_F11, Mod1Mask);
    grabKey(XK_F12, Mod1Mask);
    //grabKey(' ', Mod1Mask);
}

void YFrameWindow::manage(YFrameClient *client) {
    assert(client != 0);
    fClient = client;
    
    XSetWindowBorderWidth(app->display(),
                          client->handle(),
                          0);
    
    XAddToSaveSet(app->display(), client->clientWindow());

    client->reparent(fClientContainer, 0, 0);

    client->setFrame(this);
}

void YFrameWindow::unmanage() {
    assert(fClient != 0);
     
    if (!fClient->destroyed()) {
        int gx, gy;
#ifdef STRICT_POSITIONING
        gx = gy = 0;
#else
        client()->gravityOffsets(gx, gy);
#endif
        
        XSetWindowBorderWidth(app->display(),
                              client()->handle(),
                              client()->border());

        client()->reparent(app->root(), 
#if 1 /* !!! ICCCM specified policy :-( */
                           x() - borderX() * gx,
                           y() - borderY() * gy
#else
                           x(), y()
#endif
                          );

        XRemoveFromSaveSet(app->display(), client()->clientWindow());
    }

    client()->setFrame(0);

    fClient = 0;
}

void YFrameWindow::configureClient(const XConfigureRequestEvent &configureRequest) {
    int cx, cy, cwidth, cheight;

    client()->setBorder((configureRequest.value_mask & CWBorderWidth) ? configureRequest.border_width : client()->border());
    cx = (configureRequest.value_mask & CWX) ? configureRequest.x + DO_BORDER(client()->border()) : x() + borderX();
    cy = (configureRequest.value_mask & CWY) ? configureRequest.y + DO_BORDER(client()->border()) : y() + titleY() + borderY();
    cwidth = (configureRequest.value_mask & CWWidth) ? configureRequest.width : client()->width();
    cheight = (configureRequest.value_mask & CWHeight) ? configureRequest.height : client()->height();

    client()->constrainSize(cwidth, cheight);

    cx -= borderX();
    cy -= borderY() + titleY();
    cwidth += 2 * borderX();
    cheight += 2 * borderY() + titleY();

#if 0
    int gx, gy;
    client()->gravityOffsets(gx, gy);

    cx += borderX() * gx;
    cy += borderY() * gx;
#endif

    if (cx != x() || cy != y() ||
        (unsigned int)cwidth != width() || (unsigned int)cheight != height())
        if (style & fsMaximized) {
            style &= ~fsMaximized;
            fMaximizeButton->setCommand(cmdMaximize, cmdMaximizeVert);
        }

    setGeometry(cx, cy, cwidth, cheight);

    if (configureRequest.value_mask & CWStackMode) {
        YFrameWindow *sibling = 0;
        XWindowChanges xwc;

        if ((configureRequest.value_mask & CWSibling) &&
            XFindContext(app->display(),
                         configureRequest.above,
                         frameContext,
                         (XPointer *)&sibling) == 0)
            xwc.sibling = sibling->handle();
        else
            xwc.sibling = configureRequest.above;

        xwc.stack_mode = configureRequest.detail;

        /* !!! implement the rest, and possibly fix these: */

        if (sibling && xwc.sibling != None) { /* ICCCM suggests sibling=None */
            switch (xwc.stack_mode) {
            case Above:
                setAbove(sibling);
                break;
            case Below:
                setBelow(sibling);
                break;
            default:
                return ;
            }
            XConfigureWindow (app->display(),
                              handle(),
                              configureRequest.value_mask & (CWSibling | CWStackMode),
                              &xwc);
        } else if (xwc.sibling == None && app->top(layer()) != 0) {
            switch (xwc.stack_mode) {
            case Above:
                wmRaise();
		activate();
                break;
            case Below:
                wmLower();
                break;
            default:
                return ;
            }
        } else
            return ;
    }
}

void YFrameWindow::handleClick(const XButtonEvent &/*down*/, const XButtonEvent &up, int /*count*/) {
    if (up.button == 3) {
        updateMenu();
        windowMenu()->popup(0, this,
                            up.x_root, up.y_root, -1, -1,
                            YPopupWindow::pfCanFlipVertical |
                            YPopupWindow::pfCanFlipHorizontal |
                            YPopupWindow::pfPopupMenu);
        
    }
}


void YFrameWindow::handleCrossing(const XCrossingEvent &crossing) {
    if (crossing.type == EnterNotify && crossing.mode == NotifyNormal) {
        if (!clickFocus && visible())
            activate();
#ifndef SLOPPY_FOCUS
    } else if (crossing.type == LeaveNotify && fFocused) {
        if (crossing.detail != NotifyInferior &&
            crossing.mode == NotifyNormal)
        {
            if (!clickFocus) {
                deactivate();
            }
        }
#endif
    }
}

void YFrameWindow::raise() {
    if (this != app->top(layer())) {
        YWindow::raise();
        setAbove(app->top(layer()));
    }
}

void YFrameWindow::lower() {
    if (this != app->bottom(layer())) {
        YWindow::lower();
        setAbove(0);
    }
}

void YFrameWindow::removeFrame() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before removing", this);
#endif
    if (prev())
        prev()->setNext(next());
    else
        app->setTop(layer(), next());

    if (next())
        next()->setPrev(prev());
    else
        app->setBottom(layer(), prev());
 
    setPrev(0);
    setNext(0);

#ifdef DEBUG
    if (debug_z) dumpZorder("after removing", this);
#endif
}

void YFrameWindow::insertFrame() {
#ifdef DEBUG
    if (debug_z) dumpZorder("before inserting", this);
#endif
    setNext(app->top(layer()));
    setPrev(0);
    if (next())
        next()->setPrev(this);
    else
        app->setBottom(layer(), this);
    app->setTop(layer(), this);
#ifdef DEBUG
    if (debug_z) dumpZorder("after inserting", this);
#endif
}

void YFrameWindow::setAbove(YFrameWindow *aboveFrame) {
#ifdef DEBUG
    if (debug_z) dumpZorder("before setAbove", this, aboveFrame);
#endif
    if (aboveFrame != next() && aboveFrame != this) {
        if (prev())
            prev()->setNext(next());
        else
            app->setTop(layer(), next());

        if (next())
            next()->setPrev(prev());
        else
            app->setBottom(layer(), prev());
        
        setNext(aboveFrame);
        if (next()) {
            setPrev(next()->prev());
            next()->setPrev(this);
        } else {
            setPrev(app->bottom(layer()));
            app->setBottom(layer(), this);
        }
        if (prev())
            prev()->setNext(this);
        else
            app->setTop(layer(), this);
#ifdef DEBUG
        if (debug_z) dumpZorder("after setAbove", this, aboveFrame);
#endif
    }
}

void YFrameWindow::setBelow(YFrameWindow *belowFrame) {
    if (belowFrame != next())
        setAbove(belowFrame->next());
}

YFrameWindow *YFrameWindow::findWindow(int flags) {
    YFrameWindow *p = this;

     if (flags & fwfNext) 
         goto next;
     
     do {
         if ((flags & fwfVisible) && !p->visible())
             goto next;
         if ((flags & fwfFocusable) && !p->isFocusable())
             goto next;
         if ((flags & fwfWorkspace) && !p->visibleOn(app->activeWorkspace()))
             goto next;
         
         return p;

     next:
         if (flags & fwfBackward) 
             p = (flags & fwfLayers) ? p->prevLayer() : p->prev();
         else
             p = (flags & fwfLayers) ? p->nextLayer() : p->next();
         if (p == 0)
             if (!(flags & fwfCycle))
                 return 0;
             else if (flags & fwfBackward)
                 p = (flags & fwfLayers) ? app->bottomLayer() : app->bottom(layer());
             else 
                 p = (flags & fwfLayers) ? app->topLayer() : app->top(layer());
     } while (p != this);

     if (!(flags & fwfSame))
         return 0;
     if ((flags & fwfVisible) && !p->visible())
         return 0;
     if ((flags & fwfFocusable) && !p->isFocusable())
         return 0;
     if ((flags & fwfWorkspace) && !p->visibleOn(app->activeWorkspace()))
         return 0;
     
     return this;
}

void YFrameWindow::sendConfigure() {
    XEvent xev;
    
    xev.xconfigure.type = ConfigureNotify;
    xev.xconfigure.display = app->display();
    xev.xconfigure.event = client()->handle();
    xev.xconfigure.window = client()->handle();
    xev.xconfigure.x = x() + borderX() - DO_BORDER(client()->border());
    xev.xconfigure.y = y() + borderY()
#ifndef TITLEBAR_BOTTOM
        + titleY()
#endif
        - DO_BORDER(client()->border());
    xev.xconfigure.width = client()->width();
    xev.xconfigure.height = client()->height();
    xev.xconfigure.border_width = client()->border();

    assert(titlebar() != 0);
    xev.xconfigure.above = None; //titlebar()->handle();
    xev.xconfigure.override_redirect = False;

#ifdef DEBUG_C
    Status rc = 
#endif
        XSendEvent(app->display(),
               client()->clientWindow(),
               False,
               StructureNotifyMask,
               &xev);

#ifdef DEBUG_C
    MSG(("sent %d: x=%d, y=%d, width=%d, height=%d",
         rc,
         x(),
         y(),
         client()->width(),
         client()->height()));
#endif
}

void YFrameWindow::handleCommand(WMCommand command, void *context, unsigned int modifiers) {
    switch (command) {
    case cmdActivate: activate(); break;
    case cmdRestore: wmRestore(); break;
    case cmdMinimize: wmMinimize(); break;
    case cmdMaximize: wmMaximize(); break;
    case cmdMaximizeVert: wmMaximizeVert(); break;
    case cmdLower: wmLower(); break;
    case cmdRaise: wmRaise(); break;
    case cmdRollup: wmRollup(); break;
    case cmdClose: wmClose(); break;
    case cmdKill: wmKill(); break;
    case cmdHide: wmHide(); break;
    case cmdShow: wmShow(); break;
    case cmdMove: wmMove(); break;
    case cmdSize: wmSize(); break;
    case cmdOccupyAll: wmOccupyAll(); break;
    case cmdOccupyAllOrCurrent: wmOccupyAllOrCurrent(); break;
    case cmdOccupyWorkspace: wmOccupyWorkspace((int)context); break;
    case cmdOccupyOnlyWorkspace: wmOccupyOnlyWorkspace((int)context); break;
    case cmdMoveToWorkspace: wmMoveToWorkspace((int)context); break;
    default:
	app->handleCommand(command, context, modifiers);
        break;
    }
}

void YFrameWindow::wmMove() {
    startMoveSize(1, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmSize() {
    startMoveSize(0, 0,
                  0, 0,
                  0, 0);
}

void YFrameWindow::wmRestore() {
    if (style & fsHidden)
        wmHide();
    else if (style & fsMinimized)
        wmMinimize();
    else if (style & fsRolledup)
        wmRollup();
    else if (style & fsMaximized)
        wmMaximize();
}

void YFrameWindow::wmMinimize() {
#ifdef DEBUG_S
    MSG(("wmMinimize - Frame: %d", visible()));
    MSG(("wmMinimize - Client: %d", client()->visible()));
#endif
    if (style & fsMinimized) {
        style &= ~fsMinimized;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        hideIcon();
        changeState(NormalState);
        if (this == app->focus())
            app->focusTopWindow();
    } else {
        if (!canMinimize())
            return ;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowMin);
#endif
        style |= fsMinimized;
        changeState(IconicState);
        if (this == app->focus())
            app->focusTopWindow();
        showIcon();
    }
}

void YFrameWindow::DoMaximize(int what) {
    if (style & fsRolledup)
        wmRollup();

    XSizeHints *sh = client()->sizeHints();
    
    if (style & fsMaximized) {
        style &= ~fsMaximized;
        fMaximizeButton->setCommand(cmdMaximize, cmdMaximizeVert);

        if (sh) {
            normalWidth = normalWidth * sh->width_inc + sh->base_width;
            normalHeight = normalHeight * sh->height_inc + sh->base_height;
        }
        
        setGeometry(normalX, normalY,
                    normalWidth + 2 * borderX(),
                    normalHeight + 2 * borderY() + titleY());
        if (style & fsMinimized)
            wmMinimize();
    } else {
        if (!canMaximize())
            return ;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowMax);
#endif

        style |= fsMaximized;
        fMaximizeButton->setCommand(cmdRestore, cmdRestore);
        
        normalX = x();
        normalY = y();
        normalWidth = client()->width();
        normalHeight = client()->height();

        if (sh) {
            normalWidth = (normalWidth - sh->base_width) / sh->width_inc;
            normalHeight = (normalHeight - sh->base_height) / sh->height_inc;
        }
        
        int nx = x() + borderX();
        int ny = y() + borderY();
        int nw = (what & 2) ? app->maxWidth() : client()->width();
        int nh = (what & 1) ? app->maxHeight() - titleY() : client()->height();
        
        client()->constrainSize(nw, nh);
        
        //nx = x();
        //ny = y();
        //app->constrainPosition(x, y, nw, nh);

        if (nx + nw > int(app->maxX()))
            nx = app->minX(); //app->maxX() - nw - borderX();

        if (ny + nh + int(wsTitleBar) > int(app->maxY()))
            ny = app->minY(); //app->maxY() - nh - titleY() - borderY();

        if (nx < app->minX())
            nx = app->minX();
        if (ny < app->minY())
            ny = app->minY();
        
        nx -= borderX();
        ny -= borderY();
        nw += 2 * borderX();
        nh += 2 * borderY() + titleY();
        
        setGeometry(nx, ny, nw, nh);

        if (style & fsMinimized)
            wmMinimize();
    }
    
}

void YFrameWindow::wmMaximize() {
    DoMaximize(3);

}

void YFrameWindow::wmMaximizeVert() {
    DoMaximize(1);
}

void YFrameWindow::wmMaximizeHorz() {
    DoMaximize(2);
}

void YFrameWindow::wmRollup() {
#ifdef DEBUG_S
    MSG(("wmRollup- Frame: %d", visible()));
    MSG(("wmRollup- Client: %d", client()->visible()));
#endif
    if (style & fsRolledup) {
        style &= ~fsRolledup;
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        if (!(style & fsMinimized)) {
            changeState(NormalState);
            setGeometry(x(),
                        y()
#ifdef TITLEBAR_BOTTOM
                        - client()->height()
#endif
                        ,
                        client()->width() + 2 * borderX(),
                        client()->height() + 2 * borderY() + titleY());
        }
        if (focused() && clickFocus)
            setFocus();
    } else {
        if (!canRollup())
            return ;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowShade);
#endif
        style |= fsRolledup;
        changeState(IconicState);
        setGeometry(x(),
                    y()
#ifdef TITLEBAR_BOTTOM
                    + client()->height()
#endif
                    ,
                    client()->width() + 2 * borderX(),
                    2 * borderY() + titleY());

    }
    if (focused()) {
        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);
    }
#ifdef DEBUG_S
    MSG(("/wmRollup- Frame: %d", visible()));
    MSG(("/wmRollup- Client: %d", client()->visible()));
#endif
}

void YFrameWindow::wmHide() {
    if (style & fsHidden) {
        style &= ~fsHidden;

        if (style & fsMinimized)
            showIcon();
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowRestore);
#endif
        changeState(NormalState);
        if (fTaskBarApp == 0 && !(frameOptions() & foIgnoreTaskBar))
            fTaskBarApp = taskBar->addApp(this);
        app->focusTopWindow();
    } else {
        if (!canHide())
            return ;

        hideIcon();
#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowHide);
#endif
        style |= fsHidden;
        changeState(IconicState);
        if (fTaskBarApp) {
            taskBar->removeApp(this);
            fTaskBarApp = 0;
        }
        app->focusTopWindow();
    }
}

void YFrameWindow::wmLower() {
    if (this != app->bottom(layer())) {
        YFrameWindow *w = this;

#ifdef CONFIG_GUIEVENTS
        app->signalGuiEvent(geWindowLower);
#endif
        while (w) {
            w->doLower();
            w = w->owner();
        }
        app->restackWindows(this, -1);
        if (clickFocus)
            app->focusTopWindow();
    }
}

void YFrameWindow::doLower() {
    setAbove(0);
}

void YFrameWindow::wmRaise() {
    doRaise();
    app->restackWindows(this, 1);
}

void YFrameWindow::doRaise() {
#ifdef DEBUG
    if (debug_z) dumpZorder("wmRaise: ", this);
#endif
    if (this != app->top(layer())) {
        setAbove(app->top(layer()));
        {
            YFrameWindow *w = transient();
            while (w) {
                w->doRaise();
                w = w->nextTransient();
            }
        }
#ifdef DEBUG
        if (debug_z) dumpZorder("wmRaise after raise: ", this);
#endif
    }
}

void YFrameWindow::wmClose() {
    if (!canClose())
        return ;

    /* race condition otherwise (???) */
    XGrabServer(app->display());
    client()->getProtocols();

    if (client()->protocols() & YFrameClient::wpDeleteWindow)
        client()->sendMessage(_XA_WM_DELETE_WINDOW);
    else
        wmKill();
    XUngrabServer(app->display());
}

void YFrameWindow::wmKill() {
    // !!! must ask user!
    ///return ;
    
    if (!canClose())
        return ;
#ifdef DEBUG
    msg("No WM_DELETE_WINDOW protocol");
#endif
    XBell(app->display(), 100);
    XKillClient(app->display(), client()->clientWindow());
}

void YFrameWindow::wmPrevWindow() {
    if (next() != this)
        app->activate(findWindow(fwfNext | fwfBackward | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
}

void YFrameWindow::wmNextWindow() {
    if (next() != this) {
        wmLower();
        app->activate(findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
    }
}

void YFrameWindow::wmLastWindow() {
    if (next() != this)
        app->activate(findWindow(fwfNext | fwfVisible | fwfCycle | fwfFocusable | fwfWorkspace));
}

void YFrameWindow::loseFocus() {
    if (fFocused) {
        fFocused = false;
        if (focusOnClickClient || raiseOnClickClient)
            if (fClientContainer)
                fClientContainer->grabButtons();
        repaint();
        titlebar()->deactivate();
        if (fTaskBarApp)
            fTaskBarApp->repaint();
    }
}

void YFrameWindow::setFocus() {
    if (!fFocused) {

        fFocused = true;
        titlebar()->activate();
        repaint();
        if (fTaskBarApp)
            fTaskBarApp->repaint();
        
        if (warpPointer && phase == phaseRunning) { /* !!! ugh :-( buggy */
            Window child, root;
            int root_x, root_y, win_x, win_y;
            unsigned int mask;
            
            XQueryPointer(app->display(), app->root()->handle(), &root, &child, 
                          &root_x, &root_y, &win_x, &win_y, &mask);

            if (child != handle()) {
                XWarpPointer(app->display(), 
                             None, app->root()->handle(),
                             0, 0, 0, 0, x(), y());
            }
        }

        XSetInputFocus(app->display(),
                       client()->visible() ? client()->clientWindow() : handle(),
                       RevertToPointerRoot,
                       CurrentTime);

        if (focusOnClickClient &&
           !(raiseOnClickClient && (this != app->top(layer()))))
            fClientContainer->releaseButtons();

        if (client()->protocols() & YFrameClient::wpTakeFocus)
            client()->sendMessage(_XA_WM_TAKE_FOCUS);
    }
}

void YFrameWindow::focusOnMap() {
    if (owner() != 0) {
        if (focusOnMapTransient)
            activate();
    } else {
        if (::focusOnMap)
            activate();
    }
}

void YFrameWindow::wmShow() {
   // recover lost (offscreen) windows
    if (x() >= int(app->root()->width()) ||
        y() >= int(app->root()->height()) ||
        x() <= - int(width()) ||
        y() <= - int(height()))
    {
        int newX = x();
        int newY = y();
        
        if (x() >= int(app->root()->width()))
            newX = int(app->root()->width() - width() + borderX());
        if (y() >= int(app->root()->height()))
            newY = int(app->root()->height() - height() + borderY());
        
        if (newX < int(- borderX()))
            newX = int(- borderX());
        if (newY < int(- borderY()))
            newY = int(- borderY());
        setPosition(newX, newY);
    }

    if (style & fsHidden)
        wmHide();
    if (style & fsMinimized)
        wmMinimize();
}

void YFrameWindow::activate() {
    if (!visibleNow()) {
        for (int w = 0; w < workspaceCount; w++)
            if (visibleOn(w)) {
                app->activateWorkspace(w);
                break;
            }
    }
    // recover lost (offscreen) windows
    if (x() >= int(app->root()->width()) ||
        y() >= int(app->root()->height()) ||
        x() <= - int(width()) ||
        y() <= - int(height()))
    {
        int newX = x();
        int newY = y();
        
        if (x() >= int(app->root()->width()))
            newX = int(app->root()->width() - width() + borderX());
        if (y() >= int(app->root()->height()))
            newY = int(app->root()->height() - height() + borderY());
        
        if (newX < int(- borderX()))
            newX = int(- borderX());
        if (newY < int(- borderY()))
            newY = int(- borderY());
        setPosition(newX, newY);
    }
    
    if (raiseOnFocus)
        wmRaise();
    if (style & fsHidden)
        wmHide();
    if (style & fsMinimized)
        wmMinimize();
    if (isFocusable())
        app->setFocus(this);
}

void YFrameWindow::deactivate() {
    if (app->focus() == this)
        app->loseFocus(this);
}

void YFrameWindow::setFrameState(FrameState state) {
    fState = state;
    client()->setWMState(fState);
}

void YFrameWindow::changeState(FrameState state) {
    if (state == IconicState) {
        if (frameState() == NormalState || frameState() == WithdrawnState || (style & fsRolledup)) {
            if (visibleNow()) {
#ifdef DEBUG_S
                MSG(("->Iconic"));
#endif
                if (!(style & fsRolledup) || (style & (fsMinimized | fsHidden))) {
#ifdef DEBUG_S
                    MSG(("--- Hide Frame: %d", visible()));
#endif
                    hide();
                }
#ifdef DEBUG_S
                MSG(("--- Hide Client: %d", client()->visible()));
#endif
                fClientContainer->hide();
                client()->hide();
                if (frameState() != IconicState)
                    setFrameState(IconicState);
            } else {
                fSavedFrameState = state;
            }
        }
    } else if (state == NormalState) {
        if (frameState() == IconicState || frameState() == WithdrawnState) {
            if (visibleNow()) {
#ifdef DEBUG_S
                MSG(("->Normal"));
#endif
                if (!(style & fsRolledup)) {
#ifdef DEBUG_S
                    MSG(("--- Show Client: %d", client()->visible()));
#endif
                    client()->show();
                    fClientContainer->show();
                }
#ifdef DEBUG_S
                MSG(("--- Show Frame: %d", visible()));
#endif
                show();
                if (!(style & fsRolledup))
                    if (frameState() != NormalState)
                        setFrameState(NormalState);
            } else {
                fSavedFrameState = state;
            }
        }
    }
#ifdef DEBUG_S
    MSG(("- Frame: %d", visible()));
    MSG(("- Client: %d", client()->visible()));
#endif
}

void YFrameWindow::paint(Graphics &g, int , int , unsigned int , unsigned int ) {
    YColor *bg;

    if (focused())
        bg = activeBorderBg;
    else
        bg = inactiveBorderBg;

    g.setColor(bg);
    switch (wmLook) {
#if defined(CONFIG_LOOK_WIN95) || defined(CONFIG_LOOK_WARP4) || defined(CONFIG_LOOK_NICE)
#ifdef CONFIG_LOOK_WIN95
    case lookWin95:
#endif
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4:
#endif
#ifdef CONFIG_LOOK_NICE
    case lookNice:
#endif
        g.fillRect(1, 1, width() - 3, height() - 3);
        g.drawBorderW(0, 0, width() - 1, height() - 1, true);
        break;
#endif
#if defined(CONFIG_LOOK_MOTIF) || defined(CONFIG_LOOK_WARP3)
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif:
#endif
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3:
#endif
        g.fillRect(1, 1, width() - 2, height() - 2);
        g.draw3DRect(0, 0, width() - 1, height() - 1, true);
        g.draw3DRect(borderX() - 1, borderY() - 1,
                     width() - 2 * borderX() + 1, height() - 2 * borderY() + 1,
                     false);

#ifdef CONFIG_LOOK_MOTIF
        if (wmLook == lookMotif && canSize()) {
            YColor *b = bg->brighter();
            YColor *d = bg->darker();


            g.setColor(d);
            g.drawLine(wsCornerX - 1, 0, wsCornerX - 1, height() - 1);
            g.drawLine(width() - wsCornerX - 1, 0, width() - wsCornerX - 1, height() - 1);
            g.drawLine(0, wsCornerY - 1, width(),wsCornerY - 1);
            g.drawLine(0, height() - wsCornerY - 1, width(), height() - wsCornerY - 1);
            g.setColor(b);
            g.drawLine(wsCornerX, 0, wsCornerX, height() - 1);
            g.drawLine(width() - wsCornerX, 0, width() - wsCornerX, height() - 1);
            g.drawLine(0, wsCornerY, width(), wsCornerY);
            g.drawLine(0, height() - wsCornerY, width(), height() - wsCornerY);
        }
        break;
#endif
#endif
#ifdef CONFIG_LOOK_PIXMAP
    case lookPixmap:
        {
            int n = focused() ? 1 : 0;
            int t = (frameDecors() & fdResize) ? 0 : 1;

            g.drawPixmap(frameTL[t][n], 0, 0);
            g.repHorz(frameT[t][n], wsCornerX, 0, width() - 2 * wsCornerX);
            g.drawPixmap(frameTR[t][n], width() - wsCornerX, 0);
            g.repVert(frameL[t][n], 0, wsCornerY, height() - 2 * wsCornerY);
            g.repVert(frameR[t][n], width() - borderX(), wsCornerY, height() - 2 * wsCornerY);
            g.drawPixmap(frameBL[t][n], 0, height() - wsCornerY);
            g.repHorz(frameB[t][n], wsCornerX, height() - borderY(), width() - 2 * wsCornerX);
            g.drawPixmap(frameBR[t][n], width() - wsCornerX, height() - wsCornerY);
        }
        break;
#endif
    default:
        break;
    }
}

void YFrameWindow::setPopupActive(YPopupWindow *popup) {
    MSG(("setting popup for frame to %ld", popup));
    fPopupActive = popup;
}

void YFrameWindow::popupSystemMenu() {
    if (fPopupActive == 0)
        fMenuButton->popupMenu();
}

void YFrameWindow::updateTitle() {
    titlebar()->repaint();
    updateIconTitle();
}

void YFrameWindow::updateIconTitle() {
    if (fTaskBarApp) {
        fTaskBarApp->repaint();
        fTaskBarApp->setToolTip((char *)client()->windowTitle());
    }
}

int YFrameWindow::visibleNow() {
    return (style & fsWorkspaceHidden) ? 0 : 1;
}

void YFrameWindow::wmOccupyAllOrCurrent() {
    if (client()->allWorkspaces()) {
        client()->setWorkspaces(1 << app->activeWorkspace());
        client()->setAllWorkspaces(0);
    } else {
        client()->setAllWorkspaces(1);
    }
    if (!visibleOn(app->activeWorkspace()))
        workspaceHide();
    taskBar->relayout();
}

void YFrameWindow::wmOccupyAll() {
    if (client()->allWorkspaces()) {
        client()->setAllWorkspaces(0);
    } else {
        client()->setAllWorkspaces(1);
    }
    if (!visibleOn(app->activeWorkspace()))
        workspaceHide();
    taskBar->relayout();
}

void YFrameWindow::wmOccupyWorkspace(int workspace) {
    assert(workspace >= 0 && workspace < workspaceCount);
    unsigned long nw = client()->workspaces() ^ (1 << workspace);
    if (nw == 0)
        nw = 1 << app->activeWorkspace();
    client()->setWorkspaces(nw);
    if (!visibleOn(app->activeWorkspace()))
        workspaceHide();
    taskBar->relayout();
}

void YFrameWindow::wmOccupyOnlyWorkspace(int workspace) {
    assert(workspace >= 0 && workspace < workspaceCount);
    client()->setWorkspaces(1 << workspace);
    client()->setAllWorkspaces(0);
    if (!visibleOn(app->activeWorkspace()))
        workspaceHide();
    taskBar->relayout();
}

void YFrameWindow::wmMoveToWorkspace(int workspace) {
    // ??? is this ok?
    // alternatively, we could move window from current workspace to
    // target one, leaving others unchanged
    wmOccupyOnlyWorkspace(workspace);
}

void YFrameWindow::workspaceShow() {
    if (style & fsWorkspaceHidden) {
        style &= ~fsWorkspaceHidden;

        fState = fSavedFrameState; 
        if (frameState() == NormalState)
            show();
        if (style & fsRolledup)
            show();

        if (!taskBarShowAllWindows)
            if (fTaskBarApp)
                fTaskBarApp->setShown(1);
        
        app->focusTopWindow();
    }
}

void YFrameWindow::workspaceHide() {
    if (!(style & fsWorkspaceHidden)) {
        style |= fsWorkspaceHidden;

        fSavedFrameState = frameState();
        /*setFrameState(IconicState);
        hide();
        fClientContainer->hide();
        client()->hide();*/

        hide();
        fState = IconicState;
        
        if (!taskBarShowAllWindows)
            if (fTaskBarApp)
                fTaskBarApp->setShown(0);

        //app->focusTopWindow();
    }
}

void YFrameWindow::addToMenu(YMenu *menu) {
    YMenu::YMenuItem *item =
        new YMenu::YMenuItem((char *)client()->windowTitle(), -1, 0,
                             cmdActivateWindow, 0, this);
    if (item) {
        item->setPixmap(clientIcon()->small());
        menu->add(item);
    }
}

void YFrameWindow::getFrameHints() {
#ifndef NO_MWM_HINTS
    CARD32 decors = client()->mwmDecors();
    CARD32 functions = client()->mwmFunctions();

    fFrameFunctions = ffRollup;
    fFrameDecors = 0;
    fFrameOptions = 0;
    
    if (decors & MWM_DECOR_BORDER)      fFrameDecors |= fdBorder;
    if (decors & MWM_DECOR_RESIZEH)     fFrameDecors |= fdResize;
    if (decors & MWM_DECOR_TITLE)       fFrameDecors |= fdTitleBar;
    if (decors & MWM_DECOR_MENU)        fFrameDecors |= fdSysMenu;
    if (decors & MWM_DECOR_MAXIMIZE)    fFrameDecors |= fdMaximize;
    if (decors & MWM_DECOR_MINIMIZE)    fFrameDecors |= fdMinimize;

    if (functions & MWM_FUNC_MOVE)      fFrameFunctions |= ffMove;
    if (functions & MWM_FUNC_RESIZE)    fFrameFunctions |= ffResize;
    if (functions & MWM_FUNC_MAXIMIZE)  fFrameFunctions |= ffMaximize;
    if (functions & MWM_FUNC_MINIMIZE)
        fFrameFunctions |= ffMinimize | ffHide;
    if (functions & MWM_FUNC_CLOSE) {
        fFrameFunctions |= ffClose;
        fFrameDecors |= fdClose; /* hack */
    }
#else
    fFrameFunctions =
        ffMove | ffResize | ffClose |
        ffMinimize | ffMaximize | ffHide | ffRollup;
    fFrameDecors =
        fdTitleBar | fdSysMenu | fdBorder | fdResize |
        fdClose | fdMinimize | fdMaximize;
    fFrameOptions = 0;
#endif

#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;

    getWindowOptions(wo);
    fFrameFunctions &= ~wo.function_mask;
    fFrameFunctions |= wo.functions;
    fFrameDecors &= ~wo.decor_mask;
    fFrameDecors |= wo.decors;
    fFrameOptions &= ~wo.option_mask;
    fFrameOptions |= wo.options;
#endif
}

#ifndef NO_WINDOW_OPTIONS
static void combineOptions(WindowOption &cm, WindowOption &n) {
    if (cm.icon == 0)
        if (n.icon != 0)
            cm.icon = n.icon;
    cm.functions |= n.functions & ~cm.function_mask;
    cm.function_mask |= n.function_mask;
    cm.decors |= n.decors & ~cm.decor_mask;
    cm.decor_mask |= n.decor_mask;
    cm.options |= n.options & ~cm.option_mask;
    cm.option_mask |= n.option_mask;
    if (cm.workspace != -1)
        cm.workspace = n.workspace;
}

void YFrameWindow::getWindowOptions(WindowOption &opt) {
    memset((void *)&opt, 0, sizeof(opt));
    opt.workspace = -1;

    XClassHint *h = client()->classHint();
    WindowOption *wo;
    
    if (!h)
        return;

    if (h->res_name && h->res_class) {
        char *both = (char *) malloc(strlen(h->res_name) + 1 +
                                     strlen(h->res_class) + 1);
        if (both) {
            strcpy(both, h->res_class);
            strcat(both, ".");
            strcat(both, h->res_name);
        }
        wo = both ? getWindowOption(both, 0) : 0;
        if (wo) combineOptions(opt, *wo);
        free(both);
    }
    if (h->res_class) {
        wo = getWindowOption(h->res_class, 0);
        if (wo) combineOptions(opt, *wo);
    }
    if (h->res_name) {
        wo = getWindowOption(h->res_name, 0);
        if (wo) combineOptions(opt, *wo);
    }
    wo = getWindowOption(0, 0);
    if (wo) combineOptions(opt, *wo);
}
#endif

void YFrameWindow::getDefaultOptions() {
    delete fFrameIcon;
    fFrameIcon = 0;

#ifndef NO_WINDOW_OPTIONS
    WindowOption wo;
    
    getWindowOptions(wo);
    
    if (wo.icon)
        fFrameIcon = getIcon(wo.icon);
    if (wo.workspace >= 0 && wo.workspace < workspaceCount)
        wmOccupyWorkspace(wo.workspace);
#endif
}

YFrameWindow *YFrameWindow::nextLayer() {
    return fNextFrame ? fNextFrame : ((layer() == 1) ? app->top(0) : 0);
}
YFrameWindow *YFrameWindow::prevLayer() {
    return fPrevFrame ? fPrevFrame : ((layer() == 0) ? app->bottom(1) : 0);
}

YMenu *YFrameWindow::windowMenu() {
    //if (frameOptions() & foFullKeys)
    //    return windowMenuNoKeys;
    //else
    return ::windowMenu;
}

void YFrameWindow::showIcon() {
    if (fDesktopIcon && minimizeToDesktop)
        fDesktopIcon->show();
}

void YFrameWindow::hideIcon() {
    if (fDesktopIcon)
        fDesktopIcon->hide();
}

void YFrameWindow::addAsTransient() {
    Window fTransientFor = client()->ownerWindow();
    if (fTransientFor) {
        fOwner = app->getFrame(fTransientFor);
        if (fOwner != 0) {
            MSG(("transient for 0x%lX: 0x%lX", fTransientFor, fOwner));
            if (fOwner) {
                fNextTransient = fOwner->transient();
                fOwner->setTransient(this);
            } else {
                fTransientFor = 0; // ?

                fNextTransient = 0;
                fOwner = 0;
            }
        }
    }
}

void YFrameWindow::removeAsTransient() {
    if (fOwner) {
        MSG(("removeAsTransient"));
        
        YFrameWindow *w = fOwner->transient(), *cp = 0;

        while (w) {
            if (w == this) {
                if (cp)
                    cp->setNextTransient(nextTransient());
                else
                    fOwner->setTransient(nextTransient());
            }
            w = w->nextTransient();
        }
        fOwner = 0;
        fNextTransient = 0;
    }
}

void YFrameWindow::addTransients() {
    YFrameWindow *w = app->bottomLayer();

    while (w) {
        if (w->owner() == 0)
            w->addAsTransient();
        w = w->prevLayer();
    }
}

void YFrameWindow::removeTransients() {
    if (transient()) {
        MSG(("removeTransients"));
        YFrameWindow *w = transient(), *n;

        while (w) {
            n = w->nextTransient();
            w->setNextTransient(0);
            w->setOwner(0);
            w = n;
        }
        fTransient = 0;
    }
}

DesktopIcon::DesktopIcon(YFrameWindow *frame,
                         YWindow *aParent,
                         Window win):
    YFocusWindow(aParent, win)
{
    fFrame = frame;
    selected = 0;
    setSize(120, 24);
    static int x = 2;
    static int y = 2;
    setPosition(x, y);
    y += height() + 2;
}

DesktopIcon::~DesktopIcon() {
    
}

void DesktopIcon::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    YColor *bg = normalTaskBarAppBg;
    YColor *fg = normalTaskBarAppFg;
    int tx = 2;
    int x, y, w, h;

    g.setColor(bg);
    g.draw3DRect(0, 0, width() - 1, height() - 1, true);
    g.fillRect(1, 1, width() - 2, height() - 2);

    x = tx; y = 2;
    w = width() - 6;
    h = height() - 6;

    if (selected == 2) {
        g.setColor(menuBg->darker());
        g.draw3DRect(x, y, w, h, false);
        g.fillRect(x + 1, y + 1, w - 1, h - 1);
    } else {
        g.setColor(menuBg->brighter());
        g.drawRect(x + 1, y + 1, w, h);
        g.setColor(menuBg->darker());
        g.drawRect(x, y, w, h);
        g.setColor(menuBg);
        g.fillRect(x + 2, y + 2, w - 2, h - 2);
    }

    if (frame()->clientIcon() && frame()->clientIcon()->small()) {
        //int y = (height() - 3 - frame()->clientIcon()->small()->height()) / 2;
        g.drawMaskPixmap(frame()->clientIcon()->small(), 2 + tx + 1, 4);
    }

    char *str = (char *)frame()->client()->iconTitle();
    if (!str)
        str = (char *)frame()->client()->windowTitle();
    if (str) {
        g.setColor(fg);
        g.setFont(normalTaskBarFont);
        int ty = (height() - 1 + titleFont->height()) / 2 - titleFont->descent();
        if (ty < 2)
            ty = 2;
        g.drawChars((char *)str, 0, strlen(str),
                    tx + 4 + 16 + 2,
                    ty);
        //(yheight() - font->height()) / 2 - titleFont->descent() - 4);
    }
}

void DesktopIcon::handleButton(const XButtonEvent &button) {
    if (button.type == ButtonPress) {
        if (!(button.state & ControlMask))
            raise();
        if (button.button == 1) {
            selected = 2;
            repaint();
        }
    } else if (button.type == ButtonRelease) {
        if (button.button == 1) {
            if (selected == 2) {
                fFrame->wmRaise();
                fFrame->activate();
            }
            selected = 0;
            repaint();
        }
    }
    YWindow::handleButton(button);
}

void DesktopIcon::handleClick(const XButtonEvent &down, const XButtonEvent &up, int count) {
    if (up.button == 3) {
        frame()->updateMenu();
        frame()->windowMenu()->popup(0, frame(),
                                     up.x_root, up.y_root, -1, -1,
                                     YPopupWindow::pfCanFlipVertical |
                                     YPopupWindow::pfCanFlipHorizontal |
                                     YPopupWindow::pfPopupMenu);
    }
}

void DesktopIcon::handleCrossing(const XCrossingEvent &crossing) {
    if (selected > 0) {
        if (crossing.type == EnterNotify) {
            selected = 2;
            repaint();
        } else if (crossing.type == LeaveNotify) {
            selected = 1;
            repaint();
        }
    }

}

void DesktopIcon::handleDrag(const XButtonEvent &down, const XMotionEvent &motion) {
    if (down.button != 1) {
        int x = motion.x_root - down.x;
        int y = motion.y_root - down.y;
        x += down.x;
        y += down.y;
        x /= width() + 2; x *= width() + 2;
        y /= height() + 2; y *= height() + 2;
        if (x < 0) x = 0;
        if (y < 0) y = 0;
        x += 2;
        y += 2;
        setPosition(x, y);
    }
}

bool YFrameWindow::isModal() {
    if (!client())
        return false;

    MwmHints *mwmHints = client()->mwmHints();
    if (mwmHints && (mwmHints->flags & MWM_HINTS_INPUT_MODE))
        if (mwmHints->input_mode != MWM_INPUT_MODELESS) /* ??? */
            return true;

    if (hasModal()) // ??? is this ok
        return true;
    
    return false;
}

bool YFrameWindow::hasModal() {
    YFrameWindow *w = transient();
    while (w) {
        if (w->isModal())
            return true;
        w = w->nextTransient();
    }
    /* !!! missing code for app modal dialogs */
    return false;
}

bool YFrameWindow::isFocusable() {
    if (hasModal())
        return false;

    if (!client())
        return false;

    XWMHints *hints = client()->hints();
    
    if (!hints)
        return true;
    if (!(hints->flags & InputHint))
        return true;
    if (hints->input)
        return true;
    if (client()->protocols() & YFrameClient::wpTakeFocus)
        return true;
    return false;
}

