/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 *
 * ScrollBar
 */

#include "icewm.h"

YScrollBar::YScrollBar(YWindow *aParent): YWindow(aParent) {
    fOrientation = Vertical;
    fMinimum = fMaximum = fValue = fVisibleAmount = 0;
    fUnitIncrement = fBlockIncrement = 1;
    fListener = 0;
    fScrollTo = goNone;
}


YScrollBar::YScrollBar(Orientation anOrientation, YWindow *aParent):
    YWindow(aParent)
{
    fOrientation = anOrientation;

    fMinimum = fMaximum = fValue = fVisibleAmount = 0;
    fUnitIncrement = fBlockIncrement = 1;
    fListener = 0;
    fScrollTo = goNone;
}

YScrollBar::YScrollBar(Orientation anOrientation,
                       int aValue, int aVisibleAmount, int aMin, int aMax,
                       YWindow *aParent): YWindow(aParent)
{
    fOrientation = anOrientation;
    fMinimum = aMin;
    fMaximum = aMax;
    fVisibleAmount = aVisibleAmount;
    fValue = aValue;

    fUnitIncrement = fBlockIncrement = 1;
    fListener = 0;
    fScrollTo = goNone;
}

YScrollBar::~YScrollBar() {
}

void YScrollBar::setOrientation(Orientation anOrientation) {
    if (anOrientation != fOrientation) {
        fOrientation = anOrientation;
        repaint();
    }
}

void YScrollBar::setMaximum(int aMaximum) {
    if (aMaximum < fMinimum)
        aMaximum = fMinimum;
    
    if (aMaximum != fMaximum) {
        fMaximum = aMaximum;
        repaint();
    }
}

void YScrollBar::setMinimum(int aMinimum) {
    if (aMinimum > fMaximum)
        aMinimum = fMaximum;
    
    if (aMinimum != fMinimum) {
        fMinimum = aMinimum;
        repaint();
    }
}

void YScrollBar::setVisibleAmount(int aVisibleAmount) {
    if (fVisibleAmount > fMaximum - fMinimum)
        fVisibleAmount = fMaximum - fMinimum;
    if (fVisibleAmount < 0)
        fVisibleAmount = 0;
    
    if (aVisibleAmount != fVisibleAmount) {
        fVisibleAmount = aVisibleAmount;
        repaint();
    }
}

void YScrollBar::setUnitIncrement(int anUnitIncrement) {
    fUnitIncrement = anUnitIncrement;
}

void YScrollBar::setBlockIncrement(int aBlockIncrement) {
    fBlockIncrement = aBlockIncrement;
}

void YScrollBar::setValue(int aValue) {
    if (aValue > fMaximum - fVisibleAmount)
        aValue = fMaximum - fVisibleAmount;
    if (aValue < fMinimum)
        aValue = fMinimum;

    if (aValue != fValue) {
        fValue = aValue;
        repaint();
    }
}

void YScrollBar::setValues(int aValue, int aVisibleAmount, int aMin, int aMax) {
    int v = aValue;

    if (aMax < aMin)
        aMax = aMin;
    if (aVisibleAmount > aMax - aMin)
        aVisibleAmount = aMax - aMin;
    if (aVisibleAmount < 0)
        aVisibleAmount = 0;
    if (aValue > aMax - aVisibleAmount)
        aValue = aMax - aVisibleAmount;
    if (aValue < aMin)
        aValue = aMin;

    if (aMax != fMaximum ||
        aMin != fMinimum ||
        aValue != fValue ||
        aVisibleAmount != fVisibleAmount)
    {
        fMinimum = aMin;
        fMaximum = aMax;
        fValue = aValue;
        fVisibleAmount = aVisibleAmount;
        repaint();
        if (v != fValue)
            fListener->move(this, fValue);
    }
}

void YScrollBar::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    g.setColor(scrollBarArrow);
    if (fOrientation == Vertical) {
        g.draw3DRect(0, 0, width() - 1, width() - 1,
                     (fScrollTo == goUp) ? false : true);
        g.fillRect(1, 1, width() - 2, width() - 2);

        int w = width();
        int m = 7;
        int d = 5;
        int t = w / 2 - d / 2 - ((fScrollTo == goUp) ? 1 : 0);
        int b = t + d;

        if (fValue > fMinimum)
            g.setColor(black);
        else
            g.setColor(white);
        g.drawLine(m - d, b, m, t);
        g.drawLine(m - d + 1, b, m + 1, t);
        g.drawLine(m + d + 1, b, m + 1, t);
        g.drawLine(m + d, b, m, t);

        g.setColor(scrollBarArrow);
        g.draw3DRect(0, height() - width(), width() - 1, width() - 1,
                     (fScrollTo == goDown) ? false : true);
        g.fillRect(1, height() - width() + 1, width() - 2, width() - 2);

        m = 7;
        d = 5;
        t = w / 2 + d / 2 + ((fScrollTo == goDown) ? 1 : 0);
        t += height() - width();
        b = t - d;

        if (fValue < fMaximum - fVisibleAmount)
            g.setColor(black);
        else
            g.setColor(white);
        g.drawLine(m - d, b, m, t);
        g.drawLine(m - d + 1, b, m + 1, t);
        g.drawLine(m + d + 1, b, m + 1, t);
        g.drawLine(m + d, b, m, t);

        int h = height() - 2 * width();
        if (h > 0) {
            int min, max;
            
            if (fMaximum - fMinimum > 0) {
                min = h * fValue / (fMaximum - fMinimum);
                max = min + h * fVisibleAmount / (fMaximum - fMinimum);
            } else {
                min = 0;
                max = h;
            }
#if 0
            if (max - min < 16)
                max = min + 16;
            if (max > h)
                max = h;
#endif

            g.setColor(scrollBarBg);
            if (min > 0 && min < h)
                g.fillRect(0, width(), width(), min);

            if (max > 0 && max < h)
                g.fillRect(0, max + width(), width(), h - max);

            g.setColor(scrollBarSlider);
            g.draw3DRect(0, min + width(), width() - 1, max - min - 1,
                         (fScrollTo == goPosition) ? false : true);
            g.fillRect(1, min + width() + 1, width() - 2, max - min - 2);
        }
    } else
        abort();
}

void YScrollBar::scroll(int delta) {
    int fNewPos = fValue;

    if (delta == 0)
        return ;
    
    fNewPos += delta;

    if (fNewPos >= fMaximum - fVisibleAmount)
        fNewPos = fMaximum - fVisibleAmount;
    if (fNewPos < fMinimum)
        fNewPos = fMinimum;
    if (fNewPos != fValue) {
        delta = fNewPos - fValue;
        fValue = fNewPos;
        repaint();
        fListener->scroll(this, delta);
    }
}

void YScrollBar::move(int pos) {
    int fNewPos = pos;

    if (fNewPos >= fMaximum - fVisibleAmount)
        fNewPos = fMaximum - fVisibleAmount;
    if (fNewPos < fMinimum)
        fNewPos = fMinimum;
    if (fValue != fNewPos) {
        fValue = fNewPos;
        repaint();
        fListener->move(this, fValue);
    }
}

void YScrollBar::handleButton(const XButtonEvent &button) {
    if (button.button != 1)
        return ;
    
    int h = height() - 2 * width();

    if (h <= 0 || fMinimum >= fMaximum)
        return ;

    int min, max;

    min = h * fValue / (fMaximum - fMinimum);
    max = min + h * fVisibleAmount / (fMaximum - fMinimum);

    if (button.type == ButtonPress) {
        fScrollTo = goNone;
        if (button.y < int(width())) {
            if (fValue > 0)
                fScrollTo = goUp;
        } else if (button.y > int(height() - width())) {
            if (fValue < fMaximum - fVisibleAmount)
                fScrollTo = goDown;
        } else if (button.y < int(width()) + min) {
            fScrollTo = goPageUp;
        } else if (button.y > int(width()) + max) {
            fScrollTo = goPageDown;
        } else {
            fScrollTo = goPosition;
            fGrabDelta = button.y - min - int(width());
        }
        repaint();
    } else if (button.type == ButtonRelease) {
        switch (fScrollTo) {
        case goNone:
            break;
        case goUp:
            scroll(-fUnitIncrement);
            break;
        case goDown:
            scroll(+fUnitIncrement);
            break;
        case goPageUp:
            scroll(-fBlockIncrement);
            break;
        case goPageDown:
            scroll(+fBlockIncrement);
            break;
        case goPosition:
            break;
        }
        fScrollTo = goNone;
        repaint();
    }
}

void YScrollBar::handleMotion(const XMotionEvent &motion) {
    if (motion.state != Button1Mask)
        return ;
    
    int h = height() - 2 * width();

    if (h <= 0 || fMinimum >= fMaximum)
        return ;

    int min, max;

    min = h * fValue / (fMaximum - fMinimum);
    max = h * (fValue + fVisibleAmount) / (fMaximum - fMinimum);

    if (fScrollTo == goPosition) {
        int y = motion.y - fGrabDelta - width();
        if (y < 0) y = 0;
        int pos = y * (fMaximum - fMinimum) / h;
        move(pos);
    }
}

