/* wdefaults.c - window specific defaults
 * 
 *  WindowMaker window manager
 * 
 *  Copyright (c) 1997 Alfredo K. Kojima
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "wconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
 
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <wraster.h>

#include "generic/wwmlib.h"

#include "WindowMaker.h"
#include "window.h"
#include "screen.h"
#include "funcs.h"
#include "workspace.h"
#include "defaults.h"


extern proplist_t wAttributeDomainName;


/* type converters */
static int getBool(proplist_t, proplist_t);

#define getNotBool(pl1, pl2) (!getBool((pl1), (pl2)))

static char* getString(proplist_t, proplist_t);


static proplist_t ANoTitlebar = NULL;
static proplist_t ANoResizebar;
static proplist_t ANoMiniaturizable;
static proplist_t ANoClosable;
static proplist_t ANoHideOthers;
static proplist_t ANoMouseBindings;
static proplist_t ANoKeyBindings;
static proplist_t ANoAppIcon;
static proplist_t AKeepOnTop;
static proplist_t AOmnipresent;
static proplist_t ASkipWindowList;
static proplist_t ADontMoveOff;
static proplist_t AUnfocusable;
static proplist_t AAlwaysUserIcon;

static proplist_t AStartWorkspace;

static proplist_t AIcon;
static proplist_t AIconData;
static proplist_t AIconLabel;


static proplist_t AnyWindow;
static proplist_t No;


static void
init_wdefaults(WScreen *scr)
{
    AIcon = PLMakeString("Icon");
    AIconData = PLMakeString("IconData");
    AIconLabel = PLMakeString("IconLabel");
    
    ANoTitlebar = PLMakeString("NoTitlebar");
    ANoResizebar = PLMakeString("NoResizebar");
    ANoMiniaturizable = PLMakeString("NotMiniaturizable");
    ANoClosable = PLMakeString("NotClosable");
    ANoHideOthers = PLMakeString("NoHideOthers");
    ANoMouseBindings = PLMakeString("NoMouseBindings");
    ANoKeyBindings = PLMakeString("NoKeyBindings");
    ANoAppIcon = PLMakeString("NoAppIcon");
    AKeepOnTop = PLMakeString("KeepOnTop");
    AOmnipresent = PLMakeString("Omnipresent");
    ASkipWindowList = PLMakeString("SkipWindowList");
    ADontMoveOff = PLMakeString("KeepInsideScreen");
    AUnfocusable = PLMakeString("Unfocusable");
    AAlwaysUserIcon = PLMakeString("AlwaysUserIcon");
    
    AStartWorkspace = PLMakeString("StartWorkspace");
    
    AnyWindow = PLMakeString("*");
    No = PLMakeString("No");
    /*
    if (!scr->wattribs) {
	scr->wattribs = PLGetDomain(wAttributeDomainName);
    }*/
}



static proplist_t
get_value(proplist_t dict_win, proplist_t dict_class, proplist_t dict_name, 
	  proplist_t dict_any, proplist_t option, proplist_t default_value)
{
    proplist_t value;
    

    if (dict_win) {
	value = PLGetDictionaryEntry(dict_win, option);
	if (value)
	    return value;
    }
    
    if (dict_name) {
	value = PLGetDictionaryEntry(dict_name, option);
	if (value)
	    return value;
    }

    if (dict_class) {
	value = PLGetDictionaryEntry(dict_class, option);
	if (value)
	    return value;
    }
        
    if (dict_any) {
	value = PLGetDictionaryEntry(dict_any, option);
	if (value)
	    return value;
    }
    
    return default_value;
}


void
wDefaultFillAttributes(WScreen *scr, char *instance, char *class, 
		       WWindowAttributes *attr)
{
    proplist_t value;
    proplist_t key1, key2, key3;
    proplist_t dw, dc, dn, da;
    char buffer[256];

    memset(attr, 0xff, sizeof(WWindowAttributes));

    if (class && instance)
      key1 = PLMakeString(strcat(strcat(strcpy(buffer,instance),"."),class));
    else
      key1 = NULL;
    
    if (instance)
      key2 = PLMakeString(instance);
    else
      key2 = NULL;
    
    if (class)
      key3 = PLMakeString(class);
    else
      key3 = NULL;
    
    if (!ANoTitlebar) {
	init_wdefaults(scr);
    }

    PLSetStringCmpHook(NULL);

    if (scr->dwindow->dictionary) {
	dw = key1 ? PLGetDictionaryEntry(scr->dwindow->dictionary, key1) : NULL;
	dn = key2 ? PLGetDictionaryEntry(scr->dwindow->dictionary, key2) : NULL;
	dc = key3 ? PLGetDictionaryEntry(scr->dwindow->dictionary, key3) : NULL;
	da = PLGetDictionaryEntry(scr->dwindow->dictionary, AnyWindow);
    } else {
	dw = NULL;
	dn = NULL;
	dc = NULL;
	da = NULL;
    }
    if (key1)
      PLRelease(key1);
    if (key2)
      PLRelease(key2);
    if (key3)
      PLRelease(key3);

    /* get the data */        
    value = get_value(dw, dc, dn, da, ANoTitlebar, No);
    attr->titlebar = getNotBool(ANoTitlebar, value);
        
    value = get_value(dw, dc, dn, da, ANoResizebar, No);
    attr->resizable = getNotBool(ANoResizebar, value);

    value = get_value(dw, dc, dn, da, ANoMiniaturizable, No);
    attr->miniaturizable = getNotBool(ANoMiniaturizable, value);
    
    value = get_value(dw, dc, dn, da, ANoClosable, No);
    attr->closable = getNotBool(ANoClosable, value);

    value = get_value(dw, dc, dn, da, ANoHideOthers, No);
    attr->hide_others = getNotBool(ANoHideOthers, value);
    
    value = get_value(dw, dc, dn, da, ANoMouseBindings, No);
    attr->bind_mouse = getNotBool(ANoMouseBindings, value);
    
    value = get_value(dw, dc, dn, da, ANoKeyBindings, No);
    attr->bind_keys = getNotBool(ANoKeyBindings, value);
    
    value = get_value(dw, dc, dn, da, ANoAppIcon, No);
    attr->application = getNotBool(ANoAppIcon, value);

    value = get_value(dw, dc, dn, da, AKeepOnTop, No);
    attr->floating = getBool(AKeepOnTop, value);

    value = get_value(dw, dc, dn, da, AOmnipresent, No);
    attr->omnipresent = getBool(AOmnipresent, value);

    value = get_value(dw, dc, dn, da, ASkipWindowList, No);
    attr->skip_window_list = getBool(ASkipWindowList, value);
    
    value = get_value(dw, dc, dn, da, ADontMoveOff, No);
    attr->dont_move_off = getBool(ADontMoveOff, value);

    value = get_value(dw, dc, dn, da, AUnfocusable, No);
    attr->focusable = getNotBool(AUnfocusable, value);

    value = get_value(dw, dc, dn, da, AAlwaysUserIcon, No);
    attr->always_user_icon = getBool(AAlwaysUserIcon, value);

    /* clean up */
    PLSetStringCmpHook(StringCompareHook);
}



proplist_t
get_generic_value(WScreen *scr, char *instance, char *class, proplist_t option)
{
    proplist_t value, key, dict;
    char buffer[256];
    
    value = NULL;

    PLSetStringCmpHook(NULL);

    if (class && instance) {
	key = PLMakeString(strcat(strcat(strcpy(buffer,instance),"."),class));
	
	dict = PLGetDictionaryEntry(scr->dwindow->dictionary, key);
	PLRelease(key);

	if (dict) {
	    value = PLGetDictionaryEntry(dict, option);
	}
    }

    if (!value && instance) {
	key = PLMakeString(instance);
	
	dict = PLGetDictionaryEntry(scr->dwindow->dictionary, key);
	PLRelease(key);

	if (dict) {
	    value = PLGetDictionaryEntry(dict, option);
	}
    }

    if (!value && class) {
	key = PLMakeString(class);
	
	dict = PLGetDictionaryEntry(scr->dwindow->dictionary, key);
	PLRelease(key);

	if (dict) {
	    value = PLGetDictionaryEntry(dict, option);
	}
    }

    if (!value) {
	dict = PLGetDictionaryEntry(scr->dwindow->dictionary, AnyWindow);
	
	if (dict) {
	    value = PLGetDictionaryEntry(dict, option);
	}
    }
    
    PLSetStringCmpHook(StringCompareHook);

    return value;
}


char*
wDefaultGetIconFile(WScreen *scr, char *instance, char *class)
{
    proplist_t value;
    char *tmp;

    if (!ANoTitlebar) {
	init_wdefaults(scr);
    }

    if (!scr->dwindow->dictionary)
	return NULL;

    value = get_generic_value(scr, instance, class, AIcon);
    
    if (!value)
	return NULL;

    tmp = getString(AIcon, value);

    return tmp;
}


int
wDefaultGetStartWorkspace(WScreen *scr, char *instance, char *class)
{
    proplist_t value;
    int w, i;
    char *tmp;

    if (!ANoTitlebar) {
	init_wdefaults(scr);
    }

    if (!scr->dwindow->dictionary)
	return -1;

    value = get_generic_value(scr, instance, class, AStartWorkspace);
    
    if (!value)
	return -1;

    tmp = getString(AStartWorkspace, value);

    if (sscanf(tmp, "%i", &w)!=1) {
	w = -1;
	for (i=0; i < scr->workspace_count; i++) {
	    if (strcmp(scr->workspaces[i]->name, tmp)==0) {
		w = i;
		break;
	    }	
	}
    } else {
	w--;
    }

    return w;
}



RImage*
wDefaultGetIconLabel(WScreen *scr, char *instance, char *class);



/* --------------------------- Local ----------------------- */

static int 
getBool(proplist_t key, proplist_t value)
{
    char *val;

    if (!PLIsString(value)) {
	wWarning(_("Wrong option format for key \"%s\". Should be %s."),
		 PLGetString(key), "Boolean");
	return 0;
    }
    val = PLGetString(value);
    
    if ((val[1]=='\0' && (val[0]=='y' || val[0]=='Y' || val[0]=='T' 
			  || val[0]=='t' || val[0]=='1'))
	|| (strcasecmp(val, "YES")==0 || strcasecmp(val, "TRUE")==0)) {
	
	return 1;
    } else if ((val[1]=='\0' 
	      && (val[0]=='n' || val[0]=='N' || val[0]=='F' 
		  || val[0]=='f' || val[0]=='0')) 
	     || (strcasecmp(val, "NO")==0 || strcasecmp(val, "FALSE")==0)) {
	
	return 0;
    } else {
	wWarning(_("can't convert \"%s\" to boolean"), val);
        /* We return False if we can't convert to BOOLEAN.
         * This is because all options defaults to False.
         * -1 is not checked and thus is interpreted as True,
         * which is not good.*/
        return 0;
    }
}



static char*
getString(proplist_t key, proplist_t value)
{    
    if (!PLIsString(value)) {
	wWarning(_("Wrong option format for key \"%s\". Should be %s."),
		 PLGetString(key), "String");
	return NULL;
    }
        
    return PLGetString(value);
}
