/* xkeycaps, Copyright (c) 1993, 1996 Jamie Zawinski <jwz@netscape.com>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#ifndef _KBDDEF_H_
#define _KBDDEF_H_

#ifndef __STDC__
# undef const
# define const /**/
#endif

#include <X11/Xlib.h>	/* For KeySym and KeyCode */

/* There are several pieces of data that make up a single keyboard instance:

   - the physical shape and positions of the keys
   - the strings which are printed on those physical keys
   - the X keycodes which those keys generate
   - the X keysyms which are (by default) associated with those keycodes

   All of these data are represented by different structures, to enable 
   maximal sharing among keyboard definitions; for example, different
   countries have different keysyms loaded in by default and different
   strings printed on the keys, and many different vendors ship keyboards
   which physically resemble IBM keyboards, but have completely different
   keycode maps.
  */

struct keyboard_instance {
  const char *short_name; /* ID for the "-kbd" command-line option. */

  const char *vendor;	  /* Who makes the computer that this keyboard
			     usually comes with (e.g., "Sun", "NCD", "PC".) */
  const char *kbd_style;  /* What this particular instance is named
			     (e.g., "type-2", "104-key", "LK401".) */
  const char *server_id;  /* What X-server-vendor ID string might indicate
			     that this keyboard is the most likely one. */

  const struct keyboard_geometry *keyboard_geometry;	 /* physical layout */
  const struct key_strings *key_strings;		 /* printed surface */
  const KeyCode *keycodes;				 /* hardware codes  */
  const struct default_keycode_map *default_keycode_map; /* default xmodmap */
  int key_strings_count;
  int keycodes_count;
  int default_keycode_map_count;
};


/* These three structures represent the physical shape of a keyboard. */
struct keyboard_geometry {
  unsigned int nrows;
  const struct keyboard_row_geometry *rows;	/* the rows of keys */
  int default_scale;
  int horiz_border, vert_border;
};

struct keyboard_row_geometry {
  unsigned int nkeys;
  unsigned int height;
  const struct key_geometry *keys;	/* the keys in this row */
};

struct key_geometry {
  unsigned short width;	 /* Width of the key in some arbitrary units */
  unsigned short height; /* Height of the key in some arbitrary units */
  /* Other interesting info would be color (whether it's darker than
     the other keys, etc) and whether it's "L-shaped".) */
};


/* This structure represents the strings which are physically printed on
   the keys; there may be up to three strings per key. */
struct key_strings {
  const char *keysym_1;
  const char *keysym_2;
  const char *keysym_3;
};
  

/* The set of keycodes generated by a keyboard is represented by a simple
   array of KeyCodes, so there's no structure for it. */


/* This structure represents the keysyms associated with a keycode, that
   is the default key and modifier map. */
struct default_keycode_map {
  KeyCode keycode;
  unsigned long default_mods;
  const KeySym keysyms[8];
};


#define ROW(name, height) \
	{ sizeof (name) / sizeof (struct key_geometry), (height), name }
#define EMPTYROW(height) \
	{ 0, height, 0 }

extern const struct keyboard_instance all_kbds [];

#endif /* _KBDDEF_H_ */
