/* dlink.c
 * (c) Alexandre Frey 1996
 *
 * General routines to perform dynamic linking with bfd
 */

#include "dlink.h"

#include <stdlib.h>
#include <stdio.h>
#include <bfd.h>
#include <libiberty.h>
#include <ansidecl.h>
#include <bfdlink.h>
#include <string.h>
#include <ctype.h>

#include "alloc.h"

/* Functions defined in libbfd.h but this is not a public header */
extern PTR bfd_alloc (bfd*, size_t);
extern boolean _bfd_generic_link_add_symbols (bfd*, struct bfd_link_info*);
extern boolean _bfd_generic_link_add_one_symbol 
(struct bfd_link_info *info,
 bfd *abfd,
 const char *name,
 flagword flags,
 asection *section,
 bfd_vma value,
 const char *string,
 boolean copy,
 boolean collect,
 struct bfd_link_hash_entry **hashp);
extern boolean _bfd_generic_link_add_archive_symbols 
(bfd*, struct bfd_link_info*, 
 boolean (*)(bfd*, struct bfd_link_info*, boolean*));
 

/* The allocation functions 
 * Filled by dlink_init
 */
static void* (*internal_alloc)(size_t);
static void  (*internal_free)(void*);
static void* (*section_alloc)(size_t);
static void  (*section_free)(void*);

/* The error and warning reporting functions, idem.
 * the HINTS parameter is a malloc'ed string
 * the funstions are responsible to free it
 * error shouldn't return
 */
static void (*dlink_error) (char* desc, char* hints);
static void (*dlink_warning) (char* desc, char* hints);


/* Record the description and hint of an error
 * but call the dlink_error procedure later
 * when all the bfd internal allocated stuff
 * have been freed
 */
static char *last_desc, *last_hints;
static void dlink_notify_error (char* desc, char* hints)
{
  last_desc = desc;
  last_hints = hints;
}

static void dlink_confirm_error (void)
{
  dlink_error (last_desc, last_hints);
}

/* Always return true and do nothing */
static boolean 
dlink_true (bfd* ignore) 
{
  return true;
}

/* Our own target, an hybrid btw binary and default without any output :
 * . disables all output
 * . replaces the set_section_contents by a memory copy routine
 * . gets the final_link routine from binary_vec
 * . gets also the hash table manipulation routines from binary_vec
 * NB : this is for the output bfd
 */
bfd_target dlink_vec;

static boolean
dl_set_section_contents (bfd* abfd, asection* sect, PTR data, file_ptr pos, bfd_size_type size) 
{
#ifdef DEBUG
  fprintf (stderr, "dl_set_section_contents data = %p, sect->name = %s, sect->vma = %lx\n",
      data, sect->name, sect->vma); 
#endif
  bcopy ((const char*)data, (char*)sect->vma, sect->_cooked_size);
  return true;
}

extern bfd_target binary_vec;

static
char dlink_symbol_leading_char;

static
void init_dlink_vec (void) 
{
  CONST bfd_target *default_vec;

  {
    /* bfd_find_target needs a bfd argument ... */
    bfd dummy;
    default_vec = bfd_find_target (NULL, &dummy);
  }

  bcopy ((const char*)default_vec, (char*)&dlink_vec, sizeof (bfd_target));
  
  dlink_vec._bfd_set_section_contents = dl_set_section_contents;
  { 
    bfd_format i;
    for ( i = 0; i < bfd_type_end; i++) 
      dlink_vec._bfd_write_contents[i] = dlink_true;
  }

  dlink_vec._bfd_final_link = binary_vec._bfd_final_link;
  dlink_vec._bfd_link_hash_table_create = binary_vec._bfd_link_hash_table_create;
  dlink_vec._bfd_link_add_symbols = binary_vec._bfd_link_add_symbols;
  dlink_symbol_leading_char = dlink_vec.symbol_leading_char;
}

static boolean 
overflow (struct bfd_link_info *info,
		 const char *name,
		 const char *reloc_name, bfd_vma addend,
		 bfd *abfd, asection *section,
		 bfd_vma address) 
{
  char* hints = (char*)malloc (strlen (name)
			       + strlen (reloc_name)
			       + strlen (section->name)
			       + 30);
  sprintf (hints, "%s against %s at %s:0x%lx", 
	   reloc_name, name, section->name, address);
  dlink_warning ("Overflow reloc", hints);
  return true;
}

static boolean 
undefined_symbol (struct bfd_link_info *info,
		  const char* symbol_name,
		  bfd* input_bfd,
		  asection* input_section,
		  bfd_vma address) 
{
  char* hints = (char*) malloc (strlen (symbol_name)
				+ strlen (input_bfd->filename)
				+ strlen (input_section->name)
				+ 30);
  sprintf (hints, "%s (%s:%s + 0x%lx)",
	   symbol_name, input_bfd->filename, input_section->name, address);
  dlink_notify_error ("Undefined symbol", hints);
  return false;
}

static boolean 
dangerous (struct bfd_link_info *info,
	   const char *message,
	   bfd *abfd, asection *section,
	   bfd_vma address)
{
  char* hints = (char*) malloc (strlen (message) + strlen (section->name)
			      + 30);
  sprintf (hints, "%s at %s:0x%lx", message, section->name, address);
  dlink_warning ("Dangerous reloc", hints);
  return true;
}

/* The static dlink_info */
static struct dlink_info _static_dlink_info;
struct dlink_info *static_dlink_info = &_static_dlink_info;

#define obstack_chunk_alloc malloc
#define obstack_chunk_free free

static
void init_static (char* static_map_filename) 
{
  FILE* map;
  struct obstack o;
  
  map = fopen (static_map_filename, "r");
  if (!map) dlink_error ("Unable to open static map file", 
			 strdup(static_map_filename));
  
  obstack_init (&o);
  dlink_info_init (static_dlink_info);

  while (!feof (map))
    {
      bfd_vma value;
      struct bfd_link_hash_entry *entry;
      char* name;

      if (fscanf (map, "%lx ", &value) != 1) break;
      while (1) {
	int c = fgetc (map);
	if (isspace (c) || c == EOF) break;
	obstack_1grow (&o, (char)c);
      }
      obstack_1grow (&o, '\000');
      name = (char*) obstack_finish (&o);
      entry = bfd_link_hash_lookup (static_dlink_info->link_info.hash, 
				    name,
				    true, /* Create the entry */
				    true, /* Copy the string */
				    false);
      entry->type = bfd_link_hash_defined;
      entry->u.def.section = bfd_abs_section_ptr;
      entry->u.def.value = value;
      obstack_free (&o, name);
    }
  obstack_free (&o, NULL);

  fclose (map);
}

/* forward declaration */
static boolean 
dlink_info_add_bfd (struct dlink_info* info, bfd* abfd);

static boolean 
add_element (struct bfd_link_info *info,
	     bfd *abfd,
	     const char *name)
{
#ifdef DEBUG
  fprintf (stderr, "%s pulled %s from archive\n",
	   name, abfd->filename);
#endif
  dlink_info_add_bfd ((struct dlink_info*)info, abfd);
  abfd->link_next = info->input_bfds;
  info->input_bfds = abfd;
  return true;
}

static boolean 
multiple_definition (struct bfd_link_info *info,
			    const char* name,
			    bfd *obfd,
			    asection *osec,
			    bfd_vma oval,
			    bfd *nbfd,
			    asection *nsec,
			    bfd_vma nval)
{
  char* hints = (char*)malloc (strlen (name) + strlen(osec->name) 
			       + strlen (nsec->name) 
			       + 30);
  sprintf (hints, "%s, %s+0x%lx and %s+0x%lx", 
	   name, osec->name, oval, nsec->name, nval);
  dlink_notify_error ("Multiple definition", hints);
  return false;
}
		
static boolean 
multiple_common ()
{
  fprintf (stderr, "Multiple common definition ??!?\n");
  abort ();
  return false;
}
		    
static boolean 
add_to_set ()
{
  dlink_notify_error ("Set", NULL);
  return false;
}

static boolean 
constructor ()
{
  dlink_notify_error ("Constructor", NULL);
  return false;
}

static boolean 
warning (struct bfd_link_info *info,
	 const char *warning,
	 const char* symbol,
	 bfd* abfd,
	 asection *section,
	 bfd_vma address)
{
  char* message = (char*)malloc (strlen (warning) 
				 + strlen (symbol)
				 + (section==NULL ? 0 : strlen (section->name))
				 + 20);
  sprintf (message, "%s for %s (%s+0x%lx",
	   warning, symbol, (section==NULL? "???" : section->name), address);
  dlink_warning ("Warning", message);
  return true;
}

static boolean 
unattached_reloc ()
{
  fprintf (stderr, "Unattached reloc ??!?\n");
  abort ();
  return false;
}

static boolean 
notice ()
{
  fprintf (stderr, "Notice ??!?\n");
  abort ();
  return false;
}
     
struct bfd_link_callbacks dlink_callbacks = 
{
  add_element,
  multiple_definition,
  multiple_common,
  add_to_set,
  constructor,
  warning,
  undefined_symbol,
  overflow,
  dangerous,
  unattached_reloc,
  notice
};

struct dlink_info *
dlink_info_create (void)
{
  struct dlink_info *info = internal_alloc (sizeof (struct dlink_info));
  dlink_info_init (info);
  return info;
}


void
dlink_info_init (struct dlink_info *info)
{
  {
    bfd dummy;
    dummy.xvec = &dlink_vec;
    info->output_bfd = bfd_create ("dlink_info", &dummy);
    info->output_bfd->direction = write_direction;
  }
  info->link_info.callbacks = &dlink_callbacks;
  info->link_info.relocateable = false;
  info->link_info.shared = false;
  info->link_info.strip = strip_all;
  info->link_info.discard = discard_all;
  info->link_info.keep_memory = true;
  info->link_info.input_bfds = NULL;
  info->link_info.create_object_symbols_section = NULL;
  info->link_info.hash = bfd_link_hash_table_create (info->output_bfd);
  info->link_info.keep_hash = NULL;
  info->link_info.notice_hash = NULL;
  info->memory = dlink_memory_create (internal_alloc, internal_free,
				      section_alloc, section_free);

  /* FIXME : this is ugly */
#ifdef __alpha
  _bfd_generic_link_add_one_symbol
    ((struct bfd_link_info*)info,
     info->output_bfd,
     "_fpdata",
     BSF_EXPORT,
     bfd_abs_section_ptr,
     0,
     NULL,
     false,
     false,
     NULL);
#endif    
}

/* Copied verbatim from linker.c:generic_link_read-symbols */

/* Grab the symbols for an object file when doing a generic link.  We
   store the symbols in the outsymbols field.  We need to keep them
   around for the entire link to ensure that we only read them once.
   If we read them multiple times, we might wind up with relocs and
   the hash table pointing to different instances of the symbol
   structure.  */

static boolean
dlink_read_symbols (abfd)
     bfd *abfd;
{
  if (abfd->outsymbols == (asymbol **) NULL)
    {
      long symsize;
      long symcount;

      symsize = bfd_get_symtab_upper_bound (abfd);
      if (symsize < 0)
	return false;
      abfd->outsymbols = (asymbol **) bfd_alloc (abfd, symsize);
      if (abfd->outsymbols == NULL && symsize != 0)
	{
	  bfd_set_error (bfd_error_no_memory);
	  return false;
	}
      symcount = bfd_canonicalize_symtab (abfd, abfd->outsymbols);
      if (symcount < 0)
	return false;
      abfd->symcount = symcount;
    }

  return true;
}

static boolean 
dlink_info_add_bfd (struct dlink_info* info, bfd* abfd)
{
  asection* section;

#ifdef DEBUG
  fprintf (stderr, "Adding bfd %s\n", abfd->filename);
#endif
    
  if (! bfd_check_format (abfd, bfd_object))
    return false;

  /* Turn common symbol into defined ones (in a COMMON section)
   * As a consequence, multiple commons are not allowed
   * (what would they mean anyway in a dynamic world ??)
   */
  {
    long i;
    asymbol **s;
    
    dlink_read_symbols (abfd);
    s = abfd->outsymbols;
    
    for (i=0; i<abfd->symcount; i++)
      if (s[i]->section == bfd_com_section_ptr)
	{
	  symvalue size = s[i]->value;
	  asection* com_section = 
	    bfd_make_section_old_way (abfd, "COMMON");
	  com_section->flags = SEC_ALLOC;
	  s[i]->section = com_section;
	  s[i]->value = com_section->_raw_size;
	  com_section->_raw_size += size;
	}
  }
  
  for (section = abfd->sections; section != NULL; section=section->next)
    if ((section->flags & SEC_ALLOC)
#ifdef __alpha
	/* Exception handling sections on alpha are not supported */
	&& (strcmp (section->name, ".xdata") != 0)
	&& (strcmp (section->name, ".pdata") != 0)
#endif      
	)
      {
	
	int protection = DLINK_SEC_READ_WRITE;
	char* name = bfd_alloc (info->output_bfd, 
				sizeof(char)*(strlen (section->name)
					      + strlen(abfd->filename)
					      + 2));
	struct bfd_link_order *link_order;

	/* Read the relocations, so that non further access to the file 
	 * is needed
	 */
	{
	  long reloc_storage 
	    = bfd_get_reloc_upper_bound (abfd, section);
	  arelent** reloc_vector 
	    = (arelent**) bfd_alloc (abfd, reloc_storage);
	  bfd_canonicalize_reloc (abfd, section, 
				  reloc_vector,
				  abfd->outsymbols);
	}
	
	/* Create the output section */
	sprintf (name, "%s:%s", abfd->filename, section->name);
	section->output_section 
	  = bfd_make_section_anyway (info->output_bfd, name);
#ifdef DEBUG
	fprintf (stderr, "Allocating %s\n", name); 
#endif
	if (section->output_section == NULL) return false;
	
	/* Allocate the section with the appropriate protection */
	if (section->flags & SEC_CODE) protection = DLINK_SEC_CODE;
	if (section->flags & SEC_READONLY) protection = DLINK_SEC_READ_ONLY;
	if (strcmp (section->name, ".sbss") == 0
	    || strcmp (section->name, ".sdata") == 0
	    || strcmp (section->name, ".scommon") == 0
	    || strcmp (section->name, ".lit4") == 0
	    || strcmp (section->name, ".lit8") == 0
	    || strcmp (section->name, ".lita") == 0)
	  {
	    struct bfd_link_hash_entry *gp_entry;
	    protection |= DLINK_SEC_SMALL;
	    /* allocate the small segment if not already done */
	    dlink_memory_allocate_small_segment (info->memory);
	       
	    /* Create gp_entry if not defined */
	    gp_entry = bfd_link_hash_lookup (info->link_info.hash, "_gp",
					     true, false, false);
	    gp_entry->type = bfd_link_hash_defined;
	    gp_entry->u.def.section = bfd_abs_section_ptr;
	    gp_entry->u.def.value
	      = (bfd_vma)dlink_memory_get_gp (info->memory);
	  }

	dlink_memory_protect (info->memory);
	
	bfd_set_section_vma (info->output_bfd, section->output_section, 
			     (bfd_vma) dlink_section_allocate 
			     (info->memory,
			      bfd_section_size (abfd, section),
			      protection) );
	bfd_copy_private_section_data (abfd,section,
				       info->output_bfd,
				       section->output_section);
	
	/* The output section has always a content if we alloc it */
	bfd_set_section_flags (info->output_bfd, section->output_section,
			       section->flags | SEC_HAS_CONTENTS);
	bfd_set_section_size (info->output_bfd, section->output_section, 
			      bfd_section_size (abfd, section));
	bfd_set_section_size (abfd, section, bfd_section_size (abfd, section));
	link_order = bfd_new_link_order (info->output_bfd, 
					 section->output_section);
	link_order->offset = 0;
	link_order->size = bfd_section_size (abfd, section);
	if (section->flags & SEC_LOAD)
	  {
	    link_order->type = bfd_indirect_link_order;
	    link_order->u.indirect.section = section;
	    /* Load the section raw content now, and keep it in memory
	     * so that no further access to the file is made 
	     * FIXME : verify that !!
	     */
	    section->contents = (void*) section->output_section->vma;
	    dlink_memory_unprotect (info->memory);
	    bfd_get_section_contents (abfd, section, section->contents, 
				      0, section->_raw_size);
	    dlink_memory_protect (info->memory);
	    section->flags |= SEC_IN_MEMORY;
	  }
	else
	  {
	    link_order->type = bfd_fill_link_order;
	    link_order->u.fill.value = 0;
	  }
	
      }

  return true;
}

/* This function is adapted from linker.c:generic_link_check_archive_element_collect
 * Basically, it modifies the behavior on common symbol
 * They are treated as defined ones, since they will be turned into defined
 * in dlink_add_bfd
 */
static boolean
dlink_check_archive_element (bfd* abfd, struct bfd_link_info *info, boolean *pneeded)
{
  asymbol **pp, **ppend;

  *pneeded = false;
  
  if (! dlink_read_symbols (abfd))
    return false;
  
  pp = abfd->outsymbols;
  ppend = pp + abfd->symcount;
  for (; pp < ppend; pp++)
    {
      asymbol *p;
      struct bfd_link_hash_entry *h;
      
      p = *pp;

      /* We are only interested in globally visible symbols.  */
      if (! bfd_is_com_section (p->section)
	  && (p->flags & (BSF_GLOBAL | BSF_INDIRECT | BSF_WEAK)) == 0)
	continue;

      /* We are only interested if we know something about this
	 symbol, and it is undefined or common.  An undefined weak
	 symbol (type bfd_link_hash_undefweak) is not considered to be
	 a reference when pulling files out of an archive.  See the
	 SVR4 ABI, p. 4-27.  */
      h = bfd_link_hash_lookup (info->hash, bfd_asymbol_name (p), false,
				false, true);
      if (h == (struct bfd_link_hash_entry *) NULL
	  || (h->type != bfd_link_hash_undefined
	      && h->type != bfd_link_hash_common))
	continue;

      /* P is a symbol we are looking for.  */
      
      /* This object file defines this symbol, so pull it in.  */
      if (! (*info->callbacks->add_archive_element) (info, abfd,
						     bfd_asymbol_name (p)))
	return false;
      if (! _bfd_generic_link_add_symbols (abfd, info))
	return false;
      *pneeded = true;
      return true;
    }
 
  /* This archive element is not needed.  */
  return true;
}

/* add the symbol in hash entry ENTRY to INFO 
 * ENTRY is supposed to be of type defined
 */
static boolean
add_entry (struct dlink_info *info, struct bfd_link_hash_entry *entry)
{
  return _bfd_generic_link_add_one_symbol
    ((struct bfd_link_info *) info,
    info->output_bfd,
    entry->root.string,
    BSF_EXPORT,
    entry->u.def.section,
    entry->u.def.value,
    NULL,
    false,			/* Don't copy the string */
    false,			/* Don't collect */
    NULL);
}

/* if USE_STATIC is true, all the defined symbols
 * of the object file must be defined statically
 * Then they are added to the dlink_info
 */
static boolean 
dlink_add_object (struct dlink_info* info, 
		  bfd* input_bfd,
		  boolean use_static)
			   
{
  if (use_static)
    {
      asymbol **s;
      int i;

      dlink_read_symbols (input_bfd);
      s = input_bfd->outsymbols;
      for (i=0; i < input_bfd->symcount; i++)
	if (s[i]->flags & BSF_GLOBAL)
	  {
	    struct bfd_link_hash_entry *entry =
	      bfd_link_hash_lookup (static_dlink_info->link_info.hash,
				    s[i]->name,
				    false, false, true);
	    if (!entry)
	      {
		dlink_notify_error ("Object file not statically linked", 
				    strdup (input_bfd->filename) );
		goto problem;
	      }
	    if (!add_entry (info, entry))
	      goto problem;
	  }
      bfd_close (input_bfd);
    }
  else
    {
      dlink_info_add_bfd (info, input_bfd);
      if (!_bfd_generic_link_add_symbols(input_bfd, 
					 (struct bfd_link_info *)info))
	goto problem;
      input_bfd->link_next = info->link_info.input_bfds;
      info->link_info.input_bfds = input_bfd;
    }

  /* No problem */
  return true;

  problem : 
  bfd_close (input_bfd);
  dlink_confirm_error ();
  return false;
}

static boolean 
dlink_add_library (struct dlink_info* info,
		   bfd* input_bfd,
		   boolean use_static)
{
  bfd* linked_bfd;
  boolean already_linked;
  
  /* dload.scm often add the same library 
   * to save memory, first search the library in the info 
   */
  for (already_linked = false, linked_bfd = info->link_info.input_bfds;
       linked_bfd != NULL;
       linked_bfd = linked_bfd -> link_next)
    if (strcmp (linked_bfd->filename, input_bfd->filename) == 0)
      {
	already_linked = true;
	input_bfd = linked_bfd;
	break;
      }

  /* If USE_STATIC is true, all the symbols defined in the library
   * and which exists in the executable itself are passed to the 
   * dlink_info
   */
  if (use_static)
    {
      carsym* arsymbol;
      symindex previous = BFD_NO_MORE_SYMBOLS;
      while ((previous 
	      = bfd_get_next_mapent(input_bfd, previous, &arsymbol))
	     != BFD_NO_MORE_SYMBOLS)
	{
	  struct bfd_link_hash_entry *entry =
	    bfd_link_hash_lookup 
	    (static_dlink_info->link_info.hash, 
	     arsymbol->name, false, false, true);
	  if (entry)
	    /* A library definition is static -> add the static symbol
	       first */
	    if (! add_entry (info, entry))
	      goto problem;
	}
      
    }
  
  if (!_bfd_generic_link_add_archive_symbols 
      (input_bfd, (struct bfd_link_info*)info, dlink_check_archive_element) )
    goto problem;
  
  if (!already_linked)
    {
      input_bfd->link_next = info->link_info.input_bfds;
      info->link_info.input_bfds = input_bfd;
    }
  return true;
  
  problem :
    if (!already_linked) bfd_close (input_bfd);
  dlink_confirm_error ();
  return false;
}


/* This function dispatches to dlink_add_object or dlink_add_library
 * w.r.t the format of the file
 */
boolean 
dlink_add_file (struct dlink_info *info, 
		char *filename, 
		boolean use_static)
{
  bfd* input_bfd;

  if (info == static_dlink_info)
    dlink_error ("Can't add file to static unit",
		 strdup (filename) );

  input_bfd = bfd_openr (filename, NULL);
  
  if (!input_bfd)
    dlink_error ("Unable to open file",
		 strdup (filename) );

  if (bfd_check_format (input_bfd, bfd_object))
    return dlink_add_object (info, input_bfd, use_static);
  if (bfd_check_format (input_bfd, bfd_archive))
    return dlink_add_library (info, input_bfd, use_static);
  
  bfd_close (input_bfd);
  dlink_error ("Unrecognised format", strdup (filename));
  return false;
}


  

boolean 
dlink_info_closed_p (struct dlink_info* info)
{
  struct bfd_link_hash_entry *entry;
  for (entry = info->link_info.hash->undefs;
       entry != NULL;
       entry = entry->next)
    if (entry->type == bfd_link_hash_undefined) return false;
  
  return true;
}

boolean 
dlink_info_link (struct dlink_info* info)
{
  asection *section;
  boolean ok;
  
  dlink_memory_unprotect (info->memory);
  ok = bfd_final_link (info->output_bfd, (struct bfd_link_info*)info);
  dlink_memory_protect (info->memory);
  
  /* We may link the unit another time ... bfd doesn't allow bfd_make_section
   * when output_hash_begun is true ...
   */
  info->output_bfd->output_has_begun = false;

  if (!ok) dlink_confirm_error();

  /* Remove the link orders */
  for (section = info->output_bfd->sections;
       section != NULL;
       section = section->next)
    {
      section->link_order_head = NULL;
      section->link_order_tail = NULL;
    }

  return true;
}

static char* 
c_name_to_asm_name (char* c_name)
{
  if (dlink_symbol_leading_char != '\000')
    {
      char* asm_name = (char*) malloc ((strlen (c_name) + 2)*sizeof(char));
      sprintf (asm_name, "%c%s", dlink_symbol_leading_char, c_name);
      return asm_name;
    }
  else
    return c_name;
}

static void
free_asm_name (char* asm_name)
{
  if (dlink_symbol_leading_char != '\000')
    free (asm_name);
}
      
static      
struct bfd_link_hash_entry *
dlink_symbol_lookup (struct dlink_info *info, char* c_name)
{
  struct bfd_link_hash_entry *ret;
  char* asm_name = c_name_to_asm_name (c_name);
  ret = bfd_link_hash_lookup (info->link_info.hash, 
			      asm_name, 
			      false, false, true);
  free_asm_name (asm_name);
  return ret;
}

boolean 
dlink_info_symbol_defined_p (struct dlink_info* info, char* c_name)
{
  struct bfd_link_hash_entry *entry;
  entry = dlink_symbol_lookup (info, c_name);
  return (entry != NULL) && (entry->type == bfd_link_hash_defined);
}

bfd_vma
dlink_get_symbol_value (struct dlink_info* info, char* c_name)
{
  struct bfd_link_hash_entry *entry;
  
  entry = dlink_symbol_lookup (info, c_name);
  if (entry == NULL || entry->type != bfd_link_hash_defined)
    dlink_error ("Undefined symbol", strdup(c_name));
  
  return entry->u.def.value + entry->u.def.section->output_section->vma;
}
  
void 
dlink_init (void* (*ainternal_alloc) (size_t),
	    void  (*ainternal_free)  (void*),
	    void* (*asection_alloc) (size_t),
	    void  (*asection_free)  (void*),
	    void  (*notice_error) (char* desc, char* hints),
	    void  (*notice_warning) (char* desc, char* hints),
	    char* static_map_filename)
{
  internal_alloc = ainternal_alloc;
  internal_free = ainternal_free;
  section_alloc = asection_alloc;
  section_free  = asection_free;

  dlink_error = notice_error;
  dlink_warning = notice_warning;
  
  bfd_init ();
  init_dlink_vec ();
  init_static (static_map_filename);
}

void dlink_info_free (struct dlink_info *info)
{
  dlink_info_release (info);
  internal_free (info);
}
     
void dlink_info_release (struct dlink_info *info)
{
  bfd *input_bfd, *bfd_next;
  
  /* Free the sections */
  dlink_memory_free (info->memory);
  
  /* Free the input bfds */
  for (input_bfd = info->link_info.input_bfds;
       input_bfd != NULL;
       input_bfd = bfd_next)
    {
      bfd_next = input_bfd->link_next;
      bfd_close (input_bfd);
    }

  /* Free the output_bfd */
  bfd_close (info->output_bfd);

  /* Free the hash table of info */
  bfd_hash_table_free ((struct bfd_hash_table *)info->link_info.hash);

  /* Don't free the info structure */
}


void 
dlink_info_walk_on_undefined (struct dlink_info *info,
			      void (*proc)(const char*) )
{
  struct bfd_link_hash_entry *entry;
  for (entry = info->link_info.hash->undefs;
       entry != NULL;
       entry = entry->next)
    if (entry->type == bfd_link_hash_undefined)
      proc (entry->root.string);
}

/* EOF : dlink.c */
