;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/comptime1.9c/Ast/alphatize.scm       */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Jan  6 11:09:14 1995                          */
;*    Last change :  Mon Dec 22 07:59:16 1997 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The substitution tools module                                    */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module ast_alphatize
   (include "Ast/node.sch"
	    "Tools/location.sch")
   (import  tools_error
	    tools_shape
	    tools_error
	    type_cache
	    ast_sexp
	    ast_local
	    ast_apply
	    ast_app)
   (export  (alphatize::node what* by* loc ::node)))

;*---------------------------------------------------------------------*/
;*    alphatize ...                                                    */
;*    -------------------------------------------------------------    */
;*    This function differs from substitute, because:                  */
;*      - it operates only on variables                                */
;*      - it allocated new nodes (i.e. it does not operates on place). */
;*    -------------------------------------------------------------    */
;*    Alphatize can replace a variable by a variable                   */
;*    construction but nothing else.                                   */
;*---------------------------------------------------------------------*/
(define (alphatize what* by* loc node)
   ;; we set alpha-fnode slot and the type of the new variable
   (for-each (lambda (what by)
		(variable-fast-alpha-set! what by))
	     what*
	     by*)
   (set! *location* loc)
   (let ((res (do-alphatize node)))
      ;; we remove alpha-fast slots
      (for-each (lambda (what)
		   (variable-fast-alpha-set! what #unspecified))
		what*)
      res))

;*---------------------------------------------------------------------*/
;*    *location* ...                                                   */
;*---------------------------------------------------------------------*/
(define *location* #f)

;*---------------------------------------------------------------------*/
;*    get-location ...                                                 */
;*---------------------------------------------------------------------*/
(define (get-location node)
   (if (location? (node-loc node))
       (node-loc node)
       *location*))

;*---------------------------------------------------------------------*/
;*    do-alphatize ...                                                 */
;*---------------------------------------------------------------------*/
(define-generic (do-alphatize::node node::node))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::atom ...                                          */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::atom)
   (duplicate::atom node
      (loc (get-location node))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::var ...                                           */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::var)
   (let* ((var   (var-variable node))
	  (alpha (variable-fast-alpha var)))
      (cond
	 ((eq? alpha #unspecified)
	  (use-variable! var (node-loc node) 'value)
	  (duplicate::var node (loc (get-location node))))
	 ((variable? alpha)
	  (use-variable! alpha (node-loc node) 'value)
	  (if (fun? (variable-value alpha))
	      (instantiate::closure
		 (loc (get-location node))
		 (type (variable-type alpha))
		 (variable alpha))
	      (duplicate::var node
		 (loc (get-location node))
		 (variable alpha))))
	 ((atom? alpha)
	  (duplicate::atom alpha))
	 (else
	  (internal-error "alphatize"
			  "Illegal alphatization"
			  (shape node))))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::closure ...                                       */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::closure)
   (let* ((var   (var-variable node))
	  (alpha (variable-fast-alpha var)))
      (cond
	 ((eq? alpha #unspecified)
	  (use-variable! var (node-loc node) 'value)
	  (duplicate::closure node (loc (get-location node))))
	 ((variable? alpha)
	  (use-variable! alpha (node-loc node) 'value)
	  (duplicate::closure node
	     (loc (get-location node))
	     (variable alpha)))
	 (else
	  (internal-error "alphatize"
			  "Illegal alphatization"
			  (shape node))))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::kwote ...                                         */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::kwote)
   (duplicate::kwote node (loc (get-location node))))
       
;*---------------------------------------------------------------------*/
;*    do-alphatize ::sequence ...                                      */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::sequence)
   (duplicate::sequence node
      (loc (get-location node))
      (nodes (map do-alphatize (sequence-nodes node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::app ...                                           */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::app)
   ;; we have to enforce here a variable and not a closure (that
   ;; why the duplicate::var of the fun field).
   (duplicate::app node
      (loc (get-location node))
      (fun (let ((var (do-alphatize (app-fun node))))
	      (if (closure? var)
		  (duplicate::var var)
		  var)))
      (args (map do-alphatize (app-args node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::app-ly ...                                         */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::app-ly)
   (let ((fun (do-alphatize (app-ly-fun node)))
	 (arg (do-alphatize (app-ly-arg node))))
      (if (closure? fun)
	  (known-app-ly->node '()
			     (get-location node)
			     (duplicate::var fun)
			     arg
			     'value)
	  (duplicate::app-ly node
	     (loc (get-location node))
	     (fun fun)
	     (arg arg)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::funcall ...                                       */
;*    -------------------------------------------------------------    */
;*    When transforming a funcall into an app node we have to remove   */
;*    the extra argument which hold the closure.                       */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::funcall)
   (let ((fun  (do-alphatize (funcall-fun node)))
	 (args (map do-alphatize (funcall-args node))))
      (if (closure? fun)
	  (if (correct-arity-app? (var-variable fun)
				  (cdr args))
	      (make-app-node '()
			     (get-location node)
			     (duplicate::var fun)
			     (cdr args))
	      (user-error/location (get-location node)
				   "Illegal application"
				   "wrong number of argument(s)"
				   (shape node)))
	  (duplicate::funcall node
	     (loc (get-location node))
	     (fun fun)
	     (args args)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::pragma ...                                        */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::pragma)
   (duplicate::pragma node
      (loc (get-location node))
      (args (map do-alphatize (pragma-args node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::cast ...                                          */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::cast)
   (duplicate::cast node
      (loc (get-location node))
      (arg (do-alphatize (cast-arg node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::setq ...                                          */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::setq)
   (let* ((v     (setq-var node))
	  (var   (var-variable v))
	  (alpha (variable-fast-alpha var)))
      (cond
	 ((eq? alpha #unspecified)
	  (use-variable! var (node-loc node) 'set!)
	  (duplicate::setq node
	     (loc (get-location node))
	     (var (duplicate::var v (loc (get-location node))))
	     (value (do-alphatize (setq-value node)))))
	 ((variable? alpha)
	  (use-variable! alpha (node-loc node) 'set!)
	  (duplicate::setq node
	     (loc (get-location node))
	     (var (duplicate::var v
		     (loc (get-location node))
		     (variable alpha)))
	     (value (do-alphatize (setq-value node)))))
	 (else
	  (internal-error "alphatize"
			  "Illegal alphatization"
			  (shape node))))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::conditional ...                                   */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::conditional)
   (duplicate::conditional node
      (loc (get-location node))
      (test (do-alphatize (conditional-test node)))
      (true (do-alphatize (conditional-true node)))
      (false (do-alphatize (conditional-false node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::fail ...                                          */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::fail)
   (duplicate::fail node
      (loc (get-location node))
      (proc (do-alphatize (fail-proc node)))
      (msg  (do-alphatize (fail-msg node)))
      (obj  (do-alphatize (fail-obj node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::select ...                                        */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::select)
   (duplicate::select node
      (loc (get-location node))
      (test (do-alphatize (select-test node)))
      (clauses (map (lambda (clause)
		       (cons (car clause)
			     (do-alphatize (cdr clause))))
		    (select-clauses node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::make-box ...                                      */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::make-box)
   (duplicate::make-box node
      (loc (get-location node))
      (value (do-alphatize (make-box-value node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::box-ref ...                                       */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::box-ref)
   (duplicate::box-ref node
      (loc (get-location node))
      (var (do-alphatize (box-ref-var node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::box-set! ...                                      */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::box-set!)
   (duplicate::box-set! node
      (loc (get-location node))
      (var (do-alphatize (box-set!-var node)))
      (value (do-alphatize (box-set!-value node)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::let-fun ...                                       */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::let-fun)
   (let* ((old-locals (let-fun-locals node))
	  (new-locals (map (lambda (l)
			      (make-local-sfun (local-id l)
					       (local-type l)
					       (local-value l)))
			   old-locals)))
      (for-each (lambda (old new)
		   (let* ((old-sfun (local-value old))
			  (old-args (sfun-args old-sfun))
			  (new-args (map (lambda (l)
					    (make-local-svar (local-id l)
							     (local-type l)))
					 old-args))
			  (old-body (sfun-body old-sfun))
			  (new-body (alphatize (append old-locals old-args)
					       (append new-locals new-args)
					       (get-location node)
					       old-body))
			  (new-sfun (duplicate::sfun old-sfun
				       (args new-args)
				       (body new-body))))
		      (local-value-set! new new-sfun)))
		old-locals
		new-locals)
      (duplicate::let-fun node
	 (loc (get-location node))
	 (locals new-locals)
	 (body (alphatize old-locals
			  new-locals
			  (get-location node)
			  (let-fun-body node))))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::let-var ...                                       */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::let-var)
   (let* ((old-locals   (map car (let-var-bindings node)))
	  (new-locals   (map (lambda (l)
				;; we can't use duplicate for locals because
				;; all local variables must be allocated
				;; using the `make-local-svar' form
				;; (for the key attribution).
				(make-local-svar (local-id l) (local-type l)))
			     old-locals))
	  (new-bindings (map (lambda (binding new-local)
				(cons new-local (do-alphatize (cdr binding))))
			     (let-var-bindings node)
			     new-locals)))
      (duplicate::let-var node
	 (loc (get-location node))
	 (bindings new-bindings)
	 (body (alphatize old-locals
			  new-locals
			  (get-location node)
			  (let-var-body node))))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::set-ex-it ...                                     */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::set-ex-it)
   (let* ((old-var    (var-variable (set-ex-it-var node)))
	  (old-exit   (local-value old-var))
	  (old-hdlg   (sexit-handler old-exit))
	  (alpha-hdlg (variable-fast-alpha old-hdlg))
	  (new-var    (make-local-sexit (local-id old-var)
					(local-type old-var)
					(duplicate::sexit old-exit
					   (handler alpha-hdlg))))
								 
	  (old-body   (set-ex-it-body node)))
      (duplicate::set-ex-it node
	 (loc (get-location node))
	 (var (duplicate::var (set-ex-it-var node)
		 (loc (get-location node))
		 (variable new-var)))
	 (body (alphatize (list old-var)
			  (list new-var)
			  (get-location node)
			  old-body)))))

;*---------------------------------------------------------------------*/
;*    do-alphatize ::jump-ex-it ...                                    */
;*---------------------------------------------------------------------*/
(define-method (do-alphatize node::jump-ex-it)
   (duplicate::jump-ex-it node
      (loc (get-location node))
      (exit (do-alphatize (jump-ex-it-exit node)))
      (value (do-alphatize (jump-ex-it-value node)))))

