/* --------------------------------------------------------------------*/
/*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \   /  '                               */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome. Send them to                                          */
/*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
/*-------------------------------------------------------------------- */
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cstring.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Tue Sep  5 09:55:58 1995                          */
/*    Last change :  Fri Mar 20 09:04:56 1998 (serrano)                */
/*    -------------------------------------------------------------    */
/*    String management                                                */
/*=====================================================================*/
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <bigloo1.9c.h>
#include <varargs.h>

/*---------------------------------------------------------------------*/
/*    Les recuperations externes                                       */
/*---------------------------------------------------------------------*/
extern void perror();

/*---------------------------------------------------------------------*/
/*    string_to_bstring ...                                            */
/*---------------------------------------------------------------------*/
obj_t
string_to_bstring( char *c_string )
{
   obj_t string;
   long  len;

   if( !c_string )
      c_string = "";

   len = strlen( c_string );

   string = GC_MALLOC_ATOMIC( STRING_SIZE + len );

#if( !defined( TAG_STRING ) )
   string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	
   string->string_t.length = len;

   strcpy( &(string->string_t.char0), c_string );

   return BSTRING( string );
}

/*---------------------------------------------------------------------*/
/*    make_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
make_string( long len, char c )
{
   obj_t string;

   if( len < 0 )
      FAILURE( string_to_bstring( "make-string" ),
	       string_to_bstring( "Illegal string size" ),
	       BINT( len ) );
   {
      string = GC_MALLOC_ATOMIC( STRING_SIZE + len );

#if( !defined( TAG_STRING ) )
      string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	
      string->string_t.length = len;

      memset( &(string->string_t.char0), c, len );
      (&(string->string_t.char0))[ len ] = '\0';
		
      return BSTRING( string );
   }
}

/*---------------------------------------------------------------------*/
/*    c_constant_string_to_string ...                                  */
/*---------------------------------------------------------------------*/
obj_t
c_constant_string_to_string( char *c_string )
{
   return string_to_bstring( c_string );
}

/*---------------------------------------------------------------------*/
/*    string_append ...                                                */
/*---------------------------------------------------------------------*/
obj_t
string_append( obj_t s1, obj_t s2 )
{
   long  l1, l2;
   obj_t string;
   
   l1 = STRING( s1 ).length;
   l2 = STRING( s2 ).length;

	
   string = GC_MALLOC_ATOMIC( STRING_SIZE + l1 + l2 );

#if( !defined( TAG_STRING ) )
   string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	
   string->string_t.length = l1 + l2;

   memcpy( &(string->string_t.char0), &STRING_REF( s1, 0 ), l1 );
   memcpy( &((char *)(&(string->string_t.char0)))[ l1 ], &STRING_REF( s2, 0 ), l2 );
   ((char *)(&(string->string_t.char0)))[ l1 + l2 ] = '\0';
	
   return BSTRING( string );
}
 
/*---------------------------------------------------------------------*/
/*    c_substring ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
c_substring( obj_t src_string, long min, long max )
{
   long  len;
   obj_t dst_string;
   
   len = max - min;

   dst_string = GC_MALLOC_ATOMIC( STRING_SIZE + len );

#if( !defined( TAG_STRING ) )
   dst_string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	
   dst_string->string_t.length = len;

   memcpy( &(dst_string->string_t.char0),
	   &STRING_REF( src_string, min ),
            len );
   (&(dst_string->string_t.char0))[ len ] = '\0';

   return BSTRING( dst_string );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    blit_string ...                                                  */
/*---------------------------------------------------------------------*/
obj_t
blit_string( obj_t s1, obj_t offset1, obj_t s2, obj_t offset2, obj_t len )
{
   bcopy( &STRING_REF( s1, CINT( offset1 ) ),
	  &STRING_REF( s2, CINT( offset2 ) ),
	  CINT( len ) );
   return BUNSPEC;
}
	
/*---------------------------------------------------------------------*/
/*    integer_to_string ...                                            */
/*---------------------------------------------------------------------*/
char *
integer_to_string( long x, long radix )
{
   char *aux;

   aux = (char *)GC_MALLOC_ATOMIC( 1024 );

   switch( radix )
   {
      case 8 :
         sprintf( aux, "#o%o", x );
         break;
         
      case 10 :
         sprintf( aux, "%ld", x );
         break;
         
      case 16 :
         sprintf( aux, "#x%lx", x );
         break;
   }

   return aux;
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    internal_real_to_string ...                                      */
/*    -------------------------------------------------------------    */
/*    This code is has been provided by Alain Mellan                   */
/*    <alain.mellan@st.com> and I think it is moslty coming from       */
/*    SCM. It implements a more precise number->string conversion      */
/*    that C does with the #g format.                                  */
/*---------------------------------------------------------------------*/
static int
internal_real_to_string( double f, char *a )
{
#define IS_INF(x) ((x)==(x)/2)
   static double fx[] = { 0.0, 5e-1, 5e-2, 5e-3, 5e-4, 5e-5,
			  5e-6, 5e-7, 5e-8, 5e-9, 5e-10,
			  5e-11,5e-12,5e-13,5e-14,5e-15,
			  5e-16,5e-17,5e-18,5e-19,5e-20 };
   int efmt, dpt, d, i, wp = DOUBLE_PRECISION;
   int ch = 0;
   int exp = 0;

   if( f==0.0 )
      goto zero;	
   
   if( f < 0.0 )
   {
      f = -f;
      a[ ch++ ] = '-';
   }
   else
      if( f > 0.0 );
      else
	 goto funny;
   
   if( IS_INF( f ) )
   {
      if( ch==0 )
	 a[ ch++ ] = '+';
   funny:
      a[ ch++ ] = '#';
      a[ ch++ ] = '.';
      a[ ch++ ] = '#';
      return ch;
  }

   while( f < 1.0 )
   {
      f *= 10.0;
      exp--;
   }
   while( f > 10.0 )
   {
      f /= 10.0;
      exp++;
   }

   if( f+fx[wp] >= 10.0 )
   {
      f = 1.0;
      exp++;
   }
   
zero:
   efmt = (exp < -3) || (exp > wp+2);
   if( !efmt )
      if( exp < 0 )
      {
	 a[ ch++ ] = '0';
	 a[ ch++ ] = '.';
	 dpt = exp;
	 while( ++dpt )
	    a[ ch++ ] = '0';
      }
      else
	 dpt = exp+1;
   else
      dpt = 1;

   do {
      d = f;
      f -= d;
      a[ ch++ ] = d+'0';
      if( f < fx[ wp ] )
	 break;
      if( f+fx[ wp ] >= 1.0 )
      {
	 a[ ch-1 ]++;
	 break;
      }
      f *= 10.0;
      if( !(--dpt) )
	 a[ ch++ ] = '.';
   } while( wp-- );

   if( dpt > 0 )
      if( (dpt > 4) && (exp > 6) )
      {
	 d = (a[ 0 ]=='-'?2:1);
	 for( i = ch++; i > d; i-- )
	    a[ i ] = a[ i-1 ];
	 a[ d ] = '.';
	 efmt = 1;
      }
      else
      {
	 while( --dpt )
	    a[ ch++ ] = '0';
	 a[ ch++ ] = '.';
      }
   
   if( a[ ch-1 ]=='.' )
      /* trailing zero */
      a[ ch++ ] = '0'; 
   
   if( efmt && exp )
   {
      a[ ch++ ] = 'e';
      if( exp < 0 )
      {
	 exp = -exp;
	 a[ ch++ ] = '-';
      }
      for( i = 10; i <= exp; i *= 10 );
      for (i /= 10; i; i /= 10 )
      {
	 a[ ch++ ] = exp/i + '0';
	 exp %= i;
      }
   }
   return ch;
#undef IS_INF
}
   
/*---------------------------------------------------------------------*/
/*    real_to_string ...                                               */
/*    -------------------------------------------------------------    */
/*    Many thanks to Raj Manandhar and Alain Mellan for providing us   */
/*    with these codes.                                                */
/*---------------------------------------------------------------------*/
char *
real_to_string( double x )
{
#if( !defined( DOUBLE_PRECISION ) )
#if( defined( __STDC__ ) )
#   define FLONUM_PREC 12
#   define STRINGIZE(x) stringize(x)
#   define stringize(x) #x
#   define FLONUM_LEN  (FLONUM_PREC + 8)
#   define FORMAT      "%#." STRINGIZE( FLONUM_PREC ) "g"
#else
#   define FLONUM_LEN  20
#   define FORMAT      "%#.12g"
#endif
   char new;

   new = GC_MALLOC_ATOMIC( FLONUM_LEN + 1 );
      
   memset( new, '\0', FLONUM_LEN );
   sprintf( new, FORMAT, REAL( o ).real );
   new[ FLONUM_LEN ] = '\0';

   {
       char *newp;
       int i, simplenum;

       for( newp = new, simplenum = 1; newp; newp++ )
	 if( !isdigit( *newp ) && (*newp != '.') && (*newp != '-') )
	 {
	     /* No exponents, infinity, underflow, etc */
	     simplenum = 0;
	     break;
	 }

       if( simplenum )
       {
	  /* Chop trailing zeros. */
	  for( i = FLONUM_LEN - 1;
	      (i > 0) && ((new[ i ] == '0') || (new[ i ] == '\0'));
	      i-- );
	  new[ i ] = '\0';
       }
   }
   return new;
#else
   char *new;
   int size;

   new = (char *)GC_MALLOC_ATOMIC( 1024 );
   size = internal_real_to_string( x, new );
   new[size] = '\0';
   
   return new;
#endif   
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bigloo_strcmp ...                                                */
/*---------------------------------------------------------------------*/
bool_t
bigloo_strcmp( obj_t o1, obj_t o2 )
{
   long l1, l2;

   l1 = STRING_LENGTH( o1 );
   l2 = STRING_LENGTH( o2 );

   if( l1 == l2 )
      return !memcmp( BSTRING_TO_STRING( o1 ), BSTRING_TO_STRING( o2 ), l1 );
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    strcicmp ...                                                     */
/*---------------------------------------------------------------------*/
bool_t        
strcicmp( obj_t bst1, obj_t bst2 )
{
   long l1, l2;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   if( l1 == l2 )
   {
      char *st1 = BSTRING_TO_STRING( bst1 ), *st2 = BSTRING_TO_STRING( bst2 );
      long i;
      
      for( i = 0;
	   (i < l1) && (tolower( *st1 ) == tolower( *st2 ));
	   i++, st1++, st2++ );
      
      if( i == l1 )
	 return 1;
      else
	 return 0;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    string_lt ...                                                    */
/*---------------------------------------------------------------------*/
bool_t       
string_lt( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0; (*st1 == *st2) && (i < min); i++, st1++, st2++ );

   if( i < min )
      return *st1 < *st2;
   else
      return l1 < l2;
}

/*---------------------------------------------------------------------*/
/*    string_le ...                                                    */
/*---------------------------------------------------------------------*/
bool_t        
string_le( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0; (*st1 == *st2) && (i < min); i++, st1++, st2++ );

   if( i < min )
      return *st1 <= *st2;
   else
      return l1 <= l2;
}

/*---------------------------------------------------------------------*/
/*    string_gt ...                                                    */
/*---------------------------------------------------------------------*/
bool_t       
string_gt( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0; (*st1 == *st2) && (i < min); i++, st1++, st2++ );

   if( i < min )
      return *st1 > *st2;
   else
      return l1 > l2;
}

/*---------------------------------------------------------------------*/
/*    string_ge ...                                                    */
/*---------------------------------------------------------------------*/
bool_t        
string_ge( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0; (*st1 == *st2) && (i < min); i++, st1++, st2++ );

   if( i < min )
      return *st1 >= *st2;
   else
      return l1 >= l2;
}

/*---------------------------------------------------------------------*/
/*    string_cilt ...                                                  */
/*---------------------------------------------------------------------*/
bool_t       
string_cilt( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0; (tolower( *st1 ) == tolower( *st2 )) && (i < min); i++, st1++, st2++ );

   if( i < min )
      return tolower( *st1 ) < tolower( *st2 );
   else
      return l1 < l2;
}

/*---------------------------------------------------------------------*/
/*    string_cile ...                                                  */
/*---------------------------------------------------------------------*/
bool_t        
string_cile( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0;
        (tolower( *st1 ) == tolower( *st2 )) && (i < min);
        i++, st1++, st2++ );

   if( i < min )
      return tolower( *st1 ) <= tolower( *st2 );
   else
      return l1 <= l2;
}

/*---------------------------------------------------------------------*/
/*    string_cigt ...                                                  */
/*---------------------------------------------------------------------*/
bool_t       
string_cigt( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0;
        (tolower( *st1 ) == tolower( *st2 )) && (i < min);
        i++, st1++, st2++ );

   if( i < min )
      return tolower( *st1 ) > tolower( *st2 );
   else
      return l1 > l2;
}

/*---------------------------------------------------------------------*/
/*    string_cige ...                                                  */
/*---------------------------------------------------------------------*/
bool_t        
string_cige( obj_t bst1, obj_t bst2 )
{
   unsigned char *st1 = (unsigned char *)BSTRING_TO_STRING( bst1 );
   unsigned char *st2 = (unsigned char *)BSTRING_TO_STRING( bst2 );
   long l1, l2;
   long i, min;

   l1 = STRING_LENGTH( bst1 );
   l2 = STRING_LENGTH( bst2 );

   min = (l1 < l2) ? l1 : l2;

   for( i = 0;
        (tolower( *st1 ) == tolower( *st2 )) && (i < min);
        i++, st1++, st2++ );

   if( i < min )
      return tolower( *st1 ) >= tolower( *st2 );
   else
      return l1 >= l2;
}

/*---------------------------------------------------------------------*/
/*    escape_C_string ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine ou la representation des     */
/*    caracteres de controles a ete remplacee par ces caracteres.      */
/*    ex:     +---+---+---+---+          +---+---+---+                 */
/*            | \ | n | a | 0 |    ==>   | \n| a | 0 |                 */
/*            +---+---+---+---+          +---+---+---+                 */
/*    Cette conversion est utile pour l'interprete car les chaines     */
/*    lues ne sont pas parsees. On donne donc la possibilite de le     */
/*    faire avec cette fonction.                                       */
/*---------------------------------------------------------------------*/
obj_t
escape_C_string( unsigned char *src )
{
   /* on supprime un caractere de cette chaine car elle est rendue par le */
   /* lecteur comme etant `"tototo'. Ceci est dut au fait qu'on utilise   */
   /* la fonction `the-small-string' qui supprime le premier et le        */
   /* dernier caractere de la chaine lu. Comme les chaines etrangeres     */
   /* commence par 2 caracteres, on en supprime 1 autre maintenant.       */

   long len = strlen( (const char *)(++src) );
   unsigned char *dst;
   obj_t string;

   string = GC_MALLOC( STRING_SIZE + len );

#if( !defined( TAG_STRING ) )
   string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	

   dst = ((unsigned char *)(&string->string_t.char0));

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
	 len--;
	 
         switch( *++src ) 
         {
            case '\0' : *dst++ = '\\';
                        break;

            case 'n'  : *dst++ = '\n';
                        break;
                        
            case 't'  : *dst++ = '\t';
                        break;
                        
            case 'b'  : *dst++ = '\b';
                        break;
                        
            case 'r'  : *dst++ = '\r';
                        break;
                        
            case 'f'  : *dst++ = '\f';
                        break;
                        
            case 'v'  : *dst++ = '\v';
                        break;
                        
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '\'' : *dst++ = '\'';
                        break;
                        
            case '"'  : *dst++ = '\"';
                        break;

#if( defined( __STDC___ ) )                          
            case 'a'  : *dst++ = '\a';
                        break;

            case '?'  : *dst++ = '\?';
                        break;
#endif                        

            default   : if( isdigit( *(src) ) &&
			    isdigit( *(src+1) ) &&
			    isdigit( *(src+2) ) )
	    /* C'est une representation alpha-numerique `\???' */
	    {
	       unsigned char aux;
	       
	       aux = (*src     - '0')*64 +
	             (*(src+1) - '0')*8 +
		     (*(src+2) - '0');
	       *dst++ = aux;
	       src+=2;
	       
	       len -= 2;
	    }
	    else
	       *dst++ = *src;
		 
	    break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = len;

   return BSTRING( string );
}
       
/*---------------------------------------------------------------------*/
/*    escape_scheme_string ...                                         */
/*    -------------------------------------------------------------    */
/*    Cette fonction ressemble a la precedente mais elle filtre moins  */
/*    de caracteres                                                    */
/*---------------------------------------------------------------------*/
obj_t
escape_scheme_string( char *src )
{
   long  len = strlen( src );
   char *dst;
   obj_t string;

   string = GC_MALLOC( STRING_SIZE + len );

#if( !defined( TAG_STRING ) )
   string->string_t.header = MAKE_HEADER( STRING_TYPE, 0 );
#endif	

   dst = ((char *)(&(string->string_t.char0)));

   while( *src )
   {
      if( *src != '\\' )
         *dst++ = *src++;
      else
      {
	 len--;
	 
         switch( *++src )  
         {
            case '\\' : *dst++ = '\\';
                        break;
                        
            case '"'  : *dst++ = '"';
                        break;

            default   : *dst++ = *src;
                        break;
         }
         src++;
      }
   }
   *dst = '\0';
   
   string->string_t.length = len;

   return BSTRING( string );
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    escape_char_found ...                                            */
/*    -------------------------------------------------------------    */
/*    This variable help write_string to decide to print (or           */
/*    to not print) a `#" character before the beginning of the        */
/*    string.                                                          */
/*---------------------------------------------------------------------*/
int escape_char_found = 0;

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    string_for_read ...                                              */
/*---------------------------------------------------------------------*/
obj_t
string_for_read( obj_t bstring )
{
   unsigned char *dst;
   unsigned char *src = (unsigned char *)(BSTRING_TO_STRING( bstring ));
   long  r, w, len = STRING_LENGTH( bstring );
   obj_t res;
	
#define BUFFER_SIZE 200
   unsigned char buffer[ BUFFER_SIZE ];
	
   escape_char_found = 0;

   if( ((len * 4) + 1) > BUFFER_SIZE )
      dst = malloc( (len * 4) + 1 );
   else
      dst = buffer;
#undef BUFFER_SIZE
	
   for( r = 0, w = 0; r < len; r++ )
      switch( src[ r ] )
      {
         case '\n' : dst[ w++ ] = '\\';
                dst[ w++ ] = 'n';
                escape_char_found = 1;
                break;

         case '\t' : dst[ w++ ] = '\\';
                dst[ w++ ] = 't';
                escape_char_found = 1;
                break;

         case '\b' : dst[ w++ ] = '\\';
                dst[ w++ ] = 'b';
                escape_char_found = 1;
                break;
              
         case '\r' : dst[ w++ ] = '\\';
                dst[ w++ ] = 'r';
                escape_char_found = 1;
                break;
              
         case '\f' : dst[ w++ ] = '\\';
                dst[ w++ ] = 'f';
                escape_char_found = 1;
                break;

         case '\v' : dst[ w++ ] = '\\';
                dst[ w++ ] = 'v';
                escape_char_found = 1;
                break;

         case '"'  : dst[ w++ ] = '\\';
                dst[ w++ ] = '"';
                escape_char_found = 1;
                break;
              
         case '\\' : dst[ w++ ] = '\\';
                dst[ w++ ] = '\\';
                escape_char_found = 1;
                break;

         default :   if( (src[r] < 128) && (isprint( src[ r ] )) )
                        dst[ w++ ] = src[ r ];
                     else
           {
              escape_char_found = 1;
              sprintf( &dst[ w ], "\\%03o", ((unsigned char *)src)[ r ] );
              w += 4;
           }
      }

   dst[ w ] = '\0';
   
   res = string_to_bstring( dst );

   if( dst != buffer )
      free( dst );
   
   return res;
}
