;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime1.9b/Llib/tvector.scm         */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Mon Mar 27 09:38:41 1995                          */
;*    Last change :  Fri Apr  4 16:24:43 1997 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The typed vectors Scheme management.                             */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module __tvector

   (import  (__error                   "Llib/error.scm")
	    (__hash                    "Llib/hash.scm"))
   
   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__structure               "Llib/struct.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_output_6_10_3        "Ieee/output.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__evenv                   "Eval/evenv.scm")
	    (__reader                  "Read/reader.scm"))

   (foreign (macro bool c-tvector?         (obj)        "TVECTORP")
	    (macro long c-tvector-length   (tvector)    "TVECTOR_LENGTH")
	    
	    (macro obj tvector-descr      (tvector)     "TVECTOR_DESCR")
	    (macro obj tvector-descr-set! (tvector obj) "TVECTOR_DESCR_SET")
	    
	    (export get-tvector-descriptor "get_tvector_descriptor"))

   (export  (inline tvector?::bool       ::obj)
	    (inline tvector-length::long ::tvector)
	    (get-tvector-descriptor::obj ::symbol)
	    (declare-tvector!            ::string  ::procedure ::obj ::obj)
	    (tvector-ref                 ::tvector)
	    (tvector-id                  ::tvector)
	    (list->tvector               ::symbol ::pair)
	    (vector->tvector             ::symbol ::vector)
	    (tvector->vector             ::tvector))

   (pragma  (c-tvector? (predicate-of tvector) no-cfa-top)
	    (tvector? side-effect-free no-cfa-top)
	    (c-tvector-length side-effect-free no-cfa-top)
	    (list->tvector no-cfa-top)
	    (tvector-descr side-effect-free no-cfa-top)
	    (get-tvector-descriptor side-effect-free no-cfa-top)
	    (tvector-ref side-effect-free no-cfa-top)
	    (tvector-id side-effect-free no-cfa-top)
	    (vector->tvector no-cfa-top)
	    (tvector->vector no-cfa-top)))

;*---------------------------------------------------------------------*/
;*    tvector? ...                                                     */
;*---------------------------------------------------------------------*/
(define-inline (tvector? obj)
   (c-tvector? obj))

;*---------------------------------------------------------------------*/
;*    tvector-length ...                                               */
;*---------------------------------------------------------------------*/
(define-inline (tvector-length obj)
   (c-tvector-length obj))

;*---------------------------------------------------------------------*/
;*    tvector-id ...                                                   */
;*---------------------------------------------------------------------*/
(define (tvector-id tvect)
   (tvect-descr-id (tvector-descr tvect)))

;*---------------------------------------------------------------------*/
;*    tvec-descr ...                                                   */
;*    -------------------------------------------------------------    */
;*    The structure to implements typed-vector descriptor. A unique    */
;*    structure is allocated for all the instances of a typed-vector.  */
;*---------------------------------------------------------------------*/
(define-struct tvect-descr
   id          ;; symbol          :: the identifier
   allocate    ;; procedure       :: a allocator
   ref         ;; procedure       :: how to access the elements
   set)        ;; procedure       :: how to set an element

;*---------------------------------------------------------------------*/
;*    *tvector-table* ...                                              */
;*---------------------------------------------------------------------*/
(define *tvector-table* '())
   
;*---------------------------------------------------------------------*/
;*    get-tvector-descriptor ...                                       */
;*---------------------------------------------------------------------*/
(define (get-tvector-descriptor id::symbol)
   (let ((cell (assq id *tvector-table*)))
      (if (pair? cell)
	  (cdr cell)
	  #f)))

;*---------------------------------------------------------------------*/
;*    declare-tvector! ...                                             */
;*    -------------------------------------------------------------    */
;*    The `id' argument has to be a c-string otherwise                 */
;*    declare-tvector! cannot be performed before the constants        */
;*    intialization.                                                   */
;*---------------------------------------------------------------------*/
(define (declare-tvector! id allocate ref set)
   (let* ((id  (string->symbol (if (not (eq? *bigloo-case-sensitive*
					     #unspecified))
				   (string->bstring id)
				   (string-upcase (string->bstring id)))))
	  (old (get-tvector-descriptor id)))
      (if (not (tvect-descr? old))
	  (let ((new (tvect-descr id allocate ref set)))
	     (set! *tvector-table* (cons (cons id new) *tvector-table*))
	     new)
	  old)))
	    
;*---------------------------------------------------------------------*/
;*    tvector-ref ...                                                  */
;*---------------------------------------------------------------------*/
(define (tvector-ref tvector)
   (tvect-descr-ref (tvector-descr tvector)))
 
;*---------------------------------------------------------------------*/
;*    list->tvector ...                                                */
;*---------------------------------------------------------------------*/
(define (list->tvector id::symbol l::pair)
   (let ((descr (get-tvector-descriptor id)))
      (if (not descr)
	  (error "list->tvector" "Undeclared tvector" id)
	  (let ((allocate (tvect-descr-allocate descr))
		(set      (tvect-descr-set descr)))
	     (if (not (procedure? set))
		 (error "list->tvector"
			"Unable to convert to such tvector"
			id)
		 (let* ((len  (length l))
			(tvec (allocate len)))
		    (let loop ((l l)
			       (i 0))
		       (if (null? l)
			   tvec
			   (begin
			      (set tvec i (car l))
			      (loop (cdr l) (+fx i 1)))))))))))

;*---------------------------------------------------------------------*/
;*    vector->tvector ...                                              */
;*---------------------------------------------------------------------*/
(define (vector->tvector id::symbol v::vector)
   (let ((descr (get-tvector-descriptor id)))
      (if (not descr)
	  (error "vector->tvector" "Undeclared tvector" id)
	  (let ((allocate (tvect-descr-allocate descr))
		(set      (tvect-descr-set descr)))
	     (if (not (procedure? set))
		 (error "vector->tvector"
			"Unable to convert to such tvector"
			id)
		 (let* ((len  (vector-length v))
			(tvec (allocate len)))
		    (let loop ((i (-fx len 1)))
		       (if (=fx i -1)
			   tvec
			   (begin
			      (set tvec i (vector-ref v i))
			      (loop (-fx i 1)))))))))))

;*---------------------------------------------------------------------*/
;*    tvector->vector ...                                              */
;*---------------------------------------------------------------------*/
(define (tvector->vector tv::tvector)
   (let ((descr (tvector-descr tv)))
      (let ((allocate (tvect-descr-allocate descr))
	    (ref      (tvect-descr-ref descr)))
	 (if (not (procedure? ref))
	     (error "tvector->vector"
		    "Unable to convert to such tvector"
		    (tvector-id tv))
	     (let* ((len  (tvector-length tv))
		    (vec  (c-create-vector len)))
		(let loop ((i (-fx len 1)))
		   (if (=fx i -1)
		       vec
		       (begin
			  (vector-set! vec i (ref tv i))
			  (loop (-fx i 1))))))))))
		    
   

   
