/* Copyright (C) 1979-1997 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

#include "mysql_priv.h"
#include "sql_lex.h"
#include <m_ctype.h>
#include <thr_alarm.h>
extern "C"
{
  extern pthread_key(int, THR_KEY_abort);
}

extern int yyparse(void);

static bool check_access(THD *thd,uint access,char *db=0);
static bool check_dup(char *name,TABLE_LIST *tables);
static void mysql_init_query(THD *thd);
static void remove_escape(char *name);

static char *any_db="any";		// Special symbol for check_access

char *command_name[]={
  "Sleep", "Quit", "Init DB", "Query", "Field List", "Create DB",
  "Drop DB", "Reload","Shutdown", "Statistics", "Processes",
  "Connect","Kill","Refresh",
};


/****************************************************************************
** mysql_execute_command
** Execute command saved in thd and current_lex->sql_command
****************************************************************************/

#ifdef __WIN32__
static void  test_signal(int sig_ptr)
{
   MessageBox(NULL,"Test signal","DBUG",MB_OK);
}
static void init_signals(void)
{
  int signals[7] = {SIGINT,SIGILL,SIGFPE,SIGSEGV,SIGTERM,SIGBREAK,SIGABRT } ;
  for(int i=0 ; i < 7 ; i++)
    signal( signals[i], test_signal) ;
}
#endif


pthread_handler_decl(handle_one_connection,arg)
{
  THD *thd=(THD*) arg;
  NET *net;
  DBUG_ENTER("handle_one_connection");
  thd->thread_stack= (char*) &thd;

#ifdef __WIN32__
  init_signals();				// IRENA; testing ?
#endif
  pthread_detach_this_thread();
  if (my_thread_init() || init_thr_lock() ||
      my_pthread_setspecific_ptr(THR_THD,  thd) ||
      my_pthread_setspecific_ptr(THR_MALLOC, &thd->alloc) ||
      my_pthread_setspecific_ptr(THR_NET,  &thd->net))
  {
    close_connection(&thd->net,ER_OUT_OF_RESOURCES);
    end_thread(0);
    return 0;
  }
#ifndef __WIN32__
  sigset_t set;
  VOID(sigemptyset(&set));			/* Get mask in use */
  VOID(pthread_sigmask(SIG_UNBLOCK,&set,&thd->block_signals));
#endif

  net= &thd->net;
  thd->mysys_abort_flag= my_pthread_getspecific(int*,THR_KEY_abort);
  if (max_join_size == (ulong) ~0L)
    thd->options |= OPTION_BIG_SELECTS;

  mysql_log.write(COM_CONNECT,
		  (thd->priv_user == thd->user ? "%s@%s on %s" :
		   "%s@%s as anonymous on %s"),
		  thd->user,
		  thd->host ? thd->host : thd->ip,
		  thd->db ? thd->db : "");
  thd->alloc.free=thd->alloc.used=0;
  if (thd->db)
  {						// If login with db
    char *db_name=thd->db;
    thd->db=0;
    if (mysql_change_db(thd,db_name))
      thd->killed=1;				// Could not change db
  }
  else
    send_ok(net);				// Ready to handle questions

  while (!net->error && net->fd >= 0 && !thd->killed)
  {
    if (do_command(thd))
      break;
  }
  VOID(pthread_mutex_lock(&LOCK_thread_count));
  if (net->error && net->fd >= 0)
  {
    fprintf(stderr,"Aborted connection %ld to db: '%s' user: '%s'\n",
	    thd->thread_id,(thd->db ? thd->db : "unconnected"),
	    thd->user);
  }
  thd->mysys_abort_flag=0;			// Don't allow this anymore
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
  close_connection(net);
  end_thread(0);
  return(0);					/* purecov: deadcode */
}

static inline void free_items(THD *thd)
{
    /* This works because items are allocated with sql_alloc() */
  for (Item *item=thd->free_list ; item ; item=item->next)
    delete item;
}


	/* Execute one command from socket */

bool do_command(THD *thd)
{
  char *packet;
  uint old_timeout,packet_length;
  bool	error=0,tables_used=0;
  NET *net;
  enum enum_server_command command;
  DBUG_ENTER("do_command");

  init_sql_alloc(&thd->alloc,8192);
  net= &thd->net;
  thd->select_distinct=thd->current_tablenr=0;
  thd->dupp_field=0;
  thd->tmp_table=0;

  packet=0;
  old_timeout=net->timeout;
  net->timeout=NET_WAIT_TIMEOUT;		/* Wait max for 8 hours */
  net->last_error[0]=0;				// Clear error message

  net_new_transaction(net);
  if ((packet_length=my_net_read(net)) == packet_error)
  {
    command = COM_QUIT;
    DBUG_PRINT("general",("Got error reading command from socket %d",net->fd));
  }
  else
  {
    packet=(char*) net->buff;
    command = (enum enum_server_command) (uchar) packet[0];
    DBUG_PRINT("general",("Command on socket %d = %d (%s)",
			  net->fd, command, command_name[command]));
  }
  net->timeout=old_timeout;			/* Timeout */
  thd->command=command;
  thd->version=reload_version;
  VOID(pthread_mutex_lock(&LOCK_thread_count));
  thd->query_id=query_id;
  if (command != COM_STATISTICS)
    query_id++;
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
  switch(command) {
  case COM_INIT_DB:
    if (!mysql_change_db(thd,packet+1))
      mysql_log.write(command,"%s",thd->db);
    break;
  case COM_QUERY:
    tables_used=1;
  if (!(specialflag & SPECIAL_NO_PRIOR))
    my_pthread_setprio(pthread_self(),QUERY_PRIOR);
#ifdef FIX_LOG					// Done in client
    if (opt_log)
    {						// Remove linefeed for log
      for (char *pos=packet+1; pos=strchr(pos,'\n'); pos++)
	*pos=' ';
    }
#endif
    mysql_log.write(command,"%s",packet+1);
    DBUG_PRINT("query",("%s",packet+1));
    mysql_parse(thd,thd->query=sql_memdup((gptr) (packet+1),packet_length),
		packet_length-1);
    if (!(specialflag & SPECIAL_NO_PRIOR))
      my_pthread_setprio(pthread_self(),WAIT_PRIOR);
    DBUG_PRINT("info",("query ready"));
    break;
  case COM_FIELD_LIST:				// This isn't actually neaded
  {
    tables_used=1;
    char *table,*fields;
    if (!thd->db)
    {
      send_error(net,ER_NO_DB_ERROR);
      break;
    }
    table=packet+1;
    fields=strend(table)+1;
    mysql_log.write(command,"%s %s",table,fields);
    remove_escape(table);			// This can't have wildcards
    thd->free_list=0;
    mysqld_list_fields(thd,sql_strdup(table),sql_strdup(fields));
    free_items(thd);
    break;
  }
  case COM_QUIT:
    mysql_log.write(command,NullS);
    net->error=0;				// Don't give 'abort' message
    error=TRUE;					// End server
    break;

  case COM_CREATE_DB:
    {
      char *db=sql_strdup(packet+1);
      if (check_access(thd,CREATE_ACL,db))
	break;
      mysql_log.write(command,packet+1);
      mysql_create_db(thd,db);
      break;
    }
  case COM_DROP_DB:
    {
      char *db=sql_strdup(packet+1);
      if (check_access(thd,DROP_ACL,db))
	break;
      mysql_log.write(command,db);
      mysql_rm_db(thd,db);
      break;
    }
  case COM_REFRESH:
    {
      uint options=(uchar) packet[1];
      if (check_access(thd,RELOAD_ACL,any_db))
	break;
      mysql_log.write(command,NullS);
      reload_acl_and_cache(options);
      send_eof(net);
      break;
    }
  case COM_SHUTDOWN:
    if (check_access(thd,SHUTDOWN_ACL,any_db))
      break; /* purecov: inspected */
    mysql_log.write(command,NullS);
    send_eof(net);
    close_connection(net);
    close_thread_tables(thd);			/* Free before kill */
    free_root(&thd->alloc);
#ifdef __WIN32__
    {
	extern HANDLE hEventShutdown;
	SetEvent(hEventShutdown);
	// or:
	// HANDLE hEvent=OpenEvent(0, FALSE, "MySqlShutdown");
	// SetEvent(hEventShutdown);
	// CloseHandle(hEvent);
    }
    //IRENA raise(SIGQUIT);
#else
    if (pthread_kill(signal_thread,SIGQUIT))	/* End everything nicely */
    {
      DBUG_PRINT("error",("Got error %d from pthread_kill",errno)); /* purecov: inspected */
    }
#endif
    error=TRUE;
    break;

  case COM_STATISTICS:
  {
    mysql_log.write(command,NullS);
    char buff[200];
    sprintf((char*) buff,
	    "Uptime: %ld  Running threads: %d  Questions: %ld  Opened_tables: %ld  Reloads: %ld  Open tables: %d",
	    (ulong) (time((time_t*) 0) - start_time),
	    (int) thread_count,thd->query_id,opened_tables,reload_version,
	    cached_tables());
#ifdef SAFEMALLOC
    if (lCurMemory)				// Using SAFEMALLOC
      sprintf(strend(buff), "  Memory in use: %ldK  Max memory used: %ldK",
	      (lCurMemory+1023L)/1024L,(lMaxMemory+1023L)/1024L);
 #endif
    VOID(my_net_write(net,(byte*) buff,strlen(buff)));
    VOID(net_flush(net));
    break;
  }
  case COM_PROCESS_INFO:
    if (check_access(thd,PROCESS_ACL,any_db))
      break;
    mysql_log.write(command,NullS);
    thd->free_list=0;
    mysqld_list_processes(thd);
    free_items(thd);
    break;
  case COM_PROCESS_KILL:
  {
    ulong thread_id=(ulong) uint4korr(packet+1);

    VOID(pthread_mutex_lock(&LOCK_thread_count)); // For unlink from list
    I_List_iterator<THD> it(threads);
    THD *tmp;
    uint error=ER_NO_SUCH_THREAD;
    while ((tmp=it++))
    {
      if (tmp->thread_id == thread_id)
      {
	if ((thd->master_access & PROCESS_ACL) ||
	    !strcmp(thd->user,tmp->user))
	{
	  thr_alarm_kill(tmp->real_id);
	  tmp->killed=1;
	  if (tmp->mysys_abort_flag)
	    *tmp->mysys_abort_flag=1;
	  error=0;
	  /* Release possible user lock */
	  pthread_mutex_lock(&LOCK_user_locks);
	  if (tmp->cond)
	    pthread_cond_broadcast(tmp->cond);
	  pthread_mutex_unlock(&LOCK_user_locks);
	}
	else
	  error=ER_KILL_DENIED_ERROR;
	break;
      }
    }
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
    if (!error)
      send_ok(net);
    else
      net_printf(net,error,thread_id);
    break;
  }
  case COM_DEBUG:
    if (check_access(thd,PROCESS_ACL,any_db))
      break;					/* purecov: inspected */
    thr_print_locks();				// Write some debug info
    TERMINATE(stdout);				// Write malloc information
    mysql_log.write(command,NullS);
    send_eof(net);
    break;
  case COM_SLEEP:
  case COM_CONNECT:				// Impossible here
  default:
    send_error(net, ER_UNKNOWN_COM_ERROR);
    break;
  }
  if (tables_used)
    close_thread_tables(thd);			/* Free tables */
#ifndef DBUG_OFF
  if (test_flags & TEST_PRINT_CACHED_TABLES)
  {
    VOID(pthread_mutex_lock(&LOCK_open)); /* purecov: inspected */
    print_cached_tables(); /* purecov: inspected */
    VOID(pthread_mutex_unlock(&LOCK_open)); /* purecov: inspected */
  }
#endif

  if (thd->fatal_error)
    send_error(net,0);				// End of memory ?
  VOID(pthread_mutex_lock(&LOCK_thread_count)); // For process list
  thd->proc_info=0;
  thd->command=COM_SLEEP;
  thd->query=0;
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
  thd->packet.shrink(net_buffer_length);	// Reclaim some memory
  free_root(&thd->alloc);
  DBUG_RETURN(error);
}


void
mysql_execute_command(void)
{
  int	res=0;
  THD	*thd=current_thd;
  LEX	*lex=current_lex;
  TABLE_LIST *tables=(TABLE_LIST*) thd->table_list.first;
  DBUG_ENTER("mysql_execute_command");

  switch(lex->sql_command) {
  case SQL_SELECT:
  {
    select_result *result;
    if (lex->options & SELECT_DESCRIBE)
      lex->exchange=0;
    if (check_access(thd,lex->exchange ? SELECT_ACL | FILE_ACL : SELECT_ACL,
		     tables ? thd->db : any_db))
    {
#ifdef DELETE_ITEMS
      delete lex->having;
      delete lex->where;
#endif
      break;					// Error message is given
    }
    thd->offset_limit=lex->offset_limit;
    thd->select_limit=lex->select_limit+lex->offset_limit;
    if (thd->select_limit < lex->select_limit)
      thd->select_limit= (ulong) ~0L;			// Fix if overflow

    if (lex->exchange)
    {
      if (!(result=new select_export(lex->exchange)))
      {
	res= -1;
#ifdef DELETE_ITEMS
	delete lex->having;
	delete lex->where;
#endif
	break;
      }
    }
    else if (!(result=new select_send()))
    {
      res= -1;
#ifdef DELETE_ITEMS
      delete lex->having;
      delete lex->where;
#endif
      break;
    }

    if (!(res=open_and_lock_tables(thd,tables)))
    {
      res=mysql_select(thd,tables,thd->item_list,
		       lex->where,
		       (ORDER*) thd->order_list.first,
		       (ORDER*) thd->group_list.first,
		       lex->having,
		       (ORDER*) thd->proc_list.first,
		       lex->options | thd->options,
		       result);
    }
    delete result;
#ifdef DELETE_ITEMS
    delete lex->having;
    delete lex->where;
#endif
    break;
  }
  case SQL_CREATE_TABLE:
    if (check_access(thd,CREATE_ACL))
      goto error;				/* purecov: inspected */
    if (strlen(tables->name) > NAME_LEN)
    {
      net_printf(&thd->net,ER_WRONG_TABLE_NAME,tables->name);
      res=0;
      break;
    }
    res = mysql_create_table(thd,tables->name, thd->create_list,
			     thd->key_list,0);
    if (!res)
      send_ok(&thd->net);
    break;
  case SQL_CREATE_INDEX:
    if (check_access(thd,CREATE_ACL))
      goto error; /* purecov: inspected */
    res = mysql_create_index(thd,tables->name, lex->col_list,
			     lex->unique_flag);
    break;
  case SQL_ALTER_TABLE:
    if (check_access(thd,DB_ACL))
      goto error; /* purecov: inspected */
    if (lex->name && strlen(lex->name) > NAME_LEN)
    {
      net_printf(&thd->net,ER_WRONG_TABLE_NAME,lex->name);
      res=0;
      break;
    }
    res= mysql_alter_table(thd, lex->name, tables, thd->create_list,
			   thd->key_list, lex->drop_list, lex->alter_list,
			   lex->drop_primary, lex->duplicates);
    break;
  case SQL_UPDATE:
    if (check_access(thd,UPDATE_ACL))
      goto error;
    if (thd->item_list.elements != thd->value_list.elements)
    {
      send_error(&thd->net,ER_WRONG_VALUE_COUNT);
      DBUG_VOID_RETURN;
    }
    res = mysql_update(thd,tables,
		       thd->item_list,
		       thd->value_list,
		       lex->where);
#ifdef DELETE_ITEMS
    delete lex->where;
#endif
    break;
  case SQL_INSERT:
    if (check_access(thd,INSERT_ACL))
      goto error; /* purecov: inspected */
    res = mysql_insert(thd,tables,thd->field_list,thd->value_list,DUP_ERROR);
    break;
  case SQL_REPLACE:
    if (check_access(thd,INSERT_ACL | UPDATE_ACL | DELETE_ACL))
      goto error; /* purecov: inspected */
    res = mysql_insert(thd,tables,thd->field_list,thd->value_list,DUP_REPLACE);
    break;
  case SQL_REPLACE_SELECT:
  case SQL_INSERT_SELECT:
  {
    if (check_access(thd,lex->sql_command == SQL_INSERT ?
		     INSERT_ACL | SELECT_ACL :
		     INSERT_ACL | UPDATE_ACL | DELETE_ACL | SELECT_ACL))
    {
#ifdef DELETE_ITEMS
      delete lex->having;
#endif
      goto error; /* purecov: inspected */
    }
    select_result *result;
    thd->offset_limit=lex->offset_limit;
    thd->select_limit=lex->select_limit+lex->offset_limit;
    if (thd->select_limit < lex->select_limit)
      thd->select_limit= (ulong) ~0L;		// Fix if overflow

    if (check_dup(tables->real_name,tables->next))
    {
      net_printf(&thd->net,ER_INSERT_TABLE_USED,tables->real_name);
      DBUG_VOID_RETURN;
    }
    tables->flags=1;				// update first table
    if (!(res=open_and_lock_tables(thd,tables)))
    {
      if ((result=new select_insert(tables->table,&thd->field_list,
				    lex->sql_command == SQL_REPLACE_SELECT ?
				    DUP_REPLACE : DUP_IGNORE)))
      {
	res=mysql_select(thd,tables->next,thd->item_list,
			 lex->where,
			 (ORDER*) thd->order_list.first,
			 (ORDER*) thd->group_list.first,
			 lex->having,
			 (ORDER*) thd->proc_list.first,
			 lex->options,
			 result);
	delete result;
      }
      else
	res= -1;
    }
#ifdef DELETE_ITEMS
    delete lex->having;
    delete lex->where;
#endif
    break;
  }
  case SQL_DELETE:
    if (check_access(thd,DELETE_ACL))
      goto error; /* purecov: inspected */
    res = mysql_delete(thd,tables,lex->where);
#ifdef DELETE_ITEMS
    delete lex->where;
#endif
    break;
  case SQL_DROP_TABLE:
    if (check_access(thd,DROP_ACL))
      goto error; /* purecov: inspected */
    res = mysql_rm_table(thd,tables);
    break;
  case SQL_DROP_INDEX:
    if (check_access(thd,DROP_ACL))
      goto error; /* purecov: inspected */
    res=0;					// Always ok
    send_ok(&thd->net);
    break;
  case SQL_SHOW_DATABASES:
    res= mysqld_show_dbs(thd, (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case SQL_SHOW_STATUS:
    res= mysqld_show_status(thd);
    break;
  case SQL_SHOW_VARIABLES:
    res= mysqld_show_variables(thd, (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case SQL_SHOW_TABLES:
    if (!lex->db)
    {
      send_error(&thd->net,ER_NO_DB_ERROR);	/* purecov: inspected */
      goto error;				/* purecov: inspected */
    }
    remove_escape(lex->db);			// Fix escaped '_'
    if (strlen(lex->db) > NAME_LEN)
    {
      net_printf(&thd->net,ER_WRONG_DB_NAME, lex->db);
      goto error;
    }
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error;				/* purecov: inspected */
    res= mysqld_show_tables(thd,lex->db,
			    (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case SQL_SHOW_FIELDS:
    if (!lex->db)
    {
      send_error(&thd->net,ER_NO_DB_ERROR);	/* purecov: inspected */
      goto error;				/* purecov: inspected */
    }
    remove_escape(lex->db);			// Fix escaped '_'
    remove_escape(tables->name);
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error;				/* purecov: inspected */
    res= mysqld_show_fields(thd,lex->db,tables->name,
			    (lex->wild ? lex->wild->ptr() : NullS));
    break;
  case SQL_SHOW_KEYS:
    if (!lex->db)
    {
      send_error(&thd->net,ER_NO_DB_ERROR);	/* purecov: inspected */
      goto error;				/* purecov: inspected */
    }
    remove_escape(lex->db);			// Fix escaped '_'
    remove_escape(tables->name);
    if (check_access(thd,SELECT_ACL,lex->db))
      goto error; /* purecov: inspected */
    res= mysqld_show_keys(thd,lex->db,tables->name);
    break;
  case SQL_CHANGE_DB:
    mysql_change_db(thd,lex->db);
    break;
  case SQL_LOAD:
    if (check_access(thd,INSERT_ACL | FILE_ACL))
      goto error;
    res=mysql_load(thd, lex->exchange, tables, thd->field_list,
		   lex->duplicates);
    break;
  case SQL_SET_OPTION:
    thd->options=lex->options;
    thd->default_select_limit=lex->select_limit;
    DBUG_PRINT("info",("options: %ld  limit: %ld",
		       thd->options,thd->default_select_limit));
    send_ok(&thd->net);
    break;
  case SQL_UNLOCK_TABLES:
    if (thd->locked_tables)
    {
      thd->lock=thd->locked_tables;
      thd->locked_tables=0;			// Will be automaticly closed
    }
    send_ok(&thd->net);
    break;
  case SQL_LOCK_TABLES:
    if (thd->locked_tables)
    {
      thd->lock=thd->locked_tables;
      thd->locked_tables=0;			// Will be automaticly closed
      close_thread_tables(thd);
    }
    if (!thd->db)
    {
      send_error(&thd->net,ER_NO_DB_ERROR);	/* purecov: inspected */
      goto error;				/* purecov: inspected */
    }
    if (!(res=open_and_lock_tables(thd,tables)))
    {
      thd->locked_tables=thd->lock;
      thd->lock=0;
      send_ok(&thd->net);
    }
    break;
  case SQL_CREATE_DB:
    {
      if (check_access(thd,CREATE_ACL,lex->name))
	break;
      mysql_create_db(thd,lex->name);
      break;
    }
  case SQL_DROP_DB:
    {
      if (check_access(thd,DROP_ACL,lex->name))
	break;
      mysql_rm_db(thd,lex->name);
      break;
    }
  case SQL_CREATE_FUNCTION:
    if (check_access(thd,INSERT_ACL,"mysql"))
      break;
#ifdef HAVE_DLOPEN
    if (!(res = mysql_create_function(thd,&lex->udf)))
      send_ok(&thd->net);
#else
    res= -1;
#endif
    break;
  case SQL_DROP_FUNCTION:
    if (check_access(thd,DELETE_ACL,"mysql"))
      break;
#ifdef HAVE_DLOPEN
    if (!(res = mysql_drop_function(thd,lex->udf.name)))
      send_ok(&thd->net);
#else
    res= -1;
#endif
    break;
  case SQL_GRANT:
  default:					/* Impossible */
    send_ok(&thd->net);
    break;
  }
  if (res < 0)
    send_error(&thd->net,thd->killed ? ER_SERVER_SHUTDOWN : 0);

error:
  DBUG_VOID_RETURN;
}


/* Return true if access denied */

static bool
check_access(THD *thd,uint want_access,char *db)
{
  uint access;
  if (!db && !thd->db)
  {
    send_error(&thd->net,ER_NO_DB_ERROR); /* purecov: tested */
    return TRUE; /* purecov: tested */
  }
  if ((thd->master_access & want_access) == want_access)
    return FALSE;
  if ((want_access & ~thd->master_access) & ~DB_ACL)
  {						// We can never grant this
    net_printf(&thd->net,ER_ACCESS_DENIED_ERROR,
	       thd->priv_user,
	       thd->host ? thd->host : "unknown",
	       thd->password ? ER(ER_YES) : ER(ER_NO));/* purecov: tested */
    return TRUE;				/* purecov: tested */
  }
  if (db == any_db)
    return FALSE;				// Alloc select on anything
  if (db && (!thd->db || strcmp(db,thd->db)))
    access=acl_get(thd->host, thd->ip, thd->priv_user, db); /* purecov: inspected */
  else
    access=thd->db_access;

  if (((access | thd->master_access) & want_access) == want_access)
    return FALSE;				/* Ok */
  net_printf(&thd->net,ER_DBACCESS_DENIED_ERROR,
	     thd->priv_user,
	     thd->host ? thd->host : "unknown",
	     db ? db : thd->db ? thd->db : "unknown"); /* purecov: tested */
  return TRUE;					/* purecov: tested */
}

/****************************************************************************
	Check stack size; Send error if there isn't enough stack to continue
****************************************************************************/

#define STACK_MIN_SIZE 2048	/* We will need this much stack later */
#if STACK_DIRECTION < 0
#define used_stack(A,B) (long) (A - B)
#else
#define used_stack(A,B) (long) (B - A)
#endif

bool check_stack_overrun(THD *thd,char *buf __attribute__((unused)))
{
  long stack_used;
  if ((stack_used=used_stack(thd->thread_stack,(char*) &stack_used)) >=
      (long) (thread_stack-STACK_MIN_SIZE))
  {
    sprintf(errbuff[0],ER(ER_STACK_OVERRUN),stack_used,thread_stack);
    my_message(ER_STACK_OVERRUN,errbuff[0],MYF(0));
    thd->fatal_error=1;
    return 1;
  }
  return 0;
}

#define MY_YACC_INIT 1000			// Start with big alloc
#define MY_YACC_MAX  32000			// Because of 'short'

bool my_yyoverflow(short **yyss, YYSTYPE **yyvs, int *yystacksize)
{
  LEX	*lex=current_lex;
  int  old_info=0;
  if ((uint) *yystacksize >= MY_YACC_MAX)
    return 1;
  if (!lex->yacc_yyvs)
    old_info= *yystacksize;
  *yystacksize= set_zone((*yystacksize)*2,MY_YACC_INIT,MY_YACC_MAX);
  if (!(lex->yacc_yyvs=
	my_realloc((gptr) lex->yacc_yyvs,
		   *yystacksize*sizeof(**yyvs),
		   MYF(MY_ALLOW_ZERO_PTR | MY_FREE_ON_ERROR))) ||
      !(lex->yacc_yyss=
	my_realloc((gptr) lex->yacc_yyss,
		   *yystacksize*sizeof(**yyss),
		   MYF(MY_ALLOW_ZERO_PTR | MY_FREE_ON_ERROR))))
    return 1;
  if (old_info)
  {						// Copy old info from stack
    memcpy(lex->yacc_yyss, (gptr) *yyss, old_info*sizeof(**yyss));
    memcpy(lex->yacc_yyvs, (gptr) *yyvs, old_info*sizeof(**yyvs));
  }
  *yyss=(short*) lex->yacc_yyss;
  *yyvs=(YYSTYPE*) lex->yacc_yyvs;
  return 0;
}


/****************************************************************************
	Initialize global thd variables neaded for query
****************************************************************************/

static void
mysql_init_query(THD *thd)
{
  DBUG_ENTER("mysql_init_query");
  thd->net.last_error[0]=0;
  thd->item_list.empty();
  thd->value_list.empty();
  thd->key_list.empty();
  thd->create_list.empty();
  thd->order_list.elements=thd->table_list.elements= thd->group_list.elements=
    thd->create_list.elements=thd->proc_list.elements=0;
  thd->free_list=0;

  thd->order_list.first=0;
  thd->order_list.next= (byte**) &thd->order_list.first;
  thd->table_list.first=0;
  thd->table_list.next= (byte**) &thd->table_list.first;
  thd->group_list.first=0;
  thd->group_list.next= (byte**) &thd->group_list.first;
  thd->proc_list.first=0;
  thd->proc_list.next= (byte**) &thd->proc_list.first;
  thd->set_time();
  thd->fatal_error=0;				// Safety
  thd->last_insert_id_used=thd->query_start_used=thd->insert_id_used=0;
  DBUG_VOID_RETURN;
}


void
mysql_parse(THD *thd,char *inBuf,uint length)
{
  DBUG_ENTER("mysql_parse");
  mysql_init_query(thd);
  LEX *lex=lex_start((uchar*) inBuf,length);
  if (!lex)					// Ensure that we have memory
    DBUG_VOID_RETURN;
  if (!yyparse() && ! thd->fatal_error)
    mysql_execute_command();
  /* Free strings used by items */
#ifdef DELETE_ITEMS
  thd->item_list.delete_elements();
  thd->value_list.delete_elements();
  thd->field_list.delete_elements();
  for (ORDER *order=(ORDER*) thd->proc_list.first ; order ; order=order->next)
  {
    if (order->free_me)
      delete *order->item;
  }
#else
  free_items(thd);
#endif
  lex_end(lex);
  DBUG_VOID_RETURN;
}


inline static void
link_in_list(SQL_LIST *list,byte *element,byte **next)
{
  list->elements++;
  (*list->next)=element;
  list->next=next;
  *next=0;
}


/*****************************************************************************
** Store field definition for create
** Return 0 if ok
******************************************************************************/

bool add_field_to_list(char *field_name, enum_field_types type,
		       char *length, char *decimals,
		       uint type_modifier,int unireg_check,
		       int keyinfo, Item *default_value,char *change,
		       TYPELIB *interval)
{
  register create_field *new_field;
  THD	*thd=current_thd;
  DBUG_ENTER("add_field_to_list");

  if (strlen(field_name) > NAME_LEN)
  {
    net_printf(&thd->net, ER_TOO_LONG_IDENT, field_name); /* purecov: inspected */
    DBUG_RETURN(1);				/* purecov: inspected */
  }
  if (keyinfo)
  {
    type_modifier|=NOT_NULL_FLAG;		// Keys can't be NULL in mysql
    if (keyinfo == PRI_KEY_FLAG)		// Compatible with mysql
    {
      current_lex->col_list.push_back(new key_part_spec(field_name,0));
      thd->key_list.push_back(new Key(Key::PRIMARY,NullS,
				      current_lex->col_list));
      current_lex->col_list.empty();
    }
  }

  if (default_value && default_value->type() == Item::NULL_ITEM)
  {
    if (type_modifier & NOT_NULL_FLAG)
    {
      net_printf(&thd->net,ER_INVALID_DEFAULT,field_name);
      DBUG_RETURN(1);
    }
    default_value=0;
  }
  /* change FLOAT(precision) to FLOAT or DOUBLE */
  if (type == FIELD_TYPE_FLOAT && length && !decimals)
  {
    uint tmp_length=atoi(length);
    if (tmp_length > sizeof(double))
    {
      net_printf(&thd->net,ER_WRONG_FIELD_SPEC,field_name); /* purecov: inspected */
      DBUG_RETURN(1); /* purecov: inspected */
    }
    else if (tmp_length > sizeof(float))
      type=FIELD_TYPE_DOUBLE;
    length=0;				// Use default format
  }
  if (!(new_field=new create_field()))
    DBUG_RETURN(1);
  new_field->field=0;
  new_field->field_name=field_name;
  new_field->def=default_value;
  new_field->flags= type_modifier;
  new_field->sql_type=type;
  new_field->unireg_check= (Field::utype) unireg_check;
  new_field->decimals= decimals ? (uint) set_zone(atoi(decimals),0,30) : 0;
  new_field->length=0;
  new_field->change=change;
  new_field->interval=0;
  new_field->pack_length=0;
  if (keyinfo)
    new_field->flags|= keyinfo | NOT_NULL_FLAG; /* Key can't be null */
  if (length)
    if (!(new_field->length= (uint) atoi(length)))
      length=0; /* purecov: inspected */
  uint sign_len=type_modifier & UNSIGNED_FLAG ? 0 : 1;
  if (new_field->length && new_field->decimals &&
      new_field->length < new_field->decimals+2)
    new_field->length=new_field->decimals+2; /* purecov: inspected */

  switch (type) {
  case FIELD_TYPE_YEAR:				// Avoid warning
  case FIELD_TYPE_TINY:
    if (!length) new_field->length=3+sign_len;
    break;
  case FIELD_TYPE_SHORT:
    if (!length) new_field->length=5+sign_len;
    break;
  case FIELD_TYPE_INT24:
    if (!length) new_field->length=8+sign_len;
    break;
  case FIELD_TYPE_LONG:
    if (!length) new_field->length=10+sign_len;
    break;
  case FIELD_TYPE_LONGLONG:
    if (!length) new_field->length=20+sign_len;
    break;
  case FIELD_TYPE_STRING:
  case FIELD_TYPE_VAR_STRING:
  case FIELD_TYPE_DECIMAL:
  case FIELD_TYPE_NULL:
    break;
  case FIELD_TYPE_BLOB:
  case FIELD_TYPE_TINY_BLOB:
  case FIELD_TYPE_LONG_BLOB:
  case FIELD_TYPE_MEDIUM_BLOB:
    if (default_value)				// Allow empty as default value
    {
      String str,*res;
      res=default_value->str(&str);
      if (res->length())
      {
	net_printf(&thd->net,ER_BLOB_CANT_HAVE_DEFAULT,field_name); /* purecov: inspected */
	DBUG_RETURN(1); /* purecov: inspected */
      }
      new_field->def=0;
    }
    new_field->flags|=BLOB_FLAG;
    break;
  case FIELD_TYPE_FLOAT:
    if (!length)
    {
      new_field->length = 8+2;			// Default 2 decimals
      new_field->decimals=2;
    }
    break;
  case FIELD_TYPE_DOUBLE:
    if (!length)
    {
      new_field->length = 12+4;
      new_field->decimals=4;
    }
    break;
  case FIELD_TYPE_TIMESTAMP:
    if (!length)
      new_field->length= 14;			// Full date YYYYMMDDHHMMSS
    else
    {
      new_field->length=((new_field->length+1)/2)*2; /* purecov: inspected */
      new_field->length= min(new_field->length,14); /* purecov: inspected */
    }
    new_field->flags|= ZEROFILL_FLAG | UNSIGNED_FLAG | NOT_NULL_FLAG;
    new_field->unireg_check=Field::TIMESTAMP_FIELD;
    break;
  case FIELD_TYPE_NEWDATE:
  case FIELD_TYPE_DATE:
    new_field->length=10;
    break;
  case FIELD_TYPE_TIME:
    new_field->length=8;
    break;
  case FIELD_TYPE_DATETIME:
    new_field->length=19;
    break;
  case FIELD_TYPE_SET:
    {
      if (interval->count > sizeof(longlong)*8)
      {
	net_printf(&thd->net,ER_TOO_BIG_SET,field_name); /* purecov: inspected */
	DBUG_RETURN(1); /* purecov: inspected */
      }
      new_field->pack_length=(interval->count+7)/8;
      if (new_field->pack_length > 4)
	new_field->pack_length=8;
      new_field->interval=interval;
      new_field->length=strlen(interval->type_names[0]);
      for (char **pos=interval->type_names+1; *pos ; pos++)
      {
	uint length=strlen(*pos);
	set_if_bigger(new_field->length,length);
      }
      set_if_smaller(new_field->length,MAX_FIELD_WIDTH-1);
      if (default_value)
      {
	thd->cuted_fields=0;
	String str,*res;
	res=default_value->str(&str);
	(void) find_set(interval,res->ptr(),res->length());
	if (thd->cuted_fields)
	{
	  net_printf(&thd->net,ER_INVALID_DEFAULT,field_name);
	  DBUG_RETURN(1);
	}
      }
    }
    break;
  case FIELD_TYPE_ENUM:
    {
      new_field->interval=interval;
      new_field->pack_length=interval->count < 256 ? 1 : 2; // Should be safe
      new_field->length=0;
      for (char **pos=interval->type_names; *pos ; pos++)
	new_field->length+=strlen(*pos)+1;
      new_field->length--;
      set_if_smaller(new_field->length,MAX_FIELD_WIDTH-1);
      if (default_value)
      {
	String str,*res;
	res=default_value->str(&str);
	if (!find_enum(interval,res->ptr(),res->length()))
	{
	  net_printf(&thd->net,ER_INVALID_DEFAULT,field_name);
	  DBUG_RETURN(1);
	}
      }
      break;
    }
  }
  if (new_field->length >= MAX_FIELD_WIDTH ||
      (!new_field->length && !(new_field->flags & BLOB_FLAG)))
  {
    net_printf(&thd->net,ER_TOO_BIG_FIELDLENGTH,field_name,
	       MAX_FIELD_WIDTH-1);		/* purecov: inspected */
    DBUG_RETURN(1);				/* purecov: inspected */
  }
  if (!new_field->pack_length)
    new_field->pack_length=calc_pack_length(new_field->sql_type,
					    new_field->length);
  thd->create_list.push_back(new_field);
  DBUG_RETURN(0);
}


bool
add_proc_to_list(Item *item)
{
  ORDER *order;
  Item	**item_ptr;

  if (!(order = (ORDER *) sql_alloc(sizeof(ORDER)+sizeof(Item*))))
    return 1;
  item_ptr = (Item**) (order+1);
  *item_ptr= item;
  order->item=item_ptr;
  order->free_me=0;
  link_in_list(&current_thd->proc_list,(byte*) order,(byte**) &order->next);
  return 0;
}


/* Fix escaping of _, % and \ in database and table names (for ODBC) */

static void remove_escape(char *name)
{
  char *to;
  for (to=name; *name ; name++)
  {
#ifdef USE_BIG5CODE
    if (name[1] && isbig5code(*name,*(name+1)))
    {
      *to++= *name++;
      *to++= *name;
      continue;
    }
#endif
#ifdef USE_MB
    int l;
    if ((l = ismbchar(name, name+MBMAXLEN))) {
	while (l--)
	    *to++ = *name++;
	name--;
	continue;
    }
#endif
    if (*name == '\\' && name[1])
      name++;					// Skipp '\\'
    *to++= *name;
  }
  *to=0;
}

/****************************************************************************
** save order by and tables in own lists
****************************************************************************/


bool add_to_list(SQL_LIST &list,Item *item,bool asc)
{
  ORDER *order;
  Item	**item_ptr;
  DBUG_ENTER("add_to_list");
  if (!(order = (ORDER *) sql_alloc(sizeof(ORDER)+sizeof(Item*))))
    DBUG_RETURN(1);
  item_ptr = (Item**) (order+1);
  *item_ptr=item;
  order->item= item_ptr;
  order->asc = asc;
  order->free_me=0;
  link_in_list(&list,(byte*) order,(byte**) &order->next);
  DBUG_RETURN(0);
}


TABLE_LIST *add_table_to_list(char *name,uint length,char *alias,uint flags)
{
  register TABLE_LIST *ptr;
  THD	*thd=current_thd;
  DBUG_ENTER("add_table_to_list");

  if (!alias)
    alias=name;
  if (length > NAME_LEN)
  {
    net_printf(&thd->net,ER_WRONG_TABLE_NAME,name);
    DBUG_RETURN(0);
  }
#ifdef FN_LOWER_CASE
  if (alias == name)
    if (!(alias=sql_strmake(name,length)))     /* Alias is case sensitive */
      DBUG_RETURN(0);
  casedn_str(name);
#endif
  if (!(ptr = (TABLE_LIST *) sql_alloc(sizeof(TABLE_LIST))))
    DBUG_RETURN(0);
  ptr->db=0;
  ptr->real_name=name;
  ptr->name=alias;
  ptr->flags=flags;
  ptr->table=0;
  ptr->on_expr=0;
  ptr->natural_join=0;

  /* check that used name is unique */
  for (TABLE_LIST *table=(TABLE_LIST*) thd->table_list.first ; table ;
       table=table->next)
  {
    if (!strcmp(alias,table->name))
    {
      net_printf(&thd->net,ER_NONUNIQ_TABLE,alias); /* purecov: tested */
      DBUG_RETURN(0);				/* purecov: tested */
    }
  }
  link_in_list(&thd->table_list,(byte*) ptr,(byte**) &ptr->next);
  DBUG_RETURN(ptr);
}

void add_left_join_on(TABLE_LIST *a __attribute__((unused)),
			TABLE_LIST *b,Item *expr)
{
  b->on_expr=expr;
}


void add_left_join_natural(TABLE_LIST *a,TABLE_LIST *b)
{
  b->natural_join=a;
}

	/* Check if name is used in table list */

static bool check_dup(char *name,TABLE_LIST *tables)
{
  for (; tables ; tables=tables->next)
    if (!strcmp(name,tables->real_name))
      return 1;
  return 0;
}

void reload_acl_and_cache(uint options)
{
  select_errors=0;				/* Write if more errors */
  mysql_log.flush();				// Flush log
  if (options & REFRESH_GRANT)
    acl_reload();
  if (options & REFRESH_LOG)
  {
    mysql_log.new_file();
    mysql_update_log.new_file();
  }
  if (options & REFRESH_TABLES)
  {
    close_cached_tables();
  }
  if (options & (REFRESH_GRANT | REFRESH_TABLES))
    reload_version++;				/* Doesn't nead protection */
}
