/*
 * Ispell spellchecker front end for XFMail.
 *
 * Parts of this file are based upon the LyX frontend to ispell.
 * LyX is Copyright (C) 1995 by Matthias Ettrich, 1995-1997 LyX Team
 * and distributed under the terms of the GNU General Public License.
 *
 * Modified to work with XFMail in April 1997
 *     by Thomas Pundt <pundtt@informatik.uni-muenster.de>
 */

/* $Id: spellchecker.c,v 2.0 1997/09/23 15:55:22 gena Exp $
 */

#include <fmail.h>
#include <umail.h>
#include <choose_folder.h>
#include "configform.h"
#include <fl_edit.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include "dialogs.h"
#include "spellchecker.h"
#ifdef HAVE_REGCOMP
#include <regex.h>
#endif

static FD_config_spell *fd_spell_options;
static FD_spell_check *fd_spell_check = NULL;

static FL_OBJECT *textobj;
static FL_FORM *msg_form;

/* file handles for communication with ispell */
static FILE *out = NULL, *in = NULL;

/* set to -1 in sighandler if ispell terminates */
static int speller_alive; 

/*
 * Display error messages and close pipes, if ispell terminates.
 * Set speller_alive, so that other routines can check if ispell is still
 * alive.
 */
void 
spell_checker_cleanup(proc_info *pinfo)
{
  fd_set rdfs;
  struct timeval tv;
  char buf[1024], *p;

  FD_ZERO(&rdfs);
  FD_SET(pinfo->fd_err[0], &rdfs);
  tv.tv_sec=0;
  tv.tv_usec=0;

  if (pinfo->status!=0)
#ifdef	hpux
    if (select(pinfo->fd_err[0]+1, (int *)&rdfs, (int *)NULL, (int *)NULL, &tv)>0) {
#else
    if (select(pinfo->fd_err[0]+1, &rdfs, NULL, NULL, &tv)>0) {
#endif
      read(pinfo->fd_err[0],buf,1024);
      if ((p=strchr(buf,'\n')))
        *p='\0';
      display_msg(MSG_WARN, "Ispell", buf);
    }

  if (pinfo->fd_in[1] > 0)
    close(pinfo->fd_in[1]);
  if (pinfo->fd_out[0] > 0)
    close(pinfo->fd_out[0]);
  if (pinfo->fd_err[0] > 0)
    close(pinfo->fd_err[0]);

  speller_alive = -1;
  return;
}

/*
 * create ispell process
 * returns -1 on error, 0 on success
 */
static int 
spell_checker_setup(char *lang)
{
  char buf[1024], *p;
  int res;
  proc_info pinfo;

  init_pinfo(&pinfo);
  pinfo.handle = spell_checker_cleanup;
  pinfo.fd_in[0] = pinfo.fd_out[0] = pinfo.fd_err[0] = 0;
  res = b_getcfg_int(conf_name, "ispell_use_esc_chars", 0);

  sprintf (buf, "%s -a %s%s %s %s%s%s %s%s %s",
           b_getcfg_str(conf_name, "ispell_command", _PATH_ISPELL),
	   strlen(lang) ? "-d " : "", lang,
	   b_getcfg_int(conf_name, "ispell_accept_compound", 0) ? "-C" : "-B",
	   res ? "-w \"" : "",
	   res ? b_getcfg_str(conf_name, "ispell_escchars", "") : "", res ? "\"" : "",
	   (res=b_getcfg_int(conf_name, "ispell_use_pers_dict", 0)) ? "-p " : "",
	   res ? b_getcfg_str(conf_name, "ispell_dictionary", "") : "",
	   b_getcfg_int(conf_name, "ispell_use_input_encoding", 0) ? "-T latin1" : "");

  free(lang);

  speller_alive = 0;
  if (exec_child(buf, &pinfo) == -1) {
    read(pinfo.fd_err[0],buf,1024);
    if ((p=strchr(buf,'\n')))
      *p='\0';
    display_msg(MSG_WARN, "Ispell", buf);
    spell_checker_cleanup(&pinfo);
    return -1;
  }

  in  = fdopen(pinfo.fd_out[0],"r");
  out = fdopen(pinfo.fd_in[1],"w");

  check_extprocs();
  if (speller_alive == -1)
    return -1;

  if (!fgets(buf, 1024, in)) /* Read ispells identification message */
    return -1;               /* no identification => error */
  fputs("!\n", out);         /* Terse mode (silently accept correct words) */
  fflush(out);
  return 0;
}

/*
 * Configuration
 */
void 
spell_checker_options_update(int set_default, FD_config_spell *form) 
{
  fd_spell_options=form;
  fl_set_input(fd_spell_options->ispell_prg,
               b_getcfg_str_default(conf_name, "ispell_command", _PATH_ISPELL, set_default));
  fl_set_input(fd_spell_options->altlang_input, 
               b_getcfg_str_default(conf_name, "ispell_language", "", set_default));
  fl_set_input(fd_spell_options->perdict_input, 
               b_getcfg_str_default(conf_name, "ispell_dictionary", "", set_default));
  fl_set_input(fd_spell_options->esc_chars_input, 
               b_getcfg_str_default(conf_name, "ispell_escchars", "", set_default));
  fl_set_button(fd_spell_options->altlang,
                b_getcfg_int_default(conf_name, "ispell_use_alt_lang", 0, set_default));
  fl_set_button(fd_spell_options->buflang,
                !fl_get_button(fd_spell_options->altlang));
  fl_set_button(fd_spell_options->perdict,
                b_getcfg_int_default(conf_name, "ispell_use_pers_dict", 0, set_default));
  fl_set_button(fd_spell_options->esc_chars,
                b_getcfg_int_default(conf_name, "ispell_use_esc_chars", 0, set_default));
  fl_set_button(fd_spell_options->compounds,
                b_getcfg_int_default(conf_name, "ispell_accept_compound", 0, set_default));
  fl_set_button(fd_spell_options->inputenc,
                b_getcfg_int_default(conf_name, "ispell_use_input_encoding", 0, set_default));
  fl_set_button(fd_spell_options->citation,
                  b_getcfg_int_default(conf_name, "ispell_ignore_citation", 0, set_default));
}

void 
handle_spell_input(FD_config_spell *form)
{
  fd_spell_options=form;
  b_putcfg_str(conf_name, "ispell_command", (char*)fl_get_input(fd_spell_options->ispell_prg));
  b_putcfg_str(conf_name, "ispell_language", (char*)fl_get_input(fd_spell_options->altlang_input));
  b_putcfg_str(conf_name, "ispell_dictionary", (char*)fl_get_input(fd_spell_options->perdict_input));
  b_putcfg_str(conf_name, "ispell_escchars", (char*)fl_get_input(fd_spell_options->esc_chars_input));
  b_putcfg_int(conf_name, "ispell_use_alt_lang", fl_get_button(fd_spell_options->altlang));
  b_putcfg_int(conf_name, "ispell_use_pers_dict", fl_get_button(fd_spell_options->perdict));
  b_putcfg_int(conf_name, "ispell_accept_compound", fl_get_button(fd_spell_options->compounds));
  b_putcfg_int(conf_name, "ispell_use_esc_chars", fl_get_button(fd_spell_options->esc_chars));
  b_putcfg_int(conf_name, "ispell_use_input_encoding", fl_get_button(fd_spell_options->inputenc));
  b_putcfg_int(conf_name, "ispell_ignore_citation", fl_get_button(fd_spell_options->citation));
}

void 
spell_conf(int set_default, FD_config_spell *form)
{
  fd_spell_options = form;
  spell_checker_options_update(set_default, form);
  fl_set_focus_object(fd_spell_options->config_spell, fd_spell_options->altlang_input);
}

/*
 * spell checking
 */
typedef struct ispell_result {
  int count;
  char *reply;
  char **suggestion;
} ispell_result;

/* 
 * Send word to ispell and parse reply;
 * returns NULL, if ispell has terminated,
 * else structure with ispell reply result.
 */
static struct ispell_result *
spellcheck_word(char *word)
{
  struct ispell_result *result;
  char buf[1024], *p;
  int i;

  check_extprocs();
  if (speller_alive==-1)
    return NULL;

  fputs("^", out); fputs(word, out); fputs("\n", out); fflush(out);
  fgets(buf, 1024, in);
  
  result = (ispell_result *) calloc(sizeof(ispell_result),1);
  result->count = -1; /* "-1" indicates a correct word */
  
  switch (*buf) {
  case '\n': /* Number or when in terse mode: no problems, no next line */
  break;
  case '&': /* "& <original> <count> <offset>: <miss>, <miss>, ..." */
    result->reply = strdup(buf);
    p = strchr(result->reply+2, ' ');
    sscanf(p, "%d", &result->count);
    if (result->count) 
      result->suggestion = (char**)calloc(sizeof(char*), result->count);
    p = strtok(strchr(p, ':') + 1, ",\n") + 1;
    for (i = 0; i<result->count; i++) {
      result->suggestion[i] = p;
      p = strtok(NULL, ",\n") + 1;
    }
  case '#': /* Not found, no near misses and guesses */
  case '?': /* Not found, and no near misses, but guesses */
    if (result->count==-1) /* did we run through "case '&'" ? */
      result->count=0;
  case '*': /* Word found */
  case '+': /* Word found through affix removal */
  case '-': /* Word found through compound formation */
    /* we have to read one more line from ispell */
    check_extprocs(); 
    *buf = '\0';
    while ((*buf!='\n') && (speller_alive!=-1)) 
      fgets(buf, 1024, in);
  }
  return result;
}

static void
spellcheck_delete_result(ispell_result *result)
{
  if (result->reply)
    free(result->reply);
  if (result->suggestion)
    free(result->suggestion);
  free(result);
}

enum { SPELLER_CONT=0, SPELLER_STOP, SPELLER_DONE };

static int
spell_check_interact_forms(char *word)
{
  int line = -1;
  FL_OBJECT *obj;

  for(;;) {
    obj = fl_do_forms();

    if ((obj==fd_spell_check->stop) || (obj==fd_spell_check->done)) {
      if (out)		{
       fputs("#\n", out);
       fclose(out);
       out = NULL;	}
      return (obj==fd_spell_check->stop) ? SPELLER_STOP : SPELLER_DONE;
    }

    if ((obj==fd_spell_check->replace) || (obj==fd_spell_check->input)) {
      fl_textedit_replace_sel(textobj, (char*)fl_get_input(fd_spell_check->input));
      break;
    }

    if (obj==fd_spell_check->ignore)
      break;

    if (obj==fd_spell_check->insert) {
      fputs("*", out); fputs(word, out); fputs("\n", out); fflush(out);
      break;
    }

    if (obj==fd_spell_check->acceptword) {
      fputs("@", out); fputs(word, out); fputs("\n", out); fflush(out);
      break;
    }

    /* "simulate" a double click action in the browser */
    if (obj==fd_spell_check->browser) {
      if ((line == fl_get_browser(fd_spell_check->browser)) &&
	   (fl_get_timer(fd_spell_check->browser_timer)>0.0)) {
	fl_textedit_replace_sel(textobj, (char*)fl_get_input(fd_spell_check->input));
	break;
      }
      line = fl_get_browser(fd_spell_check->browser);
      fl_set_timer(fd_spell_check->browser_timer, 0.4);
      fl_set_input(fd_spell_check->input,
		   fl_get_browser_line(fd_spell_check->browser,
				       fl_get_browser(fd_spell_check->browser)));
    }
  }
  return SPELLER_CONT;
}

static int 
do_spell_session()
{
  ispell_result *result;
  char *word;
  int i, res=SPELLER_CONT;
  int citation=b_getcfg_int(conf_name, "ispell_ignore_citation", 0);
  char *lang = b_getcfg_int(conf_name, "ispell_use_alt_lang", 0) ? 
               b_getcfg_str(conf_name, "ispell_language", "") : "";

  if (spell_checker_setup(strdup(lang)) == -1)
    return -1;

  /*
   * check the text word by word
   */
  while (res==SPELLER_CONT) {
    if (!(word = fl_textedit_get_nextword(textobj, citation ? is_cited_line : NULL))) {
      res=SPELLER_DONE;
      break;
    }

    if (!(result = spellcheck_word(word))) {
      free(word);
      res=SPELLER_DONE;
      break;
    }

    if (result->count>=0) {
      fl_freeze_form(fd_spell_check->spell_check);
      fl_set_object_label(fd_spell_check->text, word);
      fl_clear_browser(fd_spell_check->browser);
      for (i=0; i<result->count; i++)
	fl_add_browser_line(fd_spell_check->browser, result->suggestion[i]);
      if (result->count)
	fl_set_input(fd_spell_check->input, result->suggestion[0]);
      else
	fl_set_input(fd_spell_check->input, word);
      fl_unfreeze_form(fd_spell_check->spell_check);
      res=spell_check_interact_forms(word);
    }
    spellcheck_delete_result(result);
    free(word);
  };

  if (out)	{
   fputs("#\n", out); /* Save personal dictionary */
   fclose(out);
   out = NULL;	}
  return res;
}

void spell_check_hide_form()
{
  fl_hide_form(fd_spell_check->spell_check);
  fl_free_form(fd_spell_check->spell_check);
  fl_activate_form(msg_form);
  fd_spell_check=NULL;
}

/*
 * Don't kill main application in case the window cancel button was
 * pressed. Registered via fl_set_form_atclose().
 */
int
cancel_box_cb(FL_FORM *obj, void *data)
{
  spell_check_hide_form();
  return FL_IGNORE;
}

void spell_check_done_cb(FL_OBJECT *obj, long data)
{
  if (out)	{
   fputs("#\n", out);
   fclose(out);
   out = NULL;	}
  spell_check_hide_form();
}

void spell_check_options_cb(FL_OBJECT *obj, long data)
{
  xfm_config(6);
}

void spell_check_start_cb(FL_OBJECT *obj, long data)
{
  fl_activate_object(fd_spell_check->ispell_run_objs);
  fl_deactivate_object(fd_spell_check->options);
  fl_deactivate_object(fd_spell_check->start);
  if (do_spell_session(textobj)==SPELLER_DONE) {
    spell_check_hide_form();
    return;
  }
  fl_deactivate_object(fd_spell_check->ispell_run_objs);
  fl_activate_object(fd_spell_check->options);
  fl_activate_object(fd_spell_check->start);
}

void 
show_spell_checker(FL_OBJECT *tobj, FL_FORM *form)
{
  /* ispell is in action */
  if (fd_spell_check)
    return;

  textobj=tobj;
  msg_form=form;
  fl_deactivate_form(msg_form);
  fd_spell_check = create_form_spell_check();
  fl_set_form_atclose(fd_spell_check->spell_check, cancel_box_cb, NULL);

  /* can go into dialogs.fd */
  fl_set_object_callback(fd_spell_check->start, spell_check_start_cb,0);
  fl_set_object_callback(fd_spell_check->done,spell_check_done_cb,0);
  fl_set_object_callback(fd_spell_check->options,spell_check_options_cb,0);

  fl_set_object_label(fd_spell_check->text, "");
  fl_set_input(fd_spell_check->input, "");
  fl_clear_browser(fd_spell_check->browser);

  fl_show_form(fd_spell_check->spell_check, FL_PLACE_MOUSE,FL_FULLBORDER, "Spellchecker");

  fl_deactivate_object(fd_spell_check->ispell_run_objs);
  fl_activate_object(fd_spell_check->options);
  fl_activate_object(fd_spell_check->start);
}

int
is_cited_line(char *line)
{
char *mname;
char *prefix=b_getcfg_str(conf_name, "prefix", ">");

  if (!strncmp(line, prefix, strlen(prefix)))
    return 1;
  else {
    mname = line;
    if ((*mname == ' ') || (*mname == '\t'))
      mname++;
    if ((*mname == ':') || (*mname == '>') || (*mname == '|'))
      return 1;
#ifdef HAVE_REGCOMP
    else {
      regex_t preg;
      regcomp(&preg, "^[ \t]*[a-z0-9]*>", REG_EXTENDED|REG_ICASE);
      if (regexec(&preg, mname, 0, 0, 0) == 0)  {
        regfree(&preg);
        return 1;				}
      regfree(&preg);
        }
#endif
      } 
  return 0;
}   

