
/*
 * LIB/EXPIRE.C
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 */

#include "defs.h"

Prototype int GetExpire(const char *msgid, History *h, const char *nglist, int size);
Prototype void LoadExpireCtl(History *h, int force);

typedef struct ExpireCtl {
    struct ExpireCtl	*ex_Next;
    char		*ex_Wild;
    int			ex_Hours;
    int			ex_MaxSize;
    double		ex_CrossFactor;
    double		ex_SizeFactor;
} ExpireCtl;

ExpireCtl *ExBase;

int findExpireCtl(const char *group, int ngcount, int size);

/*
 * GetExpire() return the expiration, in hours from now, for an
 * article given the msgid, history record, and group list
 */

int
GetExpire(const char *msgid, History *h, const char *nglist, int size)
{
    char group[MAXGNAME];
    const char *p;
    int hours = 65535;
    int ngcount = 0;

    /*
     * load/reload dexpire.ctl as necessary
     */

    LoadExpireCtl(h, 0);

    /*
     * Scan the grouplist and expire based on the SHORTEST expire
     * time.
     */

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;
    for (p = nglist; p; p = strchr(p + 1, ','))
	++ngcount;
    for (p = nglist; p; p = strchr(p, ',')) {
	int l;
	int hr;

	if (*p == ',')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != '*' && p[l] != '?' && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(group, p, l);
	group[l] = 0;
	hr = findExpireCtl(group, ngcount, size);
	if (hours > hr)
	    hours = hr;
    }
    if (DebugOpt > 1)
	ddprintf("ngcount=%d size=%d expire=%d (%s)\n", ngcount, size, hours, nglist);
    return(hours);
}

int
findExpireCtl(const char *group, int ngcount, int size)
{
    ExpireCtl *ex;
    ExpireCtl save = { NULL, NULL, 65535, -1, -1.0, -1.0 };
    double hr;

    for (ex = ExBase; ex; ex = ex->ex_Next) {
	if (WildCmp(ex->ex_Wild, group) == 0) {
	    if (ex->ex_Hours < save.ex_Hours)
		save.ex_Hours = ex->ex_Hours;
	    if (ex->ex_MaxSize >= 0 && (save.ex_MaxSize < 0 || ex->ex_MaxSize < save.ex_MaxSize))
		save.ex_MaxSize = ex->ex_MaxSize;
	    if (ex->ex_CrossFactor >= 0.0 && (save.ex_CrossFactor < 0.0 || ex->ex_CrossFactor < save.ex_CrossFactor))
		save.ex_CrossFactor = ex->ex_CrossFactor;
	    if (ex->ex_SizeFactor >= 0.0 && (save.ex_SizeFactor < 0.0 || ex->ex_SizeFactor < save.ex_SizeFactor))
		save.ex_SizeFactor = ex->ex_SizeFactor;
	}
    }

    hr = (double)save.ex_Hours;
    if (save.ex_CrossFactor >= 0.0)
	hr = hr * pow(save.ex_CrossFactor, (double)ngcount);
    if (save.ex_SizeFactor >= 0.0)
	hr = hr * pow(save.ex_SizeFactor, (double)size / 100000.0);
    if (save.ex_MaxSize >= 0 && size >= save.ex_MaxSize)
	hr = 0.0;
    return((int)hr);
}

uint32 ExGmtMin = (uint32)-1;
time_t ExMTime = 0;

void
LoadExpireCtl(History *h, int force)
{
    /*
     * check for dexpire.ctl file modified once a minute
     */

    if (force || h == NULL || h->gmt != ExGmtMin) {
	struct stat st = { 0 };
	FILE *fi;

	if (h)
	    ExGmtMin = h->gmt;

	fi = xfopen("r", "%s/dexpire.ctl", NewsHome);

	if (fi == NULL)  
	    syslog(LOG_EMERG, "%s/dexpire.ctl file not found", NewsHome);

	if (force || fi == NULL || 
	    (fstat(fileno(fi), &st) == 0 && st.st_mtime != ExMTime)
	) {
	    char buf[MAXGNAME+256];
	    ExpireCtl **pex = &ExBase;

	    ExMTime = st.st_mtime;	/* may be 0 if file failed to open */

	    {
		ExpireCtl *ex;

		while ((ex = ExBase) != NULL) {
		    ExBase = ex->ex_Next;
		    free(ex);
		}
	    }
	    while (fi && fgets(buf, sizeof(buf), fi) != NULL) {
		char *p;

		if (buf[0] == '/')
		    continue;
		if (buf[0] == '#')
		    continue;
		if (buf[0] == '\n')
		    continue;
		if ((p = strchr(buf, ':')) != NULL) {
		    ExpireCtl *ex = malloc(sizeof(ExpireCtl) + (p - buf) + 1);

		    ex->ex_Wild = (char *)(ex + 1);
		    ex->ex_Hours = (int)(strtod(p + 1, NULL) * 24 + 0.99);
		    ex->ex_MaxSize = -1;
		    ex->ex_CrossFactor = -1.0;
		    ex->ex_SizeFactor = -1.0;

		    memmove(ex->ex_Wild, buf, p - buf);
		    ex->ex_Wild[p-buf] = 0;
		    *pex = ex;
		    pex = &ex->ex_Next;

		    while ((p = strchr(p + 1, ':')) != NULL) {
			switch(p[1]) {
			case 'c':	/* per cross post    */
			    ex->ex_CrossFactor = strtod(p + 2, NULL);
			    break;
			case 's':	/* per million bytes */
			    ex->ex_SizeFactor = strtod(p + 2, NULL);
			    break;
			case 'm':
			    ex->ex_MaxSize = strtol(p + 2, &p, 0);

			    switch(*p) {
			    case 'g':
				ex->ex_MaxSize *= 1024;
				/* fall through */
			    case 'm':
				ex->ex_MaxSize *= 1024;
				/* fall through */
			    case 'k':
				ex->ex_MaxSize *= 1024;
				break;
			    }
			}
		    }
		}
	    }
	    *pex = NULL;
	}
	if (fi)
	    fclose(fi);
    }
}

