/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black                                    */
/*             Date   :  April 1996                                      */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* A general intonation method for implementing various simple rule      */
/* intonation systems.  It allows a list of targets to be predicted      */
/* in a way fully specified by the user without changing the C/C++ code  */
/* This was specifically designed to replace the simple intonation mode  */
/* monotone mode, and implemented generic ToBI type labels.              */
/*                                                                       */
/* This was to help Gregor Moehler do German ToBI as well as get a       */
/* we can use for a rule-based English ToBI for comparison with trained  */
/* versions                                                              */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "intonation.h"

static void check_targs(EST_Utterance &u);
static EST_Stream_Item *find_nearest_seg(EST_Utterance &u,EST_Stream_Item &t);

LISP FT_Int_Targets_General_Utt(LISP utt)
{
    // Predict F0 targets 
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item *s;
    EST_Stream_Item *targ,*seg;
    LISP gen_params, targets, t;
    LISP tfunc;  // a lisp function that returns list of targets and values

    // Create some down step accents
    gen_params = siod_get_lval("int_general_params",
				  "no general intonation simple params");
    tfunc = get_param_lisp("targ_func",gen_params,NIL);
    if (tfunc == NIL)
    {
	cerr << "Int Target General: no target function specified" << endl;
	festival_error();
    }
    
    u->create_stream("Target");
    
    for (s=u->stream("Syllable").head(); s != 0 ; s=next(s))
    {
	targets = leval(cons(tfunc,cons(utt,cons(siod_make_streamitem(s),NIL))),
			NIL);
	// Add the given targets
	for (t=targets; t != NIL; t=cdr(t))
	{
	    targ = add_target(*u,get_c_float(car(car(t))),
			      get_c_float(car(cdr(car(t)))));
	    // Add link with appropriate segment
	    seg = find_nearest_seg(*u,*targ);

	    link(*targ,*seg);
	}
    }

    check_targs(*u);

    return utt;
}

static EST_Stream_Item *find_nearest_seg(EST_Utterance &u,EST_Stream_Item &t)
{
    // Find the segment that this target falls within.
    // This naively searchs from the start of the segments,
    // this is not very efficient
    EST_Stream_Item *seg;

    for (seg=u.stream("Segment").head(); seg != 0; seg=next(seg))
    {
	if (seg->end() >= t.end())
	    return seg;
    }

    cerr << "Int Target General: target past end of segments at " <<
	ftoString(t.end()) << endl;
    festival_error();
    return NULL;
}

static void check_targs(EST_Utterance &u)
{
    // Relate targst to the segments they are contained within
    EST_Stream_Item *t;
    float l = 0.0;

    for (t=u.stream("Target").head(); t != 0; t=next(t))
    {
	if (t->end() < l)
	{
	    cerr << "Int Target General: targets out of order" << endl;
	    festival_error();
	}
	l = t->end();
    }
}
       

