/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Alan W Black                           */
/*                      Date   :  April 1996                             */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/* From words to syllables and segments using the lexicon                */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "lexicon.h"
#include "modules.h"

static EST_Stream_Item *add_syllable(EST_Utterance &u, int stress);
static LISP specified_word_pronunciation(EST_Utterance &u, 
					 EST_Stream_Item &w, LISP lpos);

LISP FT_Word_Utt(LISP utt)
{
    // Look up words in lexicon and create syllable and segment streams
    EST_Utterance *u = GETUTTVAL(utt);
    EST_Stream_Item *w;
    LISP entry,s,p,lpos;
    EST_String pos;
    EST_Stream_Item *syl,*seg;

    
    *cdebug << "Word module\n";

    u->create_stream("Syllable");
    u->create_stream("Segment");
    add_segment(*u,ph_silence());

    for (w=(u->stream("Word")).head(); w != 0; w = next(w))
    {
	pos = w->feature("pos");
	if (pos != "0")
	    lpos = rintern(pos);
	else 
	    lpos = NIL;
	//  Check if there is an explicitly given prnunciation before
	//  going to the lexicon
	if ((entry = specified_word_pronunciation(*u,*w,lpos)) == NIL)
	    entry = lex_lookup_word(w->name(),lpos);
	if (lpos == NIL)
	    w->set_feature("pos",get_c_string(car(cdr(entry))));
	for (s=car(cdr(cdr(entry))); s != NIL; s=cdr(s))
	{
	    syl = add_syllable(*u,get_c_long(car(cdr(car(s)))));
	    link(*syl,*w);
	    for (p=car(car(s)); p != NIL; p=cdr(p))
	    {
		seg = add_segment(*u,get_c_string(car(p)));
		link(*syl,*seg);
	    }
	}
    }

    return utt;
}

static LISP specified_word_pronunciation(EST_Utterance &u, 
					 EST_Stream_Item &w, LISP lpos)
{
    //  If there is a phoneme feature on w or the Token related to 
    //  w use that as the pronunciation.  Note the value will be a string
    //  from which a list can be read.
    EST_String p;

    if (((p = w.feature("phonemes")) != "0") ||
	((p = ffeature(u,w,"Token.phonemes")) != "0"))
    {
	LISP phones = read_from_string(strintern(p));

	return cons(strintern(w.name()),
		    cons(lpos,
			 cons(lex_syllabify(phones),NIL)));
    }
    else
	return NIL;

}

EST_Stream_Item *add_word(EST_Utterance &u, const EST_String &name)
{
    EST_Stream_Item item;

    item.init("Word");
    item.set_name(name);
    
    u.stream("Word").append(item);
    return u.stream("Word").tail();
}

EST_Stream_Item *add_word(EST_Utterance &u, LISP word)
{
    // Build a Word Stream_Item from the Lisp description, which may
    // contain other features
    LISP f;
    EST_Stream_Item *si_word;
    int has_name = FALSE;

    if (consp(word))
    {
	// feature form
	si_word = add_word(u,"");
	for (f=word; f != NIL; f=cdr(f))
	{
	    if (streq("name",get_c_string(car(car(f)))))
	    {
		has_name = TRUE;
		si_word->set_name(get_c_string(car(cdr(car(f)))));
	    }
	    else
		si_word->set_feature(get_c_string(car(car(f))),
				     get_c_string(car(cdr(car(f)))));
	}
	if (!has_name)
	{
	    char *msg = siod_sprint(word);
	    cerr << "add_word: word has description but no name" << endl;
	    cerr << "  " << msg << endl;
	    wfree(msg);
	    festival_error();
	}
    }
    else // just the name
	si_word = add_word(u,get_c_string(word));

    return si_word;
}

static EST_Stream_Item *add_syllable(EST_Utterance &u, int stress)
{
    EST_Stream_Item item;
    
    item.init("Syllable");
    item.set_name("syl");
    item.set_feature("stress",stress);

    u.stream("Syllable").append(item);
    return u.stream("Syllable").tail();
}

EST_Stream_Item *add_segment(EST_Utterance &u, const EST_String &s)
{
    EST_Stream_Item item;
    
    item.init("Segment");
    item.set_name(s);

    u.stream("Segment").append(item);
    return u.stream("Segment").tail();
}

EST_Val ff_word_break(EST_Utterance &u,EST_Stream_Item &s)
{
    /* Break index of word */
    EST_Relation *phr,*words;
    
    phr = s.link("Phrase");
    if (phr->head() == 0)
	return EST_Val("1");
    else
    {
	words = u.ritem("Phrase",(*phr)(phr->head())).link("Word");
	if ((words->head() != 0) &&
	    (s.addr() == u.ritem("Word",(*words)(words->tail())).addr()))
	    return EST_Val(u.ritem("Phrase",(*phr)(phr->head())).name());
    }

    char *t1 = "1";
    return EST_Val(t1);
}


