/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                Author :  Alan W Black                                 */
/*                Date   :  July 1997                                    */
/*-----------------------------------------------------------------------*/
/*  Optional 16bit linear support for /dev/audio under FreeBSD           */
/*                                                                       */
/*=======================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/stat.h>
#include "EST_cutils.h"
#include "EST_Wave.h"
#include "EST_Option.h"
#include "audioP.h"
#include "EST_io_aux.h"

#ifdef SUPPORT_FREEBSD16
#include <machine/soundcard.h>
#include <fcntl.h>

int freebsd16_supported = TRUE;

static int sb_set_sample_rate(int sbdevice, int samp_rate)
{
    int fmt;

    // As its just possible this might run on non-intel FreeBSD or
    // (NetBSD ??) I'll check the byte order
    if (EST_LITTLE_ENDIAN)
	fmt = AFMT_S16_LE;  
    else
	fmt = AFMT_S16_BE;  
    
    ioctl(sbdevice,SNDCTL_DSP_RESET,0);
    ioctl(sbdevice,SNDCTL_DSP_SPEED,&samp_rate);
    ioctl(sbdevice,SNDCTL_DSP_SETFMT,&fmt);
    
    return TRUE;
}

#define AUDIOBUFFSIZE 256

int play_freebsd16_wave(EST_Wave &inwave, EST_Option &al)
{
    int sample_rate;
    short *waveform;
    int num_samples;
    int audio;
    int i,r,n;
    (void)al;
    
    if ((audio = open("/dev/dsp",O_WRONLY)) == -1)
    {
	cerr << "FreeBSD: can't open /dev/dsp" << endl;
	return -1;
    }
    
    waveform = inwave.data();
    num_samples = inwave.num_samples();
    sample_rate = inwave.sample_rate();
    
    if (sb_set_sample_rate(audio,sample_rate) == FALSE)
    {
	cerr << "FreeBSD: unable to set sample rate " <<
	    sample_rate << endl;
	close(audio);
	return -1;
    }
    
    for (i=0; i < num_samples; i += r/2)
    {
	if (num_samples > i+AUDIOBUFFSIZE)
	    n = AUDIOBUFFSIZE;
	else
	    n = num_samples-i;
	r = write(audio,&waveform[i], n*2);
	if (r == 0)
	{
	    cerr << "FreeBSD: failed to write to buffer" <<
		sample_rate << endl;
	    close(audio);
	}
    }
    
    close(audio);
    return 1;
}

#else
int freebsd16_supported = FALSE;

int play_freebsd16_wave(EST_Wave &inwave, EST_Option &al)
{
    (void)inwave;
    (void)al;
    cerr << "FreeBSD 16bit linear not supported" << endl;
    return -1;
}

#endif
