/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1995                                 */
/*-----------------------------------------------------------------------*/
/*                   Stream Class header file                            */
/*                                                                       */
/*=======================================================================*/

#ifndef __EST_STREAM_H__
#define __EST_STREAM_H__

#include "EST_common.h"
#include "EST_String.h"
#include "EST_TList.h"
#include "EST_KV.h"
#include "EST_types.h"

class EST_Stream;

class EST_Relation : public EST_TList<int> {
private:
    EST_String p_stream_name;
public:
    const EST_String &stream_name() const {return p_stream_name;}
    void init(const EST_String &s);

    void remove_rel(int a);
    EST_Relation &operator=(const EST_Relation &a);
    friend ostream& operator << (ostream& s, const EST_Relation &list);
};

/* A class so ints floats and Strings may be happily used as one type */
enum EST_val_type {val_unset, val_int, val_float, val_string };
class EST_Val {
  private:
    EST_val_type t;
    union 
    { int ival;
      float fval; } v;
    EST_String sval;
    const int to_int() const;
    const float to_flt() const;
    const EST_String &to_str() const;
  public:
    EST_Val() {t=val_unset;}
    EST_Val(const int i) {t=val_int; v.ival=i;}
    EST_Val(const float f) {t=val_float; v.fval=f;}
    EST_Val(const double d) {t=val_float; v.fval=d;}
    EST_Val(const EST_String &s) {t=val_string; sval = s;}
    EST_Val(const char *s) {t=val_string; sval = s;}
    const EST_val_type type(void) const {return t;}
    const int Int(void) const {if (t==val_int) return v.ival; return to_int();}
    const float Float(void) const {if (t==val_float) return v.fval; return to_flt();}
    const EST_String &string(void) const
       {if (t!=val_string) to_str(); return sval;}
    const EST_String &string_only(void) const {return sval;}
    EST_Val &operator=(const int i) { t=val_int; v.ival=i; return *this;}
    EST_Val &operator=(const float f) { t=val_float; v.fval=f; return *this;}
    EST_Val &operator=(const double d) { t=val_float; v.fval=d; return *this;}
    EST_Val &operator=(const EST_String &s) { t=val_string; sval = s; return *this;}
    int operator ==(const EST_Val &a) const
    { if (t != a.t) return (1==0);
      else if (t == val_string) return (sval == a.sval);
      else if (t == val_int) return (v.ival == a.v.ival);
      else if (t == val_float) return (v.fval == a.v.fval);
      else return (1==0); }
    int operator ==(const EST_String &a) const { return (string() == a); }
    int operator ==(const char *a) const { return (string() == a); }
    int operator ==(const int &i) const { return (Int() == i); }
    int operator ==(const float &f) const { return (Float() == f); }
    int operator ==(const double &d) const { return (Float() == d); }
    int operator !=(const EST_Val &a) const { return (!(*this == a)); }
    int operator !=(const EST_String &a) const { return (string() != a); }
    int operator !=(const char *a) const { return (string() != a); }
    int operator !=(const int &i) const { return (Int() != i);}
    int operator !=(const float &f) const { return (Float() != f); }
    int operator !=(const double &d) const { return (Float() != d); }
    operator int() const { return Int(); }
    operator float() const { return Float(); }
    operator EST_String() const { return string(); }
    friend ostream& operator << (ostream &s, const EST_Val &a)
    { if (a.t == val_unset) s << "[PVAL unset]" ;
      else if (a.t == val_int)	  s << a.v.ival;
      else if (a.t == val_float) s << a.v.fval;
      else if (a.t == val_string) s << a.sval;
      return s;
    }
    EST_Val &operator=(const EST_Val &c)
    {t=c.t; if (t == val_string) sval = c.sval;
            else v=c.v;
     return *this;}
};


/* A class for containing some other (arbitrary) class       */
/* Not general enough to call itself a run-time type system  */
/* Is designed to solve the problem of holding user          */
/* specified information in a EST_Stream_Item.                   */
/* Keeps reference count to know when to delete contents     */
/*                                                           */
/* This is done on two levels EST_Contents and Contents_Data     */
class EST_Content_Data{
  private:
    int refs;
    void *data;
    void (*free_func)(void *data);
  public:
    EST_Content_Data(void *d,void (*f)(void *d)) {free_func=f; data=d; refs=1;}
    ~EST_Content_Data() { free_func(data); }
    int unref() { return --refs; }
    int ref() { return ++refs; }
    int the_refs() { return refs; }
    void *contents() { return data; }
    EST_Content_Data &operator=(const EST_Content_Data &c)
    {refs = c.refs; data = c.data; free_func = free_func; return *this;}
};

class EST_Contents{
private:
    EST_Content_Data *content_data;
    void unref_contents(void)
       { if ((content_data != 0) &&
	     (content_data->unref() == 0))
	     delete content_data;}
public:
    EST_Contents() { content_data = 0; }
    EST_Contents(void *p,void (*free_func)(void *p))
         { content_data = new EST_Content_Data(p,free_func); }
    ~EST_Contents() { unref_contents(); }
    void set_contents(void *p,void (*free_func)(void *p))
         { content_data = new EST_Content_Data(p,free_func); }
    void *get_contents() const 
         {return (content_data ? content_data->contents() : 0);}
    int refs() const { return ((content_data == 0) ? 0 :
			 content_data->the_refs());}
    EST_Contents &operator=(const EST_Contents &c)
         { unref_contents();
	   content_data = c.content_data; 
	   if (content_data != 0) content_data->ref();
           return *this; }
};

class EST_Stream_Item {
 private:
    float p_end;
    EST_String p_stream_name;
    EST_String p_name;
    int a;

    EST_Contents content;  /* some arbtirary class */

 public:
    EST_Stream_Item() {init("");}
    EST_Stream_Item(const EST_Stream_Item &item);

    void init(const EST_String &stream_name); // init function - *must* be called.
    void init(const EST_String &stream_name,void (*gc_func)(void *));

    // REORG if this is going to be private - how do we access it?
    KVL <EST_String,EST_Val> features;

    const EST_String &name() const {return p_name;}
    float end() const {return p_end;}
    int addr() const {return a;}	// "address" of this cell

    EST_TList<EST_Relation> links;		// list of list of relations
    EST_StrList fields;

    const EST_String &stream_name() const {return p_stream_name;}// name of stream
    void *contents() const {return content.get_contents();}
    EST_Val feature(const EST_String &name){return features.val_def(name,EST_Val("0"));}

    float start() const;   // derived
    float mid() const;     // derived
    float dur() const;     // derived

    void set_name(const EST_String &s) {p_name = s;}    
    void set_end(float e) {p_end = e;}
    void set_addr(int x) {a = x;}	// "address" of this cell

    void set_stream_name(const EST_String &s) {p_stream_name = s;}
//    void set_fields(const EST_String &s);

    void set_feature(const EST_String &name, int ival)
       { EST_Val pv(ival); features.add_item(name,pv); }
    void set_feature(const EST_String &name, float fval)
       { EST_Val pv(fval); features.add_item(name,pv); }
    void set_feature(const EST_String &name, EST_String sval)
       { EST_Val pv(sval); features.add_item(name,pv); }
    void set_feature(const EST_String &name, const char *cval)
       { EST_Val pv(cval); features.add_item(name,pv); }
    void set_feature_val(const EST_String &name, EST_Val sval)
       { features.add_item(name,sval); }
    void set_contents(void *c,void (*f)(void *))
        { content.set_contents(c,f); }

    void make_link(const EST_String &stream, EST_Stream_Item &item);
    void make_link(EST_Stream_Item &item);
    void make_link(const EST_String &stream, int a);
    void remove_link(const EST_String &stream, int a);

    void remove();
    
    //return list of relations
    EST_Relation *link(const EST_String &stream);

    //return ref to rel list
    EST_Relation & rlink(const EST_String &stream);     

    // Simplified link following procedures.
    void get_linked(const EST_Stream *stream, EST_Stream_Item *&item);
    void get_linked(const EST_Stream *stream, EST_Stream_Item *&item1, EST_Stream_Item *&item2);
    void get_linked(const EST_Stream *stream, EST_Stream_Item *&item1, EST_Stream_Item *&item2, EST_Stream_Item *&item3);

    EST_Stream_Item *n;		// next pointer
    EST_Stream_Item *p;		// previous pointer

    EST_Stream_Item& operator = (const EST_Stream_Item& a);
    friend ostream& operator << (ostream &s, const EST_Stream_Item &a);
};

class EST_Stream {
    EST_Stream_Item *h;		//head of list
    EST_Stream_Item *t;		//tail of list
    int next_addr;		// next unused "address" tag.
    EST_String p_stream_name;   // typically "phoneme" etc
    EST_String p_name;          // unique name, often filename

    void default_vals();
    void copy(const EST_Stream &s);    
    void (*gc_contents_func)(void *);
public:
    // construction and initialisation 
    EST_Stream(const EST_String &type);
    EST_Stream(const EST_Stream &s);
    EST_Stream();
    ~EST_Stream() { clear(); }

    void init(const EST_String &type = "blank");
    void init(const EST_String &st,void (*destroy_contents)(void *));

    // file i/o
    EST_read_status load(const EST_String &name, const EST_String &comment = ";", int literal = 1);
    EST_read_status load_type(const EST_String &filename, const EST_String &type = "", int literal = 1);
    EST_write_status save(const EST_String &filename,const EST_String &type = "", int num = 0);

    // general information
    int empty() const; // is stream empty?
    int length() const; // length of stream;

    // stream name, often linguistic class, e.g. "phoneme"
    const EST_String &stream_name() const {return p_stream_name;}
    void set_stream_name(const EST_String &a);

    // general name, often file name
    const EST_String &name() const {return p_name;}
    void set_name(const EST_String &a);

    // adding to stream
    void insert_after(EST_Stream_Item *ptr, EST_Stream_Item &item);
    void insert_before(EST_Stream_Item *ptr, EST_Stream_Item &item);

    EST_Stream_Item *append(const EST_Stream_Item &item);
  EST_Stream_Item *append(void *contents, EST_String name, void (*free_func)(void *p));

    EST_Stream_Item *prepend(const EST_Stream_Item &item);
    // removing from stream
    void clear(); // remove all items from stream
    // remove item from list, return pointer to previous
    EST_Stream_Item *remove(EST_Stream_Item *ptr); 
    void remove(int a); // remove item from list
    
    // accessing stream
    // REORG - add first() and last()
    EST_Stream_Item *head() const {return h;}	// pointer to head of list
    EST_Stream_Item *tail() const {return t;}	// pointer to tail of list
    EST_Stream_Item *item(int a) const;	// return single item from address.
    EST_Stream_Item *nth(int a) const;	// return nth item from stream.
    
    friend ostream& operator << (ostream &s, const EST_Stream &u);
    EST_Stream &operator=(const EST_Stream &s);
};

typedef EST_TList<EST_Stream> EST_StreamList;

inline EST_Stream_Item *next(EST_Stream_Item *ptr) {return (ptr == 0 ? 0 : ptr->n);}
inline EST_Stream_Item *prev(EST_Stream_Item *ptr) {return (ptr == 0 ? 0 : ptr->p);}

void link(EST_Stream_Item &a, EST_Stream_Item &b); // mutual linking utility function

EST_read_status load_mlf(EST_String filename, EST_StreamList &l);

EST_String options_stream_filetypes(void);

#endif //  __EST_STREAM_H__
