/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  June 1995                                 */
/*-----------------------------------------------------------------------*/
/*                 Stream linked list class                              */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include "EST_Stream.h"
#include "stream_io.h" // for HTK_UNITS_PER_SECOND macro

// Users may change the address of a stream_item but they are
// assigned an addr unique for this session
static int general_stream_item_addr = 1;

EST_Stream::EST_Stream(const EST_String &ps)
{
    init(ps);
}

EST_Stream::EST_Stream()
{
    default_vals();
}

EST_Stream::EST_Stream(const EST_Stream &s)
{
    default_vals();
    copy(s);
}

void EST_Stream::default_vals()
{
    p_stream_name = "";
    p_name = "";
    h = 0;
    t = 0;
    next_addr = 0;
}

void EST_Stream::copy(const EST_Stream &s)
{
    clear();
    EST_Stream_Item *ptr, item;
    p_stream_name = s.stream_name();
    p_name = s.name();

    for (ptr = s.head(); ptr != 0; ptr = next(ptr))
      append(*ptr);
}

EST_Stream &EST_Stream::operator=(const EST_Stream &s)
{
    copy(s);
    return *this;
}

void EST_Stream::init(const EST_String &st)
{
    default_vals();
    p_stream_name = st;
}

EST_Stream_Item::EST_Stream_Item(const EST_Stream_Item &x)
{
    set_name(x.name());
    n = x.n;
    p = x.p;
    a = x.addr();
    p_stream_name = x.stream_name();

    content = x.content;
    features = x.features;
    p_end = x.end();

    EST_TBI *pp;
    links.clear();
    for (pp = x.links.head(); pp != 0; pp = next(pp))
	links.append(x.links(pp));

    fields = x.fields;
//    fields.clear();
//    for (pp = x.fields.head(); pp != 0; pp = next(pp))
//	fields.append(x.fields(pp));
}


void EST_Stream_Item::make_link(const EST_String &stream, EST_Stream_Item &item)
{
    EST_TBI *pt;

    for (pt = links.head(); pt != 0; pt = next(pt))
	if (links(pt).stream_name() == stream)
	    break;

    if (pt == 0)
    {
	EST_Relation l;
	l.init(stream);
	links.append(l);
	pt = links.tail();
    }
    
    links(pt).append(item.addr());
}

void EST_Stream_Item::make_link(EST_Stream_Item &item)
{
    EST_TBI *pt;

    for (pt = links.head(); pt != 0; pt = next(pt))
	if (links(pt).stream_name() == item.stream_name())
	    break;

    if (pt == 0)
    {
	EST_Relation l;
	l.init(item.stream_name());
	links.append(l);
	pt = links.tail();
    }
    
    links(pt).append(item.addr());
}

void EST_Stream_Item::make_link(const EST_String &stream, int an)
{
    EST_TBI *pt;

    for (pt = links.head(); pt != 0; pt = next(pt))
	if (links(pt).stream_name() == stream)
	    break;

    if (pt == 0)
    {
	EST_Relation l;
	l.init(stream);
	links.append(l);
	pt = links.tail();
    }
    
    links(pt).append(an);
}

void EST_Stream_Item::remove_link(const EST_String &stream, int an)
{
    EST_TBI *pt;
    for (pt = links.head(); pt != 0; pt = next(pt))
	if (links(pt).stream_name() == stream)
	{
	    links(pt).remove_rel(an);
	    return;
	}
}

// perform mutual linking
void link(EST_Stream_Item &a, EST_Stream_Item &b)
{
    a.make_link(b);
    b.make_link(a);
}

EST_Relation *EST_Stream_Item::link(const EST_String &an)
{
    EST_TBI *pt;

//    cout << "THIS " << *this;
//    cout << "Link length " << links.length() << endl;
//    cout << "Link length " << this->links.length() << endl;
//    for (p = this->links.head(); p != 0; p = next(p))
//    {
//	cout << "Z :" << p->val << ":" << endl;
//	cout << "In link " << p->val << "x\n";
//    }

    for (pt = this->links.head(); pt != 0; pt = next(pt))
    {
//	cout << "in link " << p->val << "x\n";
	if (this->links(pt).stream_name() == an)
	    return &(links(pt));
    }

    // * HERE * - test this - it doesn't return the right thing.  //
    //    If you request a relation which doesn't exist, this makes an empty
    //    list with that relations name and adds it to the end of the
    //    relations list. Another possibility would be to return an error.

    if (pt == 0)
    {
//	cout << "creating relation of type " << a << endl;
	EST_Relation l;
	l.init(an);
	links.append(l);
	pt = links.tail();
    }
    return &(links(pt));
}

EST_Relation &EST_Stream_Item::rlink(const EST_String &an)
{
    EST_TBI *pt;

    for (pt = this->links.head(); pt != 0; pt = next(pt))
    {
	if (this->links(pt).stream_name() == an)
	    return links(pt);
    }

    if (pt == 0)
    {
	EST_Relation l;
	l.init(an);
	links.append(l);
	pt = links.tail();
    }
    return links(pt);
}


void EST_Stream_Item::get_linked(const EST_Stream *stream, 
				 EST_Stream_Item *&item1)
{
  EST_Stream_Item *item2, *item3;
  get_linked(stream, item1, item2, item3);
}


void EST_Stream_Item::get_linked(const EST_Stream *stream, 
				 EST_Stream_Item *&item1, 
				 EST_Stream_Item *&item2)
{
  EST_Stream_Item *item3;
  get_linked(stream, item1, item2, item3);
}

void EST_Stream_Item::get_linked(const EST_Stream *stream, 
				 EST_Stream_Item *&item1, 
				 EST_Stream_Item *&item2, 
				 EST_Stream_Item *&item3)
{
  EST_String stream_name(stream->stream_name());
  EST_TBI *pt;

  item1=NULL;
  item2=NULL;
  item3=NULL;

  for (pt = this->links.head(); !item1 && pt != 0; pt = next(pt))
    if (this->links(pt).stream_name() == stream_name)
      {
	EST_Relation *rel=&(links(pt));
	
	if (rel->length()>0)
	  item1 = stream->item(rel->item(0));
	if (rel->length()>1)
	  item2 = stream->item(rel->item(1));
	if (rel->length()>2)
	  item3 = stream->item(rel->item(2));
      }
}

void EST_Stream::clear()
{
    EST_Stream_Item *ptr, *nptr;

    for (ptr = head(); ptr != 0; ptr = nptr)
    {
	nptr = next(ptr);
	delete ptr;
    }

    init("");
}

EST_Stream_Item *EST_Stream::item(int a)  const
{
    EST_Stream_Item *ptr;

    for (ptr = head(); ptr != 0; ptr = next(ptr))
	if (ptr->addr() == a)
	    return ptr;

    cerr << a << ": No such address in " << stream_name() << " stream\n";
    return 0;
}

EST_Stream_Item *EST_Stream::nth(int n)  const
{
    EST_Stream_Item *ptr;
    int i;

    for (i = 0, ptr = head(); ptr != 0; ptr = next(ptr), ++i)
    {
	if (i == n)
	    return ptr;
    }

    cerr << "Request item " << n << " out of range in EST_Stream\n";
    return 0;
}

float EST_Stream_Item::start() const
{
    return (p == 0) ? 0.0 : p->end();
}

float EST_Stream_Item::mid() const
{
    return end() - (dur() / 2.0);
}

float EST_Stream_Item::dur() const
{
    return end() - start();
}

int EST_Stream::length() const
{
    EST_Stream_Item *ptr;
    int n = 0;

    for (ptr = head(); ptr != 0; ptr = next(ptr))
	++n;
    return n;
}

// set name of stream and sets all stream_items to be this also
void EST_Stream::set_stream_name(const EST_String &s)
{
    p_stream_name = s;
    EST_Stream_Item *ptr;
    for (ptr = head(); ptr != 0; ptr = next(ptr))
	ptr->set_stream_name(s);
}

void EST_Stream::set_name(const EST_String &s)
{
    p_name = s;
}

int EST_Stream::empty() const
{
    return (length() == 0) ? 1 : 0;
}

EST_Stream_Item *EST_Stream::append(const EST_Stream_Item &item)
{
    EST_Stream_Item *tmp = new EST_Stream_Item;
    tmp->init(item.stream_name());

    *tmp = item;

    if (h == 0)
    {
	h = tmp;
	tmp->p = 0;
    }
    else
    {
	t->n = tmp;
	tmp->p = t;
    }
    tmp->n = 0;

    tmp->set_addr((tmp->addr() == 0) ? next_addr++ : tmp->addr());
    t = tmp;

    return tmp;
}

EST_Stream_Item *EST_Stream::append(void *contents, EST_String name, void (*free_func)(void *p))
{
  EST_Stream_Item item;

  item.init(stream_name());
  item.set_name(name);
  item.set_contents(contents, free_func); 

  return append(item);
}


EST_Stream_Item *EST_Stream::prepend(const EST_Stream_Item &item)
{
    EST_Stream_Item *tmp = new EST_Stream_Item;
    tmp->init(item.stream_name());

    *tmp = item;

    if (t == 0)
    {
	t = tmp;
	tmp->n = 0;
    }
    else
    {
	h->p = tmp;
	tmp->n = h;
    }
    tmp->p = 0;

    tmp->set_addr((tmp->addr() == 0) ? next_addr++ : tmp->addr());
    h = tmp;

    return tmp;
}

void EST_Stream::insert_after(EST_Stream_Item *ptr, EST_Stream_Item &newitem)
{
    EST_Stream_Item *tmp;
    tmp = new EST_Stream_Item;
    (*tmp) = newitem;

    if (ptr->n == 0)
    {
	t = tmp;
	tmp->n = 0;
    }
    else
    {
	ptr->n->p = tmp;
	tmp->n = ptr->n;
    }

    tmp->p = ptr;
    ptr->n = tmp;
}

void EST_Stream::insert_before(EST_Stream_Item *ptr, EST_Stream_Item &newitem)
{
    EST_Stream_Item *tmp;
    tmp = new EST_Stream_Item;
    (*tmp) = newitem;

    if (ptr->p == 0)
    {
	h = tmp;
	tmp->p = 0;
    }
    else
    {
	ptr->p->n = tmp;
	tmp->p = ptr->p;
    }

    tmp->n = ptr;
    ptr->p = tmp;
}

EST_Stream_Item *EST_Stream::remove(EST_Stream_Item *item)
{ // removes item from list, returning a pointer to the previous item
    EST_Stream_Item *prev_item, *next_item;
    
    prev_item = item->p;
    next_item = item->n;
    
    if (next_item == 0)
	t = prev_item;
    else
	next_item->p = prev_item;
    
    if (prev_item == 0)
	h = next_item;
    else
	prev_item->n = next_item;
    
    item->remove();
    delete item;
    return prev_item;
}

void EST_Stream::remove(int a)
{
    remove(item(a));
}

void EST_Stream_Item::init(const EST_String &s)
{
    n = 0;
    p = 0;
    a = general_stream_item_addr++;
//    v = 0;
    links.clear();
    fields.clear();
    p_stream_name = s;
    p_end = 0.0;
    
}	

void EST_Stream_Item::remove(void)
{
    links.clear();
    fields.clear();
}

void EST_Relation::init(const EST_String &intype)
{
//    h = 0;
//    t = 0;
    p_stream_name = intype;
}

// removes relation and returns pointer to previous item
void EST_Relation::remove_rel(int a)
{
    EST_TBI *p;
    for (p = head(); p!= 0; p = next(p))
	if (item(p) == a)
	    remove(p);

}

EST_Relation &EST_Relation::operator=(const EST_Relation &a)
{
//    *this = DList<int>copy::(a);
//    *this = DList<int>::operator=(a);
    EST_TBI *p;

    this->clear();
    
    for (p = a.head(); p != 0; p = next(p))
	append(a.item_C(p));

    p_stream_name = a.stream_name();
    return *this;
}

ostream& operator << (ostream& os, const EST_Relation &pi)
{
    EST_TBI *pt;

    if (pi.head() == 0)
	return os;

    //    s << " (" << pi.stream_name();
    // removed brackets  so that save function could use this.
    os << " " << pi.stream_name();
    for (pt = pi.head(); pt != 0; pt = next(pt))
	os << " " << pi(pt);
    //    s << ") ";
    os << " ";
    return os;
}

EST_Stream_Item &EST_Stream_Item::operator=(const EST_Stream_Item &x)
{
    set_name(x.name());

    n = x.n;
    p = x.p;
    a = x.addr();
    p_stream_name = x.stream_name();
    content = x.content;
    p_end = x.end();
    features = x.features;
    EST_TBI *pt;

    links.clear();
    for (pt = x.links.head(); pt != 0; pt = next(pt))
	links.append(x.links(pt));

    fields = x.fields;

    return *this;
}

EST_read_status EST_Stream::load(const EST_String &filename, const EST_String &comment, int literal)
{
    EST_Stream tmp;
    EST_read_status r_val;
    EST_String myname;
    (void)literal;

    tmp.set_stream_name(p_stream_name);
    if ((r_val = load_esps_label(filename, tmp, comment))
	!= wrong_format)
    {
	myname = p_stream_name;    // so this doesn't get trashed
	*this = tmp;
	p_stream_name = myname;
    }

    return r_val;
}

// Many label formats do not have distinctive headers, therefore we
// have to explicitly say what the file type is.

EST_read_status EST_Stream::load_type(const EST_String &filename,const EST_String &type, int sample)
{
    EST_Stream tmp;
    EST_read_status r_val;

    tmp.set_stream_name(p_stream_name);
    if ((type == "htk") || (sample == 0))
	sample = 10000000;

    if ((type == "ascii") || (type == "timit") || (type == "htk"))
    {
	if ((r_val = load_sample_label(filename, tmp, sample)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "words"))
    {
	if ((r_val = load_words_label(filename, tmp)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "ogi"))
    {
	if ((r_val = load_ogi_label(filename, tmp)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "esps"))
    {
    if ((r_val = load_esps_label(filename, tmp)) 
	!= wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else 
    {
	cerr << "Unknown label type: " << type << endl;
	return read_format_error;
    }

    return read_ok;
}

EST_String options_stream_filetypes(void)
{
    // valid stream file types 
    return "esps htk spn names";
}

EST_write_status EST_Stream::save(const EST_String &filename,const EST_String &type, int n)
{
    if  (type == "")
	return save_label_esps(filename, *this, n);
    if  (type == "esps")
	return save_label_esps(filename, *this, n);
    if  (type == "htk")
	return save_label_htk(filename, *this);
    if  (type == "spn")
	return save_label_spn(filename, *this);
    if  (type == "names")
	return save_label_names(filename, *this, n);

    cerr << "Unknown label output file type " << type << endl;
    return write_error;
}

ostream& operator << (ostream &st, const EST_Stream &a)
{
    EST_Stream_Item *ptr;
    
//    st << a.stream_name() << endl;
    
    for (ptr = a.head(); ptr != 0; ptr = next(ptr))
	st << *ptr << endl;
    
    return st;
}

ostream& operator << (ostream &s, const EST_Stream_Item &a)
{
    EST_TBI *p;
    
    s.precision(3);
    s.width(4);
    s.setf(ios::left,ios::adjustfield);
    s << a.name();

    s.width(4);
    s.setf(ios::left,ios::adjustfield);
    s << a.addr();

    s.width(5);
    s.setf(ios::left,ios::adjustfield);
    s << a.end() << " :";
    
    s.setf(ios::left,ios::adjustfield);
    for (p = a.fields.head(); p != 0; p = next(p))
	s << a.fields(p) << ";";
    
    for (p = a.links.head(); p != 0; p = next(p))
	s << a.links(p);
    
    return s;
}

