/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  June 1994                              */
/*-----------------------------------------------------------------------*/
/*                    LPC/Cepstra high-level routines                    */
/*                                                                       */
/*=======================================================================*/
#include <iostream.h>
#include <fstream.h>
#include <math.h>
#include "EST_string_aux.h"
#include "EST_sigpr.h"
#include "EST_track_aux.h"
#include "EST_Option.h"
#include "EST_Wave.h"

# define MAX_LPORDER	    128
# define MAX_ORDER	    128

EST_Track make_cepstra_unnorm(EST_Wave &sig, EST_Option &op);

void ref2truearea(float *ref, float *area, int order);
void ref2logarea(float *ref, float *logarea, int order);
void ref2area(float *ref, float *area, int order);
float lpredict(float *adc, int wsize, float *acf, float *ref, float *lpc, int
	       order);
void lpc2ceps(float *lpc, int nlpc, float *ceps, int nceps);
float *Float_hamming(int size);

EST_Track make_cepstra(EST_Wave &sig, EST_Option &op)
{
    int i;
    float mean, sd;
    EST_Track cepstra = make_cepstra_unnorm(sig, op);
	
    if (op.val("cepstra_unnorm", 0) != "true"){
		
	if (op.val("cepstra_self_norm", 0) == "true")
	    normalise(cepstra);
	else{
	    mean = op.fval("cepstra_mean", 1);
	    sd = op.fval("cepstra_sd", 1);
	    for (i = 0; i < cepstra.num_channels(); ++i)
		normalise(cepstra, mean, sd, i);
	}
    }
    
    return cepstra;
}

float residual(const float *orig, float *e, int wsize, const float *a, int order);
float resynth(float *orig, float *e, int wsize, float *a, int order);

EST_Wave make_lpc_residual(EST_Wave &sig, EST_Track &lpc, EST_Option &op)
{
    EST_Wave e;
    float *a = new float[lpc.num_channels()];
    int wsize = 160;
    float *sinbuf = new float[wsize + lpc.num_channels()];
    float *r = new float[wsize + lpc.num_channels()];
    float *inbuf;
    (void)op;
    int i, j, k=0;

    inbuf = sinbuf + 12;

//    e = sig;

    e.resize(sig.num_samples() + 10000);
    e.set_sample_rate(sig.sample_rate());

    cout << "in residual2\n";
    // pre-pad with zeros
    cout << "lpc " << lpc.num_channels() << endl;
    for (j = 0; j < lpc.num_channels(); ++j)
	sinbuf[j] = 0.0;

    cout << "in residual2\n";
    for (i = 0; i < lpc.num_frames() -1 ; ++i)
    {
	cout << "frame " << i << endl;
	k = i * wsize;
	for (j = 0; j < wsize; ++j)
	    inbuf[j] = sig.a(k + j);
	for (j = 0; j < lpc.num_channels(); ++j)
	    a[j] = lpc(i, j);
	residual(inbuf, r, 160, a, lpc.num_channels());
	for (j = 0; j < wsize; ++j)
	    e.a(j + k) = (int)r[j];
    }
    cout << "e size " << e.num_samples() << " max: " << k + j << endl;

    delete [] a;
    delete [] sinbuf;
    delete [] r;
    return e;
}

EST_Wave lpc_resynth(EST_Wave &sig, EST_Track &lpc, EST_Option &op)
{
    EST_Wave e;
    float *a = new float[lpc.num_channels()];
    int wsize = 160;
    float *sinbuf = new float[wsize + lpc.num_channels()];
    float *r = new float[wsize + lpc.num_channels()];
    float *inbuf;
    (void)op;
    int i, j, k=0;

    inbuf = sinbuf + 12;

//    e = sig;

    e.resize(sig.num_samples() + 10000);
    e.set_sample_rate(sig.sample_rate());

    cout << "in residual2\n";
    // pre-pad with zeros
    cout << "lpc " << lpc.num_channels() << endl;
    for (j = 0; j < lpc.num_channels(); ++j)
	sinbuf[j] = 0.0;

    cout << "in residual2\n";
    for (i = 0; i < lpc.num_frames() -1 ; ++i)
    {
	cout << "frame " << i << endl;
	k = i * wsize;
	for (j = 0; j < wsize; ++j)
	    inbuf[j] = sig.a(k + j);
	for (j = 0; j < lpc.num_channels(); ++j)
	    a[j] = lpc(i, j);
	residual(inbuf, r, 160, a, lpc.num_channels());
	for (j = 0; j < wsize; ++j)
	    e.a(j + k) = (int)r[j];
    }
    cout << "e size " << e.num_samples() << " max: " << k + j << endl;

    delete [] a;
    delete [] sinbuf;
    delete [] r;

    return e;
}

EST_Track make_cepstra_unnorm(EST_Wave &sig, EST_Option &op)
{
    EST_Track cepstra;
    EST_String this_name;

    lpc_cepstra(cepstra, sig, op.fval("bc_frame_length", 1), 
		op.fval("bc_frame_shift", 1), 
		op.ival("bc_frame_order", 1));
    
    cepstra.fill_time(op.fval("bc_frame_length", 1));
    cepstra.set_contour_type(EST_ContourType::CEPSTRUM);
	
    // name fields c1, c2, .. etc.
    for(int i=0; i<op.ival("bc_frame_order", 1); i++){
	this_name = EST_String("c") + itoString(i+1);
	cepstra.set_field_name(this_name, i);
    }
    return cepstra;
}

int init_cepstra(EST_Option &op, EST_String bc_file)
{
    EST_Option n_op;
    EST_String bc_name;
    
    if (bc_file == ""){
	
	bc_name = op.val("broad_class_type", 1) + "_bc_opfile";
	
	if (!op.present(bc_name)){
	    cerr << "No setup found for "
		<< op.val("broad_class_type",0)
		    << " broad class analysis" << endl;
	    return -1;
	}
	
	bc_file = op.val(bc_name, 1);
    }
    
    if (n_op.load(bc_file) != read_ok){
	cerr << "Can't read op file " << bc_file << " for "
	    << op.val("broad_class_type",0)
		<< " broad class analysis" << endl;
	return -1;
    }
    
    op.override_val("cepstra_mean", n_op.val("cepstra_mean", 0));
    op.override_val("cepstra_sd", n_op.val("cepstra_sd", 0));
    op.override_val("output_delay", n_op.val("output_delay"));
    
    return 0;
}


