/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  August 1995                            */
/*-----------------------------------------------------------------------*/
/*                   EST_Track Auxiliary routines                        */
/*                                                                       */
/*=======================================================================*/

#include <math.h>
#include <stdlib.h>
#include "EST_cutils.h"
#include "EST_String.h"
#include "EST_Stream.h"
#include "EST_simplestats.h"
#include "EST_sort.h"
#include "EST_Track.h"
#include "track_io.h"
#include "EST_Option.h"
#include "EST_track_aux.h"

static int sorttest(const void *a, const void *b)
{ // for use with qsort C library function.
    float *c = (float *)a;
    float *d = (float *)b;
    float res = (*c - *d);
    if (res == 0.0)
	return 0;
    return (res < 0.0) ? -1 : 1;
}

void track_smooth(EST_Track &c, float x, EST_String stype)
{
    if (stype == "median")
	time_med_smooth(c, x);
    else 
	time_mean_smooth(c, x);
}

void time_med_smooth(EST_Track &c, float x)
{
    if (c.space_type() != "FIXED")
    {
	cerr << "Error: Time smoothing can only operate on fixed contours\n";
	return;
    }
    // want to check for divide by zero
    if (c.shift() == 0.0)
    {
	cerr << "Error in smoothing: time spacing problem\n";
	return;
    }
    int n = (int)(x / c.shift());
    for (int i = 0; i < c.num_channels(); ++i)
	simple_med_smooth(c, n, i);
}

void time_mean_smooth(EST_Track &c, float x)
{
    int j;
    EST_Track t;
    int n = (int)(x / c.shift());

    for (j = 0; j < c.num_channels(); ++j)
	simple_mean_smooth(c, n, j);
}

void simple_med_smooth(EST_Track &c, int n, int channel) 
{// simple median smoother of order n
    int i, j, h, k;
    float *a = new float[c.num_frames()];
    float *m = new float[n];
    h = n/2;
    
    // sort start using < n order smoothing
    for (i = 0; i < h; ++i)
    {
	k = (i * 2) + 1;
	for (j = 0; j < k; ++j)
	    m[j] = c.a(j, channel);
	qsort(m, k, sizeof(float), sorttest);
	a[i] = m[i];
    }
    
    // sort main section using n order smoothing
    for (i = h; i < c.num_frames() - h; ++i)
    {
	for (j = 0; j < n; ++j)
	    m[j] = c.a(i - h + j, channel);
	
	qsort(m, n, sizeof(float), sorttest);
	a[i] = m[h];
    }
    // sort end section using < n order smoothing
    for (; i < c.num_frames(); ++i)
    {
	k = ((c.num_frames() - i)* 2) -1;
	for (j = 0; j < k; ++j)
	    m[j] = c.a(i - (k/2) + j, channel);
	qsort(m, k, sizeof(float), sorttest);
	a[i] = m[k/2];
    }
    
    for (i = 0; i < c.num_frames(); ++i)
	c.a(i) = a[i];
    
    delete [] a;
    delete [] m;
}

void simple_mean_smooth(EST_Track &c, int n, int channel)
{ // simple mean smoother of order n
    int i, j, h, k=1;
    float *a = new float[c.num_frames()];
    float sum;
    h = n/2;
    
    for (i = 0; i < h; ++i)
    {
	k = (i * 2) + 1;
	sum = 0.0;
	for (j = 0; j < k; ++j)
	    sum += c.a(j, channel);
	a[i] = sum /(float) k;
    }
    
    for (i = h; i < c.num_frames() - h; ++i)
    {
	sum = 0.0;
	for (j = 0; j < n; ++j)
	    sum += c.a(i - h + j, channel);
	a[i] = sum /(float) k;
    }
    
    for (; i < c.num_frames(); ++i)
    {
	k = ((c.num_frames() - i)* 2) -1;
	sum = 0.0;
	for (j = 0; j < k; ++j)
	    sum += c.a(i - (k/2) + j, channel);
	a[i] = sum /(float) k;
    }
    
    for (i = 0; i < c.num_frames(); ++i)
	c.a(i) = a[i];
    
    delete [] a;
}

void absolute(EST_Track &tr)
{
    int i, j;
    for (i = 0; i < tr.num_frames(); ++i)
	for (j = 0; j < tr.num_channels(); ++j)
	    tr.a(i, j) = fabs(tr.a(i, j));
}

void normalise(EST_Track &tr)
{
    EST_FVector mean, sd;
    
    meansd(tr, mean, sd);
    normalise(tr, mean, sd);
}

/* Normalise a list of tracks */
void normalise(EST_TrackList &trlist, EST_FVector &mean, EST_FVector &sd)
{
    for (EST_TBI *p = trlist.head(); p; p = next(p))
	normalise(trlist(p), mean, sd);
}

/* Normalise by subtracting the mean and dividing by TWICE the
   standard deviation. */
void normalise(EST_Track &tr, EST_FVector &mean, EST_FVector &sd)
{
    for (int i = 0; i < tr.num_channels(); ++i)
	normalise(tr, mean(i), sd(i), i);
}

void normalise(EST_Track &tr, float mean, float sd, int channel)
{
    int i;
    for (i = 0; i < tr.num_frames(); ++i)
	if (!tr.track_break(i))
	    tr.a(i, channel) = (tr.a(i, channel) - mean) / (sd * 2.0);
}

EST_Track differentiate(EST_Track &c, float samp_int)
{
    // differentiate track. SEE ALSO delta(EST_Track, int) which does
    // this in a more spohisticated way !!!
    
    EST_Track diff;
    int i, j;
    float dist;
    
    if (samp_int != 0.0)
	c.sample(samp_int);
    
    diff.copy_setup(c);
    diff.resize(c.num_frames() - 1, c.num_channels());
    
    for (i = 0; i < diff.num_frames(); ++i)
    {
	dist = c.t(i + 1) - c.t(i);
	for (j = 0; j < diff.num_channels(); ++j)
	    diff.a(i, j) = (c.track_break(i) || c.track_break(i + 1)) ? 0.0 
		: (c.a(i + 1) - c.a(i)) / dist;
	diff.t(i) = c.t(i) + (dist / 2.0);
    }
    
    diff.amin = -100;
    diff.amax = 100;
    
    return diff;
}

EST_Track difference(EST_Track &a, EST_Track &b)
{
    int i, j;
    
    int size = Lof(a.num_frames(), b.num_frames());
    EST_Track diff = a;
    
    // ERROR REORG - this needs to return a proper error
    if (a.num_channels() != b.num_channels())
    {
	cerr << "Error: Can't compare " << a.num_channels() << 
	    " channel EST_Track with " << b.num_channels() << " channel EST_Track\n";
	return diff;
    }
    
    for (i = 0; i < size; ++i)
	for (j = 0; j < a.num_channels(); ++j)
	    diff.a(i, j) = a.a(i, j) - b.a(i, j);
    
    return diff;
}

void meansd(EST_Track &tr, float &mean, float &sd, int channel)
{
    float var;
    int i, n;

    for (n = 0, i = 0, mean = 0.0; i < tr.num_frames(); ++i)
	if (!tr.track_break(i))
	{
	    mean += tr.a(i, channel);
	    ++n;
	}
    
    mean /= n;
    
    for (i = 0, var = 0.0; i < tr.num_frames(); ++i)
	var += tr.track_break(i) ? 0.0 : pow(tr.a(i, channel) - mean, 2.0);
    
    var /= n;
    sd = sqrt(var);
}

float rms_error(EST_Track &a, EST_Track &b, int channel)
{
    int i;
    int size = Lof(a.num_frames(), b.num_frames());
    float sum = 0;
    
    for (i = 0; i < size; ++i)
	if (a.val(i) && b.val(i))
	    sum += pow((a.a(i, channel) - b.a(i, channel)), 2.0);
    
    sum = sqrt(sum / size);
    return sum;
}

float abs_error(EST_Track &a, EST_Track &b, int channel)
{
    int i;
    int size = Lof(a.num_frames(), b.num_frames());
    float sum = 0;
    for (i = 0; i < size; ++i)
    {
	cout << i << " " << a.a(i, channel) << " " << b.a(i, channel) << endl;
	if (a.val(i) && b.val(i))
	    sum += fabs(a.a(i, channel) - b.a(i, channel));
    }
    return sum / size;
}

float correlation(EST_Track &a, EST_Track &b, int channel)
{
    int i;
    int size = Lof(a.num_frames(), b.num_frames());
    float predict,real;
    EST_SuffStats x,y,xx,yy,xy,se,e;
    float cor,error;
    
    for (i = 0; i < size; ++i)
	if (a.val(i) && b.val(i))
	{
	    cout << a.a(i, channel) << " " << b.a(i, channel) << endl;
	    predict = b.a(i, channel);
	    real = a.a(i, channel);
	    x += predict;
	    y += real;
	    error = predict-real;
	    se += error*error;
	    e += fabs(error);
	    xx += predict*predict;
	    yy += real*real;
	    xy += predict*real;
	}
    
    cor = (xy.mean() - (x.mean()*y.mean()))/
        (sqrt(xx.mean()-(x.mean()*x.mean())) *
	 sqrt(yy.mean()-(y.mean()*y.mean())));
    
    cout << xy.mean()  << " " << x.mean() << " " << y.mean() << " "
        << xx.mean() << " " << yy.mean() << endl;
    
    cout << "RMSE " << sqrt(se.mean()) << " Correlation is " << cor 
        << " Mean (abs) Error " << e.mean() << " (" << e.stddev() << ")" 
            << endl;
    return cor;
}

void meansd(EST_Track &a, EST_FVector &m, EST_FVector &sd)
{
    int i;
    
    m.resize(a.num_channels());
    sd.resize(a.num_channels());
    
    for (i = 0; i < a.num_channels(); ++i)
	meansd(a, m(i), sd(i), i);
}

void meansd(EST_TrackList &tl, float &mean, float &sd, int channel)
{
    EST_TBI *p;
    float var=0.0;
    int i, n;

    n = 0;
    mean = 0.0;

    for (p = tl.head(); p; p = next(p))
	for (i = 0; i < tl(p).num_frames(); ++i)
	    if (!tl(p).track_break(i))
	    {
		mean += tl(p).a(i, channel);
		++n;
	    }

    mean /= n;

    for (p = tl.head(); p; p = next(p))
	for (i = 0; i < tl(p).num_frames(); ++i)
	    if (!tl(p).track_break(i))
		var +=  pow(tl(p).a(i, channel) - mean, 2.0);

    var /= n;
    sd = sqrt(var);
}

void meansd(EST_TrackList &tl, EST_FVector &m, EST_FVector &sd)
{
    int i;
    
    m.resize(tl.first().num_channels());
    sd.resize(tl.first().num_channels());
    
    for (i = 0; i < tl.first().num_channels(); ++i)
	meansd(tl, m(i), sd(i), i);
}

EST_FVector rms_error(EST_Track &a, EST_Track &b)
{
    int i;
    EST_FVector e;
    
    // ERROR REORG - this needs to return a proper error
    if (a.num_channels() != b.num_channels())
    {
	cerr << "Error: Can't compare " << a.num_channels() << 
	    " channel EST_Track with " << b.num_channels() << " channel EST_Track\n";
	return e;
    }
    e.resize(a.num_channels());
    for (i = 0; i < a.num_channels(); ++i)
	e(i) = rms_error(a, b, i);
    
    return e;
}

EST_FVector abs_error(EST_Track &a, EST_Track &b)
{
    int i;
    EST_FVector e;
    
    // ERROR REORG - this needs to return a proper error
    if (a.num_channels() != b.num_channels())
    {
	cerr << "Error: Can't compare " << a.num_channels() << 
	    " channel EST_Track with " << b.num_channels() << " channel EST_Track\n";
	return e;
    }
    e.resize(a.num_channels());
    for (i = 0; i < a.num_channels(); ++i)
	e(i) = abs_error(a, b, i);
    
    return e;
}

EST_FVector correlation(EST_Track &a, EST_Track &b)
{
    int i;
    EST_FVector cor;
    
    // ERROR REORG - this needs to return a proper error
    if (a.num_channels() != b.num_channels())
    {
	cerr << "Error: Can't compare " << a.num_channels() << 
	    " channel EST_Track with " << b.num_channels() << " channel EST_Track\n";
	return cor;
    }
    cor.resize(a.num_channels());
    for (i = 0; i < a.num_channels(); ++i)
	cor(i) = correlation(a, b, i);
    
    return cor;
}

EST_Track error(EST_Track &ref, EST_Track &test, int relax)
{
    int i, j, k, l;
    EST_Track diff;
    diff = ref;
    int t;
    
    // relaxation allows an error to be ignored near boundaries. The
    // degree of relation specifies how many frames can be ignored.
    
    int *r = new int[relax*3];
    
    for (l = 0; l < ref.num_channels(); ++l)
	for (i = 0; i < ref.num_frames(); ++i)
	{
	    t = 0;
	    for (k = 0, j = Gof((i - relax), 0); j < i + relax + 1; ++j, ++k)
	    {
		if (ref.a(i, l) > 0.5)
		    r[k] = ((j < test.num_frames()) && (test.a(j, l)> 0.6)) ?1 
			: 0;
		else
		    r[k] = ((j < test.num_frames()) && (test.a(j, l)< 0.4)) ?1 
			: 0;
		
		t |= r[k];
	    }
	    diff.a(i, l) = t;
	}
    
    delete [] r;
    return diff;
}

int track_divide(EST_TList<EST_Track> &mtfr, EST_Track &fv, EST_Stream &key)
{
    EST_Track a;
    EST_Stream_Item  *k, t;
    float kstart, length;
    int i, j, l, n;
    
    mtfr.clear();
    
    if ((key.tail())->end() < (fv.t(fv.num_frames() - 1)))
    {
	cerr << "Key file must extend beyond end of EST_Track\n";
	cerr << "key end: " << key.tail()->end() << " EST_Track end: " 
	    << fv.t(fv.num_frames() - 1) << endl;
	return -1;
    }
    
    k = key.head();
    a.set_name(k->name());
    kstart = 0.0;
    
    length = k->end() - kstart;
    n = (int)(length / (float) fv.shift()) + 2;
    a.resize(n, fv.num_channels());
    
    for (i = 0, l = 0; i < fv.num_frames(); ++i, ++l)
    {
	for (j = 0; j < fv.num_channels(); ++j)
	    a(l, j) = fv(i, j);
	
	if (fv.t(i) > k->end())
	{
	    a.set_num_frames(l + 1);
	    mtfr.append(a);
	    
	    kstart = k->end();
	    k = next(k);
	    a.set_name(k->name());
	    length = k->end() - kstart;
	    n = (int)(length / (float) fv.shift()) + 2;
	    //	    cout << "n frames: " << n << endl;
	    a.resize(n, fv.num_channels());
	    a.fill_time(fv.shift());
	    
	    //	for (j = 0; j < fv.order(); ++j)
	    //	    a(0, j) = fv(i, j);
	    l = -1;
	}
    }
    a.set_num_frames(l);
    mtfr.append(a);
    return 0;
}

void align_to_track(EST_Track &tr, float &start, float &end)
{
  int is, ie;

  // cout << " in " << start << " " << end << "\n";

  is = tr.index(start);
  ie = tr.index(end);

  // cout << " indexes " << is << " " << ie << "\n";
  
  start = tr.t(is);
  end   = tr.t(ie);
  // cout << " out " << start << " " << end << "\n";
}
void align_to_track(EST_Track &tr, int &start, int &end, int sample_rate)
{
    float start_t = start/(float)sample_rate;
    float   end_t =   end/(float)sample_rate;

    
    // cout << "align " << start_t << " " << end_t << " " << sample_rate << "\n";
    align_to_track(tr, start_t, end_t);
    // cout << " gives " << start_t << " " << end_t << "\n";

    start = (int)(start_t*sample_rate + 0.5);
      end = (int)(  end_t*sample_rate + 0.5);
}



EST_Track extract(EST_Track &orig, float start, float end)
{
    EST_Track ret;
    int new_num_frames;
    
    ret.copy_setup(orig);
    
    int i, j;
    int is = 0, ie = 0;
    
    is = orig.index(start);
    ie = orig.index(end);
    
    // check in case above results in negative length
    new_num_frames = (ie - is) > 0 ?ie - is : 0;
    ret.resize(new_num_frames, orig.num_channels());
    
    for (i = 0; i < new_num_frames; ++i)
    {
	for (j = 0; j < orig.num_channels(); ++j)
	    ret.a(i, j) = orig.a(i + is, j);
	ret.t(i) = orig.t(i + is);
	if (orig.track_break(i + is))
	    ret.set_break(i);
	else
	    ret.set_value(i);
    }
    return ret;
}

int get_order(const EST_Track &t)
{
  int order;
  if (t.has_channel(channel_coef0))
    if (t.has_channel(channel_coefN))
      order = t.channel_position(channel_coefN) - t.channel_position(channel_coef0);
    else
      order = t.num_channels()-t.channel_position(channel_coef0)-1;
  else
    {
      cout << "No coefficients in track\n";
      order=0;
    }
  return order;
}

// Note, these assumes a length channel
int sum_lengths(const EST_Track &t, int start_frame, int end_frame)
{
  int l=0;

  if (end_frame < 0)
    end_frame = t.num_frames();

  for(int i=start_frame; i<end_frame; i++)
    l += (int)t.a(i, channel_length);

  return l;
}

int *get_start_positions(const EST_Track &t)
{
  int *pos = new int[t.num_frames()];
  int p=0;

  for(int i=0; i<t.num_frames(); p += (int)t.a(i, channel_length), i++)
    pos[i] = p;

  return pos;
}

void extract_channel(EST_Track &orig, EST_Track &nt, EST_IList &ch_list)
{
    int new_ch, i, j, k;
    EST_TBI *p;
    new_ch = ch_list.length();

    nt.clear();
    nt.copy_setup(orig);
    nt.resize(orig.num_frames(), new_ch);

    for (i = 0, p = ch_list.head(); p; p = next(p), ++i)
    {
	k = ch_list(p);
	cout << "Extracting track " << k << endl;
	if (k >= orig.num_channels())
	{
	    cerr << "Tried to extract channel number " << k 
		<< " from track with only " << orig.num_channels() 
		    << " channels\n";
	    return;
	}
	for (j = 0; j < orig.num_frames(); ++j)
	    nt.a(j, i) = orig.a(j, k);
	nt.set_field_name(orig.field_name(k), i);
    }
    for (j = 0; j < orig.num_frames(); ++j)
	nt.t(j) = orig.t(j);

}

void ParallelTracks(EST_Track &a, EST_TrackList &list,const EST_String &style)
{
    // Make multi channel track out of list of tracks. There are two
    // "styles". "0" means take the size of the first track in the list,
    // "1" means take the size of the longest as the number of frames in
    // the created track.
    EST_TBI *p;
    int num_channels, num_frames;
    int i, j, k, n;
    
    a.clear();
    for (num_channels=0,p=list.head(); p; p=next(p))
	num_channels += list(p).num_channels();

    if (style == "first")
	num_frames = list.first().num_frames();
    else
    {   
	if (style != "longest")
	    cerr << "EST_Track: unknown combine style \"" << style << 
		"\" assuming longest" << endl;
	for (num_frames = 0, p = list.head(); p; p = next(p))
	    if (num_frames < list(p).num_frames())
		num_frames = list(p).num_frames();
    }
    
    a.resize(num_frames, num_channels);
    a.fill_amp(0.0);
    
    for (k = 0, p = list.head(); p; p = next(p))
    {
	n = Lof(num_frames, list(p).num_frames());
	for (j = 0; j < list(p).num_channels(); ++j, ++k)
	    for (i = 0; i < n; ++i)
		a(i, k) = list(p).a(i, j);
    }
}

int read_track(EST_Track &tr, const EST_String &in_file, EST_Option &al)
{

    float ishift = 0;

    if (al.present("ishift"))
	ishift = al.fval("ishift",1);
    else if (al.present("time_channel"))
	ishift = 1.0;		// doesn't matter, will be reset by track
    
    if (tr.load(in_file, ishift) != format_ok)
	return -1;
    
    tr.create_map();
    
    // cout << "f0 "<< tr.has_channel(channel_f0) << ".\n";
    
    if (al.present("time_channel") && tr.has_channel(al.sval("time_channel")))
    {
	cout << " time from channel " << al.sval("time_channel") << "\n";
	tr.channel_to_time(al.sval("time_channel"), al.fval("time_scale"));
    }
    
    //    cout << tr;
    return 0;
}

void track_info(EST_Track &t)
{
    cout <<  t.name() << endl;
    cout << "Number of frames: " << t.num_frames() << endl;
    cout << "Number of channels: " << t.num_channels() << endl;
    cout << "File type: " << t.file_type() << endl;
    cout << "Frame shift: " << t.shift() << endl;
    for (int i = 0; i < t.num_channels(); ++i)
	cout << "Channel: " << i << ": " << t.field_name(i) << endl;
}

EST_String options_track_filetypes(void)
{
    // Returns list of currently support track filetypes
    // Should be extracted from the list in EST_Track
    
    return "esps, xmg, htk, ascii";
}
