/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "ATraps.h"

#include "Byteswapping.h"		// Canonical
#include "DebugMgr.h"			// Debug::InDebugger
#include "ErrorHandling.h"		// Errors::ReportError
#include "Miscellaneous.h"		// StMemoryMapper
#include "Profiling.h"			// StDisableAllProfiling


// ===========================================================================
//		 ATrap
// ===========================================================================

long			ATrap::fgCallCount;

const uae_u16	kATrapReturnTrapNum		= 0x0C;
const uae_u16	kOpcode_ROMCall			= m68kTrapInstr + sysDispatchTrapNum;
const uae_u16	kOpcode_ATrapReturn		= m68kTrapInstr + kATrapReturnTrapNum;


/***********************************************************************
 *
 * FUNCTION:	ATrap::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::Initialize (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::Reset (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::ATrap
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ATrap::ATrap (void) :
#if (__GNUC__ != 2)
/* Can't call the default constructor because there isn't one defined */
/* on a struct as there is with a class under GCC 2.8.1 */
	fNewRegisters (),
#endif
	fMapper (fStack, kStackSize)
{
	// Get the registers.

	Registers::GetRegisters (fNewRegisters);

	// We don't necessarily want to go through life in Trace mode, so turn it
	// off when making ATrap::Call's.

	fNewRegisters.t0 = fNewRegisters.t1 = 0;
	fNewRegisters.sr &= ~((1 << 15) | (1 << 14));

	// If the SPCFLAG_SAVE_STATE bit is set, turn it off.  It will get restored
	// when we restore the register state.  But if we leave it on, that will cause
	// the state to get saved the instant we make this ATrap call, and again
	// the instant we return from it.  If we happen to make several ATrap calls
	// in a row, we'll end up saving the state frequently.  I've seen this cause
	// Gremlins to pause for up to 40 seconds when auto-saving a file (as opposed
	// to the normal 1/2 second).

	fNewRegisters.spcflags &= ~SPCFLAG_SAVE_STATE;
	fNewRegisters.spcflags &= ~SPCFLAG_SAVE_SUSPENDED_STATE;
	fNewRegisters.spcflags &= ~SPCFLAG_SAVE_ROOT_STATE;
	fNewRegisters.spcflags &= ~SPCFLAG_LOAD_ROOT_STATE;
	fNewRegisters.spcflags &= ~SPCFLAG_NEXT_GREMLIN_FROM_SUSPENDED_STATE;
	fNewRegisters.spcflags &= ~SPCFLAG_NEXT_GREMLIN_FROM_ROOT_STATE;

	// Jam the interrupt level so that we aren't interrupted.

	fNewRegisters.intmask = 7;

	// Make sure the SR is up-to-date.  This code is taken from
	// Registers::UpdateSRFromRegisters.  We can't call that function
	// directly as it works on the global variable "regs".

	fNewRegisters.sr = ((fNewRegisters.t1 << 15) | (fNewRegisters.t0 << 14)
			| (fNewRegisters.s << 13) | (fNewRegisters.m << 12) | (fNewRegisters.intmask << 8)
			| (XFLG << 4) | (NFLG << 3) | (ZFLG << 2) | (VFLG << 1) 
			|  CFLG);

	// Make sure the CPU is not stopped.  I suppose that we could force the CPU
	// to no longer be stopped, but I'd rather that the Palm OS itself woke up
	// first before we try making calls into it.  Therefore, anything making
	// an out-of-the-blue Palm OS call via this class (that is, a call outside
	// of the context of a Palm OS function head- or tailpatch) should first
	// bring the CPU to a halt by calling Emulator::ExecuteUntilATrap first.

	assert (fNewRegisters.stopped == 0);

	// Give ourselves our own private stack.  We'll want this in case
	// we're in the debugger and the stack pointer is hosed.

	m68k_areg (fNewRegisters, 7) = (uaecptr) &fStack[kStackSize];
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::~ATrap
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ATrap::~ATrap (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::Call
 *
 * DESCRIPTION:	Calls the given pseudo-ATrap.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

long ATrap::Call (uae_u16 trapWord)
{
	// Up until now, the registers in "regs" have been left alone.  If any
	// values were pushed on the stack, the stack position was reflected in
	// fNewRegisters.  Now's the time to move those values from fNewRegisters
	// to regs.

	regstruct	oldRegisters;
	Registers::GetRegisters (oldRegisters);
	Registers::SetRegisters (fNewRegisters);

	// Make the call.

	fgCallCount++;
	long	breakReason = this->DoCall(trapWord);
	fgCallCount--;

	// Remember the resulting register values so that we can report them to
	// the user when they call GetD0 and/or GetA0.

	Registers::GetRegisters (fNewRegisters);

	// Put things back the way they were.

	Registers::SetRegisters (oldRegisters);

	return breakReason;
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::PushByte
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::PushByte (uae_u8 iByte)
{
	StDisableAllProfiling	stopper;

	m68k_areg (fNewRegisters, 7) -= 2;
	put_byte (m68k_areg (fNewRegisters, 7), iByte);
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::PushWord
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::PushWord (uae_u16 iWord)
{
	StDisableAllProfiling	stopper;

	m68k_areg (fNewRegisters, 7) -= 2;
	put_word (m68k_areg (fNewRegisters, 7), iWord);
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::PushLong
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::PushLong (uae_u32 iLong)
{
	StDisableAllProfiling	stopper;

	m68k_areg (fNewRegisters, 7) -= 4;
	put_long (m68k_areg (fNewRegisters, 7), iLong);
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::SetNewDReg
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::SetNewDReg (int regNum, uae_u32 value)
{
	m68k_dreg (fNewRegisters, regNum) = value;
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::SetNewAReg
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void ATrap::SetNewAReg (int regNum, uae_u32 value)
{
	m68k_areg (fNewRegisters, regNum) = value;
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::GetD0
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 ATrap::GetD0 (void)
{
	return m68k_dreg (fNewRegisters, 0);
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::GetA0
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

uae_u32 ATrap::GetA0 (void)
{
	return m68k_areg (fNewRegisters, 0);
}


/***********************************************************************
 *
 * FUNCTION:	ATrap::DoCall
 *
 * DESCRIPTION:	Calls the pseudo-ATrap. When we return, D0 or A0 should
 *				hold the result, the parameters will still be on the
 *				stack with the SP pointing to them, and the PC will be
 *				restored to what it was before this function was called.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

long ATrap::DoCall (uae_u16 trapWord)
{
	// Stop all profiling activities. Stop cycle counting and stop the
	// recording of function entries and exits.  We want our calls to
	// ROM functions to be as transparent as possible.

	StDisableAllProfiling	stopper;


	// Assert that the function we're trying to call is implemented.
	//
	// Oops...bad test...this doesn't work when we're calling a library.
	// Instead, since we now invoke ROM functions by creating a TRAP $F
	// sequence, we'll let our TRAP $F handler deal with validating the
	// function call (it does that anyway).

//	assert (LowMem::GetTrapAddress (trapWord));

	// Make sure that we're not in the debugger when calling a system function.
	// We really don't know what the state of the machine is, and we're not
	// in a position to move it into a "good state" (that is, a state where
	// we feel confident that we can insert a call to the ROM here).

	// This is a pretty good assert, but I'm taking it out for now. RPC packets
	// sent to the debugger funnel through here, which triggers the assert.
	// Perhaps another approach would be to clear the "inDebugger" flag when
	// processing RPC packets, but I don't know what side-effects that might have.

//	assert (Debug::InDebugger () == false);

	// We call the ROM function by dummying up a sequence of 68xxx instructions
	// for it.  The sequence of instructions is:
	//
	//			TRAP	$F
	//			DC.W	<dispatch number>
	//			TRAP	$C
	//
	// The first two words invoke the function (calling any head- or tailpatches
	// along the way).  The third word allows the emulator to regain control
	// after the function has returned.
	//
	// Note: this gets a little ugly on little-endian machines.  The following
	// instructions are stored on the emulator's stack.  This memory is mapped
	// into the emulated address space in such a fashion that no byteswapping of
	// word or long values occurs.  Thus, we have to put the data into Big Endian
	// format when putting it into the array.
	//
	// However, opcodes are a special case.  They are optimized in the emulator
	// for fast access.  Opcodes are *always* fetched a word at a time in host-
	// endian order.  Thus, the opcodes below have to be stored in host-endian
	// order.  That's why there's no call to Canonical to put them into Big
	// Endian order.

	uae_u16	code[] = { kOpcode_ROMCall, trapWord, kOpcode_ATrapReturn };

	// Oh, OK, we do have to byteswap the trapWord.  Opcodes are fetched with
	// do_get_mem_word, which always gets the value in host byte order.  The
	// trapWord is fetched with get_word, which gets values according to the
	// rules of the memory bank.  For the dummy bank, the defined byte order
	// is Big Endian.

	Canonical (code[1]);

	// Tell the CPU loop that it should break when it hits the trap exception.

	CBreakOnException	breaker (kException_Trap0 + kATrapReturnTrapNum);

	// Set up the address banks so that pointers to variables on the stack
	// (like with that "code" array above) will resolve to their
	// actual physical locations.  This also helps us in our ROMStubs where
	// we may have passed the addresses of stack-based variables.

	StMemoryMapper	mapper (code, 64 * 1024L);

	// Point the PC to our code.  Remember the original PC so that we can
	// restore it when we're done.
	
	uaecptr	oldPC = m68k_getpc ();
	m68k_setpc ((uaecptr) code);

	// Execute until the next break.

	long	breakReason = Emulator::ExecuteUntilBreak ();

	// Restore the original PC.

	m68k_setpc (oldPC);

	// If we broke for the reason we're expecting, we're cool.

	if ((breakReason & kBreak_Exception) == (kException_Trap0 + kATrapReturnTrapNum))
	{
		breakReason = 0;
	}

	// Otherwise, display an error.

	else
	{
		char	buffer[200];
		sprintf (buffer, "An exception occured while the emulator was calling the "
				"Palm OS function \"%s\".  The emulator is now in an unstable state "
				"and will quit (eventually I'll change this so it just resets).",
				::GetTrapName (trapWord));
		Errors::DoDialog (NULL, buffer, Errors::kOK);

		abort ();	// !!! We should just reset instead.
	}

	return breakReason;
}
