/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "CPU_MT.h"

#include "CPU_REG.h"			// Emulator::Execute, etc.
#include "DebugMgr.h"			// InDebugger
#include "Logging.h"			// LogAppendMsg
#include "Platform.h"			// gEnableSounds
#include "Strings.r.h"			// kStr_ values

#define PRINTF	if (true) ; else LogAppendMsg

// ======================================================================
//	Globals and constants
// ======================================================================


/***********************************************************************
 *
 * FUNCTION:	CPU constructor
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CPU::CPU() :
	omni_thread (),
	fRunMutex (),
	fRunCondition (&fRunMutex),
	fStopMutex (),
	fStopCondition (&fStopMutex),
	fSleepMutex (),
	fSleepCondition (&fSleepMutex),
	fRunning (false),
	fStopRequest (true),
	fQuitRequest (false)
{
}


/***********************************************************************
 *
 * FUNCTION:	CPU destructor
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

CPU::~CPU()
{
	// !!! Implicitly call DestroyThread?

	assert (fRunning == false);
	assert (fQuitRequest = true);
}


/***********************************************************************
 *
 * FUNCTION:	CPU::CreateThread
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CPU::CreateThread (void)
{
	// Start the emulator thread (which will immediately block).

	this->start_undetached ();
}


/***********************************************************************
 *
 * FUNCTION:	CPU::DestroyThread
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CPU::DestroyThread (void)
{
	// Should already be stopped, but let's just make sure.

	Bool	dummy;
	this->StopThread (dummy, dummy, kStopOnADime);

	// Tell the thread to quit.

	fQuitRequest = true;

	// OK, start it up again so that it can die.

	this->StartThread ();

	// Wait for the CPU thread to finish.
	//
	// !!! race condition? what if the thread has already finished?
	// !!! experimentation seems to show that Omnithreads handles this
	// !!! case OK, but nonetheless, this warrants re-thinking.
	//
	// NOTE: the original implementation of join() would delete "self"
	// with a call to "delete this". I'm not sure why it does that, but
	// since I create my thread object as a global variable instead of
	// as an object on the heap, I had to remove that operation.

	this->join (NULL);
}


/***********************************************************************
 *
 * FUNCTION:	CPU::StartThread
 *
 * DESCRIPTION: Start up the CPU thread.  If the thread is already
 *				running, do nothing.  Otherwise, unsignal the event
 *				that flags when the CPU stops, and unblock the thread
 *				by signalling gCPUBeginEvent.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CPU::StartThread (void)
{
	PRINTF ("CPU::Start: Entering.");

#if 0
	// !!! Note, there might be a race-condition problem here.	If the
	// CPU thread is in the process of stopping, but hasn't yet cleared
	// fRunning, then Start will do nothing, and the CPU thread will
	// eventually halt.  That may not be what's wanted.  Care needs to
	// be take that this method is called only when we *know* the thread
	// has been stopped.

	if (fRunning)
	{
		PRINTF ("CPU::Start: Already running; exitting.");
		return;
	}
#endif

	// Unblock CPU::run_undetached.

	PRINTF ("CPU::Start: Unblocking CPU::run_undetached.");

	fRunMutex.lock ();
	fStopRequest = false;
	fRunning = true;	// Need to set this here.  If not, then an immediate call
						// to Stop will find it false (if the CPU thread hasn't had
						// a chance to set it to true, yet) and will think that the
						// CPU thread is stopped instead of just starting up.
	fRunCondition.broadcast ();
	fRunMutex.unlock ();

	PRINTF ("CPU::Start: Exitting.");
}


/***********************************************************************
 *
 * FUNCTION:	CPU::StopThread
 *
 * DESCRIPTION: Stop the CPU thread, returning whether or not the thread
 *				was running when this function was called (TRUE == it
 *				was running, FALSE == it was already stopped).	If the
 *				thread is running, this function asks it to stop, and
 *				then waits FOREVER for it to stop.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void CPU::StopThread (Bool& wasStopped, Bool& isStopped, int how)
{
	PRINTF ("CPU::Stop: Entering");

	// See if the CPU thread is already stopped and waiting on its mutex.
	// If fRunning is false, then it's either blocked or about to be blocked.

	omni_mutex_lock lock (fStopMutex);

	PRINTF ("CPU::Stop: fRunning = %s", fRunning ? "TRUE" : "FALSE");
	wasStopped = !fRunning;

	if (fRunning)
	{
		// The CPU thread is running.  Let's see if it's blocked on the UI thread
		// (which is probably what's calling this method, in which case our
		// waiting on the CPU thread would cause a deadlock).  If we're in that
		// small window where the CPU thread is just about to show an error
		// dialog but hasn't set gShowingDialog yet, then at least we can fall
		// back on using a timeout while blocking on it.

		if (gShowingDialog)
		{
			// The CPU thread is blocked on the UI; it's probably showing a dialog.
			// This means that its running and will continue to run.

			PRINTF ("CPU::Stop: CPU thread blocked on UI thread.");
		}
		else
		{
			// The CPU thread is running and it doesn't appear to be blocked on
			// the UI thread.  Ask it nicely to stop.

			Bool	old;

			fStopRequest = true;

			if (how == kStopOnADime)
			{
				PRINTF ("CPU::Stop: Setting break request");
				Emulator::SetBreakReason (kBreak_StopRequest);
			}
			else
			{
				PRINTF ("CPU::Stop: Need to stop on A-Trap");
				old = Emulator::SetBreakOnException (kException_Trap15, true);
			}

			// Wake it up if it's sleeping.

			PRINTF ("CPU::Stop: Waking up sleeping thread");
			fSleepMutex.lock ();
			fSleepCondition.broadcast ();
			fSleepMutex.unlock ();

			// Now wait for it to exit.
			//
			// The fRunMutex is what indicates whether or not the CPU thread is
			// running.  If acquired, the thread is running.  If not, then
			// the thread is blocked waiting for something to start it up again.
			//
			// What we'd like to do here is wait until that mutex if free'd.
			// However, we can't just try to aquire the mutex and block waiting
			// for it to be released by the CPU thread.  It's possible that the
			// CPU thread is busy and won't be releasing the mutex for a while.
			// In that case, we'd like to block for a small amount of time, and
			// then gracefully fail if fRunMutex is not released.
			//
			// The only omnithread object that waits with a timeout period is
			// omni_condition.	However, the method that does this -- timedwait
			// -- doesn't quite do what we want.  It *releases* its associated
			// mutex and then blocks waiting to re-acquire it.	This means that
			// we must first have acquired the mutex first.  But acquiring the
			// mutex is what we just said we might be able to do.
			//
			// Therefore, we use a second mutex -- fStopMutex -- to signal that
			// the CPU thread is *about* to stop.  This mutex is acquired by
			// the CPU thread solely for the purpose of signalling with it.
			// We can make use of that here: we do a timed wait on fStopMutex.
			// If we get signalled, we know the CPU thread is about to stop and
			// so we can safely block on fRunMutex.  If fStopMutex doesn't
			// get signalled, then we timeout and fail gracefully.
			//
			// To take care of the case where this function was entered *after*
			// fStopMutex was signalled, we have a fRunning boolean.  This value is
			// set to true just before the CPU thread is started, and set to
			// false just before fStopMutex is signalled.  In other words, it
			// is false between the time fStopMutex is signalled and
			// fRunMutex is released.  So we can look at this variable to know
			// if the thread is about to stop without having to rely on catching
			// the signal on fStopMutex.

			const long		kTimeout = 1000;
			unsigned long	secs, nsecs;
			this->MSecsToSecsNSecs (kTimeout, secs, nsecs);

			PRINTF ("CPU::Stop: Waiting for stop condition");
			int signalled = fStopCondition.timedwait (secs, nsecs);
			PRINTF ("CPU::Stop: fStopCondition.timedwait returned %d", signalled);

			if (how == kStopOnATrap)
			{
				(void) Emulator::SetBreakOnException (kException_Trap15, old);
			}
		}
	}

	// Get the current state of the thread.

	PRINTF ("CPU::Stop: fRunning = %s", fRunning ? "TRUE" : "FALSE");
	isStopped = !fRunning;

	// If "isStopped" is true, then the CPU thread is either stopped or
	// is about to stop.  Ensure that it's the former by blocking on the
	// mutex it releases when it really stops.

	if (isStopped)
	{
		PRINTF ("CPU::Stop: Waiting on fRunMutex");
		fRunMutex.lock ();
		fRunMutex.unlock ();
		PRINTF ("CPU::Stop: Done waiting on fRunMutex");
	}

	// It didn't stop.	Make sure it continues to not stop.  That is, when
	// it finally stops, make sure it automatically restarts itself.

	else if (!wasStopped)
	{
		PRINTF ("CPU::Stop: Not stopped, restarting");
		fStopRequest = false;
	}

	PRINTF ("CPU::Stop: Leaving");
}


/***********************************************************************
 *
 * FUNCTION:	CPU::Sleep
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void CPU::Sleep (long timeout)
{
	unsigned long	secs, nsecs;
	this->MSecsToSecsNSecs (timeout, secs, nsecs);

	fSleepMutex.lock ();
	fSleepCondition.timedwait (secs, nsecs);
	fSleepMutex.unlock ();
}


/***********************************************************************
 *
 * FUNCTION:	CPU::run_undetached
 *
 * DESCRIPTION: Thread procedure for the CPU thread.  The CPU thread
 *				just executes emulated 68K instructions.  The only time
 *				it ever stops is if an exception occurs (or when it's
 *				requested).  If either of these occurs, the CPU loop
 *				exits back to here.
 *
 * PARAMETERS:	none
 *
 * RETURNED:	nothing
 *
 ***********************************************************************/

void* CPU::run_undetached (void*)
{
	omni_mutex_lock lock (fRunMutex);

	while (1)
	{
		// Wait for the begin signal...

		while (fStopRequest && !fQuitRequest)
		{
			// Wake up anyone waiting for the thread to stop.

			PRINTF ("run_undetached: Setting fRunning to false, broadcasting sleep");
			fStopMutex.lock ();
			fRunning = false;	// Set this here so that WaitForStop can react
								// to it when it unblocks on the stop mutex.  It gets
								// set back to TRUE in StartThread.
			fStopCondition.broadcast ();
			fStopMutex.unlock ();

			// Release fRunMutex and wait to get signalled.  When
			// signalled, see if we're now runnable.  If not, keep going
			// back to sleep.

			PRINTF ("run_undetached: Waiting for !fStopRequest or fQuitRequest.");
			fRunCondition.wait ();
		}

		if (fQuitRequest)
			break;

		PRINTF ("run_undetached: Running...");

		long breakReason = Emulator::Execute();

		// If we're stopping because of an exception, make sure we do stop.
		// Normally, fStopRequest is only set externally if some other thread
		// wants us to stop.

		if ((breakReason & ~kBreak_StopRequest) != 0)
		{
			fStopRequest = true;
		}

		PRINTF ("run_undetached: Stopping...breakReason == 0x%08X", breakReason);
	}

	PRINTF ("run_undetached: Quitting...");
	fRunning = false;

	return NULL;
}


/***********************************************************************
 *
 * FUNCTION:	CPU::MSecsToSecsNSecs
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	.
 *
 * RETURNED:	nothing.
 *
 ***********************************************************************/

void CPU::MSecsToSecsNSecs (long msecs, unsigned long& secs, unsigned long& nsecs)
{
	const long	kMillisecondsPerSecond = 1000;
	const long	kNanosecondsPerMillisecond = 1000000;

	secs = msecs / kMillisecondsPerSecond;
	nsecs = msecs * kNanosecondsPerMillisecond;

	this->get_time (&secs, &nsecs, secs, nsecs);
}


Bool	CPUStopper::fgInstantiated;

// ---------------------------------------------------------------------------
//		 CPUStopper::CPUStopper
// ---------------------------------------------------------------------------

CPUStopper::CPUStopper (int how) :
	fCPU (Platform::GetCPU ()),
	fNowStopped (false)
{
#if !defined(NDEBUG) && !defined(__MACOS__)
	assert (omni_thread::self() != (omni_thread*) Platform::GetCPU());
#endif

	if (fgInstantiated++ == 0)
	{
		if (how != kStopManually)
		{
			(void) this->Stop (how);
		}
	}
	else
	{
		fNowStopped = true;
	}
}


// ---------------------------------------------------------------------------
//		 CPUStopper::~CPUStopper
// ---------------------------------------------------------------------------

CPUStopper::~CPUStopper (void)
{
	if (--fgInstantiated == 0)
	{
		this->Start ();
	}
}


// ---------------------------------------------------------------------------
//		 CPUStopper::Stop
// ---------------------------------------------------------------------------

Bool CPUStopper::Stop (int how)
{
	PRINTF ("CPUStopper::Stop: Entering");

	if (!fCPU)
	{
		PRINTF ("CPUStopper::Stop: Not stopping: no CPU");

		// Set fNowStopped to false.  This way, clients calling Stopped
		// will get false, which means that they won't try to call into
		// CPU methods when there's no CPU object.

		fNowStopped = false;
	}
	else
	{
		PRINTF ("CPUStopper::Stop: Stopping");

		Bool	wasStopped;
		fCPU->StopThread (wasStopped, fNowStopped, how);
	}

	PRINTF ("CPUStopper::Stop: Leaving");

	return this->Stopped ();
}


// ---------------------------------------------------------------------------
//		 CPUStopper::Start
// ---------------------------------------------------------------------------

void CPUStopper::Start (void)
{
	PRINTF ("CPUStopper::Start: Entering");

	// Only check the cases where the CPU thread is NOW stopped.
	// Otherwise, the CPU is still running, and we don't need to
	// restart it.
	//
	// Note: fWasInDebugger is valid only if fNowStopped is TRUE.

	if (!fCPU)
	{
		PRINTF ("CPUStopper::Start: Not restarting: no CPU");
	}
	else if (!fNowStopped)
	{
		PRINTF ("CPUStopper::Start: Not restarting: fNowStopped = FALSE");
	}
	else
	{
		// If we've determined that we need to restart the CPU thread,
		// then do it.

		if (Emulator::Runable())
		{
			PRINTF ("CPUStopper::Start: Restarting");

			fCPU->StartThread ();
		}
		else
		{
			PRINTF ("CPUStopper::Start: Not restarting: inDebugger = %s, suspended = %s",
				Debug::InDebugger() ? "TRUE" : "FALSE",
				Emulator::Suspended() ? "TRUE" : "FALSE");
		}
	}

	PRINTF ("CPUStopper::Start: Leaving");
}


// ---------------------------------------------------------------------------
//		 CPUStopper::ShowError
// ---------------------------------------------------------------------------

void CPUStopper::ShowError (const char* msg)
{
	/*	!!!
		CPU_MT.cpp: CPUStopper::CPUStopper: The warning message in this
		function is the one that's most often quoted back to me in messages from
		people asking for help with Copilot. Generally the problem is they've
		got half a ROM file and the boot sequence hangs, at which time they
		attempt to load an application. This message is sort of inappropriate at
		this point, because it leads them to believe they should turn the
		emulator "on" first. Not knowing how to do that, they click on the
		nearest mailto: link and I hear about it. Perhaps Poser users will run
		into this problem less often because of the better startup code, but
		nevertheless I think this message could use a bit more explanation.
						
						--Greg
	*/

	if (msg)
	{
		char	buffer[200];
		string	format (Platform::GetString (kStr_EmulatorOff));
		sprintf (buffer, format.c_str (), msg);
		string	app (Platform::GetString (kStr_AppName));

		Platform::CommonDialog (app.c_str (), buffer, Errors::kErrorAlert | Errors::kOK);
	}
}
