/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "DebugMgr.h"

#include "ATraps.h"				// ATrap
#include "Byteswapping.h"		// Canonical
#include "CPU_MT.h"				// CPUStopper
#include "CPU_REG.h"			// LowMem_GetGlobal, LowMem_SetGlobal, LowMem::GetTrapAddress
#include "Crc.h"				// Crc16CalcBlock
#include "ErrorHandling.h"		// kError_NoError, ReportUnhandledException
#include "Logging.h"			// gErrLog
#include "MetaMemory.h"			// MetaMemory::MarkCPUBreak
#include "Platform.h"			// Platform::SendPacket
#include "RAM_ROM.h"			// CEnableFullAccess
#include "SessionFile.h"		// SessionFile
#include "SLP.h"				// SLP::EventCallback
#include "SocketMessaging.h"	// CSocket::Write
#include "StringData.h"			// kExceptionNames
#include "Strings.r.h"			// kStr_ values
#include "SystemPacket.h"		// SystemPacket::
#include "TrapPatches.h"		// RemoveAllTailpatches, InstallAllTailpatches
#include "UAE_Utils.h"			// uae_memcpy

#define kOpcode_ADD		0x0697	// ADD.L X, (A7)
#define kOpcode_LINK	0x4E50
#define kOpcode_RTE		0x4E73
#define kOpcode_RTD		0x4E74
#define kOpcode_RTS		0x4E75
#define kOpcode_JMP		0x4ED0


// Types

#pragma mark Types

#define PacketName(command) (kPacketNames[(command) & 0x7F])

// Functions

#pragma mark Functions

static uae_u32		PrvGetAddressRegister	(int num);
static uae_u32		PrvGetDataRegister		(int num);
static Bool			PrvBPEquals				(uae_u32 a, uae_u32 b);
static Bool			PrvBPNotEquals			(uae_u32 a, uae_u32 b);
static Bool			PrvBPGreater			(uae_u32 a, uae_u32 b);
static Bool			PrvBPGreaterOrEqual		(uae_u32 a, uae_u32 b);
static Bool			PrvBPLess				(uae_u32 a, uae_u32 b);
static Bool			PrvBPLessOrEqual		(uae_u32 a, uae_u32 b);
static const char*	PrvSkipWhite			(const char* p);

static Bool			PrvParseDecimal			(const char **ps, int *i);
static Bool			PrvParseUnsigned		(const char **ps, uae_u32 *u);


struct NamedConditionType
{
	const char*	name;
	compareFun	function;
};

static NamedConditionType kConditions[] =
{
	{"==",	&PrvBPEquals},
	{"!=",	&PrvBPNotEquals},
	{">",	&PrvBPGreater},
	{">=",	&PrvBPGreaterOrEqual},
	{"<",	&PrvBPLess},
	{"<=",	&PrvBPLessOrEqual},
	{NULL,	NULL}
};



#define PRINTF	if (!LogHLDebugger ()) ; else LogAppendMsg

#pragma mark -


// ===========================================================================
//		 Debug
// ===========================================================================

DebugGlobalsType		gDebuggerGlobals;

// If we're listening for a debugger connection, the first three sockets
// contain references to those listening entities.	As soon as a connection
// is made, we set gConnectedDebugSocket and NULL out the other three,
// deleting any listening sockets that are no longer needed.

static CSocket*			gDebuggerSocket1;	 // TCP socket on user-defined port
static CSocket*			gDebuggerSocket2;	 // TCP socket on port 2000
static CSocket*			gDebuggerSocket3;	 // Platform-specific socket
static CTCPSocket*		gConnectedDebugSocket;


/***********************************************************************
 *
 * FUNCTION:	Debug::Startup
 *
 * DESCRIPTION: Create all the sockets we'll need for the application
 *				and start them listening for clients.  Call this once
 *				at application startup.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Startup (void)
{
	Debug::CreateListeningSockets ();
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Shutdown
 *
 * DESCRIPTION: Delete the sockets we use.	CSocket::Shutdown would do
 *				this anyway, but it's a good idea to do it here, too,
 *				so as to drop the references in our global variables.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Shutdown (void)
{
	Debug::DeleteListeningSockets ();

	if (gConnectedDebugSocket)
	{
		gConnectedDebugSocket->Close ();
		delete gConnectedDebugSocket;
		gConnectedDebugSocket = NULL;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::ConnectedToTCPDebugger
 *
 * DESCRIPTION: Return whether or not we're connected to a debugger (or
 *				reasonably think so; we don't check for aborted
 *				or broken connections).  This function is called during
 *				Palm OS bootup to determine if the 'gdbS' feature needs
 *				to be installed.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	True if we were connected to a debugger the last time
 *				we checked.
 *
 ***********************************************************************/

Bool Debug::ConnectedToTCPDebugger (void)
{
	return Debug::GetTCPDebuggerSocket () != NULL;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::GetTCPDebuggerSocket
 *
 * DESCRIPTION: Return the socket that's connected to the debugger (if
 *				any).
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The pointer to the CSocket object connected to an
 *				external debugger.	Returns NULL if not connected to
 *				a debugger.
 *
 ***********************************************************************/

CTCPSocket* Debug::GetTCPDebuggerSocket (void)
{
	return gConnectedDebugSocket;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::GetDebuggerSocket
 *
 * DESCRIPTION: Return the socket that's connected to the debugger (if
 *				any).
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The pointer to the CSocket object connected to an
 *				external debugger.	Returns NULL if not connected to
 *				a debugger.
 *
 ***********************************************************************/

CSocket* Debug::GetDebuggerSocket (void)
{
	// Get the preferred (TCP) socket.

	CSocket* s = Debug::GetTCPDebuggerSocket ();

	// If we don't have one (we're not connected via such a socket),
	// try the backward-compatible, platform-specific socket.

	if (s == NULL)
		s = gDebuggerSocket3;

	// If we finally have a socket, return it.

	return s;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Initialize
 *
 * DESCRIPTION: Initialize our debugging hooks.	 Sets flags to tell the
 *				CPU loop to exit if any of the exceptions that we're
 *				interested in are encountered.	If an exception is
 *				encountered, the emulator calls Debug::EnterDebugger.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

/*-----------------------------------------------------------------------------------*\

	Notes on side-stepping the ROM debugger:

		The debugger manipulates the following low-memory globals:
		
		Byte	dbgWasEntered;		// set true the first time debugger is entered
		Byte	dbgInDebugger;		// true if in debugger
		Byte	dbgTracing;		// tracing in debugger
		Ptr	dbgGlobalsP;		// pointer to dbgGlobals
		Ptr	dbgSerGlobalsP;		// pointer to Debugger Serial globals

		We set dbgWasEntered and dbgInDebugger in EnterDebugger.  We clear
		dbgInDebugger in ExitDebugger.	Both of these values are examined by
		other parts of the ROM.
		
		dbgTracing is set when the ROM debugger needs to single step.  It needs
		to single-step if there are breakpoints in the ROM, or if we need to
		execute over an installed breakpoint.  We have other ways to do these
		(we can actually install breakpoints into the ROM, and our CPU loop has
		a mechanism for fetching the opcode that is currently overwritten by
		a breakpoint), so we never have to set dbgTracing to true.

		dbgGlobalsP appears to be accessed only by the ROM debugger code.  Since
		we should intercept everything that would cause that code to be accessed,
		we shouldn't have to do anything with that block.

		dbgSerGlobalsP is initialized by DbgInit.  None of the ROM debugger code
		looks at it.  Only the serial code does.

	May want to patch the following traps:

		ConGetS
		ConPutS
		
		sysTrapDbgSrcMessage
		sysTrapDbgMessage
		sysTrapDbgGetMessage
		sysTrapDbgCommSettings

\*-----------------------------------------------------------------------------------*/

void Debug::Initialize (void)
{
	memset (&gDebuggerGlobals, 0, sizeof (gDebuggerGlobals));

	// Install functions that will cause the CPU loop to exit if we're
	// connected to an external debugger.

	Software::InstallExceptionHandler (kException_BusErr, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_AddressErr, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_IllegalInstr, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_DivideByZero, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_Chk, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_Trap, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_Privilege, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_Trace, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_ATrap, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_FTrap, Debug::BreakIfConnected);
	Software::InstallExceptionHandler (kException_AutoVec7, Debug::BreakIfConnected);

	Software::InstallExceptionHandler (kException_Trap0 + sysDbgBreakpointTrapNum, Debug::BreakIfConnected);

	// This one's a little different.  We want to exit the CPU loop
	// on a TRAP 8, but we want to do it conditionally.  If desired,
	// HandleTrap8 will cause the CPU loop to exit (leaving the PC just
	// after the TRAP instruction).	 Otherwise, it will treat the TRAP
	// like a NOP.

	Software::InstallExceptionHandler (kException_Trap0 + sysDbgTrapNum,
									   Debug::HandleTrap8);

#if 0	// !!! TBD

	// Install Debugger Traps into trap table
	dispatchTableP[ SysTrapIndex (sysTrapDbgSrcMessage) ] = (Ptr)DbgMessage;
	dispatchTableP[ SysTrapIndex (sysTrapDbgMessage) ] = (Ptr)DbgMessage;
	dispatchTableP[ SysTrapIndex (sysTrapDbgGetMessage) ] = (Ptr)DbgGetMessage;
	dispatchTableP[ SysTrapIndex (sysTrapDbgCommSettings) ] = (Ptr)DbgCommSettings;

#endif
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Reset (void)
{
	gDebuggerGlobals.firstEntrance	= true;
	gDebuggerGlobals.inDebugger = false;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Save
 *
 * DESCRIPTION: Standard save function.	 Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Save (SessionFile& f)
{
	const long	kCurrentVersion = 1;

	Chunk		chunk;
	ChunkStream s (chunk);

	s << kCurrentVersion;

	s << gDebuggerGlobals.inDebugger;
	s << gDebuggerGlobals.ignoreDbgBreaks;
	s << gDebuggerGlobals.reEntered;
	s << gDebuggerGlobals.firstEntrance;
	s << gDebuggerGlobals.stepSpy;
	s << gDebuggerGlobals.breakingOnATrap;
	s << gDebuggerGlobals.continueOverATrap;
	s << gDebuggerGlobals.continueOverBP;

	s << gDebuggerGlobals.checkTrapWordOnExit;
	s << gDebuggerGlobals.trapWord;
	s << gDebuggerGlobals.refNum;

	int ii;
	for (ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		s << (uaecptr) gDebuggerGlobals.bp[ii].addr;
		s << gDebuggerGlobals.bp[ii].enabled;
		s << gDebuggerGlobals.bp[ii].installed;

		s << gDebuggerGlobals.bpOpcode[ii];

		if (gDebuggerGlobals.bpCondition[ii])
		{
			// Save only the source string; we can reconstruct
			// the rest of the fields from that.

			s << gDebuggerGlobals.bpCondition[ii]->source;
		}
		else
		{
			s << "";
		}
	}

	for (ii = 0; ii < dbgTotalTrapBreaks; ++ii)
	{
		s << gDebuggerGlobals.trapBreak[ii];
		s << gDebuggerGlobals.trapParam[ii];
	}

	s << gDebuggerGlobals.ssAddr;
	s << gDebuggerGlobals.ssValue;

	s << gDebuggerGlobals.excType;

	s << gDebuggerGlobals.watchEnabled;
	s << gDebuggerGlobals.watchAddr;
	s << gDebuggerGlobals.watchBytes;

	f.WriteDebugInfo (chunk);
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Load
 *
 * DESCRIPTION: Standard load function.	 Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Load (SessionFile& f)
{
	Chunk	chunk;
	if (f.ReadDebugInfo (chunk))
	{
		long		version;
		ChunkStream s (chunk);

		s >> version;

		if (version >= 1)
		{
			s >> gDebuggerGlobals.inDebugger;
			s >> gDebuggerGlobals.ignoreDbgBreaks;
			s >> gDebuggerGlobals.reEntered;
			s >> gDebuggerGlobals.firstEntrance;
			s >> gDebuggerGlobals.stepSpy;
			s >> gDebuggerGlobals.breakingOnATrap;
			s >> gDebuggerGlobals.continueOverATrap;
			s >> gDebuggerGlobals.continueOverBP;

			s >> gDebuggerGlobals.checkTrapWordOnExit;
			s >> gDebuggerGlobals.trapWord;
			s >> gDebuggerGlobals.refNum;

			int ii;
			for (ii = 0; ii < dbgTotalBreakpoints; ++ii)
			{
				uaecptr temp;
				s >> temp; gDebuggerGlobals.bp[ii].addr = (Ptr) temp;
				s >> gDebuggerGlobals.bp[ii].enabled;
				s >> gDebuggerGlobals.bp[ii].installed;

				s >> gDebuggerGlobals.bpOpcode[ii];

				string	source;
				s >> source;

				if (source.size () > 0)
				{
					BreakpointCondition*	bc = NewBreakpointCondition (source.c_str ());
					gDebuggerGlobals.bpCondition[ii] = bc;
				}
			}

			for (ii = 0; ii < dbgTotalTrapBreaks; ++ii)
			{
				s >> gDebuggerGlobals.trapBreak[ii];
				s >> gDebuggerGlobals.trapParam[ii];
			}

			s >> gDebuggerGlobals.ssAddr;
			s >> gDebuggerGlobals.ssValue;

			s >> gDebuggerGlobals.excType;

			s >> gDebuggerGlobals.watchEnabled;
			s >> gDebuggerGlobals.watchAddr;
			s >> gDebuggerGlobals.watchBytes;
		}
	}
	else
	{
		f.SetCanReload (false);
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	Debug::HandleNewPacket
 *
 * DESCRIPTION: Completely handle a packet sent from an external
 *				debugger, setting any state and sending a reply if
 *				necessary.
 *
 * PARAMETERS:	slp - a reference to a SerialLink Protocol object that
 *					contains the packet information and the horse...uh,
 *					socket it rode in on.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode Debug::HandleNewPacket (SLP& slp)
{
	ErrCode result = kError_NoError;

	PRINTF ("Entering Debug::HandleNewPacket.");

	const SlkPktHeaderType& header	= slp.Header ();
	const SysPktBodyType&	body	= slp.Body ();
	const SlkPktFooterType& footer	= slp.Footer ();

	assert ((header.dest == slkSocketDebugger) || (header.dest == slkSocketConsole));

	// If the packet is for the debugger socket, then we pretty much need
	// to have the CPU stopped.  If not, when the PalmDebugger user executes
	// "g", the restored registers may not be what the machine expects.
	//
	// As a courtesy, we stop the CPU if we get a "state" packet.  That way,
	// the user can enter the debugger by typing "att" in the PalmDebugger and
	// doesn't have to fuss with shortcut-.1.
	//
	// Otherwise, we ignore the packet.

	if (header.dest == slkSocketDebugger)
	{
		if (!Debug::InDebugger ())
		{
			if (body.command == sysPktStateCmd)
			{
				// This will send the state packet as well as set our
				// "in the debugger" state.

				result = Debug::EnterDebugger (kException_Trap0 + sysDbgTrapNum, m68k_getpc (), &slp);
				goto Exit;
			}

			PRINTF ("Packet is for the debugger, we're not in debug mode, and the packet is not a state command; leaving Debug::HandleNewPacket.");

			// This packet is for the debugger socket, we're not in the
			// debugger, and the packet is not a "state" command, so just
			// ignore this request.

			goto Exit;
		}
	}

	{
		CEnableFullAccess	munge;	// Remove blocks on memory access.

		// Goose the ROM, so it doesn't go to sleep while we're exchanging packets.
		// Do this by hand; we can't call ROM functions as subroutines while in
		// the debugger (we don't know what state the ROM is in).

		//	EvtResetAutoOffTimer ();
		LowMem_SetGlobal (sysAutoOffEvtTicks, LowMem_GetGlobal (hwrCurTicks));

		switch (body.command)
		{
			case sysPktStateCmd:
				result = SystemPacket::SendState (slp);
				break;

			case sysPktReadMemCmd:
				result = SystemPacket::ReadMem (slp);
				break;

			case sysPktWriteMemCmd:
				result = SystemPacket::WriteMem (slp);
				break;

			case sysPktSingleStepCmd:
				// I don't think there's anything to do here.  I think that
				// single-stepping is performed by the debugger setting the
				// trace bit in the SR and sending a sysPktContinueCmd.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktGetRtnNameCmd:
				result = SystemPacket::SendRoutineName (slp);
				break;

			case sysPktReadRegsCmd:
				result = SystemPacket::ReadRegs (slp);
				break;

			case sysPktWriteRegsCmd:
				result = SystemPacket::WriteRegs (slp);
				break;

			case sysPktContinueCmd:
				result = SystemPacket::Continue (slp);
				break;

			case sysPktRPCCmd:
				result = SystemPacket::RPC (slp);
				break;

			case sysPktGetBreakpointsCmd:
				result = SystemPacket::GetBreakpoints (slp);
				break;

			case sysPktSetBreakpointsCmd:
				result = SystemPacket::SetBreakpoints (slp);
				break;

//			case sysPktRemoteUIUpdCmd:
//				// Sent TO debugger; never received FROM debugger.
//				break;

//			case sysPktRemoteEvtCmd:
//				// Sent TO debugger; never received FROM debugger.
//				break;

			case sysPktDbgBreakToggleCmd:
				result = SystemPacket::ToggleBreak (slp);
				break;

			case sysPktFlashCmd:
				// Not supported in this release.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktCommCmd:
				// Not supported in this release.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktGetTrapBreaksCmd:
				result = SystemPacket::GetTrapBreaks (slp);
				break;

			case sysPktSetTrapBreaksCmd:
				result = SystemPacket::SetTrapBreaks (slp);
				break;

			case sysPktGremlinsCmd:
				// Not supported in this release.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktFindCmd:
				result = SystemPacket::Find (slp);
				break;

			case sysPktGetTrapConditionsCmd:
				result = SystemPacket::GetTrapConditions (slp);
				break;

			case sysPktSetTrapConditionsCmd:
				result = SystemPacket::SetTrapConditions (slp);
				break;

			case sysPktChecksumCmd:
				// Not supported in this release.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktExecFlashCmd:
				// Not supported in this release.
				PRINTF ("   Not supported in this release.");
				break;

			case sysPktRemoteMsgCmd:
				// Nothing for us to do.  With a serial link,
				// these are sent to clear the input port.
				PRINTF ("   Should not be *receiving these!");
				break;

			case 0xFF:
			{
				// This is a special comand only for Poser

				char fn[256];
				strcpy (fn, (char*) &body.data[14]);

				FileRefList	fileList;
				fileList.push_back (fn);

				::LoadPalmFileList (fileList);
				break;
			}

			default:
				break;
		}
	}

 Exit:
	PRINTF ("Exiting Debug::HandleNewPacket.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::BreakIfConnected
 *
 * DESCRIPTION: Handle the given exception.  If we're connected to a
 *				debugger, we want to let it handle it, so change the CPU
 *				state and return TRUE (to say that we handled the
 *				execption).  Otherwise return false to perform default
 *				handling (which means handing it off to the emulated ROM).
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Debug::BreakIfConnected (uae_s32 exceptionNumber, uaecptr oldpc)
{
	// Dump any logged information to a file.

	LogDump ();

	// Attempt to enter the debugger.  If we succeed, set the break-reason
	// so that the CPU loop exits.	Then return true to say that we
	// handled the exception.

	if (Debug::EnterDebugger (exceptionNumber, oldpc, NULL) == kError_NoError)
	{
		Emulator::SetBreakReason (exceptionNumber);

		return true;
	}

	// We were unable to enter the debugger (perhaps there was no external
	// debugger running).  Return false to cause normal processing to occur
	// (which is probably to let the emulated ROM handle the exception,
	// either by contacting a debugger over the serial port, or by showing
	// a fatal exception error).

//	return false;

	// Ohhh...let's not do that.  Letting the ROM debug stubs take over isn't
	// really a good idea; they tend to hang the ROM, and the user wouldn't
	// like that.  Instead, let's show a dialog that lets them Reset.

	int button;
	uae_u16 opcode = (oldpc & 1) ? 0 : get_word (oldpc);	// Protect against odd PC.
	if (exceptionNumber == kException_ATrap &&
		(opcode == 0xA9EB || opcode == 0xA9EC || opcode == 0xA9EE))
	{
		button = Errors::ReportSANEUsage ();
	}
	else
	{
		button = Errors::ReportUnhandledException (exceptionNumber);
	}

	// (Exception is thrown if button == kDebug or kReset).

	Emulator::HandleDlgButton (button, oldpc);

	return true;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::HandleTrap8
 *
 * DESCRIPTION: The CPU just encountered a TRAP 8, which is what gets
 *				compiled into the developer's code when he "calls" the
 *				DbgBreak or DbgSrcBreak "function".  HandleTrap8
 *				determines whether or not this TRAP 8 should be handled
 *				as an exception (in which case we'd enter the debugger)
 *				or should be skipped like a NOP. If "ignoreDbgBreaks" is
 *				true, then merely return TRUE to say that we completely
 *				handled the exception.	Otherwise, change the CPU state
 *				to non-running before we return TRUE.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Debug::HandleTrap8 (uae_s32 exceptionNumber, uaecptr oldpc)
{
	if (gDebuggerGlobals.ignoreDbgBreaks)
	{
		return true;
	}

	return Debug::BreakIfConnected (exceptionNumber, oldpc);
}


/***********************************************************************
 *
 * FUNCTION:	Debug::HandleSystemCall
 *
 * DESCRIPTION: An A-Trap is about to be executed.	This function is
 *				called to determine if we want to break on this A-Trap.
 *				If there are A-Trap breaks registered with the debugger,
 *				and if the "continueOverATrap" flag is not set (if it were set,
 *				that would mean that we'd just exited the debugger, and
 *				don't need to re-break on the A-Trap that caused us to
 *				enter the debugger), then scan the break table to see if
 *				the A-Trap we're about to execute is one we want to
 *				break on.
 *
 *				If it is, set the CPU break reason to kException_Trace,
 *				which will cause the CPU to break at the end of this
 *				opcode.	 Also return TRUE, which tells the normal A-Trap
 *				handler that we completely handled this A-Trap, and that
 *				it doesn't need to do any trap dispatcher stuff.  Finally,
 *				back-up the PC so that it points to the A-Trap again so
 *				that when we resume execution we'll start at that A-Trap.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Debug::HandleSystemCall (const SystemCallContext& context)
{
	Bool	doBreak = false;

	if (!gDebuggerGlobals.continueOverATrap && gDebuggerGlobals.breakingOnATrap)
	{
		doBreak = Debug::MustBreakOnTrapSystemCall (context.fTrapWord, context.fExtra);

		// If we're supposed to break here and we can successfully contact an
		// external debugger, set the CPU state so that the CPU loop breaks at
		// the end of the current instruction.	Also back up the PC to point to
		// the TRAP instruction instead of the $Axxx selector.
		//
		// Ooops...gotta make sure we fudge the PC _before_ sending the state
		// to the debugger...

		if (doBreak)
		{
			uaecptr curpc = m68k_getpc ();
			m68k_setpc (context.fPC);

			if (Debug::EnterDebugger (kException_Trace, context.fPC, NULL) == kError_NoError)
			{
				Emulator::SetBreakReason (kException_Trace);

				// Check again on the way to to see if there's still
				// a breakpoint at this location.  If so, we need to
				// set *another* flag to make sure we skip over the
				// breakpoint.	Otherwise, we'd just break here over
				// and over again.

				gDebuggerGlobals.checkTrapWordOnExit = true;
				gDebuggerGlobals.trapWord = context.fTrapWord;
				gDebuggerGlobals.refNum = context.fExtra;
			}
			else
			{
				m68k_setpc (curpc);
			}
		}
	}

	// Clear the flag telling us to inhibit the ATrapBreak check.

	gDebuggerGlobals.continueOverATrap = false;

	return doBreak;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::EnterDebugger
 *
 * DESCRIPTION: Put the emulator into "debug mode".  This pretty much
 *				consists of setting up a bunch of flags, figuring out
 *				why we entered the debugger, and telling the external
 *				debugger that we've entered debug mode and what the
 *				current CPU state is.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

//
//	Reasons for entering the debugger:
//
//		Single step:
//			reason == kException_Trace
//
//			Normal processing.
//
//		Step spy:
//			reason == kException_Trace
//
//			Normal processing.
//
//		User breakpoint (temporary or otherwise):
//			reason == kException_Trap0 + sysDbgBreakpointTrapNum
//
//			Normal processing.
//
//		Compiled breakpoint (DbgBreak, DbgSrcBreak):
//			reason == kException_Trap0 + sysDbgTrapNum
//
//			Ignore if DbgGlobalsType.ignoreDbgBreaks is true.
//			
//			Actually, now we catch this one a little earlier.  In
//			Software_ProcessException, we call an installed exception
//			handler that leads to calling Debug::HandleTrap8.  That
//			function will return TRUE if ignoreDbgBreaks is true, which
//			will effectively turn the TRAP 8 into a NOP.  Otherwise, it
//			returns false, which will lead to EnterDebugger being called
//			to handle the exception.
//
//		"A-Trap" break:
//			reason == kException_Trace
//
//			When looking for an A-Trap to break on, the debugger uses
//			a special trap dispatcher.	Called on every TRAP F, when it
//			finds a trap to break on, it sets DbgGlobalsType.breakingOnATrap,
//			sets the tracing SR bit, and then calls the original trap
//			dispatcher.
//
//			Because the trap dispatcher is run in supervisor mode, the
//			tracing bit doesn't do anything until it RTE's.
//
//			Normal processing.
//
//		Exception (bus error, address error, etc):
//			reason == exception code
//
//			Normal processing.
//
//	The debugger is silently entered if reason == kException_Trace.
//	Otherwise, it displays a message saying what exception occured.

ErrCode Debug::EnterDebugger (uae_s32 reason, uaecptr startPC, SLP* slp)
{
	PRINTF ("Entering Debug::EnterDebugger.");

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// If we entered due to a breakpoint (TRAP 0), backup the PC.

	uaecptr curpc = m68k_getpc ();
	if (reason == (kException_Trap0 + sysDbgBreakpointTrapNum))
	{
		m68k_setpc (startPC);
	}

	// Save the reason for our entering the debugger.

	gDebuggerGlobals.excType = (Word) (reason * 4);

	// If we're re-entering the debugger, note that fact.

	gDebuggerGlobals.reEntered = LowMem_GetGlobal (dbgInDebugger) != 0;

	// Turn off sound in case it was on when the debugger was entered.

	HWRegisters::TurnSoundOff ();

	// Send a state message packet notifying the host that we've entered.
	// If we can't send the packet (perhaps because there is no external
	// debugger listening) return false saying that we failed to enter
	// the debugger.  When we return false, we indicate that perhaps the
	// emulated ROM debugger should take over.
	//
	// Concurrency note: in the multi-threaded version of the emulator,
	// EnterDebugger will be called from one thread (the CPU thread),
	// while the UI thread will be listening for packets from the
	// external debugger.  When SendState sends the packet from this
	// thread, the external debugger could receive it and send us back
	// a message immediately.  The question is, is this OK?	 Are there
	// problems with the UI thread trying to debug us before the CPU
	// thread has had a chance to stop itself? Let's see: the UI thread
	// will receive the message, and start to handle it by telling the
	// CPU thread to stop.	That means that we should have a chance to
	// clean up (set our flags, remove breakpoints, etc.) and stop before
	// the UI thread tries to debug us.

	ErrCode result = kError_NoError;

	if (!slp)
	{
		CSocket*	debuggerSocket = Debug::GetDebuggerSocket ();
		if (debuggerSocket)
		{
			SLP	newSLP (debuggerSocket);
			result = SystemPacket::SendState (newSLP);
		}
		else
		{
			result = 1;	// !!! Need a not "connected to debugger" error!
		}
	}
	else
	{
		result = SystemPacket::SendState (*slp);
	}

	if (result == kError_NoError)
	{
		// Flag to the ROM that we're in the debugger. "dbgInDebugger" gets
		// cleared in ExitDebugger. "dbgWasDebugger" stays set to true.

		LowMem_SetGlobal (dbgWasEntered, true);
		LowMem_SetGlobal (dbgInDebugger, true);
		gDebuggerGlobals.inDebugger = true;

		PRINTF ("Entered debug mode.");
	}
	else
	{
		// Reverse the pc-munging that we did earlier.

		m68k_setpc (curpc);

		PRINTF ("Failed to enter debug mode.");
	}

	PRINTF ("Exiting Debug::EnterDebugger.");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::ExitDebugger
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode Debug::ExitDebugger (void)
{
	CEnableFullAccess	munge;	// Remove blocks on memory access.

	// If we're continuing, but we're on a breakpoint, set a boolean
	// that causes us to ignore the breakpoint when we hit it.

	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		if (gDebuggerGlobals.bp[ii].enabled == false)
			continue;

		uaecptr addr = (uaecptr) gDebuggerGlobals.bp[ii].addr;
		if (!addr)
			continue;

		if (addr == m68k_getpc ())
			gDebuggerGlobals.continueOverBP = true;
	}

	// Check to see if we (a) stopped here because of a request
	// to break on a particular system call and (b) that we're
	// still requested to break on this call.  If so, set a flag
	// to skip over the next break-on-system-call.	Otherwise,
	// we'd just break at this location over and over again.

	if (gDebuggerGlobals.checkTrapWordOnExit)
	{
		gDebuggerGlobals.checkTrapWordOnExit = false;

		if (MustBreakOnTrapSystemCall (gDebuggerGlobals.trapWord,
									   gDebuggerGlobals.refNum))
		{
			// Set the flag that tells this function to not break
			// the next time it's entered.

			gDebuggerGlobals.continueOverATrap = true;
		}
	}

	// Set flags that we're no longer in the debugger.

	gDebuggerGlobals.excType = 0;
	LowMem_SetGlobal (dbgInDebugger, false);
	gDebuggerGlobals.inDebugger = false;

	return kError_NoError;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::InDebugger
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

Bool Debug::InDebugger (void)
{
	return gDebuggerGlobals.inDebugger;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::HandleCPUBreak
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::HandleCPUBreak (void)
{
	// Don't break on soft breakpoints if we're calling the ROM internally.

	if (ATrap::DoingCall ())
		return;

	// Don't break on soft breakpoints if we're exiting the debugger.

	if (gDebuggerGlobals.continueOverBP)
	{
		gDebuggerGlobals.continueOverBP = false;
		return;
	}

	ConditionalBreak (m68k_getpc ());
}


/***********************************************************************
 *
 * FUNCTION:	Debug::InstallCPUBreaks
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::InstallCPUBreaks (void)
{
	// Install the breakpoints.
	
	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		if (gDebuggerGlobals.bp[ii].enabled)
		{
			MetaMemory::MarkCPUBreak ((uaecptr) gDebuggerGlobals.bp[ii].addr);
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::RemoveCPUBreaks
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::RemoveCPUBreaks (void)
{
	// Remove the breakpoints.
	
	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		if (gDebuggerGlobals.bp[ii].enabled)
		{
			MetaMemory::UnmarkCPUBreak ((uaecptr) gDebuggerGlobals.bp[ii].addr);
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::NewBreakpointCondition
 *
 * DESCRIPTION: Create a new breakpoint condition by parsing the given
 *				source string.	Returns NULL on parse error.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

BreakpointCondition* Debug::NewBreakpointCondition (const char* sourceString)
{
	const char* source = sourceString;
	registerFun regType;
	int		regNum;
	Bool		indirect = false;
	uae_u32		indirectOffset;
	int		size = 4;

	compareFun	condition;
	int		value;

	source = PrvSkipWhite (source);
	if (!source)
		return NULL;

	if (isdigit (*source))
	{
		indirect = true;
		if (!PrvParseUnsigned (&source, &indirectOffset))
			return NULL;

		source = PrvSkipWhite (source);

		if (*source != '(')
			return NULL;

		++source;
		source = PrvSkipWhite (source);
	}

	switch (tolower (*source++))
	{
	case 'd':
		regType = PrvGetDataRegister;
		break;

	case 'a':
		regType = PrvGetAddressRegister;
		break;

	default:
		return NULL;
	}

	source = PrvSkipWhite (source);

	if (!PrvParseDecimal (&source, &regNum))
		return NULL;

	source = PrvSkipWhite (source);

	if (indirect)
	{
		if (*source != ')')
			return NULL;

		++source;
		source = PrvSkipWhite (source);
	}

	if (*source == '.')
	{
		++source;

		switch (*source)
		{
		case 'b': size = 1; break;
		case 'w': size = 2; break;
		case 'l': size = 4; break;
		default: return NULL;
		}

		++source;
		source = PrvSkipWhite (source);
	}

	condition = NULL;
	for (NamedConditionType* c = kConditions; c->name; ++c)
	{
		if (!strncmp (source, c->name, strlen (c->name)))
		{
			condition = c->function;
			source += strlen (c->name);
			break;
		}
	}

	if (!condition)
		return NULL;

	if (sscanf (source, "%i", &value) != 1)
		return NULL;

	BreakpointCondition* bc = new BreakpointCondition;

	bc->regType		= regType;
	bc->regNum			= regNum;
	bc->indirect		= indirect;
	bc->indirectOffset	= indirectOffset;
	bc->size			= size;

	bc->condition		= condition;
	bc->value			= value;
	bc->source			= _strdup (sourceString);

	return bc;
}


/***********************************************************************
 *
 * FUNCTION:	Debug::SetBreakpoint
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::SetBreakpoint (int index, uaecptr addr, BreakpointCondition* c)
{
	ClearBreakpoint (index);

	gDebuggerGlobals.bp[index].enabled		= true;
	gDebuggerGlobals.bp[index].installed	= false;
	gDebuggerGlobals.bp[index].addr			= (Ptr) addr;
	gDebuggerGlobals.bpCondition[index]	= c;

	Emulator::InstallCPUBreaks ();
}


/***********************************************************************
 *
 * FUNCTION:	Debug::ClearBreakpoint
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::ClearBreakpoint (int index)
{
	gDebuggerGlobals.bp[index].enabled = false;

	if (gDebuggerGlobals.bpCondition[index])
	{
		if (gDebuggerGlobals.bpCondition[index]->source)
			free (gDebuggerGlobals.bpCondition[index]->source);

		delete gDebuggerGlobals.bpCondition[index];
		gDebuggerGlobals.bpCondition[index] = NULL;
	}

	Emulator::InstallCPUBreaks ();
}


#pragma mark -

/***********************************************************************
 *
 * FUNCTION:	Debug::MustBreakOnTrapSystemCall
 *
 * DESCRIPTION: Test the given trapWord (and optional refNum) to see
 *				if this is a combination that we expect to break on.
 *
 * PARAMETERS:	trapWord - the system function dispatch number (Axxx)
 *					to test.
 *
 *				refNum - the library reference number to test.	This
 *					value is used only if trapWord is in the range for
 *					library function calls.
 *
 * RETURNED:	True if we should break on this combination.
 *
 ***********************************************************************/

Bool Debug::MustBreakOnTrapSystemCall (uae_u16 trapWord, uae_u16 refNum)
{
	Bool	doBreak = false;
	uae_u16 trapIndex = SysTrapIndex (trapWord);

	// Do different compares for system traps and library traps.

	if (IsSystemTrap (trapWord))
	{
		for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		{
			if (trapIndex == gDebuggerGlobals.trapBreak[ii])
			{
				doBreak = true;
				break;
			}
		}
	}
	else
	{
		for (int ii = 0; ii < dbgTotalTrapBreaks; ++ii)
		{
			if (trapIndex == gDebuggerGlobals.trapBreak[ii] &&
				refNum == gDebuggerGlobals.trapParam[ii])
			{
				doBreak = true;
				break;
			}
		}
	}

	return doBreak;
}


////// breakpoint conditions

uae_u32 PrvGetAddressRegister (int num)
{
	return m68k_areg (regs, num);
}

uae_u32 PrvGetDataRegister (int num)
{
	return m68k_dreg (regs, num);
}

Bool PrvBPEquals (uae_u32 a, uae_u32 b)
{
	return a == b;
}

Bool PrvBPNotEquals (uae_u32 a, uae_u32 b)
{
	return a != b;
}

// Comparisons are unsigned for now.  Would it be more useful if they were signed?

Bool PrvBPGreater (uae_u32 a, uae_u32 b)
{
	return a > b;
}

Bool PrvBPGreaterOrEqual (uae_u32 a, uae_u32 b)
{
	return a >= b;
}

Bool PrvBPLess (uae_u32 a, uae_u32 b)
{
	return a < b;
}

Bool PrvBPLessOrEqual (uae_u32 a, uae_u32 b)
{
	return a <= b;
}

const char* PrvSkipWhite (const char* p)
{
	while (p && isspace (*p))
		++p;

	return p;
}

/* Parse a signed decimal integer */

Bool PrvParseDecimal (const char **ps, int *i)
{
	const char *s = *ps;

	if (sscanf (s, "%d", i) != 1)
		return false;

	while (isdigit (*s))
		++s;

	*ps = s;

	return true;
}

/* Parse an unsigned integer which may be either decimal or hex (e.g. "0xabc")
 */
Bool PrvParseUnsigned (const char **ps, uae_u32 *u)
{
	const char *s = *ps;

	if (sscanf (s, "%li", u) != 1)
		return false;

	if (s[0] == '0' && (s[1] == 'x' || s[1] == 'X'))	/* hex */
	{
		s += 2;
		while (isxdigit (*s))
			++s;
	}
	else /* decimal */
	{
		while (isdigit (*s))
			++s;
	}

	*ps = s;

	return true;
}

Bool BreakpointCondition::Evaluate (void)
{
	uae_u32 r = regType (regNum);

	if (indirect)
	{
		if (size == 4)
			r = get_long (r + indirectOffset);
		else if (size == 2)
			r = get_word (r + indirectOffset);
		else if (size == 1)
			r = get_byte (r + indirectOffset);
		else
			return false;
	}
	else
	{
		if (size == 2)
			r = (uae_u16) r;
		else if (size == 1)
			r = (uae_u8) r;
	}

	return condition (r, value);
}


/***********************************************************************
 *
 * FUNCTION:	Debug::ConditionalBreak
 *
 * DESCRIPTION: .
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::ConditionalBreak (uaecptr startPC)
{
	Ptr bpAddress = (Ptr) startPC;

	// Loop over the breakpoints to see if we've hit one.

	for (int ii = 0; ii < dbgTotalBreakpoints; ++ii)
	{
		// This breakpoint is not enabled, so just skip over it.

		if (!gDebuggerGlobals.bp[ii].enabled)
			continue;

		// This breakpoint is not for this PC, so just skip over it.

		if (gDebuggerGlobals.bp[ii].addr != bpAddress)
			continue;

		// If there is a condition for this breakpoint but it evaluates
		// to false, just skip over the breakpoint.

		BreakpointCondition*	bc = gDebuggerGlobals.bpCondition[ii];

		if (bc && bc->Evaluate () == false)
			continue;

		// Clear temporary breakpoint if we hit it.

		if (bpAddress == gDebuggerGlobals.bp[dbgTempBPIndex].addr)
		{
			gDebuggerGlobals.bp[dbgTempBPIndex].enabled = false;
		}

		// Break into the debugger.
		BreakIfConnected (kException_Trap0, startPC);
		break;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::CreateListeningSockets
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

static void PrvFireUpSocket (CSocket*& s)
{
	if (s)
	{
		if (s->Open () != errNone)
		{
			s->Delete();
			s = NULL;
		}
	}
}

void Debug::CreateListeningSockets (void)
{
	Preference<long>	portPref (kPrefKeyDebuggerSocketPort);

	assert (gDebuggerSocket1 == NULL);
	assert (gDebuggerSocket2 == NULL);
	assert (gDebuggerSocket3 == NULL);

	if (*portPref != 0)
	{
		gDebuggerSocket1 = new CTCPSocket (&Debug::EventCallback, *portPref);
		gDebuggerSocket2 = new CTCPSocket (&Debug::EventCallback, 2000);
	}

	gDebuggerSocket3 = Platform::CreateDebuggerSocket ();

	::PrvFireUpSocket (gDebuggerSocket1);
	::PrvFireUpSocket (gDebuggerSocket2);
	::PrvFireUpSocket (gDebuggerSocket3);
}


/***********************************************************************
 *
 * FUNCTION:	Debug::DeleteListeningSockets
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void Debug::DeleteListeningSockets (void)
{
	if (gDebuggerSocket1)
	{
		gDebuggerSocket1->Close ();
		gDebuggerSocket1->Delete();
		gDebuggerSocket1 = NULL;
	}

	if (gDebuggerSocket2)
	{
		gDebuggerSocket2->Close ();
		gDebuggerSocket2->Delete();
		gDebuggerSocket2 = NULL;
	}

	if (gDebuggerSocket3)
	{
		gDebuggerSocket3->Close ();
		gDebuggerSocket3->Delete();
		gDebuggerSocket3 = NULL;
	}
}


/***********************************************************************
 *
 * FUNCTION:	Debug::EventCallback
 *
 * DESCRIPTION: Callback function for TCP-based debugger-related
 *				sockets.  This function takes care of installing and
 *				removing the 'gdbS' Feature (for gdb support), and
 *				forwards debugger packets to the Debug sub-system.
 *
 * PARAMETERS:	s - the socket that connected, disconnected, or received
 *					some data.
 *
 *				event - a code indicating what happened.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void Debug::EventCallback (CSocket* s, int event)
{
	switch (event)
	{
		case CSocket::kConnected:
		{
			assert (gDebuggerSocket1 == s ||
					gDebuggerSocket2 == s ||
					gDebuggerSocket3 == s);
			assert (gConnectedDebugSocket == NULL);

			// We've connected on one of the TCP sockets we were listening
			// on.	Delete the other listening sockets so that we don't
			// connect with it, too.  We actually delete the other
			// Sockets so that we don't accidentally start listening with
			// them again (our TCPSockets are pretty tenacious when it comes
			// to auto-starting the listening process).

			if (s == gDebuggerSocket1)
			{
				gConnectedDebugSocket = (CTCPSocket*) gDebuggerSocket1;
				gDebuggerSocket1 = NULL;
			}
			else if (s == gDebuggerSocket2)
			{
				gConnectedDebugSocket = (CTCPSocket*) gDebuggerSocket2;
				gDebuggerSocket2 = NULL;
			}
			else	// s == gDebuggerSocket3
			{
				gDebuggerSocket3 = NULL;
			}

			Debug::DeleteListeningSockets ();

			// If we're listening on a socket, install the 'gdbS' feature.	The
			// existance of this feature causes programs written the the prc tools
			// to enter the debugger  when they're launched.

			if (Patches::UIInitialized ())
			{

#if !defined (__MACOS__)
				CPUStopper	stopper (kStopOnATrap);
				if (stopper.Stopped ())
#endif
				{
					::FtrSet ('gdbS', 0, 0x12BEEF34);
				}
			}
			break;
		}

		case CSocket::kDataReceived:
		{
			break;
		}

		case CSocket::kDisconnected:
		{
			// Let's start listening for a new debugger connection.

			assert (gDebuggerSocket1 == NULL);
			assert (gDebuggerSocket2 == NULL);
			assert (gDebuggerSocket3 == NULL);
			assert (gConnectedDebugSocket == s);

			gConnectedDebugSocket = NULL;
			s->Delete();

			Debug::CreateListeningSockets ();

			if (Patches::UIInitialized ())
			{
#if !defined (__MACOS__)
				CPUStopper	stopper (kStopOnATrap);
				if (stopper.Stopped ())
#endif
				{
					::FtrUnregister ('gdbS', 0);
				}
			}

			break;
		}
	}

	SLP::EventCallback (s, event);
}


