/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "HostControl.h"
#include "HostControlPrv.h"

#include "ATraps.h" 			// DoingCall
#include "CPU_REG.h"			// Emulator::SetBreakReasonMask
#include "EmulatorTypes.h"		// StringList, ByteList
#include "Hordes.h"				// Hordes::IsOn
#include "LoadApplication.h"	// LoadApplication
#include "Logging.h"			// LogFile
#include "Miscellaneous.h"		// GetDeviceTextList, GetMemoryTextList
#include "Platform.h"			// Platform::GetShortVersionString
#include "RAM_ROM.h"			// Memory::MapPhysicalMemory
#include "EmRPC.h"				// RPC::HandlingPacket, RPC::DeferCurrentPacket
#include "Startup.h"			// Startup::ScheduleNewHorde
#include "Strings.r.h"			// kStr_ProfileResults
#include "Tracing.h"			// Tracer
#include "UAE_Utils.h"			// uae_strlen, uae_strcpy


typedef void					(*HostHandler) (void);

static void 		HostInit (void);
static HostHandler	HostGetHandler (long selector);

static void 		CollectParameters (int stackOffset, const string& fmt, ByteList& stackData, StringList& stringData);
static void 		PushShort (int& callerStackOffset, ByteList& stackData);
static void 		PushLong (int& callerStackOffset, ByteList& stackData);
static void 		PushDouble (int& callerStackOffset, ByteList& stackData);
static void 		PushLongDouble (int& callerStackOffset, ByteList& stackData);
static void 		PushString (int& callerStackOffset, ByteList& stackData, StringList& stringData);

static string		ToString (uaecptr);
static FILE*		ToFILE (uaecptr);

// Write to this "file" if you want to intertwine your
// output with that created by any host logging facilities
// (such as event logging).

#define hostLogFile ((HostFILE*) -1)
#define hostLogFILE ((FILE*) -1)


// Macros for extracting parameters from the emulated stack

#define PARAMETER_SIZE(x)	\
	(sizeof (((StackFrame*) 0)->x))

#define PARAMETER_OFFSET(x) \
	(m68k_areg (regs, 7) + offsetof (StackFrame, x))

#define GET_PARAMETER(x)		\
	((PARAMETER_SIZE(x) == sizeof (char)) ? get_byte (PARAMETER_OFFSET(x)) :	\
	 (PARAMETER_SIZE(x) == sizeof (short)) ? get_word (PARAMETER_OFFSET(x)) :	\
											get_long (PARAMETER_OFFSET(x)))

inline int __fclose (FILE* f)
{
	if (f == hostLogFILE)
	{
		return 0;
	}

	return fclose (f);
}

inline int __feof (FILE* f)
{
	return feof (f);
}

inline int __ferror (FILE* f)
{
	return ferror (f);
}

inline int __fflush (FILE* f)
{
	if (f == hostLogFILE)
	{
		return 0;
	}

	return fflush (f);
}

inline int __fgetc (FILE* f)
{
	if (f == hostLogFILE)
	{
		return EOF;
	}

	return fgetc (f);
}

inline int __fgetpos(FILE* f, fpos_t* p)
{
	if (f == hostLogFILE)
	{
		return -1;
	}

	return fgetpos (f, p);
}

inline char* __fgets (char* s, int n, FILE* f)
{
	if (f == hostLogFILE)
	{
		return NULL;
	}

	return fgets (s, n, f);
}

inline int __vfprintf (FILE* f, const char* fmt, va_list args)
{
	if (f == hostLogFILE)
	{
		return LogGetStdLog ()->VPrintf (fmt, args);
	}

	return vfprintf (f, fmt, args);
}

inline int __fputc (int c, FILE* f)
{
	if (f == hostLogFILE)
	{
		return LogGetStdLog ()->Printf ("%c", c);
	}

	return fputc (c, f);
}

inline int __fputs (const char* s, FILE* f)
{
	if (f == hostLogFILE)
	{
		return LogGetStdLog ()->Printf ("%s", s);
	}

	return fputs (s, f);
}

inline size_t __fread (void* buffer, size_t size, size_t count, FILE* f)
{
	if (f == hostLogFILE)
	{
		return 0;
	}

	return fread (buffer, size, count, f);
}

inline int __fseek (FILE* f, long offset, int origin)
{
	if (f == hostLogFILE)
	{
		return -1;
	}

	return fseek (f, offset, origin);
}

inline int __fsetpos (FILE* f, const fpos_t* pos)
{
	if (f == hostLogFILE)
	{
		return -1;
	}

	return fsetpos (f, pos);
}

inline long __ftell (FILE* f)
{
	if (f == hostLogFILE)
	{
		return -1;
	}

	return ftell (f);
}

inline size_t __fwrite (const void* buffer, size_t size, size_t count, FILE* f)
{
	if (f == hostLogFILE)
	{
		return LogGetStdLog ()->Write (buffer, size * count);
	}

	return fwrite (buffer, size, count, f);
}


// The following functions define a bunch of StackFrame structs.
// These structs need to mirror the format of parameters pushed
// onto the stack by the emulated code, and so need to be packed
// to 2-byte boundaries.
//
// The pragmas are reversed at the end of the file.

#include "PalmPack.h"



HostHandler gHandlerTable [hostSelectorLastTrapNumber];


// ---------------------------------------------------------------------------
//		 HandleHostControlCall
// ---------------------------------------------------------------------------

CallROMType HandleHostControlCall (void)
{
	static int	inited;
	if (!inited)
	{
		HostInit ();
		inited = true;
	}

	uae_u16 	selector = get_word (m68k_areg (regs, 7) + 0);
	HostHandler fn = HostGetHandler (selector);

	if (fn)
	{
		fn ();
	}
	else
	{
		// Return NULL/0 for unknown selectors
		m68k_areg (regs, 0) = 0;
		m68k_dreg (regs, 0) = 0;
	}

	return kSkipROM;
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostGetHostVersion
// ---------------------------------------------------------------------------

static void _HostGetHostVersion (void)
{
	// long HostHostGetVersion (void)

	struct StackFrame
	{
		short	_selector;
	};

	enum { kMajor, kMinor, kFix, kBuild };

	int major		= 0;
	int minor		= 0;
	int fix 		= 0;
	int stage		= sysROMStageRelease;
	int buildNum	= 0;
	int state		= kMajor;

	string				version (Platform::GetShortVersionString ());
	string::iterator	iter;

	for (iter = version.begin (); iter != version.end (); ++iter)
	{
		char	ch = *iter;

		switch (state)
		{
			case kMajor:
				if (isdigit (ch))
					major = major * 10 + ch - '0';
				else if (ch == '.')
					state = kMinor;
				else
					goto VersionParseDone;
				break;

			case kMinor:
			case kFix:
				if (isdigit (ch))
				{
					if (state == kMinor)
						minor = minor * 10 + ch - '0';
					else
						fix = fix * 10 + ch - '0';
				}
				else if (ch == '.')
				{
					if (state == kMinor)
						state = kFix;
					else
						goto VersionParseDone;
				}
				else if (ch == 'd')
				{
					stage = sysROMStageDevelopment;
					state = kBuild;
				}
				else if (ch == 'a')
				{
					stage = sysROMStageAlpha;
					state = kBuild;
				}
				else if (ch == 'b')
				{
					stage = sysROMStageBeta;
					state = kBuild;
				}
				else
					goto VersionParseDone;
				break;

			case kBuild:
				if (isdigit (ch))
					buildNum = buildNum * 10 + ch - '0';
				else
					goto VersionParseDone;
				break;
		}
	}

VersionParseDone:
	
	m68k_dreg (regs, 0) = sysMakeROMVersion (major, minor, fix, stage, buildNum);
}


// ---------------------------------------------------------------------------
//		 _HostGetHostID
// ---------------------------------------------------------------------------

static void _HostGetHostID (void)
{
	// HostHostID HostHostGetID (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = hostIDPalmOSEmulator;
}


// ---------------------------------------------------------------------------
//		 _HostGetHostPlatform
// ---------------------------------------------------------------------------

static void _HostGetHostPlatform (void)
{
	// HostPlatform HostHostGetPlatform (void)

	struct StackFrame
	{
		short	_selector;
	};

#if defined (_WINDOWS)
	m68k_dreg (regs, 0) = hostPlatformWindows;
#elif defined (__MACOS__)
	m68k_dreg (regs, 0) = hostPlatformMacintosh;
#elif defined (UNIX)
	m68k_dreg (regs, 0) = hostPlatformUnix;
#else
	#error "Unsupported platform"
#endif
}


// ---------------------------------------------------------------------------
//		 _HostIsSelectorImplemented
// ---------------------------------------------------------------------------

static void _HostIsSelectorImplemented (void)
{
	// HostBool HostHostIsSelectorImplemented (long)

	struct StackFrame
	{
		short	_selector;
		long	selector;
	};

	long		selector = GET_PARAMETER (selector);
	HostHandler fn = HostGetHandler (selector);

	m68k_dreg (regs, 0) = (fn != NULL);
}


// ---------------------------------------------------------------------------
//		 _HostGestalt
// ---------------------------------------------------------------------------

static void _HostGestalt (void)
{
	// HostErr HostHostGestalt (long gestSel, long* response)

	struct StackFrame
	{
		short	_selector;
		long	gestSel;
		long*	response;
	};

	m68k_dreg (regs, 0) = hostErrUnknownGestaltSelector;
}


// ---------------------------------------------------------------------------
//		 _HostIsCallingTrap
// ---------------------------------------------------------------------------

static void _HostIsCallingTrap (void)
{
	// HostBool HostIsCallingTrap (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = ATrap::DoingCall();
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostProfileInit
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileInit (void)
{
	// HostErr HostProfileInit (long maxCalls, long maxDepth)

	struct StackFrame
	{
		short	_selector;
		long	maxCalls;
		long	maxDepth;
	};

	long	maxCalls = GET_PARAMETER (maxCalls);
	long	maxDepth = GET_PARAMETER (maxDepth);

	ProfileInit (maxCalls, maxDepth);

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


// ---------------------------------------------------------------------------
//		 _HostProfileStart
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileStart (void)
{
	// HostErr HostProfileStart (void)

	struct StackFrame
	{
		short	_selector;
	};

	ProfileStart ();

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


// ---------------------------------------------------------------------------
//		 _HostProfileStop
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileStop (void)
{
	// HostErr HostProfileStop (void)

	struct StackFrame
	{
		short	_selector;
	};

	ProfileStop ();

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


// ---------------------------------------------------------------------------
//		 _HostProfileDump
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileDump (void)
{
	// HostErr HostProfileDump (const char* filename)

	struct StackFrame
	{
		short		_selector;
		const char* filename;
	};

	// Get the caller's parameters.

	uaecptr filename = GET_PARAMETER (filename);

	// Check the parameters.

	string	fileName;
	if (filename)
	{
		fileName = ToString (filename);
	}
	else
	{
		fileName = Platform::GetString (kStr_ProfileResults);
	}

	ProfileDump (fileName.c_str ());

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


// ---------------------------------------------------------------------------
//		 _HostProfileCleanup
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileCleanup (void)
{
	// HostErr HostProfileCleanup (void)

	struct StackFrame
	{
		short	_selector;
	};

	ProfileCleanup ();

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


// ---------------------------------------------------------------------------
//		 _HostProfileDetailFn
// ---------------------------------------------------------------------------
#if HAS_PROFILING

static void _HostProfileDetailFn (void)
{
	// HostErr _HostProfileDetailFn (void* addr, HostBool logDetails)

	struct StackFrame
	{
		short		_selector;
		void*		addr;
		HostBool	logInstructions;
	};

	uaecptr	addr = GET_PARAMETER (addr);
	Bool	logInstructions = GET_PARAMETER (logInstructions);

	ProfileDetailFn (addr, logInstructions);

	m68k_dreg (regs, 0) = 0;	// !!! TBD
}

#endif


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostErrNo
// ---------------------------------------------------------------------------

static void _HostErrNo (void)
{
	// long HostErrNo (void)

	struct StackFrame
	{
		short	_selector;
	};

	// Return the result.

	m68k_dreg (regs, 0) = (uae_u32) errno;
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostFClose
// ---------------------------------------------------------------------------

static void _HostFClose (void)
{
	// long HostFClose (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __fclose (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFEOF
// ---------------------------------------------------------------------------

static void _HostFEOF (void)
{
	// long HostFEOF (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = 1;	// At end of file (right choice?)
		return;
	}

	// Call the function.

	int 	result = __feof (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFError
// ---------------------------------------------------------------------------

static void _HostFError (void)
{
	// long HostFError (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __ferror (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFFlush
// ---------------------------------------------------------------------------

static void _HostFFlush (void)
{
	// long HostFFlush (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __fflush (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFGetC
// ---------------------------------------------------------------------------

static void _HostFGetC (void)
{
	// long HostFGetC (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;	// No file, no data...
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __fgetc (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFGetPos
// ---------------------------------------------------------------------------

static void _HostFGetPos (void)
{
	// long HostFGetPos (HostFILE*, long*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
		long*		posP;
	};

	// Get the caller's parameters.

	FILE*	f		= ToFILE (GET_PARAMETER (f));
	uaecptr posP	= GET_PARAMETER (posP);

	// Check the parameters.

	if (!f || !posP)
	{
		m68k_dreg (regs, 0) = 1;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	fpos_t	pos;
	int 	result = __fgetpos (f, &pos);

	// If the function succeeded, return the position in
	// the memory location pointed to by "posP".

	if (result == 0)	// success
	{
		put_long (posP, (long) pos);
	}

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFGetS
// ---------------------------------------------------------------------------

static void _HostFGetS (void)
{
	// char* HostFGetS (char*, long n, HostFILE*)

	struct StackFrame
	{
		short		_selector;
		char*		s;
		long		n;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	uaecptr s = GET_PARAMETER (s);
	uae_u32 n = GET_PARAMETER (n);
	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f || !s)
	{
		m68k_dreg (regs, 0) = UAE_NULL;
		errno = hostErrInvalidParameter;
		return;
	}

	// Create a temporary string and read the data into it
	// so that we can later sort out the wordswapping.

	if (n > 0)
	{
		string	tempString (n, 'a');

		// Call the function.

		char*	result = __fgets (&tempString[0], (int) n, f);

		// If we were able to read the string, copy it into the
		// user's buffer (using uae_strcpy to take care of real
		// <-> emulated memory mapping.  If the read failed,
		// return NULL.

		if (result != 0)
		{
			uae_strcpy (s, tempString.c_str ());
		}
		else
		{
			s = UAE_NULL;
		}
	}

	// Return the result.

	m68k_areg (regs, 0) = s;
}


// ---------------------------------------------------------------------------
//		 _HostFOpen
// ---------------------------------------------------------------------------

static void _HostFOpen (void)
{
	// HostFILE* HostFOpen (const char*, const char*)

	struct StackFrame
	{
		short		_selector;
		const char* name;
		const char* mode;
	};

	// Get the caller's parameters.

	uaecptr name = GET_PARAMETER (name);
	uaecptr mode = GET_PARAMETER (mode);

	// Check the parameters.

	if (!name || !mode)
	{
		m68k_areg (regs, 0) = UAE_NULL;
		errno = hostErrInvalidParameter;
		return;
	}

	// Copy the strings from emulated memory into real
	// memory; that way, we un-wordswap the strings
	// on little-endian platforms.

	string	name2 (ToString (name));
	string	mode2 (ToString (mode));

	// Call the function.

	FILE*	result = fopen (name2.c_str (), mode2.c_str ());

	// Return the result.

	m68k_areg (regs, 0) = (uae_u32) result;
}


// ---------------------------------------------------------------------------
//		 _HostFPrintF
// ---------------------------------------------------------------------------

static void _HostFPrintF (void)
{
	// long HostFPrintF (HostFILE*, const char*, ...)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
		const char* fmt;
	};

	// Get the caller's parameters.

	FILE*	f	= ToFILE (GET_PARAMETER (f));
	uaecptr fmt = GET_PARAMETER (fmt);

	// Check the parameters.

	if (!f || !fmt)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;
		errno = hostErrInvalidParameter;
		return;
	}

	// Make a copy of the format string. We'll need it eventually
	// when we call vfprintf, and getting it now will allow us to
	// access the format string characters more quickly.

	string	fmt2 (ToString (fmt));

	// Collect the specified parameters. We need to make copies of everything
	// so that it's in the right endian order and to reverse any effects
	// of wordswapping.  The values specified on the stack (the integers,
	// chars, doubles, pointers, etc.) get converted and placed in stackData.
	// The data pointed to by the pointers gets converted and placed in stringData.

	ByteList	stackData;
	StringList stringData;

	::CollectParameters (sizeof(short) + sizeof(HostFILE*) + sizeof(char*), fmt2, stackData, stringData);

	// Write everything out to the file using vfprintf.

	int 	result = __vfprintf (f, fmt2.c_str (), (va_list) &stackData[0]);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFPutC
// ---------------------------------------------------------------------------

static void _HostFPutC (void)
{
	// long HostFPutC (long c, HostFILE*)

	struct StackFrame
	{
		short		_selector;
		long		c;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	uae_u32 ch	= GET_PARAMETER (c);
	FILE*	f	= ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __fputc ((int) ch, f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFPutS
// ---------------------------------------------------------------------------

static void _HostFPutS (void)
{
	// long HostFPutS (const char*, HostFILE*)

	struct StackFrame
	{
		short		_selector;
		const char* s;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	uaecptr str = GET_PARAMETER (s);
	FILE*	f	= ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f || !str)
	{
		m68k_dreg (regs, 0) = (uae_u32) EOF;
		errno = hostErrInvalidParameter;
		return;
	}

	// Copy the string from emulated memory into real
	// memory; that way, we un-wordswap the string
	// on little-endian platforms.

	string	str2 (ToString (str));

	// Call the function.

	int 	result = __fputs (str2.c_str (), f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFRead
// ---------------------------------------------------------------------------

static void _HostFRead (void)
{
	// long HostFRead (void*, long, long, HostFILE*)

	struct StackFrame
	{
		short		_selector;
		void*		buffer;
		long		size;
		long		count;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	uaecptr buffer	= GET_PARAMETER (buffer);
	uae_u32 size	= GET_PARAMETER (size);
	uae_u32 count	= GET_PARAMETER (count);
	FILE*	f		= ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f || !buffer)
	{
		m68k_dreg (regs, 0) = 0;
		errno = hostErrInvalidParameter;
		return;
	}

	// Allocate a temporary buffer to hold the data read from
	// disk. We use a temporary buffer so that we can sort out
	// real <-> emulated memory mapping issues with uae_memcpy.

	void*	tempBuffer = malloc (size * count);

	// Call the function.

	size_t	result = __fread (tempBuffer, size, count, f);

	// If the read succeeded, copy the data into the user's buffer.

	if (result)
	{
		uae_memcpy (buffer, tempBuffer, size * result);
	}
	
	free (tempBuffer);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostRemove
// ---------------------------------------------------------------------------

static void _HostRemove (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostRename
// ---------------------------------------------------------------------------

static void _HostRename (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostFReopen
// ---------------------------------------------------------------------------

static void _HostFReopen (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostFScanF
// ---------------------------------------------------------------------------

static void _HostFScanF (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostFSeek
// ---------------------------------------------------------------------------

static void _HostFSeek (void)
{
	// long HostFSeek (HostFILE*, long offset, long origin)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
		long		offset;
		long		origin;
	};

	// Get the caller's parameters.

	FILE*	f		= ToFILE (GET_PARAMETER (f));
	uae_u32 offset	= GET_PARAMETER (offset);
	uae_u32 origin	= GET_PARAMETER (origin);

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) -1;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result	= __fseek (f, offset, (int) origin);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFSetPos
// ---------------------------------------------------------------------------

static void _HostFSetPos (void)
{
	// long HostFSetPos (HostFILE*, long)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
		long*		pos;
	};

	// Get the caller's parameters.

	FILE*	f	= ToFILE (GET_PARAMETER (f));
	uaecptr pos = GET_PARAMETER (pos);

	fpos_t	myPos = get_long (pos);

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = 1;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	int 	result = __fsetpos (f, &myPos);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFTell
// ---------------------------------------------------------------------------

static void _HostFTell (void)
{
	// long HostFTell (HostFILE*)

	struct StackFrame
	{
		short		_selector;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	FILE*	f = ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f)
	{
		m68k_dreg (regs, 0) = (uae_u32) -1;
		errno = hostErrInvalidParameter;
		return;
	}

	// Call the function.

	long	result = __ftell (f);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostFWrite
// ---------------------------------------------------------------------------

static void _HostFWrite (void)
{
	// long HostFWrite (const void*, long, long, HostFILE*)

	struct StackFrame
	{
		short		_selector;
		const void* buffer;
		long		size;
		long		count;
		HostFILE*	f;
	};

	// Get the caller's parameters.

	uaecptr buffer	= GET_PARAMETER (buffer);
	uae_u32 size	= GET_PARAMETER (size);
	uae_u32 count	= GET_PARAMETER (count);
	FILE*	f		= ToFILE (GET_PARAMETER (f));

	// Check the parameters.

	if (!f || !buffer)
	{
		m68k_dreg (regs, 0) = 0;
		errno = hostErrInvalidParameter;
		return;
	}

	// Allocate a temporary buffer to hold the data being written
	// to disk. We use a temporary buffer so that we can sort out
	// real <-> emulated memory mapping issues with uae_memcpy.

	void*	tempBuffer = malloc (size * count);
	uae_memcpy (tempBuffer, buffer, size * count);

	// Call the function.

	size_t	result = __fwrite (tempBuffer, size, count, f);

	free (tempBuffer);

	// Return the result.

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostTmpFile
// ---------------------------------------------------------------------------

static void _HostTmpFile (void)
{
	// HostFILE* HostTmpFile (void)

	struct StackFrame
	{
		short	_selector;
	};

	// Get the caller's parameters.

	// Call the function.

	FILE*		result = tmpfile ();

	// Return the result.

	m68k_areg (regs, 0) = (uae_u32) result;
}


// ---------------------------------------------------------------------------
//		 _HostTmpNam
// ---------------------------------------------------------------------------

static void _HostTmpNam (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostGetEnv
// ---------------------------------------------------------------------------

static void _HostGetEnv (void)
{
	// char* HostGetEnv (const char*)

	struct StackFrame
	{
		short		_selector;
		const char* name;
	};

	// Get the caller's parameters.

	uaecptr name	= GET_PARAMETER (name);

	// Check the parameters.

	if (!name)
	{
		m68k_areg (regs, 0) = UAE_NULL;
		errno = hostErrInvalidParameter;
		return;
	}

	// Copy the string from emulated memory into real
	// memory; that way, we un-wordswap the string
	// on little-endian platforms.

	string	name2 (ToString (name));

	// Call the function.

	char*	value = getenv (name2.c_str ());

	// Map the string into emulated space.

	if (value != NULL)
	{
		Memory::MapPhysicalMemory (value, strlen (value) + 1);
	}

	// Return the result.

	m68k_areg (regs, 0) = (uae_u32) value;
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostMalloc
// ---------------------------------------------------------------------------

static void _HostMalloc (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostRealloc
// ---------------------------------------------------------------------------

static void _HostRealloc (void)
{
}


// ---------------------------------------------------------------------------
//		 _HostFree
// ---------------------------------------------------------------------------

static void _HostFree (void)
{
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostGremlinIsRunning
// ---------------------------------------------------------------------------

static void _HostGremlinIsRunning (void)
{
	// HostBool HostGremlinIsRunning (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = Hordes::IsOn () ? 1 : 0;
}


// ---------------------------------------------------------------------------
//		 _HostGremlinNumber
// ---------------------------------------------------------------------------

static void _HostGremlinNumber (void)
{
	// long HostGremlinNumber (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = Hordes::GremlinNumber ();
}


// ---------------------------------------------------------------------------
//		 _HostGremlinCounter
// ---------------------------------------------------------------------------

static void _HostGremlinCounter (void)
{
	// long HostGremlinCounter (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = Hordes::EventCounter ();
}


// ---------------------------------------------------------------------------
//		 _HostGremlinLimit
// ---------------------------------------------------------------------------

static void _HostGremlinLimit (void)
{
	// long HostGremlinLimit (void)

	struct StackFrame
	{
		short	_selector;
	};

	m68k_dreg (regs, 0) = Hordes::EventLimit ();
}


// ---------------------------------------------------------------------------
//		 _HostGremlinNew
// ---------------------------------------------------------------------------

static void _HostGremlinNew (void)
{
	// HostErr HostGremlinNew (void)

	struct StackFrame
	{
		short		_selector;
		const HostGremlinInfo*	info;
	};

	uaecptr		infoP		= GET_PARAMETER (info);

	// Get the easy parts.

	HordeInfo	info;

	info.fStartNumber		= get_long (infoP + offsetof(HostGremlinInfo, fFirstGremlin));
	info.fStopNumber		= get_long (infoP + offsetof(HostGremlinInfo, fLastGremlin));
	info.fSaveFrequency		= get_long (infoP + offsetof(HostGremlinInfo, fSaveFrequency));
	info.fSwitchDepth		= get_long (infoP + offsetof(HostGremlinInfo, fSwitchDepth));
	info.fMaxDepth 			= get_long (infoP + offsetof(HostGremlinInfo, fMaxDepth));

	// Get the list of installed applications so that we can compare it to
	// the list of applications requested by the caller.

	DatabaseInfoList	installedAppList;
	::GetDatabases (installedAppList, kApplicationsOnly);

	// Get the list of applications requested by the caller.  Break up the
	// string into a list of individual names, and check to see if the whole
	// thing was preceded by a '-'.

	string		appNames	= ToString (infoP + offsetof(HostGremlinInfo, fAppNames));

	Bool		exclude = false;
	if (appNames[0] == '-')
	{
		exclude = true;
		appNames = appNames.substr(1);
	}

	StringList	requestedAppList;
	::SeperateList (requestedAppList, appNames, ',');

	// For each requested application, see if it's installed in the device.
	// If so, add it to the list of applications Gremlins should be run on.

	StringList::iterator	iter1 = requestedAppList.begin();
	while (iter1 != requestedAppList.end())
	{
		// Get the application info based on the given name.

		DatabaseInfoList::iterator	iter2 = installedAppList.begin ();
		while (iter2 != installedAppList.end ())
		{
			Bool	addIt;

			if (exclude)
				addIt = *iter1 != iter2->name;
			else
				addIt = *iter1 == iter2->name;

			if (addIt)
			{
				info.fAppList.push_back (*iter2);
			}

			++iter2;
		}

		++iter1;
	}

	// Start up the gremlin sub-system.

//	Hordes::New (info);

	Startup::ScheduleNewHorde (info, StringList());

	m68k_dreg (regs, 0) = errNone;
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostImportFile
// ---------------------------------------------------------------------------

static void _HostImportFile (void)
{
	// HostErr HostImportFile (const char* fileName)

	struct StackFrame
	{
		short		_selector;
		const char* fileName;
	};

	// Get the caller's parameters.

	uaecptr str = GET_PARAMETER (fileName);

	// Check the parameters.

	if (!str)
	{
		m68k_dreg (regs, 0) = hostErrInvalidParameter;
		return;
	}

	// Copy the string from emulated memory into real
	// memory; that way, we un-wordswap the string
	// on little-endian platforms.

	string	fileName (ToString (str));

	ErrCode result = kError_NoError;
	try
	{
		FileReference	fileRef (fileName);
		FileRefList		fileList;
		fileList.push_back (fileRef);
		::LoadPalmFileList (fileList);
	}
	catch (ErrCode errCode)
	{
		result = errCode;
	}

	m68k_dreg (regs, 0) = result;
}


// ---------------------------------------------------------------------------
//		 _HostExportFile
// ---------------------------------------------------------------------------

static void _HostExportFile (void)
{
	// HostErr HostExportFile (const char* dbName, long cardNum, const char* fileName)

	struct StackFrame
	{
		short		_selector;
		const char* fileName;
		long		cardNum;
		const char* dbName;
	};

	// Get the caller's parameters.

	uaecptr fName	= GET_PARAMETER (fileName);
	long	cardNum = GET_PARAMETER (cardNum);
	uaecptr dbName	= GET_PARAMETER (dbName);

	// Check the parameters.

	if (!fName || !dbName)
	{
		m68k_dreg (regs, 0) = hostErrInvalidParameter;
		return;
	}

	// Copy the string from emulated memory into real
	// memory; that way, we un-wordswap the string
	// on little-endian platforms.

	string	fileName (ToString (fName));
	string	databaseName (ToString (dbName));

	ErrCode result = kError_NoError;
	try
	{
		FileReference	fileRef (fileName);
		FileHandle		file (fileRef, kCreateAlways | kOpenReadWrite,
								'PRC ', 'PPia');
		::SavePalmFile (file, cardNum, databaseName.c_str ());
	}
	catch (ErrCode errCode)
	{
		result = errCode;
	}

	m68k_dreg (regs, 0) = result;
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostSetPreference
// ---------------------------------------------------------------------------

static void _HostSetPreference (void)
{
	// void HostSetPreference (const char*, const char*);

	struct StackFrame
	{
		short		_selector;
		const char* key;
		const char* value;
	};

	string	key 	= ToString (GET_PARAMETER (key));
	string	value	= ToString (GET_PARAMETER (value));

	Preference<string>	pref (key.c_str ());
	pref = value;
}


// ---------------------------------------------------------------------------
//		 _HostGetPreference
// ---------------------------------------------------------------------------

static void _HostGetPreference (void)
{
	// HostBool HostGetPreference (const char*, char*);

	struct StackFrame
	{
		short		_selector;
		const char* key;
		char*		value;
	};

	string	key 	= ToString (GET_PARAMETER (key));
	uaecptr value	= GET_PARAMETER (value);

	Preference<string>	pref (key.c_str ());

	if (pref.Loaded ())
	{
		uae_strcpy (value, pref->c_str ());
		m68k_dreg (regs, 0) = 1;
	}
	else
	{
		m68k_dreg (regs, 0) = 0;
	}
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostLogFile
// ---------------------------------------------------------------------------

static void _HostLogFile (void)
{
	m68k_areg (regs, 0) = (uae_u32) hostLogFile;
}


// ---------------------------------------------------------------------------
//		 _HostSetLogFileSize
// ---------------------------------------------------------------------------

static void _HostSetLogFileSize (void)
{
	// void HostSetLogFileSize (long)

	struct StackFrame
	{
		short	_selector;
		long	newSize;
	};

	long	newSize = GET_PARAMETER (newSize);

	LogGetStdLog ()->SetLogSize (newSize);

	m68k_dreg (regs, 0) = 0;
}

#pragma mark -

// ---------------------------------------------------------------------------
//		 _HostSessionCreate
// ---------------------------------------------------------------------------

static void _HostSessionCreate (void)
{
	// HostErr HostSessionCreate(const char* device, long ramSize, const char* romPath)

	struct StackFrame
	{
		short		_selector;
		const char*	device;
		long		ramSize;
		const char*	romPath;
	};

	string		deviceStr	= ToString (GET_PARAMETER (device));
	RAMSizeType	ramSize		= GET_PARAMETER (ramSize);
	string		romPathStr	= ToString (GET_PARAMETER (romPath));

	// See if it's OK to create a session.

	if (Platform::SessionRunning())
	{
		m68k_dreg (regs, 0) = hostErrSessionRunning;
		return;
	}

	// Convert the device string into a DeviceType

	DeviceType		device = kDeviceUnspecified;

	DeviceTextList	devices;
	::GetDeviceTextList (devices);

	DeviceTextList::iterator	deviceIter = devices.begin();
	while (deviceIter != devices.end())
	{
		if (_stricmp (deviceStr.c_str(), deviceIter->second.c_str()) == 0)
		{
			device = deviceIter->first;
			break;
		}
	}

	if (device == kDeviceUnspecified)
	{
		m68k_dreg (regs, 0) = hostErrInvalidDeviceType;
		return;
	}

	// Validate the RAM size

	Bool			sizeOK = false;
	MemoryTextList	sizes;
	::GetMemoryTextList (sizes);

	MemoryTextList::iterator	sizeIter = sizes.begin();
	while (sizeIter != sizes.end())
	{
		if (ramSize == sizeIter->first)
		{
			sizeOK = true;
			break;
		}
	}

	if (!sizeOK)
	{
		m68k_dreg (regs, 0) = hostErrInvalidRAMSize;
		return;
	}

	// Convert the ROM file string into a FileReference.

	FileReference	romRef(romPathStr);
	if (!romRef.Exists())
	{
		m68k_dreg (regs, 0) = hostErrFileNotFound;
		return;
	}

	// Kick this all off.

	Configuration	cfg (device, ramSize, romRef);
	Startup::ScheduleCreateSession (cfg);
	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSessionOpen
// ---------------------------------------------------------------------------

static void _HostSessionOpen (void)
{
	// HostErr HostSessionOpen (const char* psfFileName);

	struct StackFrame
	{
		short		_selector;
		const char*	psfFileName;
	};

	string	psfFileName	= ToString (GET_PARAMETER (psfFileName));

	// See if it's OK to open a session.

	if (Platform::SessionRunning())
	{
		m68k_dreg (regs, 0) = hostErrSessionRunning;
		return;
	}

	// Validate the file to open.

	FileReference	psfFileRef(psfFileName);
	if (!psfFileRef.Exists())
	{
		m68k_dreg (regs, 0) = hostErrFileNotFound;
		return;
	}

	// Kick this all off.

	Startup::ScheduleOpenSession (psfFileRef);
	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSessionClose
// ---------------------------------------------------------------------------

static void _HostSessionClose (void)
{
	// HostErr HostSessionClose (const char* saveFileName)

	struct StackFrame
	{
		short		_selector;
		const char*	saveFileName;
	};

	string	saveFileName	= ToString (GET_PARAMETER (saveFileName));

	// See if it's OK to close a session.

	if (!Platform::SessionRunning())
	{
		m68k_dreg (regs, 0) = hostErrSessionNotRunning;
		return;
	}

	FileReference	saveFileRef(saveFileName);

	// Kick this all off.

	Startup::ScheduleCloseSession (saveFileName);
	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSessionQuit
// ---------------------------------------------------------------------------

static void _HostSessionQuit (void)
{
	// HostErr HostSessionQuit (void)

	struct StackFrame
	{
		short		_selector;
	};

	// See if it's OK to quit Poser.

	if (Platform::SessionRunning())
	{
//		m68k_dreg (regs, 0) = hostErrSessionRunning;
//		return;
	}

	// Kick this all off.

	Startup::ScheduleQuit ();

	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSignalSend
// ---------------------------------------------------------------------------
// Called by anyone wanting to send a signal to any waiting scripts.

static void _HostSignalSend (void)
{
	// HostErr HostSignalSend (HostSignal signalNumber)

	struct StackFrame
	{
		short		_selector;
		HostSignal	signalNumber;
	};

	HostSignal	signalNumber = GET_PARAMETER(signalNumber);

	RPC::SignalWaiters (signalNumber);

	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSignalWait
// ---------------------------------------------------------------------------
// Called by scripts that want to get a signal sent from HostSignalSend.

static void _HostSignalWait (void)
{
	// HostErr HostSignalWait (long timeout)

	struct StackFrame
	{
		short	_selector;
		long	timeout;
	};

	long	timeout = GET_PARAMETER(timeout);

	// Unblock the CPU thread if it's suspended from a previous
	// HostSignalSend call.

	m68k_dreg (regs, 0) = Emulator::Resume();

	if (RPC::HandlingPacket())
	{
		RPC::DeferCurrentPacket(timeout);

		if (Patches::UIInitialized ())
		{
			::EvtWakeup ();	// Wake up the process in case the caller is looking
							// for an idle event (which would never otherwise
							// happen if EvtGetEvent has already been called and
							// is blocking).
		}
	}

	m68k_dreg (regs, 0) = errNone;
}


// ---------------------------------------------------------------------------
//		 _HostSignalResume
// ---------------------------------------------------------------------------
// Called by scripts to restart the emulator after it has sent a signal and
// then suspended itself.

static void _HostSignalResume (void)
{
	// HostErr HostSignalResume (void)

	struct StackFrame
	{
		short		_selector;
	};

	m68k_dreg (regs, 0) = Emulator::Resume();
}


#if defined (_WINDOWS)
// ---------------------------------------------------------------------------
//		 _HostTraceInit
// ---------------------------------------------------------------------------

static void _HostTraceInit (void)
{
	Tracer::Init();
}


// ---------------------------------------------------------------------------
//		 _HostTraceClose
// ---------------------------------------------------------------------------

static void _HostTraceClose (void)
{
	Tracer::Close();
}


// ---------------------------------------------------------------------------
//		 _HostTraceOutputT
// ---------------------------------------------------------------------------

static void _HostTraceOutputT (void)
{
	// void HostTraceOutputT (unsigned short, const char*, ...)

	struct StackFrame
	{
		short			_selector;
		unsigned short	module;
		const char*		fmt;
	};

	// Get the caller's parameters.

	unsigned short module = GET_PARAMETER (module);
	uaecptr fmt = GET_PARAMETER (fmt);

	// Check the parameters.

	if (!fmt)
	{
		errno = hostErrInvalidParameter;
		return;
	}

	// Make a copy of the format string.

	string	fmt2 (ToString (fmt));

	// Collect the specified parameters. We need to make copies of everything
	// so that it's in the right endian order and to reverse any effects
	// of wordswapping.  The values specified on the stack (the integers,
	// chars, doubles, pointers, etc.) get converted and placed in stackData.
	// The data pointed to by the pointers gets converted and placed in stringData.

	ByteList	stackData;
	StringList stringData;

	::CollectParameters (sizeof(short) + sizeof(short) + sizeof(char*), fmt2, stackData, stringData);

	// Write everything out

	Tracer::OutputVT( module, fmt2.c_str (), (va_list) &stackData[0]);
}


// ---------------------------------------------------------------------------
//		 _HostTraceOutputTL
// ---------------------------------------------------------------------------

static void _HostTraceOutputTL (void)
{
	// void HostTraceOutputTL (unsigned short, const char*, ...)

	struct StackFrame
	{
		short			_selector;
		unsigned short	module;
		const char*		fmt;
	};

	// Get the caller's parameters.

	unsigned short module = GET_PARAMETER (module);
	uaecptr fmt = GET_PARAMETER (fmt);

	// Check the parameters.

	if (!fmt)
	{
		errno = hostErrInvalidParameter;
		return;
	}

	// Make a copy of the format string.

	string	fmt2 (ToString (fmt));

	// Collect the specified parameters. We need to make copies of everything
	// so that it's in the right endian order and to reverse any effects
	// of wordswapping.  The values specified on the stack (the integers,
	// chars, doubles, pointers, etc.) get converted and placed in stackData.
	// The data pointed to by the pointers gets converted and placed in stringData.

	ByteList	stackData;
	StringList stringData;

	::CollectParameters (sizeof(short) + sizeof(short) + sizeof(char*), fmt2, stackData, stringData);

	// Write everything out

	Tracer::OutputVTL( module, fmt2.c_str (), (va_list) &stackData[0]);
}


// ---------------------------------------------------------------------------
//		 _HostOutputVT
// ---------------------------------------------------------------------------

static void _HostTraceOutputVT (void)
{
	// void HostTraceOutputVT (unsigned short, const char*, char*)

	struct StackFrame
	{
		short			_selector;
		unsigned short	module;
		const char*		fmt;
	};

	// Get the caller's parameters.

	unsigned short module = GET_PARAMETER (module);
	uaecptr fmt = GET_PARAMETER (fmt);

	// Check the parameters.

	if (!fmt)
	{
		errno = hostErrInvalidParameter;
		return;
	}

	// Make a copy of the format string.

	string	fmt2 (ToString (fmt));

	// Collect the specified parameters. We need to make copies of everything
	// so that it's in the right endian order and to reverse any effects
	// of wordswapping.  The values specified on the stack (the integers,
	// chars, doubles, pointers, etc.) get converted and placed in stackData.
	// The data pointed to by the pointers gets converted and placed in stringData.

	ByteList	stackData;
	StringList stringData;

	::CollectParameters (sizeof(short) + sizeof(short) + sizeof(char*), fmt2, stackData, stringData);

	// Write everything out

	Tracer::OutputVT( module, fmt2.c_str (), (va_list) &stackData[0]);
}


// ---------------------------------------------------------------------------
//		 _HostOutputVTL
// ---------------------------------------------------------------------------

static void _HostTraceOutputVTL (void)
{
	// void HostTraceOutputVTL (unsigned short, const char*, char*)

	struct StackFrame
	{
		short			_selector;
		unsigned short	module;
		const char*		fmt;
	};

	// Get the caller's parameters.

	unsigned short module = GET_PARAMETER (module);
	uaecptr fmt = GET_PARAMETER (fmt);

	// Check the parameters.

	if (!fmt)
	{
		errno = hostErrInvalidParameter;
		return;
	}

	// Make a copy of the format string.

	string	fmt2 (ToString (fmt));

	// Collect the specified parameters. We need to make copies of everything
	// so that it's in the right endian order and to reverse any effects
	// of wordswapping.  The values specified on the stack (the integers,
	// chars, doubles, pointers, etc.) get converted and placed in stackData.
	// The data pointed to by the pointers gets converted and placed in stringData.

	ByteList	stackData;
	StringList stringData;

	::CollectParameters (sizeof(short) + sizeof(short) + sizeof(char*), fmt2, stackData, stringData);

	// Write everything out

	Tracer::OutputVTL( module, fmt2.c_str (), (va_list) &stackData[0]);
}


// ---------------------------------------------------------------------------
//		 _HostOutputB
// ---------------------------------------------------------------------------

static void _HostTraceOutputB (void)
{
	// void HostTraceOutputB (unsigned short, const char*, unsigned long )

	struct StackFrame
	{
		short			_selector;
		unsigned short	module;
		const char*		buff;
		unsigned long	length;
	};
	
	void*			tempBuff;

	// Get the caller's parameters.

	unsigned short	module	= GET_PARAMETER (module);
	uaecptr			buff	= GET_PARAMETER (buff);
	unsigned long	length 	= GET_PARAMETER (length);

	// Check the parameters.

	if (!buff || !length)
	{
		errno = hostErrInvalidParameter;
		return;
	}
	
	tempBuff = malloc(length);

	if (tempBuff)
	{	
		uae_memcpy(tempBuff, buff, length);
		Tracer::OutputB(module, (const unsigned char*) tempBuff, length);
		free(tempBuff);
	}
	else
	{
		errno = hostErrInvalidParameter;
		return;
	}
}

#endif	// _WINDOWS

#pragma mark -

// ---------------------------------------------------------------------------
//		 HostInit
// ---------------------------------------------------------------------------

void HostInit (void)
{
	memset (gHandlerTable, 0, sizeof (gHandlerTable));

	gHandlerTable [hostSelectorGetHostVersion]	= _HostGetHostVersion;
	gHandlerTable [hostSelectorGetHostID]		= _HostGetHostID;
	gHandlerTable [hostSelectorGetHostPlatform] = _HostGetHostPlatform;
	gHandlerTable [hostSelectorIsSelectorImplemented] = _HostIsSelectorImplemented;
	gHandlerTable [hostSelectorGestalt] 		= _HostGestalt;
	gHandlerTable [hostSelectorIsCallingTrap]	= _HostIsCallingTrap;

#if HAS_PROFILING
	gHandlerTable [hostSelectorProfileInit] 	= _HostProfileInit;
	gHandlerTable [hostSelectorProfileStart]	= _HostProfileStart;
	gHandlerTable [hostSelectorProfileStop] 	= _HostProfileStop;
	gHandlerTable [hostSelectorProfileDump] 	= _HostProfileDump;
	gHandlerTable [hostSelectorProfileCleanup]	= _HostProfileCleanup;
	gHandlerTable [hostSelectorProfileDetailFn]	= _HostProfileDetailFn;
#endif

	gHandlerTable [hostSelectorErrNo]			= _HostErrNo;

	gHandlerTable [hostSelectorFClose]			= _HostFClose;
	gHandlerTable [hostSelectorFEOF]			= _HostFEOF;
	gHandlerTable [hostSelectorFError]			= _HostFError;
	gHandlerTable [hostSelectorFFlush]			= _HostFFlush;
	gHandlerTable [hostSelectorFGetC]			= _HostFGetC;
	gHandlerTable [hostSelectorFGetPos] 		= _HostFGetPos;
	gHandlerTable [hostSelectorFGetS]			= _HostFGetS;
	gHandlerTable [hostSelectorFOpen]			= _HostFOpen;
	gHandlerTable [hostSelectorFPrintF] 		= _HostFPrintF;
	gHandlerTable [hostSelectorFPutC]			= _HostFPutC;
	gHandlerTable [hostSelectorFPutS]			= _HostFPutS;
	gHandlerTable [hostSelectorFRead]			= _HostFRead;
//	gHandlerTable [hostSelectorRemove]			= _HostRemove;
//	gHandlerTable [hostSelectorRename]			= _HostRename;
//	gHandlerTable [hostSelectorFReopen] 		= _HostFReopen;
//	gHandlerTable [hostSelectorFScanF]			= _HostFScanF;
	gHandlerTable [hostSelectorFSeek]			= _HostFSeek;
	gHandlerTable [hostSelectorFSetPos] 		= _HostFSetPos;
	gHandlerTable [hostSelectorFTell]			= _HostFTell;
	gHandlerTable [hostSelectorFWrite]			= _HostFWrite;
	gHandlerTable [hostSelectorTmpFile] 		= _HostTmpFile;
//	gHandlerTable [hostSelectorTmpNam]			= _HostTmpNam;
	gHandlerTable [hostSelectorGetEnv]			= _HostGetEnv;

//	gHandlerTable [hostSelectorMalloc]			= _HostMalloc;
//	gHandlerTable [hostSelectorRealloc] 		= _HostRealloc;
//	gHandlerTable [hostSelectorFree]			= _HostFree;

	gHandlerTable [hostSelectorGremlinIsRunning]= _HostGremlinIsRunning;
	gHandlerTable [hostSelectorGremlinNumber]	= _HostGremlinNumber;
	gHandlerTable [hostSelectorGremlinCounter]	= _HostGremlinCounter;
	gHandlerTable [hostSelectorGremlinLimit]	= _HostGremlinLimit;
	gHandlerTable [hostSelectorGremlinNew]		= _HostGremlinNew;

	gHandlerTable [hostSelectorImportFile]		= _HostImportFile;
	gHandlerTable [hostSelectorExportFile]		= _HostExportFile;

	gHandlerTable [hostSelectorGetPreference]	= _HostSetPreference;
	gHandlerTable [hostSelectorSetPreference]	= _HostGetPreference;

	gHandlerTable [hostSelectorLogFile] 		= _HostLogFile;
	gHandlerTable [hostSelectorSetLogFileSize]	= _HostSetLogFileSize;

//	gHandlerTable [hostSelectorSessionCreate]	= _HostSessionCreate;
//	gHandlerTable [hostSelectorSessionOpen] 	= _HostSessionOpen;
	gHandlerTable [hostSelectorSessionClose]	= _HostSessionClose;
	gHandlerTable [hostSelectorSessionQuit]		= _HostSessionQuit;
	gHandlerTable [hostSelectorSignalSend]		= _HostSignalSend;
	gHandlerTable [hostSelectorSignalWait] 		= _HostSignalWait;
	gHandlerTable [hostSelectorSignalResume]	= _HostSignalResume;

#if defined (_WINDOWS)
	gHandlerTable [hostSelectorTraceInit]		= _HostTraceInit;
	gHandlerTable [hostSelectorTraceClose] 		= _HostTraceClose;
	gHandlerTable [hostSelectorTraceOutputT]	= _HostTraceOutputT;
	gHandlerTable [hostSelectorTraceOutputTL] 	= _HostTraceOutputTL;
	gHandlerTable [hostSelectorTraceOutputVT] 	= _HostTraceOutputVT;
	gHandlerTable [hostSelectorTraceOutputVTL]	= _HostTraceOutputVTL;
	gHandlerTable [hostSelectorTraceOutputB]	= _HostTraceOutputB;
#endif
}


// ---------------------------------------------------------------------------
//		 HostGetHandler
// ---------------------------------------------------------------------------

HostHandler HostGetHandler (long selector)
{
	HostHandler fn = NULL;

	// Hack for GremlinIsOn; see comments at head of HostTraps.h.

	if ((selector & 0xFF00) == 0)
	{
		selector = hostSelectorGremlinIsRunning;
	}

	if (selector >= 0 && selector < hostSelectorLastTrapNumber)
	{
		fn = gHandlerTable [selector];
	}

	return fn;
}


// ---------------------------------------------------------------------------
//		 CollectParameters
// ---------------------------------------------------------------------------

void CollectParameters (int stackOffset, const string& fmt, ByteList& stackData, StringList& stringData)
{
	// Skip past the first few items on the stack.

	int callerStackOffset = stackOffset;

	// Start parsing up the format string.

	string::const_iterator	iter;

	for (iter = fmt.begin (); iter != fmt.end ();)
	{
		char	ch = *iter++;

		/*
			Format specification:

				% ? 12 .4 h d
				| |  |  | | |
				| |  |  | | +-- conversion letter
				| |  |  | +---- size modifier (l, L, or h)
				| |  |  +------ precision
				| |  +--------- minimum width field
				| +------------ flags (one or more of +, -, #, 0, or space)
				+-------------- start of specification
		*/

		if (ch == '%')
		{
			ch = *iter++;

			// Skip over any flags.

			while (ch == '+' || ch == '-' || ch == ' ' || ch == '#' || ch == '0')
			{
				ch = *iter++;
			}

			// Skip over any minimum width field.

			if (ch == '*')
			{
				ch = *iter++;
			}
			else
			{
				while (ch >= '0' && ch <= '9')
				{
					ch = *iter++;
				}
			}

			// Skip over any precision.

			if (ch == '.')
			{
				ch = *iter++;

				while (ch >= '0' && ch <= '9')
				{
					ch = *iter++;
				}
			}

			// Get any size modifier.

			enum {kSizeNone, kSizeLongInt, kSizeLongDouble, kSizeShortInt};

			int mod = kSizeNone;

			if (ch == 'l')
				mod = kSizeLongInt;
			else if (ch == 'L')
				mod = kSizeLongDouble;
			else if (ch == 'h')
				mod = kSizeShortInt;

			// If there was a modifier, it's been handled,
			// so skip over it.

			if (mod != kSizeNone)
				ch = *iter++;

			switch (ch)
			{
				case 'd':
				case 'i':
				case 'u':
				case 'o':
				case 'x':
				case 'X':
					// int, short, or long
					if (mod == kSizeNone || mod == kSizeShortInt)
						PushShort (callerStackOffset, stackData);
					else
						PushLong (callerStackOffset, stackData);
					break;

				case 'f':
				case 'e':
				case 'E':
				case 'g':
				case 'G':
					// double or long double
					if (mod == kSizeNone)
						PushDouble (callerStackOffset, stackData);
					else
						PushLongDouble (callerStackOffset, stackData);
					break;

				case 'c':
					// int or wint_t
					if (mod == kSizeNone)
						PushShort (callerStackOffset, stackData);
#if defined (_MSC_VER)
					else if (sizeof (wint_t) == 2)
						PushShort (callerStackOffset, stackData);
#endif
					else
						PushLong (callerStackOffset, stackData);
					break;

				case 's':
					PushString (callerStackOffset, stackData, stringData);
					break;

				case 'p':
					PushLong (callerStackOffset, stackData);
					break;

				case 'n':
					// pointer
					// !!! Not supported for now...
					break;

				case '%':
					// none
					break;

				default:
					// Bad conversion will soon ensue...
					break;
			}
		}
	}
}


// ---------------------------------------------------------------------------
//		 PushShort
// ---------------------------------------------------------------------------

void PushShort (int& callerStackOffset, ByteList& stackData)
{
	// Read a 2-byte int from the caller's stack, and push it
	// onto our stack as a 4-byte int.

	uae_u16 value = get_word (m68k_areg (regs, 7) + callerStackOffset);
	callerStackOffset += sizeof (value);

	ByteList::size_type	oldSize = stackData.size ();
	stackData.insert (stackData.end (), sizeof (int), 0);	// Make space for 4 more bytes

	*(int*) &stackData[oldSize] = value;
}


// ---------------------------------------------------------------------------
//		 PushLong
// ---------------------------------------------------------------------------

void PushLong (int& callerStackOffset, ByteList& stackData)
{
	// Read a 4-byte long int from the caller's stack, and push it
	// onto our stack as a 4-byte long int.

	uae_u32 value = get_long (m68k_areg (regs, 7) + callerStackOffset);
	callerStackOffset += sizeof (value);

	ByteList::size_type	oldSize = stackData.size ();
	stackData.insert (stackData.end (), sizeof (long), 0);	// Make space for 4 more bytes

	*(long*) &stackData[oldSize] = value;
}


// ---------------------------------------------------------------------------
//		 PushDouble
// ---------------------------------------------------------------------------

void PushDouble (int& callerStackOffset, ByteList& stackData)
{
	UNUSED_PARAM(callerStackOffset)
	UNUSED_PARAM(stackData)
}


// ---------------------------------------------------------------------------
//		 PushLongDouble
// ---------------------------------------------------------------------------

void PushLongDouble (int& callerStackOffset, ByteList& stackData)
{
	UNUSED_PARAM(callerStackOffset)
	UNUSED_PARAM(stackData)
}


// ---------------------------------------------------------------------------
//		 PushString
// ---------------------------------------------------------------------------

void PushString (int& callerStackOffset, ByteList& stackData, StringList& stringData)
{
	// Get the string pointer and clone the string into a new string object.

	uaecptr stringPtr = get_long (m68k_areg (regs, 7) + callerStackOffset);
	string	strCopy;
	size_t	strLen = uae_strlen (stringPtr);

	if (strLen > 0)
	{
		strCopy.resize (strLen);
		uae_strcpy (&strCopy[0], stringPtr);
	}

	// Add this string to the string array.

	stringData.push_back (strCopy);

	// In the stack data byte array, add a pointer to the string. 

	ByteList::size_type	oldSize = stackData.size ();
	stackData.insert (stackData.end (), sizeof (char*), 0); // Make space for 4 more bytes
	*(uae_u32*) &stackData[oldSize] = (uae_u32) (*(stringData.end () - 1)).c_str ();

	// Bump the caller's stack pointer by the size of a string pointer.

	callerStackOffset += sizeof (char*);
}


// ---------------------------------------------------------------------------
//		 ToString
// ---------------------------------------------------------------------------

string ToString (uaecptr s)
{
	string	result;

	size_t	sLen = uae_strlen (s);
	if (sLen > 0)
	{
		result.resize (sLen);
		uae_strcpy (&result[0], s);
	}

	return result;
}


// ---------------------------------------------------------------------------
//		 ToFILE
// ---------------------------------------------------------------------------

FILE* ToFILE (uaecptr f)
{
	if ((HostFILE*) f == hostLogFile)
	{
		return hostLogFILE;
	}

	return (FILE*) f;
}


#include "PalmPackPop.h"
