/* Network configuration during Debian installation
 * 
 * 1. Get a list of all available network interfaces from /proc/net/dev
 * 2. Let the user chose thich one to configure (either manually or using
 *    DHCP/BOOTP): IP address, netmask, broadcast address, default gateway
 * 3. Ask for the hostname, domain name and DNS servers (use predefined
 *    values we might have received from a DHCP server)
 * 4. Write the network configuration to /etc/network/interfaces (non-PCMCIA
 *    devices, at least loopback) or /etc/pcmcia/network.opts (PCMCIA devices)
 * 5. Activate the configuration
 */

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h> 
#include "dbootstrap.h"
#include "lang.h"
#include "util.h"

/*
 * Strings and other constants
 */
#define NC_ETC_DIR         "/etc"
#define NC_NETWORKS_FILE   "/etc/networks"
#define NC_RESOLV_FILE     "/etc/resolv.conf"
#define NC_ETC_NETWORK_DIR "/etc/network"
#define NC_INTERFACES_FILE "/etc/network/interfaces"
#define NC_HOSTS_FILE      "/etc/hosts"
#define NC_PCMCIA_DIR      "/etc/pcmcia"
#define NC_PCMCIA_FILE     "/etc/pcmcia/network.opts"

#define MAX_INTERFACES (16)
#define IP4_ADDR_SZ (4)
#define IP4_ADDR_LN (3 * IP4_ADDR_SZ + 4)
typedef struct ip4_addr
{
  int i[IP4_ADDR_SZ];
} ip4_addr_t;


/*
 * Global variables for this module
 */
char *host = NULL;
char *domain = NULL;
char *transceiver = NULL;
char *netinterface = NULL;
char *nameserver = NULL;
ip4_addr_t ipaddr = {{192, 168, 1, 1}}; 
ip4_addr_t netmask = {{255, 255, 255, 0}};
ip4_addr_t network = {{192, 168, 1, 0}};
ip4_addr_t broadcast = {{192, 168, 1, 255}};
ip4_addr_t gateway = {{0, 0, 0, 0}};
int has_gw = 0;
int has_pcmcia = 0;
int use_dhcp = 0;
struct stat nc_statbuf;

/*
 * Function prototypes
 */
extern int check_dir(const char *);
extern int activate_pcmcia();
int atoIP4(char *, ip4_addr_t *);
char *IP4toa(ip4_addr_t);
char *IP4tostr(char *, ip4_addr_t);
int ask_for_IP4(ip4_addr_t *, char *, char *, char *);
char *get_ifdsc(const char *);
int check_ip4_nameserver(char *);
int get_host();
int get_netinterface();
int get_ifconfig();
int get_domain();
int get_dnsconfig();
int get_pcmcia_transceiver();
int write_common_network();
int write_pcmcia_network();
int write_static_network();
int write_dhcp_network();
int activate_static_network();
int configure_static_network();


/*
 * Checks a string with an IPv4-address for validity and converts it into an
 * ip4_addr_t type;
 */
int atoIP4(char* addr, ip4_addr_t* iaddr)
{
  char* end;
  char* tmp_a;
  char* current = strdup(addr);
  char* next = NULL;
  int ix = 0;
  int tmp;

  tmp_a = current;
  end = current + strlen(current) + 1;
  while(next !=  end)
    {
      next = strchr(current, '.');
      if(next == NULL)
	next = end;
      else
	{
	  *next = '\0';
	  next++;
	}
      
      if(ix == IP4_ADDR_SZ)
	{
	  free(tmp_a);
	  return 255;
	}
      else
	{
	  tmp = atoi(current);
	  if ((tmp < 0) || (tmp > 255))
	    {
	      free(tmp_a);
	      return 255;
	    }
	  iaddr->i[ix++] = tmp;
	  current = next;
	}
    }
  free(tmp_a);
  if(ix != IP4_ADDR_SZ)
    return 255;
  return 0;
}

char* IP4toa(ip4_addr_t iaddr)
{
  int length = 3 * IP4_ADDR_SZ + 4;
  char* addr = calloc(length, sizeof(char));
  snprintf(addr, length, "%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return addr;
}

char* IP4tostr(char *s, ip4_addr_t iaddr)
{
  snprintf(s, 3 * IP4_ADDR_SZ + 4, "%d.%d.%d.%d",
	   iaddr.i[0],
	   iaddr.i[1],
	   iaddr.i[2],
	   iaddr.i[3]);
  return s;
}

/* Ask for a generic IPv4 address */
int ask_for_IP4(ip4_addr_t *ip, char *title, char *mesg, char *err)
{
  char *def;
  char *addr = IP4toa(*ip);
  for (;;) {
    def = addr;
    addr = inputBox(title, mesg, def);
    free(def);
    if ((addr == NULL) || (strlen(addr) == 0)) {
      return 255;
    }
    if (atoIP4(addr, ip) != 0) {
      problemBox(err, _("Problem"));
    }
    else {
      break;
    }
  }
  free(addr);
  return 0;
}



/*
 * Get a description for an interface (i.e. "Ethernet" for ethX).
 */
char
*get_ifdsc (const char *ifp)
{
  int i;
  struct if_alist_struct {
    char *interface;
    char *description;
  } interface_alist[] = {
    /*
     * A _("string") is an element of a char *[], and the linker will
     * know where to find the elements.  This works with `pointerize';
     * perhaps that's different with `gettext'... though if it expands
     * to a function call, this initializer should still work fine.
     * Also see the 1.66 version of choose_cdrom(), which uses the
     * similar technique.  If it works there, it will work here.
     */
    {"eth",    _("Ethernet or Fast Ethernet")},
    {"pcmcia", _("PC-Card (PCMCIA) Ethernet or Token Ring")},
    {"tr",     _("Token Ring")},
    {"arc",    _("Arcnet")},
    {"slip",   _("Serial-line IP")},
    {"plip",   _("Parallel-line IP")},
    {NULL,     NULL}};

  for (i = 0; interface_alist[i].interface != NULL; i++)
    if (! strncmp(ifp, interface_alist[i].interface, 
                  strlen (interface_alist[i].interface)))
      return strdup (interface_alist[i].description);
  return NULL;
}



/* Check for valid IPv4 addesses in nameservers' string */
int check_ip4_nameserver(char *ns) {
  ip4_addr_t ip;
  char msg[100];
  char *buf = strdup(ns);
  char *cur = buf;
  char *tmp, *end;
  end = buf + strlen(ns);
  do
    {
    if ((tmp = strstr(cur, " ")) != NULL)
      *tmp = '\0';
    if (atoIP4(cur, &ip) != 0)
      {
      snprintf(msg, 100, _("%s isn't a valid IP address!"), cur);
      problemBox(msg, _("Problem"));
      free(buf);
      return -1;
      }
    cur = (tmp == NULL) ? end : tmp + 1;  
    }
  while (cur < end);
  free(buf);
  return 0;
}



/*
 * Get the hostname of this system
 */
int get_host()
{
  char *def = NULL;

  for (;;) {
    if(host)
      def = host;
    else
      def = strdup("debian");

    snprintf(prtbuf, PRTBUFSIZE,
             _("Every Debian system has a name - even if it is not on a network. This name is called its \"hostname\". You should now specify a hostname for your new system.\n\nIf your system is going to be on a network, you should talk to the administrator of the network before you choose a hostname. If not, you may choose any name you like.\n\nThe hostname must not contain dots or underscores, and you must not append the domain name here.\n\nIf you can't think of a hostname for your new system, you may press <ENTER> to use the default hostname of \"%s\".\nPlease enter your Host name"), def);

    host = inputBox(prtbuf, _("Choose the Host name"), def);
    free(def);
    def = NULL;

    if (host == NULL)
      return 255;

    if (strlen(host) == 0) {
      free(host);
      host = NULL;
      return 255;
    }

    if(strchr(host, ' ')) {
      problemBox(_("The host name must be one word."), _("Problem"));
    } else {
      if (strchr(host, '.')) {
        problemBox(_("The host name must not contain dots."), _("Problem"));
      } else {
        if (isdigit(*host)) {
          problemBox(_("The host name contains leading numerical character(s).\nNote that the RFC does not allow this."),
                     _("Problem"));
        } else {
          break;
        }
      }
    }
  }
  return 0;
}

/*
 * Get all available interfaces from the kernel and ask the user which one
 * he wants to configure
 */
int get_netinterface()
{
  int i = 0, ret, ifcnt = is_network_up(1); /* the total number of interfaces */
  struct d_choices *opt = calloc(ifcnt, sizeof(struct d_choices));
  char *inter;

  getif_start();
  while ((inter = getif(1)) != NULL) {
    opt[i].tag = strdup(inter);
    opt[i++].string = get_ifdsc(inter);
  }
  getif_end();

  if (ifcnt==0) {
    if(netinterface) {
      free(netinterface);
      netinterface=NULL;
    }
    INFOMSG("no network devices found");
    problemBox(_("No network interfaces were found.   That means that the installation system was unable to find a network device.  If you do have a network card, then it is possible that the module for it hasn't been selected yet.  Go back to the 'Configure Device Driver Modules' menu and load the required kernel modules first.\n\nIf your only network connection is PPP, don't worry -- you will be asked if you want PPP later in the process."),
               _("Problem"));
    ret = 0;
  } else if (ifcnt == 1) {
    /* only one interface found, so use it */
    netinterface = strdup(opt[0].tag);
    ret = 0;
  } else {
    ret = menuBox(_("Choose the type of your primary network interface that you will need for installing Debian (via NFS or HTTP)"),
		_("Choose Network Interface"), opt, ifcnt, 1);
    if (ret >= 0) {
      netinterface = strdup(opt[ret].tag);
      ret = 0;
    } else {
      if(netinterface) {
	free(netinterface);
        netinterface=NULL;
      }
      ret = 255;
    }
  }

  has_pcmcia = 0;
  if (netinterface && NAME_ISDIR("/proc/bus/pccard", &nc_statbuf)) {
    snprintf(prtbuf, PRTBUFSIZE, _("The interface %s (%s) seems to be a PCMCIA card. Is this correct?"), 
             netinterface, get_ifdsc(netinterface));
    if (bootargs.isquiet || yesNoBox(prtbuf, _("Confirmation")) == DLG_YES) {
      has_pcmcia = 1;
    }
  }

  for (i=0; i<ifcnt; i++) {
    free(opt[i].tag);
    free(opt[i].string);
  }
  free(opt);
  return ret;
}



/*
 * Get IPv4 address, netmask and default gateway
 */
int get_ifconfig() {
  int ret, ix;

  /* Get the IP address */
  snprintf(prtbuf, PRTBUFSIZE,
           _("What is the IP address of system %s?"), host);
  ret = ask_for_IP4(&ipaddr, prtbuf,
                    _("Choose the IP Address"),
                    _("You entered an invalid IP address.  Ask your network administrator for the proper value."));
  if (ret == 255) return ret;

  /* Get the netmask */
  ret = ask_for_IP4(&netmask, _("What is your netmask?"),
                    _("Choose the Network Mask"),
		    _("You entered an invalid netmask.  Ask your network administrator for the proper value."));
  if (ret == 255) return ret;

  /*
   * Generate the network, broadcast and default gateway address
   * Default gateway address is network with last number set to "1", 
   * or "2" if "1" is the local address.
   */
  for (ix = 0; ix < IP4_ADDR_SZ; ix++) {
    gateway.i[ix] = network.i[ix] = ipaddr.i[ix] & netmask.i[ix];
    broadcast.i[ix] = (~netmask.i[ix] & 255) | ipaddr.i[ix];
  }
  gateway.i[IP4_ADDR_SZ - 1] |=
    (ipaddr.i[IP4_ADDR_SZ - 1] == (network.i[IP4_ADDR_SZ - 1] | 1)) ? 2 : 1;

  /* Get the gateway */
  has_gw = 1;
  ret = ask_for_IP4(&gateway, _("A gateway system is one that connects your network to other networks such as your company wide-area net or the Internet.\n\nIf there is no gateway in your subnet just leave this field blank."),
                    _("What is your IP gateway address?"),
                    _("You entered an invalid gateway address.  Ask your network administrator for the proper value."));
  if (ret == 255) has_gw = 0;

  return 0;
}



/*
 * Get the domain name of this system
 */
int get_domain() {
  char *def;
  
  for(;;) {
    if (domain) {
      def = strdup(domain);
      free(domain);
      domain = NULL;
    } else {
      def = strdup("");
    }
    domain = inputBox(_("As with individual systems, every network has a name. This name is called the domain name.\n\nPlease enter your domain name or leave this field empty if you don't have a domain."),
                      _("Choose the Domain name"), def);
    free(def);
    if (!domain) return 255;

    if (strchr(domain, ' ') || strchr(domain, '_')) {
      problemBox(_("Domain names cannot contain spaces or underscores"),
		 _("Problem"));
    } else {
      if (strlen(domain)==0) {
        free(domain);
        domain = NULL;
      }
      break;
    }
  }
  return 0;
}



/*
 * Get the IPv4 addresses of up to 3 name servers
 */
int get_dnsconfig() {
  char *def;
  
  for(;;) {
    if (nameserver) {
      def = strdup(nameserver);
      free(nameserver);
      nameserver = NULL;
    } else {
      if (has_gw) {
        def = IP4toa(gateway);
      } else {
        def = strdup("");
      }
    }

    nameserver = inputBox(_("Please enter the IP addresses (not host names) of up to 3 name servers, separated by spaces. Do not use commas. The servers will be queried in the order in which you enter them.\n\nIf you don't want to use any name servers just leave this field blank."),
                          _("Choose the DNS Server Addresses"), def);
    free(def);

    if (!nameserver) return 255;
    if (strlen(nameserver) == 0) {
      free(nameserver);
      nameserver = NULL;
      break;
    }
    
    if (check_ip4_nameserver(nameserver) == 0) {
      break;
    }
  }
  return 0;
}



/*
 * Get the PCMCIA transceiver type
 */
int get_pcmcia_transceiver() {
  for (;;) {
    transceiver = inputBox(_("Some PCMCIA Cards can detect the transceiver type although this might take some time. Others cannot detect it at all. Possible values for this parameter are \"auto\", \"10baseT\", \"10base2\", \"aui\", and \"100baseT\", or a blank for cards that don't support transceiver selection.\nPlease specify what transceiver type you would like to use. If you aren't sure, just use the default value of \"auto\"."),
			     _("PCMCIA Transceiver"), transceiver);
    if (transceiver) {
      if (transceiver[0] == '\0') {
        break;
      }
      if ((strcmp(transceiver, "auto") == 0 ) ||
          (strcmp(transceiver, "10baseT") == 0 ) ||
          (strcmp(transceiver, "10base2") == 0 ) ||
          (strcmp(transceiver, "aui") == 0 ) ||
          (strcmp(transceiver, "100baseT") == 0 )) {
         break;
      }
    }
  }
  return 0;
}

static int write_hostname() {
    FILE *p_file;
    const char *path;

    check_target_path();
    path = target_path(NC_ETC_DIR);
    if (check_dir(path) == -1) {
	mkdir(path, 0755);
    }

    if ((p_file = fopen(target_path(NC_HOSTNAME_FILE), "w"))) {
	fprintf(p_file, "%s\n", host);
	fclose(p_file);
    } else
	return 255;

    /* Just do this for now */
    if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
	fprintf(p_file, "127.0.0.1\t%s\n", host);
	fclose(p_file);
    } else
	return 255;

    return 0;
}

/*
 * Write the network configuration files that are common to a static,
 * DHCP and PCMCIA configuration.
 */
int write_common_network() {
  FILE *p_file;
  const char* path;
  int ret = 0;

  /* Create /target/etc and /target/etc/network if they don't exist */
  check_target_path();
  path = target_path(NC_ETC_DIR);
  if (check_dir(path) == -1) {
    mkdir(path, 0755);
  }
  path = target_path(NC_ETC_NETWORK_DIR);
  if (check_dir(path) == -1) {
    mkdir(path, 0755);
  }

  write_hostname();
  
  if (!netinterface || use_dhcp) {
    if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
      /* write_static_network will overwrite this file so we can just assume
       * that we use DHCP or have no netinterface at all
       */
      fprintf(p_file, "127.0.0.1\t%s\tlocalhost\n", host);
      fclose(p_file);
    } else {
      ret = 255;
    }
  }

  if ((p_file = fopen(target_path(NC_INTERFACES_FILE), "w"))) {
    fprintf(p_file, "# /etc/network/interfaces -- configuration file for ifup(8), ifdown(8)\n\n");
    fprintf(p_file, "# The loopback interface\n");
    fprintf(p_file, "iface lo inet loopback\n");
    fclose(p_file);
  } else {
    ret = 255;
  }

  return ret;
}



/*
 * Write the PCMCIA network configuration (static and DHCP)
 */
int write_pcmcia_network() {
  FILE *p_file;
  char *tmp, *ns, *aux;
  const char* path;
  int ix;

  path = target_path(NC_PCMCIA_DIR);
  if (check_dir(path) == -1) mkdir(path, 0755);
      
  if ((p_file = fopen(target_path(NC_PCMCIA_FILE), "w"))) {
    fprintf(p_file, "# Network adapter configuration\n#\n# Automatically generated during the Debian installation\n#\n");
    fprintf(p_file, "# The address format is \"scheme,socket,instance,hwaddr\".\n#\ncase \"$ADDRESS\" in\n*,*,*,*)\n");
    fprintf(p_file, "\t# Transceiver selection, for some cards -- see 'man ifport'\n");
    if (transceiver) {
      fprintf(p_file, "\tIF_PORT=\"%s\"\n", transceiver);
    } else {
      fprintf(p_file, "\tIF_PORT=\"\"\n");
    }
    fprintf(p_file, "\t# Use /sbin/pump for BOOTP/DHCP? [y/n]\n");
    if(use_dhcp) {
      fprintf(p_file, "\tPUMP=\"y\"\n");
    } else {
      fprintf(p_file, "\tPUMP=\"n\"\n");
      fprintf(p_file, "\t# Host's IP address, netmask, network address, broadcast address\n");
      fprintf(p_file, "\tIPADDR=\"%s\"\n", IP4tostr(prtbuf, ipaddr));
      fprintf(p_file, "\tNETMASK=\"%s\"\n", IP4tostr(prtbuf, netmask));
      fprintf(p_file, "\tNETWORK=\"%s\"\n", IP4tostr(prtbuf, network));
      fprintf(p_file, "\tBROADCAST=\"%s\"\n", IP4tostr(prtbuf, broadcast));
      if (has_gw) {
        fprintf(p_file, "\tGATEWAY=\"%s\"\n", IP4tostr(prtbuf, gateway));
      } else {
        fprintf(p_file, "\tGATEWAY=\"\"\n");
      }
      if (domain) {
        fprintf(p_file, "\tDOMAIN=\"%s\"\n", domain);
      }
      if (nameserver) {
        aux = ns = strdup(nameserver);
        ix = 1;
        while ((tmp = strchr(ns, ' ')) != NULL) {
          *tmp = '\0';
          tmp++;
          fprintf(p_file, "\tDNS_%d=\"%s\"\n", ix, ns);
          ns = tmp;
          ix++;
        }
        fprintf(p_file, "\tDNS_%d=\"%s\"\n", ix, ns);
        free(aux);
      } else {
        fprintf(p_file, "\tDNS_1=\"\"\n");
      }
    }
    fprintf(p_file, "\t;;\nesac\n");
    fclose(p_file);
    return 0;
  } else {
    return 255;
  }
}


/*
 * Write the static network configuration
 * NOTE: This function must not be called with netinterface == NULL
 */
int write_static_network() {
  FILE *p_file;
  char *tmp, *ns, *aux;
  int ret;

  ret = write_common_network();
  if (ret == 255) return ret;

  if ((p_file = fopen(target_path(NC_HOSTS_FILE), "w"))) {
    fprintf(p_file, "127.0.0.1\tlocalhost\n");
    if (domain) {
      fprintf(p_file, "%s\t%s.%s\t%s\n", IP4tostr(prtbuf, ipaddr),
                                         host, domain, host);
    } else {
      fprintf(p_file, "%s\t%s\n", IP4tostr(prtbuf, ipaddr), host);
    }
    fclose(p_file);
  } else {
    ret = 255;
  }

  if ((p_file = fopen(target_path(NC_NETWORKS_FILE), "w"))) {
    fprintf(p_file, "localnet %s\n", IP4tostr(prtbuf, network));
    fclose(p_file);
  } else {
    ret = 255;
  }

  if ((p_file = fopen(target_path(NC_RESOLV_FILE), "w"))) {
    if (domain) fprintf(p_file, "search %s\n", domain);
    if (nameserver) {
      aux = ns = strdup(nameserver);
      while ((tmp = strchr(ns, ' ')) != NULL) {
        *tmp = '\0';
        tmp++;
        fprintf(p_file, "nameserver %s\n", ns);
        ns = tmp;
      }
      fprintf(p_file, "nameserver %s\n", ns);
      free(aux);
    }
    fclose(p_file);
    /* if this is a live CD filesystem, symlink to /tmp/resolv.conf too */
    if (is_cdrom_image() == 2) {
	unlink("/tmp/resolv.conf");
	symlink(target_path(NC_RESOLV_FILE), "/tmp/resolv.conf");
    }
  } else {
    ret = 255;
  }

  /*
   * Only write /etc/network/interfaces if we have a non-PCMCIA interface.
   */
  if (has_pcmcia) {
    ret = write_pcmcia_network();
  } else {
    if ((p_file = fopen(target_path(NC_INTERFACES_FILE), "a"))) {
      fprintf(p_file, "\n# The first network card - this entry was created during the Debian installation\n");
      fprintf(p_file, "# (network, broadcast and gateway are optional)\n");
      fprintf(p_file, "iface %s inet static\n", netinterface);
      fprintf(p_file, "\taddress %s\n", IP4tostr(prtbuf, ipaddr));
      fprintf(p_file, "\tnetmask %s\n", IP4tostr(prtbuf, netmask));
      fprintf(p_file, "\tnetwork %s\n", IP4tostr(prtbuf, network));
      fprintf(p_file, "\tbroadcast %s\n", IP4tostr(prtbuf, broadcast));
      if (has_gw) {
        fprintf(p_file, "\tgateway %s\n", IP4tostr(prtbuf, gateway));
      }
      fclose(p_file);
    } else {
      ret = 255;
    }
  }
  return ret;
}



/*
 * Write the network configuration for a DHCP setup
 */
int write_dhcp_network() {
  FILE *p_file;
  int ret;

  ret = write_common_network();
  if (ret == 255) return ret;

  /* Copy /etc/resolv.conf (created by pump) to the target system */
  copyFile(NC_RESOLV_FILE, target_path(NC_RESOLV_FILE), 0, 0);

  if (has_pcmcia) {
    write_pcmcia_network();
  } else {
    p_file = fopen(target_path(NC_INTERFACES_FILE), "a");
    fprintf(p_file, "\n# The first network card - this entry was created during the Debian installation\n");
    fprintf(p_file, "iface %s inet dhcp\n", netinterface);
    fclose(p_file);
  }
  return 0;
}



/*
 * Activates the static network configuration (we don't add the network
 * route to the interfaces, this is not required for kernel 2.2)
 */
int activate_static_network()
{
  int ret = 0;
  if (NAME_ISEXE("/sbin/ifconfig", &nc_statbuf)) {
#ifndef _TESTING_
    unlink(NC_RESOLV_FILE);
    symlink(target_path(NC_RESOLV_FILE), NC_RESOLV_FILE);
#endif

    /*
     * Always activate the loopback interface and don't
     * activate PCMCIA (this is done in activate_pcmcia)
     */

    ret = execlog("/sbin/ifconfig lo 127.0.0.1", LOG_INFO);
    if (!use_dhcp && !has_pcmcia && (ret==0) && netinterface) {
      char cmdline[MAXLINE];
      snprintf(cmdline, MAXLINE-5, "/sbin/ifconfig %s %s netmask ",
               netinterface, IP4tostr(prtbuf, ipaddr));
      strncat(cmdline, IP4tostr(prtbuf, netmask), MAXLINE-strlen(cmdline)-5);
      strncat(cmdline, " broadcast ", MAXLINE-strlen(cmdline)-5);
      strncat(cmdline, IP4tostr(prtbuf, broadcast), MAXLINE-strlen(cmdline)-5);
      ret = execlog(cmdline, LOG_INFO);
      if (has_gw && (ret==0)) {
        snprintf(cmdline, MAXLINE-5, "/sbin/route add default gw %s metric 1",
                 IP4tostr(prtbuf, gateway));
        ret = execlog(cmdline, LOG_INFO);
      }
    }
  }
  return ret;
}



/*
 * Configuration for static (i.e. not DHCP) network setups. This
 * function is called by configure_network() and calls the other
 * get_XXX() functions
 */
int configure_static_network() {
  int ret;
  has_gw = 0;
  transceiver = strdup("auto");

  if (!netinterface) {
    write_common_network();
  } else {
    ret = get_ifconfig();
    if (ret == 255) return ret;

    ret = get_domain();
    if (ret == 255) return ret;

    ret = get_dnsconfig();
    if (ret == 255) return ret;

    if (has_pcmcia) {
      ret = get_pcmcia_transceiver();
      if (ret == 255) return ret;
    }

    write_static_network();

    if (has_pcmcia && activate_pcmcia()) {
        problemBox(_("As soon as you configure the PCMCIA support, your network configuration will be automatically activated."),
                   _("PCMCIA Networking"));
    }
  }

  ret = activate_static_network();

  return ret;
}

int configure_hostname() {
    int ret;

    ret = get_host();
    if(ret == 255) return ret;
    ret = write_hostname();
    return ret;
}

/*
 * Main configuration routine: First get the network interface to
 * configure, then ask about DHCP/BOOTP.
 */
int configure_network() {
  int ret;
  struct stat hf;
  use_dhcp = 0;

  /* TODO: we need to see if there is already a network configuration, and
   * the interfaces are just not up. Give the user a choice to simply
   * activate it. This will require us to read in the config files and
   * setup out vars from there, however. */

  if (!NAME_ISREG(target_path(NC_HOSTNAME_FILE),&hf) || !host) {
    ret = get_host();
    if (ret == 255) return ret;
  }

  ret = get_netinterface();
  if (ret == 255) return ret;

  /* DHCP is only supported on Ethernet */
  if (netinterface &&
      (strncmp(netinterface, "eth", 3) == 0) &&
      (NAME_ISEXE("/sbin/pump", &nc_statbuf))) {
    snprintf(prtbuf, PRTBUFSIZE, 
	     _("Configuration for %s (%s)\n\nDo you want to use DHCP or BOOTP to automatically configure this interface? You'll need a DHCP or BOOTP server in the local network for this to work."), 
	     netinterface, get_ifdsc(netinterface));
    if (yesNoBox(prtbuf, _("Automatic Network Configuration")) == DLG_NO) {
      ret = configure_static_network();
    } else {
      pleaseWaitBox(_("Trying to get network configuration using DHCP/BOOTP...\n\nThis may take a few minutes, be patient."));
      /* Kill pump first, in case it is already running - currently disabled
       * because of problems with PCMCIA cards
       */
#if 0
      execlog("/sbin/pump -k", LOG_DEBUG);
#endif
      snprintf(prtbuf, MAXLINE-5, "/sbin/pump -i %s -h %s", netinterface, host);
      if (execlog(prtbuf, LOG_INFO)) {
	boxPopWindow();
        snprintf(prtbuf, PRTBUFSIZE, _("Automatic configuration using DHCP/BOOTP failed for %s (%s).\n\nDo you want to manually configure this interface?"), netinterface, get_ifdsc(netinterface));
        if (yesNoBox(prtbuf, _("Automatic Network Configuration")) == DLG_YES) {
          ret = configure_static_network();
        } else {
	  return DLG_CANCEL;
        }
      } else {
	boxPopWindow();
        use_dhcp = 1;
        if (!bootargs.isquiet) {
          problemBox(_("The network has been successfully configured using DHCP/BOOTP."),
                     _("Information"));
        }
        ret = write_dhcp_network();
        ret = activate_static_network();     /* For loopback interface */
      }
    }
  } else {
    ret = configure_static_network();
  }
  return ret;
}



#ifdef _TESTING_
/*
 * To test, compile using: make netconfig_test
 */
int main()
{
  LOAD_TRMFILE("test.trm");
  get_kver();
  if (check_dir("targetdir_test") == -1)
    mkdir("targetdir_test", 0755);

  boxInit();

  configure_network();

  boxFinished();
  return 0;
}
#endif
