/* cfengine for GNU
 
        Copyright (C) 1995
        Free Software Foundation, Inc.
 
   This file is part of GNU cfengine - written and maintained 
   by Mark Burgess, Dept of Computing and Engineering, Oslo College,
   Dept. of Theoretical physics, University of Oslo
 
   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any
   later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA

*/
 

/********************************************************************/
/*                                                                  */
/* EDITING of simple textfiles (Toolkit)                            */
/*                                                                  */
/********************************************************************/

#include "cf.defs.h"
#include "cf.extern.h"

/********************************************************************/
/* EDIT Data structure routines                                     */

/********************************************************************/

DoEditHomeFiles(ptr)

struct Edit *ptr;

{ DIR *dirh, *dirh2;
  struct dirent *dirp, *dirp2;
  char username[maxvarsize], *sp;
  char homedir[bufsize],dest[bufsize];
  struct passwd *pw;
  struct stat statbuf;
  struct Item *ip;
  uid_t uid;
  
if (!MountPathDefined())
   {
   CfLog(cfinform,"Mountpattern is undefined\n","");
   return;
   }

for (ip = VMOUNTLIST; ip != NULL; ip=ip->next)
   {
   if (IsExcluded(ip->classes))
      {
      continue;
      }
   
   if ((dirh = opendir(ip->name)) == NULL)
      {
      sprintf(OUTPUT,"Can't open directory %s\n",ip->name);
      CfLog(cferror,OUTPUT,"opendir");
      return;
      }

   for (dirp = readdir(dirh); dirp != NULL; dirp = readdir(dirh))
      {
      if (!SensibleFile(dirp->d_name,ip->name,NULL))
         {
         continue;
         }

      strcpy(homedir,ip->name);
      AddSlash(homedir);
      strcat(homedir,dirp->d_name);

      if (! IsHomeDir(homedir))
         {
         continue;
         }

      if ((dirh2 = opendir(homedir)) == NULL)
         {
         sprintf(OUTPUT,"Can't open directory%s\n",homedir);
         CfLog(cferror,OUTPUT,"opendir");
         return;
         }

      for (dirp2 = readdir(dirh2); dirp2 != NULL; dirp2 = readdir(dirh2))
         {
         if (!SensibleFile(dirp2->d_name,homedir,NULL))
	    {
	    continue;
	    }

         strcpy(username,dirp2->d_name);
         strcpy(dest,homedir);
         AddSlash(dest);
         strcat(dest,dirp2->d_name);
         AddSlash(dest);
         sp = ptr->fname + strlen("home/");
         strcat(dest,sp);

         if (stat(dest,&statbuf))
            {
            Verbose("%s: file %s doesn't exist for editing, skipping\n",VPREFIX,dest);
            continue;
            }
      
         if ((pw = getpwnam(username)) == NULL)
            {
            Debug2("cfengine: directory corresponds to no user %s - ignoring\n",username);
            continue;
            }
         else
            {
            Debug2("(Setting user id to %s)\n",username);
            }

         uid = statbuf.st_uid;

         WrapDoEditFile(ptr,dest);
      
         chown(dest,uid,sameowner);
         }
      closedir(dirh2);
      }
   closedir(dirh);
   }
}

/********************************************************************/

WrapDoEditFile(ptr)

struct Edit *ptr;

{ struct stat statbuf,statbuf2;
  char linkname[bufsize];
  char realname[bufsize];

Debug("WrapDoEditFile(%s)\n",ptr->fname);
  
if (lstat(ptr->fname,&statbuf) != -1)
   {
   if (S_ISLNK(statbuf.st_mode))
      {
      Verbose("%s: File %s is a link, editing real file instead\n",VPREFIX,ptr->fname);
      
      bzero(linkname,bufsize);
      bzero(realname,bufsize);
      
      if (readlink(ptr->fname,linkname,bufsize) == -1)
	 {
	 sprintf(OUTPUT,"Cannot read link %s\n",ptr->fname);
	 CfLog(cferror,OUTPUT,"readlink");
	 return;
	 }
      
      if (linkname[0] != '/')
	 {
	 strcpy(realname,ptr->fname);
	 ChopLastNode(realname);
	 AddSlash(realname);
	 }
      
      if (BufferOverflow(realname,linkname))
	 {
	 sprintf(OUTPUT,"(culprit %s in editfiles)\n",ptr->fname);
	 CfLog(cferror,OUTPUT,"");
	 return;
	 }
      
      if (stat(ptr->fname,&statbuf2) != -1)
	 {
	 if (statbuf2.st_uid != statbuf.st_uid)
	    {
	    /* Link to /etc/passwd? ouch! */
	    sprintf(OUTPUT,"Forbidden to edit a link to another user's file with privilege (%s)",ptr->fname);
	    CfLog(cfinform,OUTPUT,"");
	    return;
	    }
	 }
      
      strcat(realname,linkname);
      DoEditFile(ptr,realname);
      }
   else
      {
      DoEditFile(ptr,ptr->fname);
      }
   }
else
   {
   DoEditFile(ptr,ptr->fname);
   }
}

/********************************************************************/

DoEditFile(ptr,filename)

struct Edit *ptr;
char *filename;


   /* Many of the functions called here are defined in the */
   /* item.c toolkit since they operate on linked lists    */

{ struct Edlist *ep, *loopstart, *loopend, *ThrowAbort();
  struct Item *filestart = NULL, *newlineptr;
  char *currenteditscript, *searchstr;
  struct stat tmpstat;
  char spliton = ':';
  int todo = 0;
  int foreachloop = false;
  FILE *loop_fp;

Debug("DoEditFile(%s)\n",filename);
filestart = NULL;
currenteditscript = NULL;
searchstr = NULL;
bzero(ACTIONBUFF,bufsize);
AUTOCREATED = false;
IMAGEBACKUP = true;

for (ep = ptr->actions; ep != NULL; ep=ep->next)
   {
   if (!IsExcluded(ep->classes))
      {
      todo++;
      }
   }

if (todo == 0)   /* Because classes are stored per edit, not per file */
   {
   return;
   }

if (!GetLock(ASUniqueName("editfile"),CanonifyName(ptr->fname),VIFELAPSED,VEXPIREAFTER,VUQNAME,CFSTARTTIME))
   {
   return;
   }

CheckEditSwitches(filename,ptr->actions);

if (! LoadItemList(&filestart,filename))
   {
   CfLog(cfverbose,"File was marked for editing\n","");
   ReleaseCurrentLock();
   return;
   }

NUMBEROFEDITS = 0;
EDITVERBOSE = VERBOSE;
CURRENTLINENUMBER = 1;
CURRENTLINEPTR = filestart;
COMMENTSTART = "# ";
COMMENTEND = "";
EDITGROUPLEVEL = 0;
SEARCHREPLACELEVEL = 0;
FOREACHLEVEL = 0;
loopstart = NULL;

Verbose("%s: Begin editing %s\n",VPREFIX,filename);
ep = ptr->actions;

while (ep != NULL)
   {
   if (IsExcluded(ep->classes))
      {
      ep = ep->next;
      continue;
      }

   Debug2("Edit action: %s\n",VEDITNAMES[ep->code]);

   switch(ep->code)
      {
      case NoEdit:
      case AutoCreate:
               break;

      case CatchAbort:
	          EditVerbose("%s: Caught Exception\n",VPREFIX);
	          break;

      case SplitOn:
	          spliton = *(ep->data);
		  EditVerbose("%s: Split lines by %c\n",VPREFIX,spliton);
		  break;

      case DeleteLinesStarting:
               while (DeleteItemStarting(&filestart,ep->data))
                  {
                  }
               break;

      case DeleteLinesContaining:
               while (DeleteItemContaining(&filestart,ep->data))
                  {
                  }
               break;

      case DeleteLinesAfterThisMatching:

               if ((filestart == NULL) || (CURRENTLINEPTR == NULL))
		  {
		  break;
		  }
	       else if (CURRENTLINEPTR->next != NULL)
		  {
                  while (DeleteItemMatching(&(CURRENTLINEPTR->next),ep->data))
                    {
	   	    }
		  }
               break;	       

      case DeleteLinesMatching:
               while (DeleteItemMatching(&filestart,ep->data))
                 {
		 }
               break;
	       
      case Append:
              AppendItem(&filestart,ep->data,NULL);
              break;

      case AppendIfNoSuchLine:
               if (! IsItemIn(filestart,ep->data))
                  {
                  AppendItem(&filestart,ep->data,NULL);
                  }
               break;

      case SetLine:
               strcpy(ACTIONBUFF,ep->data);
               EditVerbose("%s: Set current line to %s\n",VPREFIX,ACTIONBUFF);
               break;

      case AppendIfNoLineMatching:

               if (strcmp(ACTIONBUFF,"") == 0)
                  {
                  sprintf(OUTPUT,"%s: SetLine not set when calling AppendIfNoLineMatching %s\n",VPREFIX,ep->data);
		  CfLog(cferror,OUTPUT,"");
                  continue;
                  }
	       
               if (LocateNextItemMatching(filestart,ep->data) == NULL)
	          {
                  AppendItem(&filestart,ACTIONBUFF,NULL);
                  }
               break;

      case Prepend:
               PrependItem(&filestart,ep->data,NULL);
               break;

      case PrependIfNoSuchLine:
               if (! IsItemIn(filestart,ep->data))
                  {
                  PrependItem(&filestart,ep->data,NULL);
                  }
               break;

      case PrependIfNoLineMatching:

               if (strcmp(ACTIONBUFF,"") == 0)
                  {
                  printf("%s: SetLine not set when calling PrependIfNoLineMatching %s\n",VPREFIX,ep->data);
                  continue;
                  }

               if (LocateNextItemMatching(filestart,ep->data) == NULL)
                  {
                  PrependItem(&filestart,ACTIONBUFF,NULL);
                  }
               break;

      case WarnIfNoSuchLine:
               if (LocateNextItemMatching(filestart,ep->data) == NULL)
                  {
                  printf("%s: Warning, file %s has no line matching %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfLineMatching:
               if (LocateNextItemMatching(filestart,ep->data) != NULL)
                  {
                  printf("%s: Warning, file %s has a line matching %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfNoLineMatching:
               if (LocateNextItemMatching(filestart,ep->data) == NULL)
                  {
                  printf("%s: Warning, file %s has a no line matching %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfLineStarting:
               if (LocateNextItemStarting(filestart,ep->data) != NULL)
                  {
                  printf("%s: Warning, file %s has a line starting %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfNoLineStarting:
               if (LocateNextItemStarting(filestart,ep->data) == NULL)
                  {
                  printf("%s: Warning, file %s has no line starting %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfLineContaining:
               if (LocateNextItemContaining(filestart,ep->data) != NULL)
                  {
                  printf("%s: Warning, file %s has a line containing %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case WarnIfNoLineContaining:
               if (LocateNextItemContaining(filestart,ep->data) == NULL)
                  {
                  printf("%s: Warning, file %s has no line containing %s\n",VPREFIX,filename,ep->data);
                  }
               break;

      case SetCommentStart:
               COMMENTSTART = ep->data;
               break;

      case SetCommentEnd:
               COMMENTEND = ep->data;
               break;

      case CommentLinesMatching:
               while (CommentItemMatching(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  }
               break;

      case CommentLinesStarting:
               while (CommentItemStarting(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  }
               break;

      case CommentLinesContaining:
               while (CommentItemContaining(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  }
               break;

      case HashCommentLinesContaining:
               while (CommentItemContaining(&filestart,ep->data,"# ",""))
                  {
                  }
               break;

      case HashCommentLinesStarting:
               while (CommentItemStarting(&filestart,ep->data,"# ",""))
                  {
                  }
               break;

      case HashCommentLinesMatching:
               while (CommentItemMatching(&filestart,ep->data,"# ",""))
                  {
                  }
               break;

      case SlashCommentLinesContaining:
               while (CommentItemContaining(&filestart,ep->data,"//",""))
                  {
                  }
               break;

      case SlashCommentLinesStarting:
               while (CommentItemStarting(&filestart,ep->data,"//",""))
                  {
                  }
               break;

      case SlashCommentLinesMatching:
               while (CommentItemMatching(&filestart,ep->data,"//",""))
                  {
                  }
               break;

      case PercentCommentLinesContaining:
               while (CommentItemContaining(&filestart,ep->data,"%",""))
                  {
                  }
               break;

      case PercentCommentLinesStarting:
               while (CommentItemStarting(&filestart,ep->data,"%",""))
                  {
                  }
               break;

      case PercentCommentLinesMatching:
               while (CommentItemMatching(&filestart,ep->data,"%",""))
                  {
                  }
               break;

      case ResetSearch:
               if (!ResetEditSearch(ep->data,filestart))
                  {
                  printf("%s: ResetSearch Failed in %s, aborting editing\n",VPREFIX,filename);
                  goto abort;
                  }
               break;

      case LocateLineMatching:
               newlineptr = LocateItemMatchingRegExp(CURRENTLINEPTR,ep->data);

               if (newlineptr == NULL)
                  {
                  EditVerbose("%s: LocateLineMatchingRegexp failed in %s, aborting editing\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case InsertLine:
	          if (filestart == NULL)
		     {
		     AppendItem(&filestart,ep->data,NULL);
		     }
		  else
		     {
		     InsertItemAfter(&filestart,CURRENTLINEPTR,ep->data);
		     }
                  break;

      case InsertFile:
	          InsertFileAfter(&filestart,CURRENTLINEPTR,ep->data);
		  break;

      case IncrementPointer:
               if (! IncrementEditPointer(ep->data,filestart))     /* edittools */
                  {
                  printf ("%s: IncrementPointer failed in %s, aborting editing\n",VPREFIX,filename);
  	          ep = ThrowAbort(ep);
                  }

               break;
	       
     case ReplaceLineWith:
               if (!ReplaceEditLineWith(ep))
                  {
                  printf("%s: aborting edit of file %s\n",VPREFIX,filename);
                  continue;
                  }
               break;

      case DeleteToLineMatching:
               if (! DeleteToRegExp(&filestart,ep->data))
                  {
                  EditVerbose("%s: Nothing matched DeleteToLineMatching regular expression\n",VPREFIX);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX ,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case DeleteNLines:
               if (! DeleteSeveralLines(&filestart,ep->data))
                  {
                  EditVerbose("%s: Could not delete %s lines from file\n",VPREFIX,ep->data);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case HashCommentToLineMatching:
               if (! CommentToRegExp(&filestart,ep->data,"#",""))
                  {
                  EditVerbose("%s: Nothing matched HashCommentToLineMatching regular expression\n",VPREFIX);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case PercentCommentToLineMatching:
               if (! CommentToRegExp(&filestart,ep->data,"%",""))
                  {
                  EditVerbose("%s: Nothing matched PercentCommentToLineMatching regular expression\n",VPREFIX);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case CommentToLineMatching:
               if (! CommentToRegExp(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  EditVerbose("%s: Nothing matched CommentToLineMatching regular expression\n",VPREFIX);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case CommentNLines:
               if (! CommentSeveralLines(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  EditVerbose("%s: Could not comment %s lines from file\n",VPREFIX,ep->data);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;
	       
      case UnCommentNLines:
               if (! UnCommentSeveralLines(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  EditVerbose("%s: Could not comment %s lines from file\n",VPREFIX,ep->data);
                  EditVerbose("%s: Aborting file editing of %s.\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case UnCommentLinesContaining:
               while (UnCommentItemContaining(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  }
               break;

      case UnCommentLinesMatching:
               while (UnCommentItemMatching(&filestart,ep->data,COMMENTSTART,COMMENTEND))
                  {
                  }
               break;
	       
      case SetScript:
               currenteditscript = ep->data;
               break;

      case RunScript:
               if (! RunEditScript(ep->data,filename,&filestart))
                  {
                  printf("%s: Aborting further edits to %s\n",VPREFIX,filename);
                  ep = ThrowAbort(ep);
                  }
               break;

      case RunScriptIfNoLineMatching:
               if (! LocateNextItemMatching(filestart,ep->data))
                  {
                  if (! RunEditScript(currenteditscript,filename,&filestart))
                     {
                     printf("%s: Aborting further edits to %s\n",VPREFIX,filename);
                     ep = ThrowAbort(ep);
                     }
                  }
               break;

      case RunScriptIfLineMatching:
               if (LocateNextItemMatching(filestart,ep->data))
                  {
                  if (! RunEditScript(currenteditscript,filename,&filestart))
                     {
                     printf("%s: Aborting further edits to %s\n",VPREFIX,filename);
                     ep = ThrowAbort(ep);
                     }
                  }
               break;

      case EmptyEntireFilePlease:
               EditVerbose("Emptying entire file\n");
               DeleteItemList(filestart);
	       filestart = NULL;
	       CURRENTLINEPTR = NULL;
	       CURRENTLINENUMBER=0;
               NUMBEROFEDITS++;
               break;

      case GotoLastLine:
               GotoLastItem(filestart);
               break;

      case BreakIfLineMatches:
               if (LineMatches(CURRENTLINEPTR->name,ep->data))
                  {
                  EditVerbose("Break! %s\n",ep->data);
                  goto abort;
                  }
               break;

      case BeginGroupIfNoMatch:
	       if (CURRENTLINEPTR == NULL || CURRENTLINEPTR->name == NULL )
		  {
		  EditVerbose("(Begin Group - no match for %s - file empty)\n",ep->data);
		  break;
		  }
	       
               if (LineMatches(CURRENTLINEPTR->name,ep->data))
                  {
                  EditVerbose("(Begin Group - skipping %s)\n",ep->data);
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - no match for %s)\n",ep->data);
                  }
               break;

     case BeginGroupIfNoLineMatching:
               if (LocateItemMatchingRegExp(filestart,ep->data) != 0)
                  {
                  EditVerbose("(Begin Group - skipping)\n");
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - no line matching %s)\n",ep->data);
                  }
               break;

      case BeginGroupIfNoLineContaining:
               if (LocateNextItemContaining(filestart,ep->data) != 0)
                  {
                  EditVerbose("(Begin Group - skipping, string matched)\n");
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - no line containing %s)\n",ep->data);
                  }
               break;

      case BeginGroupIfNoSuchLine:
               if (IsItemIn(filestart,ep->data))
                  {
                  EditVerbose("(Begin Group - skipping, line exists)\n");
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - no line %s)\n",ep->data);
                  }
               break;


     case BeginGroupIfFileIsNewer:
               if ((!AUTOCREATED) && (!FileIsNewer(filename,ep->data)))
                  {
                  EditVerbose("(Begin Group - skipping, file is older)\n");
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - new file %s)\n",ep->data);
                  }
               break;

     case BeginGroupIfFileExists:
               if (stat(ep->data,&tmpstat) == -1)
                  {
                  EditVerbose("(Begin Group - file unreadable/no such file - skipping)\n");
                  while(ep->code != EndGroup)
                     {
                     ep=ep->next;
                     }
                  }
               else
                  {
                  EditVerbose("(Begin Group - found file %s)\n",ep->data);
                  }
               break;
      case EndGroup:
               EditVerbose("(End Group)\n");
               break;

      case ReplaceAll:
               searchstr = ep->data;
               break;

      case With:
               GlobalReplace(&filestart,searchstr,ep->data);
               break;

      case FixEndOfLine:
               DoFixEndOfLine(filestart,ep->data);
	       break;

      case AbortAtLineMatching:
               EDABORTMODE = true;
	       strcpy(VEDITABORT,ep->data);
	       break;

      case UnsetAbort:
               EDABORTMODE = false;
	       break;

      case AutoMountDirectResources:
	       HandleAutomountRescources(&filestart,ep->data);
	       break;

      case ForEachLineIn:
	       if (loopstart == NULL)
		  {
	          loopstart = ep;

		  if ((loop_fp = fopen(ep->data,"r")) == NULL)
		     {
		     EditVerbose("Couldn't open %s\n",ep->data);
		     while(ep->code != EndLoop) /* skip over loop */
		        {
		        ep = ep->next;
		        }
		     break;
		     }
		  
		  EditVerbose("Starting ForEach loop with %s\n",ep->data);
		  continue;
		  }
	       else
		  {
	          if (!feof(loop_fp))
		     {
		     bzero(ACTIONBUFF,bufsize);
		     ReadLine(ACTIONBUFF,bufsize,loop_fp); /* Like SetLine */
		     }
		  else
		     {
		     EditVerbose("EndForeach\n");
		     
		     fclose(loop_fp);
		     loopstart = NULL;
		     
		     while(ep->code != EndLoop)
		        {
		        ep = ep->next;
		        }
		     }
		  }
	       
	       break;

      case EndLoop:
	       loopend = ep;
   	       ep = loopstart;
	       continue;

      case ReplaceLinesMatchingField:
	       ReplaceWithFieldMatch(&filestart,ep->data,ACTIONBUFF,spliton,filename);
	       break;

      case AppendToLineIfNotContains:
	       AppendToLine(CURRENTLINEPTR,ep->data,filename);
	       break;

      case DefineClasses:
	       break;

      default: printf("%s: Unknown action in editing of file %s\n",VPREFIX,filename);
               break;
      }

   ep = ep->next;
   }

abort :  

EditVerbose("%s: End editing %s\n",VPREFIX,filename);
EditVerbose(".....................................................................\n");

EDITVERBOSE = false;

if (DONTDO || CompareToFile(filestart,filename))
   {
   NUMBEROFEDITS = 0;
   }
 
if ((! DONTDO) && (NUMBEROFEDITS > 0))
   {
   SaveItemList(filestart,filename);
   AddEditfileClasses(ptr);
   }

ResetOutputRoute('d','d');
ReleaseCurrentLock();

DeleteItemList(filestart);
}

/********************************************************************/

IncrementEditPointer(str,liststart)

char *str;
struct Item *liststart;

{ int i,n = 0;
  struct Item *ip;

sscanf(str,"%d", &n);

if (n == 0)
   {
   printf("%s: Illegal increment value: %s\n",VPREFIX,str);
   return false;
   }

Debug("IncrementEditPointer(%d)\n",n);

if (CURRENTLINEPTR == NULL)  /* is prev undefined, set to line 1 */
   {
   if (liststart == NULL)
      {
      EditVerbose("cannot increment line pointer in empty file\n");
      return true;
      }
   else
      {
      CURRENTLINEPTR=liststart;
      CURRENTLINENUMBER=1;
      }
   }


if (n < 0)
   {
   if (CURRENTLINENUMBER + n < 1)
      {
      EditVerbose("%s: pointer decrements to before start of file!\n",VPREFIX);
      EditVerbose("%s: pointer stopped at start of file!\n",VPREFIX);
      CURRENTLINEPTR=liststart;
      CURRENTLINENUMBER=1;      
      return true;
      }

   i = 1;

   for (ip = liststart; ip != CURRENTLINEPTR; ip=ip->next, i++)
      {
      if (i == CURRENTLINENUMBER + n)
         {
         CURRENTLINENUMBER += n;
         CURRENTLINEPTR = ip;
	 Debug2("Current line (%d) starts: %20.20s ...\n",CURRENTLINENUMBER,CURRENTLINEPTR->name);

         return true;
         }
      }
   }

for (i = 0; i < n; i++)
   {
   if (CURRENTLINEPTR->next != NULL)
      {
      CURRENTLINEPTR = CURRENTLINEPTR->next;
      CURRENTLINENUMBER++;

      EditVerbose("incrementing line pointer to line %d\n",CURRENTLINENUMBER);
      }
   else
      {
      EditVerbose("inc pointer failed, still at %d\n",CURRENTLINENUMBER);
      }
   }

Debug2("Current line starts: %20s ...\n",CURRENTLINEPTR->name);

return true;
}

/********************************************************************/

ResetEditSearch (str,list)

char *str;
struct Item *list;

{ int i = 1 ,n = -1;
  struct Item *ip;

sscanf(str,"%d", &n);

if (n < 1)
   {
   printf("%s: Illegal reset value: %s\n",VPREFIX,str);
   return false;
   }

for (ip = list; (i < n) && (ip != NULL); ip=ip->next, i++)
   {
   }

if (i < n || ip == NULL)
   {
   printf("%s: Search for (%s) begins after end of file!!\n",VPREFIX,str);
   return false;
   }

EditVerbose("resetting pointers to line %d\n",n);

CURRENTLINENUMBER = n;
CURRENTLINEPTR = ip;

return true;
}

/********************************************************************/

ReplaceEditLineWith (editptr)

struct Edlist *editptr;

{ char *sp;

if (strcmp(editptr->data,CURRENTLINEPTR->name) == 0)
   {
   EditVerbose("ReplaceLineWith - line does not need correction.\n");
   return true;
   }

if ((sp = malloc(strlen(editptr->data)+1)) == NULL)
   {
   printf("%s: Memory allocation failed in ReplaceEditLineWith, aborting edit.\n",VPREFIX);
   return false;
   }

EditVerbose("Replacing line %d with %10s...\n",CURRENTLINENUMBER,editptr->data);
strcpy(sp,editptr->data);
free (CURRENTLINEPTR->name);
CURRENTLINEPTR->name = sp;
NUMBEROFEDITS++;
return true;
}

/********************************************************************/

RunEditScript (script,fname,filestart)

char *script, *fname;
struct Item **filestart;

{ FILE *pp;
  char buffer[bufsize];

if (script == NULL)
   {
   printf("%s: No script defined for with SetScript\n",VPREFIX);
   return false;
   }

if (DONTDO)
   {
   return true;
   }

if (NUMBEROFEDITS > 0)
   {
   SaveItemList(*filestart,fname);
   }

DeleteItemList(*filestart);

sprintf (buffer,"%s %s %s  2>&1",script,fname,CLASSTEXT[VSYSTEMHARDCLASS]);

EditVerbose("Running command: %s\n",buffer);

if ((pp = popen(buffer,"r")) == NULL)
   {
   printf("%s: edit script %s failed to open.\n",VPREFIX,fname);
   return false;
   }

while (!feof(pp))   
   {
   ReadLine(CURRENTITEM,bufsize,pp);

   if (!feof(pp))
      {
      EditVerbose("%s\n",CURRENTITEM);
      }
   }

pclose(pp);

*filestart = 0;

if (! LoadItemList(filestart,fname))
   {
   printf("          File was marked for editing\n");
   return false;
   }

NUMBEROFEDITS = 0;
CURRENTLINENUMBER = 1;
CURRENTLINEPTR = *filestart;
return true;
}

/************************************************************/

DoFixEndOfLine(list,type)  /* fix end of line char format */

  /* Assumes that extra_space macro allows enough space */
  /* in the allocated strings to add a few characters */

struct Item *list;
char *type;

{ struct Item *ip;
  char *sp;
  int gotCR;

EditVerbose("Checking end of line conventions: type = %s\n",type);

if (strcmp("unix",type) == 0 || strcmp("UNIX",type) == 0)
   {
   for (ip = list; ip != NULL; ip=ip->next)
      {
      for (sp = ip->name; *sp != '\0'; sp++)
	 {
	 if (*sp == (char)13)
	    {
	    *sp = '\0';
	    NUMBEROFEDITS++;
	    }
	 }
      }
   return;
   }

if (strcmp("dos",type) == 0 || strcmp("DOS",type) == 0)
   {
   for (ip = list; ip != NULL; ip = ip->next)
      {
      gotCR = false;
      
      for (sp = ip->name; *sp !='\0'; sp++)
	 {
	 if (*sp == (char)13)
	    {
	    gotCR = true;
	    }
	 }

      if (!gotCR)
	 {
	 *sp = (char)13;
	 *(sp+1) = '\0';
	 NUMBEROFEDITS++;
	 }
      }
   return;
   }

printf("%s: Unknown file format: %s\n",VPREFIX,type);
}

/**************************************************************/

HandleAutomountRescources(filestart,opts)

struct Item **filestart;
char *opts;

{ struct Item *ip;
  char buffer[bufsize];
  char *sp;

for (ip = VMOUNTABLES; ip != NULL; ip=ip->next)
   {
   for (sp = ip->name; *sp != ':'; sp++)
      {
      }

   sp++;
   sprintf(buffer,"%s\t%s\t%s",sp,opts,ip->name);

   if (LocateNextItemContaining(*filestart,sp) == NULL)
      {
      AppendItem(filestart,buffer,"");
      NUMBEROFEDITS++;
      }
   else
      {
      EditVerbose("have a server for %s\n",sp);
      }
   }
}

/**************************************************************/

CheckEditSwitches(filename,actions)

char *filename;
struct Edlist *actions;

{ struct stat statbuf;
  struct Edlist *ep;
  char inform = 'd', log = 'd';
  int fd;

for (ep = actions; ep != NULL; ep=ep->next)
   {
   if (IsExcluded(ep->classes))
      {
      continue;
      }

   switch(ep->code)
      {
      case AutoCreate: if (!DONTDO)
	                  {
			  if (stat(filename,&statbuf) == -1)
			     {
			     if ((fd = creat(filename,0644)) == -1)
				{
				sprintf(OUTPUT,"Unable to create file %s\n",filename);
				CfLog(cfinform,OUTPUT,"creat");
				}
			     else
				{
				AUTOCREATED = true;
				close(fd);
				}
			     sprintf(OUTPUT,"Creating file %s, mode 644\n",filename);
			     CfLog(cfinform,OUTPUT,"");
			     return;
			     }
			  }
                       break;
      
      case EditBackup: if (strcmp("false",ToLowerStr(ep->data)) == 0 || strcmp("off",ToLowerStr(ep->data)) == 0)
	                  {
			  IMAGEBACKUP = false;
	                  }
                       break;
		       
      case EditLog:    if (strcmp(ToLowerStr(ep->data),"true") == 0 || strcmp(ToLowerStr(ep->data),"on") == 0)
	                  {
			  log = 't';
			  break;
			  }

                       if (strcmp(ToLowerStr(ep->data),"false") == 0 || strcmp(ToLowerStr(ep->data),"off") == 0)
                          {
			  log = 'f';
			  break;
			  }

      case EditInform: if (strcmp(ToLowerStr(ep->data),"true") == 0 || strcmp(ToLowerStr(ep->data),"on") == 0)
	                  {
			  inform = 't';
			  break;
			  }

                       if (strcmp(ToLowerStr(ep->data),"false") == 0 || strcmp(ToLowerStr(ep->data),"off") == 0)
                          {
			  inform = 'f';
			  break;
			  }

	  
      }
   }

ResetOutputRoute(log,inform);
}


/**************************************************************/
/* Level 3                                                    */
/**************************************************************/

AddEditfileClasses (list)

struct Edit *list;

{ char *sp, currentitem[maxvarsize];
  struct Edlist *ep;

for (ep = list->actions; ep != NULL; ep=ep->next)
   {
   if (ep->code == DefineClasses)
      {
      break;
      }
   }

if (ep == NULL)
   {
   return;
   }

Debug("Entering AddEditfileClasses(%s)\n",ep->data);
 
for (sp = ep->data; *sp != '\0'; sp++)
   {
   currentitem[0] = '\0';

   sscanf(sp,"%[^,:.]",currentitem);

   sp += strlen(currentitem);

   AddClassToHeap(currentitem);
   }
}

/**************************************************************/

struct Edlist *ThrowAbort(from)

struct Edlist *from;

{ struct Edlist *ep, *last;
 
for (ep = from; ep != NULL; ep=ep->next)
   {
   if (ep->code == CatchAbort)
      {
      return ep;
      }
   
   last = ep;
   }

return last; 
}
