
/*
 * ui.c:  User Interface: all those lovely colors
 */

#include <std.h>
#include <ops.h>

#include <func/apt.h>

#include <list/screens.h>
#include <list/packages.h>
#include <list/colors.h>

#include <sys/time.h>
#include <sys/socket.h>

#include <sys/ioctl.h>
#include <fcntl.h>

#include <parsers/color.h>

#include "acquire.h"
#include "formatting.h"
#include "coreui.h"
#include "dialogs.h"

WINDOW *wtop = NULL;
WINDOW *wmain = NULL;
WINDOW *wstat = NULL;
WINDOW *wdtop = NULL;
WINDOW *wdesc = NULL;

int TOP_LINES;
int STAT_LINES;
int MAIN_LINES;
int DTOP_LINES;
int DESC_LINES;

vector < string > *curDescLines;
int desc_width, desc_bound_start, desc_bound_end;

/*
 *  Redraws the current screen (main + statusbar)
 */
void ui_redraw_current()
{
	ui_redraw_main();
	paint_status();
	paint_top();
	paint_description();
}

/*
 *  Refreshes all the windows
 */
void ui_refresh_current()
{
	redrawwin(wmain);
	wrefresh(wmain);
	redrawwin(wstat);
	wrefresh(wstat);
	redrawwin(wtop);
	wrefresh(wtop);

	if (wdesc != NULL)
	{
		redrawwin(wdesc);
		wrefresh(wdesc);
		redrawwin(wdtop);
		wrefresh(wdtop);
	}
}

/*
 *  Redraws the current screen list window only
 */
void ui_redraw_main()
{
	register unsigned i;
	PackageList::Package * p;

	if (screen->Packages->Length() < MAIN_LINES)
		werase(wmain);

	for (i = 0, p = screen->boundstart; p != NULL; i++, p = p->next)
	{
		if (p == screen->boundend->next)
			break;
		mv_print_line(p, i);
	}

	wrefresh(wmain);
}

/*
 * Redraw a single line for speed. Ncurses is SLOW.
 */
void ui_redraw_line()
{
	register unsigned i;
	PackageList::Package * p;

	for (i = 0, p = screen->Cursor; p != screen->boundstart; i++, p = p->prev);

	mv_print_line(screen->Cursor, i);

	wrefresh(wmain);
}

/*
 * Modify screen boundaries to set up for a scroll
 */
void ui_scroll_up()
{
	wscrl(wmain, -1);

	screen->_boundstart--;
	screen->boundstart = screen->boundstart->prev;

	screen->_boundend--;
	screen->boundend = screen->boundend->prev;
}

void ui_scroll_down()
{
	wscrl(wmain, 1);

	screen->_boundstart++;
	screen->boundstart = screen->boundstart->next;

	screen->_boundend++;
	screen->boundend = screen->boundend->next;
}

/*
 * Wrapper functions to check whether we need to call draw_first() or draw_again()
 */
void ui_draw_next_screen()
{
	if (screen->next == NULL)
		return;

	screen = screen->next;

	ui_redraw_current();
}

void ui_draw_prev_screen()
{
	if (screen->prev == NULL)
		return;
	screen = screen->prev;

	ui_redraw_current();
}

/*
 * Key wrappers:  PageUp/PageDown button.
 */

void ui_page_up()
{
	register int i;

	if (screen->Packages->Length() == 0)
		return;
	if (screen->boundstart->prev == NULL)
		return;

	i = 0;
	while (i < (MAIN_LINES - 1))
	{
		if (screen->boundstart->prev == NULL)
			break;

		screen->boundstart = screen->boundstart->prev;
		screen->_boundstart--;
		screen->boundend = screen->boundend->prev;
		screen->_boundend--;
		screen->Cursor = screen->Cursor->prev;
		screen->_cursor--;

		i++;
	}

	while (screen->_cursor > screen->_boundstart)
	{
		screen->Cursor = screen->Cursor->prev;
		screen->_cursor--;
	}

	ui_redraw_current();
}

void ui_page_down()
{
	register int i;

	if (screen->Packages->Length() == 0)
		return;
	if (screen->boundend->next == NULL)
		return;

	i = 0;
	while (i < (MAIN_LINES - 1))
	{
		if (screen->boundend->next == NULL)
			break;

		screen->boundstart = screen->boundstart->next;
		screen->_boundstart++;
		screen->boundend = screen->boundend->next;
		screen->_boundend++;
		screen->Cursor = screen->Cursor->next;
		screen->_cursor++;

		i++;
	}

	while (screen->_cursor < screen->_boundend)
	{
		screen->Cursor = screen->Cursor->next;
		screen->_cursor++;
	}

	ui_redraw_current();
}

/*
 * Key wrappers: Home/End button; Go to the start/end
 */

void ui_seek_end()
{
	if (screen->Packages->Length() == 0)
		return;

	while (screen->boundend->next != NULL)
	{
		screen->boundstart = screen->boundstart->next;
		screen->_boundstart++;
		screen->boundend = screen->boundend->next;
		screen->_boundend++;
	}

	screen->Cursor = screen->boundend;

	screen->_cursor = screen->Packages->Length() - 1;	// Are at the end!

	ui_redraw_current();
}

void ui_seek_start()
{
	if (screen->Packages->Length() == 0)
		return;

	while (screen->boundstart->prev != NULL)
	{
		screen->boundstart = screen->boundstart->prev;
		screen->_boundstart--;
		screen->boundend = screen->boundend->prev;
		screen->_boundend--;
	}

	screen->Cursor = screen->boundstart;

	screen->_cursor = 0;	// Are at the start!

	ui_redraw_current();
}

/*
 * Key wrappers: Up/Down button;  change the current selected package (BLUE!)
 */

void ui_arrow_up()
{
	int rows;
	PackageList::Package * p;

	// Check to see if we're at the start of the list...
	if (screen->Cursor->prev == NULL)
		return;

	// Scroll down one if we're at the the boundaries
	if ((screen->Cursor == screen->boundstart) && (screen->Packages->Length() > (MAIN_LINES - 1)))
		ui_scroll_up();

	// How far from the top was the old cursor
	for (rows = 0, p = screen->Cursor; p != screen->boundstart; rows++, p = p->prev);

	// Move to the next package
	screen->Cursor = screen->Cursor->prev;

	// Paint over the old, and the new..
	mv_print_line(screen->Cursor->next, rows);
	mv_print_line(screen->Cursor, rows - 1);

	screen->_cursor--;

	wrefresh(wmain);

	// Reformat the status bar
	paint_status();
	paint_description();
}

void ui_arrow_down()
{
	int rows;
	PackageList::Package * p;

	// Check to see if we're at the end of the list...
	if (screen->Cursor->next == NULL)
		return;

	// Scroll down one if we're at the the boundaries
	if ((screen->Cursor == screen->boundend) && (screen->Packages->Length() > (MAIN_LINES - 1)))
		ui_scroll_down();

	// How far from the top was the old cursor
	for (rows = 0, p = screen->Cursor; p != screen->boundstart; rows++, p = p->prev);

	// Move to the next package
	screen->Cursor = screen->Cursor->next;

	// Paint over the old, and the new.. simple non-dependency packages...
	mv_print_line(screen->Cursor->prev, rows);
	mv_print_line(screen->Cursor, rows + 1);

	screen->_cursor++;

	wrefresh(wmain);

	// Reformat the status bar
	paint_status();
	paint_description();
}

/*
 * Initwindows must be called to initialise all the ncurses functions that
 * will be used throughout the program.
 */

void ui_first_init()
{
	static bool initialized = false;

	if (initialized == true)
		return;

	initialized = true;

	initscr();
	cbreak();
	start_color();

	curs_set(0);

	signal(SIGWINCH, ui_resize);

	STAT_LINES = 1;
	TOP_LINES = 2;
	MAIN_LINES = LINES - STAT_LINES - TOP_LINES;
	DTOP_LINES = 1;
	DESC_LINES = 9;
}

void ui_initwindows()
{
	curs_set(0);

	wtop = newwin(TOP_LINES, COLS, 0, 0);
	wstat = newwin(STAT_LINES, COLS, LINES - 1, 0);
	wmain = newwin(MAIN_LINES, COLS, TOP_LINES, 0);

	keypad(wmain, true);
	keypad(wstat, true);
	keypad(wtop, true);
	scrollok(wmain, TRUE);

	wbkgd(wmain, Pair(CAPT_COLOR_DEFAULT, CAPT_COLOR_BACK));
	wbkgd(wtop, Pair(CAPT_COLOR_TOP_FG, CAPT_COLOR_TOP_BG));
	wbkgd(wstat, Pair(CAPT_COLOR_STAT_FG, CAPT_COLOR_STAT_BG));

	wrefresh(wmain);
	wrefresh(wtop);
	wrefresh(wstat);
}

/*
 * MUST be called before an exit() or will result in your
 * termIOS settings being stuffed up.
 */

void ui_finishwindows()
{
	delwin(wmain);
	delwin(wstat);
	delwin(wtop);
}

void ui_full_cleanup()
{
	ui_finishwindows();
	capt_clean_exit();
}

void ui_resize(int sig)
{
	int fd = open("/dev/tty", O_RDONLY);

	if (fd != -1)
	{
		struct winsize w;

		if (ioctl(fd, TIOCGWINSZ, &w) != -1)
		{
			resizeterm(w.ws_row, w.ws_col);
		}
		else
		{
			beep();
			DEBUGEXIT("Can't get window size from tty!");
		}
		close(fd);
	}
	else
	{
		beep();
		DEBUGEXIT("Can't open /dev/tty!");
	}

	STAT_LINES = 1;
	TOP_LINES = 2;
	MAIN_LINES = LINES - STAT_LINES - TOP_LINES;
	DTOP_LINES = 1;
	DESC_LINES = 9;

	ui_finishwindows();
	ui_initwindows();
	screen.align_all_bounds();
	ui_redraw_current();
	return;
}

/*
 * Display a whole package on the screen by seeking into
 * the packages file and grabbing the entry.
 */
void ui_DisplayPackage()
{
	pkgCache::VerIterator CandV = (*Cache)[screen->Cursor->pkg].CandidateVerIter(*Cache);
	pkgCache::VerIterator InstV = (*Cache)[screen->Cursor->pkg].InstVerIter(*Cache);

	string s = Recs->Lookup(CandV.FileList()).LongDesc();

	char Left[100], Right[100];

	werase(wstat);

	sprintf(Left, "[Available: %s]", CandV.VerStr());
	mvwprintw(wstat, 0, 1, Left);

	if (InstV.end() == false)
	{
		sprintf(Right, "[Installed: %s]", InstV.VerStr());
		mvwprintw(wstat, 0, COLS - strlen(Right) - 1, Right);
	}

	wrefresh(wstat);

	ui_dialog(Pair(COLOR_YELLOW, COLOR_BLUE) | A_BOLD, (char *) screen->Cursor->pkg.Name(), "%s", s.c_str());

	paint_status();

	wrefresh(wmain);
	wrefresh(wtop);
	wrefresh(wstat);
}

/*
 * Search text entry function with options
 */
void ui_search_entry(char **retstr, unsigned long *options)
{
	int i = 0, j = 0;

	**retstr = 0;

	int dispstrlen = ui_search_prompt(*options, *retstr);

	WINDOW *old_dwin = wdesc;

	while (1)
	{
		int c = wgetch(wstat);

		switch (c)
		{
		case ' ':
			break;

		case 15:	// Ctrl-O

			*(*retstr + j) = 0;

			ui_search_help();

			break;

		case 6:	// Ctrl-F
			*options = *options ^ SEARCH_DESC;
			*(*retstr + j + 1) = 0;
			dispstrlen = ui_search_prompt(*options, *retstr);
			break;

		case 9:	// Ctrl-I
			*options = *options ^ SEARCH_CASE;
			*(*retstr + j + 1) = 0;
			dispstrlen = ui_search_prompt(*options, *retstr);
			break;

		case 18:	// Ctrl-R
			*options = *options ^ SEARCH_REGEX;
			*(*retstr + j + 1) = 0;
			dispstrlen = ui_search_prompt(*options, *retstr);
			break;

		case '\n':
		case '\r':
			*(*retstr + j) = 0;

			curs_set(0);

			if (old_dwin != wdesc)
				ui_desc_window_hide();

			ui_redraw_current();

			return;

		case KEY_BACKSPACE:
			if (j == 0)
				break;

			if (i < j)
			{
				char *x = *retstr + i;
				char *y = (char *) malloc(j - i);

				memcpy(y, x, j - i);

				memcpy(x - 1, y, j - i);

				free(y);

				i--;
			}
			else
			{
				*(*retstr + i) = 0;

				i--;
			}

			mvwdelch(wstat, 0, i + dispstrlen);
			wmove(wstat, 0, i + dispstrlen);

			j--;

			wrefresh(wstat);

			break;

		case KEY_LEFT:
			if (j == 0)
				break;

			i--;

			wmove(wstat, 0, i + dispstrlen);

			break;

		case KEY_RIGHT:
			if (j == (signed) (COLS - dispstrlen - 1))
				break;

			if (i == j)
				break;

			i++;

			wmove(wstat, 0, i + dispstrlen);

			break;

		case 27:
			break;

		default:
			if (j == (signed) (COLS - dispstrlen - 1))
				break;

			if (i < j)
			{
				char *x = *retstr + i;
				char *y = (char *) malloc(j - i);

				memcpy(y, x, j - i);

				*x = c;

				memcpy(x + 1, y, j - i);

				free(y);

				mvwinsch(wstat, 0, i + dispstrlen, c);

				i++;
			}
			else
			{
				mvwaddch(wstat, 0, i + dispstrlen, c);

				*(*retstr + i) = c;

				i++;
			}

			wmove(wstat, 0, i + dispstrlen);

			j++;

			wrefresh(wstat);

			break;
		}
	}
}

void ui_textentry(char *dispstr, char **retstr)
{
	int i = 0, j = 0;

	**retstr = 0;

	curs_set(1);

	UniqueWindow TextEntry(3, COLS, (LINES - 3) / 2, 0);

	TextEntry.ChangeTitle(dispstr);

	int dispstrlen = 1;

	while (1)
	{
		int c = TextEntry.Getch();

		switch (c)
		{
		case ' ':
			break;

		case 15:	// Ctrl-O

			*(*retstr + j) = 0;

			break;

		case 6:	// Ctrl-F
			*(*retstr + j + 1) = 0;
			break;

		case 9:	// Ctrl-I
			*(*retstr + j + 1) = 0;
			break;

		case 18:	// Ctrl-R
			*(*retstr + j + 1) = 0;
			break;

		case '\n':
		case '\r':
			*(*retstr + j) = 0;

			curs_set(0);

			paint_status();
			paint_top();

			return;

		case KEY_BACKSPACE:
			if (j == 0)
				break;

			if (i < j)
			{
				char *x = *retstr + i;
				char *y = (char *) malloc(j - i);

				memcpy(y, x, j - i);

				memcpy(x - 1, y, j - i);

				free(y);

				i--;
			}
			else
			{
				*(*retstr + i) = 0;

				i--;
			}

			mvwdelch(TextEntry.body(), 0, i + dispstrlen);
			wmove(TextEntry.body(), 0, i + dispstrlen);

			j--;

			wrefresh(TextEntry.body());

			break;

		case KEY_LEFT:
			if (j == 0)
				break;

			i--;

			wmove(wstat, 0, i + dispstrlen);

			break;

		case KEY_RIGHT:
			if (j == (signed) (COLS - dispstrlen - 1))
				break;

			if (i == j)
				break;

			i++;

			wmove(TextEntry.body(), 0, i + dispstrlen);

			break;

		case 27:
			break;

		default:
			if (j == (signed) (COLS - dispstrlen - 1))
				break;

			if (i < j)
			{
				char *x = *retstr + i;
				char *y = (char *) malloc(j - i);

				memcpy(y, x, j - i);

				*x = c;

				memcpy(x + 1, y, j - i);

				free(y);

				mvwinsch(TextEntry.body(), 0, i + dispstrlen, c);

				i++;
			}
			else
			{
				mvwaddch(TextEntry.body(), 0, i + dispstrlen, c);

				*(*retstr + i) = c;

				i++;
			}

			wmove(TextEntry.body(), 0, i + dispstrlen);

			j++;

			wrefresh(TextEntry.body());

			break;
		}
	}
}

void ui_seek_nextmatched()
{
	PackageList::Package * moved_to, *P;

	moved_to = screen->Packages->LocateNext();

	if (moved_to == NULL)
		return;

	int I, J;

	for (I = 0, P = screen->Packages->Head(); P != moved_to; P = P->next, I++);

	if (screen->_cursor < I)
	{
		for (J = screen->_cursor; J < I; J++)
		{
			if (I > screen->_boundend)
			{
				++screen->_boundstart;
				++screen->_boundend;
			}
			++screen->_cursor;
		}
	}
	else if (screen->_cursor > I)
	{
		for (J = screen->_cursor; J > I; J--)
		{
			if (I < screen->_boundstart)
			{
				--screen->_boundstart;
				--screen->_boundend;
			}
			--screen->_cursor;
		}
	}

	screen.boundreset(screen.Current());
	ui_redraw_current();
}


/*
 * Turn the description windows on
 */
void ui_desc_window_show(void)
{
	MAIN_LINES -= (DESC_LINES + DTOP_LINES);
	screen->_boundend -= (DESC_LINES + DTOP_LINES);

	wresize(wmain, MAIN_LINES, COLS);

	wdtop = newwin(DTOP_LINES, COLS, LINES - STAT_LINES - DESC_LINES - DTOP_LINES, 0);
	wdesc = newwin(DESC_LINES, COLS, LINES - STAT_LINES - DESC_LINES, 0);

	keypad(wdtop, true);
	keypad(wdesc, true);

	wbkgd(wdtop, Pair(CAPT_COLOR_DTOP_FG, CAPT_COLOR_DTOP_BG));
	wbkgd(wdesc, Pair(CAPT_COLOR_DESC_FG, CAPT_COLOR_DESC_BG) | A_BOLD);

	scrollok(wdesc, true);

	wrefresh(wdtop);
	wrefresh(wdesc);

	screen.boundreset(screen.Current());
}

/*
 * Turn the description windows off
 */
void ui_desc_window_hide(void)
{
	MAIN_LINES += (DESC_LINES + DTOP_LINES);
	screen->_boundend += (DESC_LINES + DTOP_LINES);

	delwin(wdtop);
	delwin(wdesc);

	wdtop = NULL;
	wdesc = NULL;

	wresize(wmain, MAIN_LINES, COLS);

	screen.boundreset(screen.Current());
}

/*
 * Open a help window for search options
 */
void ui_search_help()
{
	if (wdesc == NULL)
		ui_desc_window_show();

	werase(wdtop);
	mvwprintw(wdtop, 0, 0, " Search options (all control characters) ");
	wrefresh(wdtop);

	werase(wdesc);
	mvwprintw(wdesc, 0, 0, "\n  ^F  Full-text Search of Package Descriptions  \n" "  ^I  Ignore Case  \n" "  ^R  Regular-expression Match  \n");

	wrefresh(wdesc);

	// Move the cursor back
	wrefresh(wstat);
}

int ui_search_prompt(unsigned long options, char *buf)
{
	string dispstr;

	curs_set(1);
	scrollok(wstat, false);
	wattrset(wstat, Pair(COLOR_WHITE, COLOR_BLUE));
	mvwblankl(wstat, 0, COLS);
	wmove(wstat, 0, 0);
	wattrset(wstat, Pair(COLOR_GREEN, COLOR_BLUE));

	dispstr = " locate ";

	if (options & SEARCH_CASE)
		dispstr += "ignore-case ";
	if (options & SEARCH_DESC)
		dispstr += "full-text ";
	if (options & SEARCH_REGEX)
		dispstr += "regex ";

	dispstr += "(^O for help): ";

	wprintw(wstat, (char *) dispstr.c_str());

	wattrset(wstat, Pair(COLOR_WHITE, COLOR_BLUE));

	wprintw(wstat, buf);

	wclrtoeol(wstat);
	wrefresh(wstat);

	return dispstr.length();
}

void ui_scroll_descwin_down()
{
	if (wdesc == NULL)
		return;

	if (screen->Packages->IsDivider(screen->Cursor))
		return;

	if (desc_bound_end == desc_width)
	{
		return;
	}

	desc_bound_end++;
	desc_bound_start++;

	scrollok(wdesc, true);
	wscrl(wdesc, 1);
	scrollok(wdesc, false);
	mvwprintw(wdesc, DESC_LINES - 1, 0, (char *) (*curDescLines)[desc_bound_end - 1].c_str());

	wrefresh(wdesc);
}

void ui_scroll_descwin_up()
{
	if (wdesc == NULL)
		return;

	if (screen->Packages->IsDivider(screen->Cursor))
		return;

	if (desc_bound_start == 0)
		return;

	desc_bound_start--;
	desc_bound_end--;

	scrollok(wdesc, true);
	wscrl(wdesc, -1);
	scrollok(wdesc, false);
	mvwprintw(wdesc, 0, 0, (char *) (*curDescLines)[desc_bound_start].c_str());

	wrefresh(wdesc);
}

void ui_divider_prev()
{
	PackageList::Package * P = screen->Cursor;
	int I = screen->_cursor, J = 0;

	if (screen->Packages->IsDivider(P))
	{
		P = P->prev;
		I--;
	}

	while (P != NULL && !screen->Packages->IsDivider(P))
	{
		P = P->prev;
		I--;
	}

	if (P == NULL)
		return;

	if (screen->_cursor < I)
	{
		for (J = screen->_cursor; J < I; J++)
		{
			if (I > screen->_boundend)
			{
				++screen->_boundstart;
				++screen->_boundend;
			}
			++screen->_cursor;
		}
	}
	else if (screen->_cursor > I)
	{
		for (J = screen->_cursor; J > I; J--)
		{
			if (I < screen->_boundstart)
			{
				--screen->_boundstart;
				--screen->_boundend;
			}
			--screen->_cursor;
		}
	}

	screen.boundreset(screen.Current());
	ui_redraw_current();
}

void ui_divider_next()
{
	PackageList::Package * P = screen->Cursor;
	int I = 0, J = 0;

	if (screen->Packages->IsDivider(P))
	{
		P = P->next;
		I++;
	}

	while (P != NULL && !screen->Packages->IsDivider(P))
	{
		P = P->next;
		I++;
	}

	if (P == NULL)
		return;

	I = screen->_cursor + I;

	if (screen->_cursor < I)
	{
		for (J = screen->_cursor; J < I; J++)
		{
			if (I > screen->_boundend)
			{
				++screen->_boundstart;
				++screen->_boundend;
			}
			++screen->_cursor;
		}
	}
	else if (screen->_cursor > I)
	{
		for (J = screen->_cursor; J > I; J--)
		{
			if (I < screen->_boundstart)
			{
				--screen->_boundstart;
				--screen->_boundend;
			}
			--screen->_cursor;
		}
	}

	screen.boundreset(screen.Current());
	ui_redraw_current();
}
