/*
    ext2_unix_io.c -- ext2 unix I/O code
    Copyright (C) 1999 Lennert Buytenhek <lbuijten@cs.leidenuniv.nl>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "ext2.h"

/* pseudo-header.... */

#ifndef BLKGETSIZE
#define BLKGETSIZE _IO(0x12,96)
#endif

loff_t llseek(unsigned int fd, loff_t offset, unsigned int whence);

struct my_cookie
{
	int logsize;

	int fdread;
	int fdwrite;
	loff_t readoff;
	loff_t writeoff;
};

/* ...then this must be pseudo-code  :-) */

static void  do_close        (void *cookie);
static void  do_sync         (void *cookie);
static blk_t do_get_size     (void *cookie);
static int   do_read         (void *cookie, void *ptr, blk_t block, blk_t numblocks);
static void  do_set_blocksize(void *cookie, int logsize);
static int   do_write        (void *cookie, void *ptr, blk_t block, blk_t numblocks);

struct ext2_dev_ops ops =
{
	close:		do_close,
	get_size:	do_get_size,
	read:		do_read,
	set_blocksize:	do_set_blocksize,
	sync:		do_sync,
	write:		do_write
};



static void do_close(void *cookie)
{
	struct my_cookie *monster = cookie;

	fdatasync(monster->fdwrite);
	close(monster->fdread);
	close(monster->fdwrite);

	monster->fdread = 0;
	monster->fdwrite = 0;
}

static void do_sync(void *cookie)
{
	struct my_cookie *monster = cookie;

	fdatasync(monster->fdwrite);
}

static blk_t do_get_size(void *cookie)
{
	struct my_cookie *monster = cookie;
	struct stat st;
	blk_t size;

	monster = cookie;

	if (fstat(monster->fdread, &st) < 0)
	{
		fprintf(stderr, "this isn't supposed to happen!\n");
		exit(-1);
	}

	/* Thanks to Rolf Jakob for this one (cant do SEEK_END on block devs) */
	if (S_ISBLK(st.st_mode))
	{
		if (ioctl(monster->fdread, BLKGETSIZE, &size) < 0)
		{
			perror("ioctl");
			size = 0;
		}
	}
	else
	{
		size = llseek(monster->fdread, 0, SEEK_END) >> 9;
		llseek(monster->fdread, monster->readoff, SEEK_SET);
	}

	return size >> (monster->logsize - 9);
}

static int do_read(void *cookie, void *ptr, blk_t block, blk_t numblocks)
{
	struct my_cookie *monster = cookie;
	loff_t offset;

	offset = ((loff_t)block) << monster->logsize;
	if (monster->readoff != offset)
		llseek(monster->fdread, offset, SEEK_SET);

	read(monster->fdread, ptr, numblocks << monster->logsize);
	monster->readoff = ((loff_t)block+numblocks) << monster->logsize;

	return 1;   /* FIXXXME */
}

static void do_set_blocksize(void *cookie, int logsize)
{
	struct my_cookie *monster = cookie;

	monster->logsize = logsize;
} 

static int do_write(void *cookie, void *ptr, blk_t block, blk_t numblocks)
{
	struct my_cookie *monster = cookie;
	loff_t offset;

	offset = ((loff_t)block) << monster->logsize;
	if (monster->writeoff != offset)
		llseek(monster->fdwrite, offset, SEEK_SET);

	write(monster->fdwrite, ptr, numblocks << monster->logsize);
	monster->writeoff = ((loff_t)block+numblocks) << monster->logsize;

	return 1; /* FIXXXME */
}

struct ext2_dev_handle *ext2_make_dev_handle_from_file(char *dev)
{
	struct ext2_dev_handle *dh;
	struct my_cookie *monster;

	if ((dh = malloc(sizeof(struct ext2_dev_handle))) == NULL)
		goto error;

	if ((monster = malloc(sizeof(struct my_cookie))) == NULL)
		goto error_free_dh;

	if ((monster->fdread = open(dev, O_RDONLY)) < 0)
		goto error_free_cookie;

	if ((monster->fdwrite = open(dev, O_WRONLY)) < 0)
		goto error_close_read;

	monster->readoff = 0;
	monster->writeoff = 0;
	monster->logsize = 9;

	dh->ops = &ops;
	dh->cookie = monster;

	return dh;

 error_close_read:
	close(monster->fdread);

 error_free_cookie:
	free(monster);

 error_free_dh:
	free(dh);

 error:
	return NULL;
}
