#------------------------------------------------------------------------------
# Contains common Wcb procedures.
#
# Copyright (c) 1999  Csaba Nemethi (E-mail: csaba.nemethi@t-online.de)
#------------------------------------------------------------------------------

#
# Basic procedures
# ================
#

#------------------------------------------------------------------------------
# wcb::callback
#
# Sets or retrieves the callbacks for the widget w, the argument when, and the
# command corresponding to option.  when can be "before" or "after", and option
# can take one of the following values:
#
# - "insert", "delete", or "motion",		for an entry or text widget;
# - "activate", "selset", or "selclear",	for a listbox widget.
#
# If no arguments after the option parameter are specified then the procedure
# just returns the current before- or after-callback list, respectively, for
# the given widget operation.
#
# Otherwise:
#
# - if called for the first time for this widget, it replaces the Tcl command w
#   with a new procedure in which the execution of the widget operations
#   associated with the above values of option is preceded by calls to the
#   corresponding before-callbacks and followed by calls to the corresponding
#   after-callbacks, in the global scope;
# - it sets the callback list to the one built from these arguments and returns
#   the new list.
#
# When a callback is invoked, the name of the original Tcl command for the
# widget w as well as the command arguments are automatically appended to it as
# parameters.
#------------------------------------------------------------------------------
proc wcb::callback {w when option args} {
    if {![winfo exists $w]} {
	error "bad window path name \"$w\""
    }

    set whenLen [string length $when]
    if {[string match $when* before] && $whenLen >= 1} {
	set when before
    } elseif {[string match $when* after] && $whenLen >= 1} {
	set when after
    } else {
	error "bad second argument \"$when\"; must be before or after"
    }

    set opLen [string length $option]
    if {[string match $option* insert] && $opLen >= 1} {
	set option insert
    } elseif {[string match $option* delete] && $opLen >= 1} {
	set option delete
    } elseif {[string match $option* motion] && $opLen >= 1} {
	set option motion
    } elseif {[string match $option* activate] && $opLen >= 1} {
	set option activate
    } elseif {[string match $option* selset] && $opLen >= 4} {
	set option selset
    } elseif {[string match $option* selclear] && $opLen >= 4} {
	set option selclear
    #
    # Deprecated:
    #
    } elseif {[string match $option* "sel set"] && $opLen >= 5} {
	set option selset
    } elseif {[string match $option* "sel clear"] && $opLen >= 5} {
	set option selclear
    }

    variable data
    set callbacks [expr {[info exists data($w-$when-$option)] ?
			 $data($w-$when-$option) : {}}]

    set argsLen [llength $args]

    switch -exact [winfo class $w] {
	Entry {
	    if {$option != "insert" && $option != "delete" &&
		$option != "motion"} {
		error "bad option \"$option\"; must be insert, delete,\
		       or motion"
	    }
	    if {$argsLen == 0} {
		return $callbacks
	    } else {
		if {![catch {rename $w ::_$w}]} {
		    buildNewEntryCmd $w
		}
		set data($w-$when-$option) $args
	    }
	}

	Listbox {
	    if {$option != "activate" &&
		$option != "selset" && $option != "selclear"} {
		error "bad option \"$option\"; must be activate, selset,\
		       or selclear"
	    }
	    if {$argsLen == 0} {
		return $callbacks
	    } else {
		if {![catch {rename $w ::_$w}]} {
		    buildNewListboxCmd $w
		}
		set data($w-$when-$option) $args
	    }
	}

	Text {
	    if {$option != "insert" && $option != "delete" &&
		$option != "motion"} {
		error "bad option \"$option\"; must be insert, delete,\
		       or motion"
	    }
	    if {$argsLen == 0} {
		return $callbacks
	    } else {
		if {![catch {rename $w ::_$w}]} {
		    buildNewTextCmd $w
		}
		set data($w-$when-$option) $args
	    }
	}

	default {
	    error "window \"$w\" is not an entry, listbox, or text widget"
	}
    }
}

#------------------------------------------------------------------------------
# wcb::cancel
#
# If invoked from a before-callback for a widget command, this procedure
# cancels the execution of that command and calls script in the global scope.
#------------------------------------------------------------------------------
proc wcb::cancel {{script bell}} {
    variable data
    set data(doIt) 0

    if {[llength $script] != 0} {
	uplevel #0 $script
    }
}

#------------------------------------------------------------------------------
# wcb::replace
#
# If invoked from a before-callback for a widget command, this procedure
# replaces the arguments having the indices first through last of that command
# with the values given in args.  The new argument list will be passed to the
# remaining callbacks for that command, too.  The arguments are numbered from 0.
#------------------------------------------------------------------------------
proc wcb::replace {first last args} {
    variable data
    set data(args) [eval lreplace {$data(args) $first $last} $args]
}

#------------------------------------------------------------------------------
# wcb::extend
#
# If invoked from a before-callback for a widget command, this procedure
# appends the values given in args to the argument list of that command.  The
# new argument list will be passed to the remaining callbacks for that command,
# too.
#------------------------------------------------------------------------------
proc wcb::extend args {
    variable data
    eval lappend data(args) $args
}

#
# Private procedure
# =================
#

#------------------------------------------------------------------------------
# wcb::processCmd
#
# Invokes all before-callbacks set for the widget w and the command identified
# by option, then executes the script "w option args", and finally invokes all
# after-callbacks.
#------------------------------------------------------------------------------
proc wcb::processCmd {w wcbOp cmdOp args} {
    variable data
    set data(doIt) 1
    set data(args) $args
    set orig [list _$w]

    if {[info exists data($w-before-$wcbOp)]} {
	foreach cb $data($w-before-$wcbOp) {
	    if {!$data(doIt)} {
		break
	    }
	    if {[llength $cb] != 0} {
		uplevel #0 $cb $orig $data(args)
	    }
	}
    }

    if {!$data(doIt)} {
	return
    }

    eval {_$w} $cmdOp $data(args)

    if {[info exists data($w-after-$wcbOp)]} {
	foreach cb $data($w-after-$wcbOp) {
	    if {[llength $cb] != 0} {
		uplevel #0 $cb $orig $data(args)
	    }
	}
    }
}
