#include "system.h"

#include <rpmlib.h>
#include <rpmio.h>

/* =============================================================== */

FD_t fdNew(void) {
    FD_t fd = (FD_t) xmalloc(sizeof(struct _FD));
    if (fd == NULL)
	return NULL;
    fd->fd_fd = -1;
    fd->fd_bzd = NULL;
    fd->fd_gzd = NULL;
    fd->fd_url = NULL;
    return fd;
}

int fdValid(FD_t fd) {
    return (fd != NULL && fd->fd_fd >= 0);
}

int fdFileno(FD_t fd) {
    return (fd != NULL ? fd->fd_fd : -1);
}

FD_t fdOpen(const char *pathname, int flags, mode_t mode) {
    FD_t fd;
    int fdno;
    if ((fdno = open(pathname, flags, mode)) < 0)
	return NULL;
    fd = fdNew();
    fd->fd_fd = fdno;
    return fd;
}

FD_t fdDup(int fdno) {
    FD_t fd;
    int nfdno;
    if ((nfdno = dup(fdno)) < 0)
	return NULL;
    fd = fdNew();
    fd->fd_fd = nfdno;
    return fd;
}

off_t fdLseek(FD_t fd, off_t offset, int whence) {
    off_t rc = lseek(fdFileno(fd), offset, whence);
    return rc;
}

ssize_t fdRead(FD_t fd, void * buf, size_t count) {
    ssize_t rc = read(fdFileno(fd), buf, count);
    return rc;
}

ssize_t fdWrite(FD_t fd, const void * buf, size_t count) {
    ssize_t rc = write(fdFileno(fd), buf, count);
    return rc;
}

int fdClose(FD_t fd) {
    int fdno;

    if (fd != NULL && (fdno = fdFileno(fd)) >= 0) {
	fd->fd_fd = -1;
	fd->fd_bzd = NULL;
	fd->fd_gzd = NULL;
	free(fd);
    	return close(fdno);
    }
    return -2;
}

FILE *fdFdopen(FD_t fd, const char *mode) {
    FILE *f = fdopen(fdFileno(fd), mode);
    if (f != NULL) {
	fd->fd_fd = -1;
	free(fd);
	return f;
    }
    return NULL;
}

/* =============================================================== */
/* Support for GZIP library.
 */
#ifdef	HAVE_ZLIB_H

#include <zlib.h>

gzFile * gzdFileno(FD_t fd) {
    return (fd != NULL ? ((gzFile *)fd->fd_gzd) : NULL);
}

FD_t gzdOpen(const char *pathname, const char *mode) {
    FD_t fd;
    gzFile *gzfile;;
    if ((gzfile = gzopen(pathname, mode)) == NULL)
	return NULL;
    fd = fdNew();
    fd->fd_gzd = gzfile;
    return fd;
}

FD_t gzdFdopen(FD_t fd, const char *mode) {
    gzFile *gzfile  = gzdopen(fdFileno(fd), mode);
    if (gzfile != NULL) {
	fd->fd_fd = -1;
	fd->fd_gzd = gzfile;
	return fd;
    }
    return NULL;
}

ssize_t gzdRead(FD_t fd, void * buf, size_t count) {
    ssize_t rc = gzread(gzdFileno(fd), buf, count);
    return rc;
}

ssize_t gzdWrite(FD_t fd, const void * buf, size_t count) {
    ssize_t rc = gzwrite(gzdFileno(fd), (void *)buf, count);
    return rc;
}

#if 0	/* XXX zlib-1.0.4 has not */
off_t gzdLseek(FD_t fd, off_t offset, int whence) {
    return gzseek(gzdFileno(fd), offset, whence);
}
#endif

int gzdFlush(FD_t fd) {
    return gzflush(gzdFileno(fd), Z_SYNC_FLUSH);	/* XXX W2DO? */
}

const char * gzdStrerror(FD_t fd) {
#if 0
    static char *zlib_err [] = {
	"OK"
	"Errno",
	"Stream", 
	"Data", 
	"Memory",
	"Buffer",
	"Version"
    };  
#endif
    int zerror = 0;
    const char *zerrstr = gzerror(gzdFileno(fd), &zerror);

    switch (zerror) {
    case Z_ERRNO:
	return strerror(errno);
	/*@notreached@*/ break;
    default:
	return zerrstr;
	/*@notreached@*/ break;
    }
    /*@notreached@*/
}

int gzdClose(FD_t fd) {
    gzFile *gzfile;
    int zerror;

    if (fd != NULL && (gzfile = gzdFileno(fd)) != NULL) {
	fd->fd_fd = -1;
	fd->fd_bzd = NULL;
	fd->fd_gzd = NULL;
	free(fd);
    	zerror = gzclose(gzfile);
	return 0;
    }
    return -2;
}

#endif	/* HAVE_ZLIB_H */
