/* This program is part of the MBR test suite. */
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <assert.h>
#include <errno.h>
#include "mbr.h"

#define TMP_EXT ".tmp"

typedef u_int8_t uchar;

extern uchar _binary_target_b_start[];
extern uchar _binary_target_b_end[];
#define TARGET_START _binary_target_b_start
#define TARGET_END _binary_target_b_end

#define N_SECTOR (4*17*4)

/* I prefer to look at this as documenting a disk image I once had,
   not as knowing the format of a DOSEMU image. */
const uchar dosemu_header[128]=
{
  0x44, 0x4F, 0x53, 0x45, 0x4D, 0x55, 0x00, 0x04,
  0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x04, /* 16 */
  0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 32 */ 
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 48 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 64 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 80 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 96 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 112 */
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* 128 */
};

struct pt_record
{
  uchar boot, start_head, start_sector, start_track;
  uchar type, end_head, end_sector, end_track;
  uchar start[4], length[4];
};

struct mbr
{
  uchar pad[446];
  struct pt_record p1, p2, p3, p4;
  uchar sig[2];
};

const struct mbr mbr_init=
{
  {0,}, /* Pad */
  {0x80, 0,  2, 0, 0x83, 2,  8, 0, {0x01}, {0x29}},
  {0x00, 2,  9, 0, 0x83, 3, 17, 0, {0x2A}, {0x1A}},
  {0x00, 0,  1, 1, 0x83, 1, 12, 2, {0x44}, {0x61}},
  {0x00, 1, 13, 2, 0x83, 3, 17, 3, {0xA5}, {0x6B}},
  {0x55, 0xAA}
};

const char *progname="(unknown)";

void do_write(FILE *f, const char *path, const void *buf,
	      size_t len)
{
  size_t res=fwrite(buf, 1, len, f);
  if(res!=len)
  {
    if(ferror(f))
      fprintf(stderr, "%s:%s: Error during write: %s\n",
	      progname, path, strerror(errno));
    else
      fprintf(stderr, "%s:%s: EOF during write.\n",
	      progname, path);
    exit(1);
  }
}

/* The structure of the disk image is:
 *   1. The DOSEMU header
 *   2. The MBR
 *   3. Lots of copies of target.b
 */
int main(int argc, const char *argv[])
{
  char *path_tmp;
  int path_len;
  FILE *f;
  int i;

  progname=argv[0];
  assert(TARGET_END-TARGET_START==512);
  assert(sizeof(mbr_init)==512);

  if(argc!=2)
  {
    fprintf(stderr, "Usage: %s <file>\n", argv[0]);
    return 1;
  }

  /* Reserve enough space for the path and extension */
  path_len=strlen(argv[1]);
  path_tmp=malloc(path_len+sizeof(TMP_EXT));
  if(path_tmp==NULL)
  {
    fprintf(stderr, "%s: Memory allocation failed (%d bytes)\n",
	    argv[0], path_len+sizeof(TMP_EXT));
    return 1;
  }

  memcpy(path_tmp, argv[1], path_len);
  strcpy(path_tmp+path_len, TMP_EXT);

  f=fopen(path_tmp, "wb");
  if(f==NULL)
  {
    fprintf(stderr, "%s:%s: Failed to open file: %s\n",
	    argv[0], path_tmp, strerror(errno));
    return 1;
  };

  do_write(f, path_tmp, dosemu_header, sizeof(dosemu_header));
  do_write(f, path_tmp, &mbr_init, sizeof(mbr_init));
  /* The MBR is the first sector (sector 0) */
  for(i=1; i<N_SECTOR; i++)
  {
    uchar timeout=20+i%100;
    uchar drive=20+i/100;
    TARGET_START[0x1ba]=timeout;
    TARGET_START[0x1bb]=drive;
    do_write(f, path_tmp, TARGET_START, TARGET_END-TARGET_START);
  }
  if(fclose(f)!=0)
  {
    fprintf(stderr, "%s:%s: Failed to close file: %s\n",
	    argv[0], path_tmp, strerror(errno));
    return 1;
  }
  if(rename(path_tmp, argv[1])<0)
  {
    fprintf(stderr, "%s:%s: Rename failed: %s\n",
	    argv[0], argv[1], strerror(errno));
    return 1;
  }
  return 0;
}

/*  */
/* Local Variables: */
/* mode:c */
/* c-indentation-style:"bsd" */
/* c-basic-offset:2 */
/* End: */
