------------------------------------------------------------------------------
--                                                                          --
--                 ASIS-for-GNAT IMPLEMENTATION COMPONENTS                  --
--                                                                          --
--                         A 4 G . C O N T T . D P                          --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--            Copyright (c) 1995-1999, Free Software Foundation, Inc.       --
--                                                                          --
-- ASIS-for-GNAT is free software; you can redistribute it and/or modify it --
-- under terms of the  GNU General Public License  as published by the Free --
-- Software Foundation;  either version 2,  or  (at your option)  any later --
-- version. ASIS-for-GNAT is distributed  in the hope  that it will be use- --
-- ful, but WITHOUT ANY WARRANTY; without even the implied warranty of MER- --
-- CHANTABILITY or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General  --
-- Public License for more details. You should have received a copy of the  --
-- GNU General Public License  distributed with ASIS-for-GNAT; see file     --
-- COPYING. If not, write to the Free Software Foundation,  59 Temple Place --
-- - Suite 330,  Boston, MA 02111-1307, USA.                                --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- ASIS-for-GNAT was originally developed  by the ASIS-for-GNAT team at the --
-- Software  Engineering  Laboratory  of  the Swiss  Federal  Institute  of --
-- Technology (LGL-EPFL) in Lausanne,  Switzerland, in cooperation with the --
-- Scientific  Research  Computer  Center of  Moscow State University (SRCC --
-- MSU), Russia,  with funding partially provided  by grants from the Swiss --
-- National  Science  Foundation  and  the  Swiss  Academy  of  Engineering --
-- Sciences.  ASIS-for-GNAT is now maintained by  Ada Core Technologies Inc --
-- (http://www.gnat.com).                                                   --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Deallocation;

with Asis.Set_Get; use Asis.Set_Get;

with A4G.Contt.UT; use A4G.Contt.UT;

with Atree;        use Atree;
with Nlists;       use Nlists;
with Namet;        use Namet;
with Sinfo;        use Sinfo;
with Lib;          use Lib;

package body A4G.Contt.DP is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Get_First_Stub (Body_Node : Node_Id) return Node_Id;
   function Get_Next_Stub  (Stub_Node : Node_Id) return Node_Id;
   --  these two functions implement the iterator through the body stubs
   --  contained in the given compilation unit. The iterator should
   --  be started from calling Get_First_Stub for the node pointed to
   --  the body (that is, for the node of ..._Body kind). The Empty node
   --  is returned if there is no first/next body stub node

   procedure Set_All_Unit_Dependencies (C : Context_Id; U : Unit_Id);
   --  Computes the full lists of supporters and dependants of U in C from
   --  the list of direct supporters of U and sets these lists as values
   --  of Supporters and Dependents lists in the Unit Table

   procedure Add_Unit_Supporters (U : Unit_Id; L : in out Elist_Id);
   --  Add all the supporters of U, excluding U itself to L. This procedure
   --  traverses all the transitive semantic dependencies.

   --------------------------------------
   -- Dynamic Unit_Id list abstraction --
   --------------------------------------
   --  All the subprograms implementing Unit_Id list abstraction do not
   --  reset Context

   --  Is this package body the right place for defining this abstraction?
   --  May be, we should move it into A4G.A_Types???

   type Unit_Id_List_Access is access Unit_Id_List;
   Tmp_Unit_Id_List_Access : Unit_Id_List_Access;

   procedure Free is new Ada.Unchecked_Deallocation
     (Unit_Id_List, Unit_Id_List_Access);

   function In_Unit_Id_List
     (U : Unit_Id;
      L : Unit_Id_List_Access)
       return Boolean;
   --  Checks if U is a member of L.

   procedure Append_Unit_To_List
     (U : Unit_Id;
      L : in out Unit_Id_List_Access);
   --  (Unconditionally) appends U to L.

   procedure Add_To_Unit_Id_List
     (U : Unit_Id;
      L : in out Unit_Id_List_Access);
   --  If not In_Unit_Id_List (U, L), U is appended to L (if L is null,
   --  new Unit_Id_List value is created)

   -------------------
   -- Add_To_Parent --
   -------------------

   procedure Add_To_Parent (C : Context_Id; U : Unit_Id) is
      Parent_Id : Unit_Id;
      Unit_Kind : Unit_Kinds := Kind (C, U);
   begin

      if U = Standard_Id then
         return;
      end if;

      Reset_Context (C); -- ???

      Get_Name_String (C, U, Norm_Ada_Name);

      if Not_Root (C, U) then
         Form_Parent_Name;

         if Unit_Kind in A_Subunit then
            A_Name_Buffer (A_Name_Len) := 'b';
         end if;

         Parent_Id := Name_Find (C);
         --  Parent_Id cannot be Nil_Unit here

            Append_Elmt
              (Unit => U,
               To   => Unit_Table.Table (Parent_Id).Subunits_Or_Childs);
      else
         Append_Elmt
           (Unit => U,
            To   => Unit_Table.Table (Standard_Id).Subunits_Or_Childs);
      end if;

   end Add_To_Parent;

   -------------------------
   -- Add_Unit_Supporters --
   -------------------------

   procedure Add_Unit_Supporters (U : Unit_Id; L : in out Elist_Id) is
      Supporters : Elist_Id        renames Unit_Table.Table (U).Supporters;
      Direct_Supporters : Elist_Id renames
         Unit_Table.Table (U).Direct_Supporters;

      Next_Support_Elmt : Elmt_Id;
      Next_Support_Unit : Unit_Id;

   begin

      if Is_Empty_Elmt_List (Direct_Supporters) then
         --  end of the recursion
         return;

      elsif not Is_Empty_Elmt_List (Supporters) then
         --  no need to traverse indirect dependencies

         Next_Support_Elmt := First_Elmt (Supporters);

         while Present (Next_Support_Elmt) loop
            Next_Support_Unit := Unit (Next_Support_Elmt);

            Add_To_Elmt_List
              (Unit => Next_Support_Unit,
               List => L);

            Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);

         end loop;

      else
         --  And here we have to traverse the recursive dependencies:

         Next_Support_Elmt := First_Elmt (Direct_Supporters);

         while Present (Next_Support_Elmt) loop
            Next_Support_Unit := Unit (Next_Support_Elmt);

            --  Here we can not be sure, that if Next_Support_Unit already
            --  is in the list, all its supporters also are in the list
            Add_To_Elmt_List
              (Unit => Next_Support_Unit,
               List => L);

            Add_Unit_Supporters (Next_Support_Unit, L);

            Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);

         end loop;

      end if;

   end Add_Unit_Supporters;

   -------------------------
   -- Append_Subunit_Name --
   -------------------------

   procedure Append_Subunit_Name (Def_S_Name : Node_Id) is
   begin
      Get_Decoded_Name_String (Chars (Def_S_Name));
      A_Name_Buffer (A_Name_Len - 1) := '.';
      A_Name_Buffer (A_Name_Len .. A_Name_Len + Name_Len - 1) :=
         Name_Buffer (1 .. Name_Len);
      A_Name_Len := A_Name_Len + Name_Len + 1;
      A_Name_Buffer (A_Name_Len - 1) := '%';
      A_Name_Buffer (A_Name_Len)     := 'b';
   end Append_Subunit_Name;

   --------------------
   -- Get_First_Stub --
   --------------------

   function Get_First_Stub (Body_Node : Node_Id) return Node_Id is
      Decls : List_Id;
      Decl  : Node_Id;
   begin
      --  pragma Assert (Nkind (Body) = ...)
      Decls := Declarations (Body_Node);

      if No (Decls) then
         return Empty;
      else
         Decl := Nlists.First (Decls);

         while Present (Decl) loop

            if Nkind (Decl) in N_Body_Stub then
               return Decl;
            end if;

            Decl := Next (Decl);
         end loop;
         return Empty;
      end if;

   end Get_First_Stub;

   -------------------
   -- Get_Next_Stub --
   -------------------

   function Get_Next_Stub  (Stub_Node : Node_Id) return Node_Id is
      Next_Decl : Node_Id;
   begin
      Next_Decl := Next (Stub_Node);

      while Present (Next_Decl) loop

         if Nkind (Next_Decl) in N_Body_Stub then
            return Next_Decl;
         end if;

         Next_Decl := Next (Next_Decl);
      end loop;
      return Empty;
   end Get_Next_Stub;

   ------------------
   -- Process_Stub --
   ------------------

   procedure Process_Stub (C : Context_Id; U : Unit_Id; Stub : Node_Id) is
      Def_S_Name     : Node_Id;
      Subunit_Id     : Unit_Id;
   begin
      --  We should save (and then restore) the content of A_Name_Buffer in
      --  case when more then one stub is to be processed. (A_Name_Buffer
      --  contains the Ada name of the parent body)

      NB_Save;

      if Nkind (Stub) = N_Subprogram_Body_Stub then
         Def_S_Name := Defining_Unit_Name (Specification (Stub));
      else
         Def_S_Name := Defining_Identifier (Stub);
      end if;

      Append_Subunit_Name (Def_S_Name);

      Subunit_Id := Name_Find (C);

      if No (Subunit_Id) then
         Subunit_Id := Allocate_Nonexistent_Unit_Entry (C);
         Append_Elmt (Unit => Subunit_Id,
                      To   => Unit_Table.Table (U).Subunits_Or_Childs);
      end if;

      NB_Restore;

   end Process_Stub;

   --------------------------
   -- Set_All_Dependencies --
   --------------------------

   procedure Set_All_Dependencies (C : Context_Id) is
   begin
      for U in First_Unit_Id + 1 .. Last_Unit (C) loop
         --  First_Unit_Id corresponds to Standard
         Set_All_Unit_Dependencies (C, U);
      end loop;
   end Set_All_Dependencies;

   -------------------------------
   -- Set_All_Unit_Dependencies --
   -------------------------------

   procedure Set_All_Unit_Dependencies (C : Context_Id; U : Unit_Id) is
      Supporters        : Elist_Id renames Unit_Table.Table (U).Supporters;
      Direct_Supporters : Elist_Id renames
         Unit_Table.Table (U).Direct_Supporters;

      Next_Support_Elmt : Elmt_Id;
      Next_Support_Unit : Unit_Id;
   begin

      --  Setting all the unit supporters
      Next_Support_Elmt := First_Elmt (Direct_Supporters);

      while Present (Next_Support_Elmt) loop
         Next_Support_Unit := Unit (Next_Support_Elmt);

         --  If Next_Support_Unit already is in Supporters list,
         --  all its supportes also are already included in Supporters.

         if not In_Elmt_List (Next_Support_Unit, Supporters) then
            Append_Elmt
              (Unit => Next_Support_Unit,
               To   => Supporters);

            Add_Unit_Supporters (Next_Support_Unit, Supporters);
         end if;

         Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);

      end loop;

      --  And now - adding U as depended unit to the list of Dependents for
      --  all its supporters

      Next_Support_Elmt := First_Elmt (Supporters);

      while Present (Next_Support_Elmt) loop
         Next_Support_Unit := Unit (Next_Support_Elmt);

         Append_Elmt
           (Unit => U,
            To   => Unit_Table.Table (Next_Support_Unit).Dependents);

         Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);
      end loop;

   end Set_All_Unit_Dependencies;

   ---------------------------
   -- Set_Direct_Dependents --
   ---------------------------
   procedure Set_Direct_Dependents
     (C   : Context_Id;
      U   : Unit_Id)
   is
      Next_Support_Elmt : Elmt_Id;
      Next_Support_Unit : Unit_Id;
   begin
      Next_Support_Elmt := First_Elmt (Unit_Table.Table (U).Direct_Supporters);

      while Present (Next_Support_Elmt) loop
         Next_Support_Unit := Unit (Next_Support_Elmt);

         Append_Elmt
           (Unit => U,
            To   => Unit_Table.Table (Next_Support_Unit).Direct_Dependents);

         Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);
      end loop;

   end Set_Direct_Dependents;

   ------------------------
   -- Set_All_Supporters --
   ------------------------

   procedure Set_All_Supporters
     (Compilation_Units : in Asis.Compilation_Unit_List;
      Dependent_Units   : in Asis.Compilation_Unit_List;
      The_Context       : in Asis.Context;
      Result            : in out Compilation_Unit_List_Access)
   is
      Cont            : Context_Id := Current_Context;
      Arg_List        : Unit_Id_List (1 .. Compilation_Units'Length) :=
                        (others => Nil_Unit);
      Arg_List_Len    : Natural := 0;

      Dep_List        : Unit_Id_List (1 .. Dependent_Units'Length) :=
                        (others => Nil_Unit);
      Dep_List_Len    : Natural := 0;

      Result_List     : Unit_Id_List_Access := null;

      Next_Unit   : Unit_Id;

      Next_Support_Elmt : Elmt_Id;
      Next_Support_Unit : Unit_Id;

      function In_List
        (U : Unit_Id;
         L : Unit_Id_List;
         Up_To : Natural)
         return Boolean;
      --  Checks if U is a member of the first Up_To components of L. (If
      --  Up_To is 0, False is returned

      function In_List
        (U : Unit_Id;
         L : Unit_Id_List;
         Up_To : Natural)
         return Boolean
      is
         Len    : Natural := Natural'Min (Up_To, L'Length);
         Result : Boolean := False;
      begin
         for I in 1 .. Len loop
            if L (I) = U then
               Result := True;
               exit;
            end if;
         end loop;

         return Result;

      end In_List;

   begin  --  Set_All_Supporters
      --  For the current version, we are supposing, that we have only one
      --  Context opened at a time

      --  ??? Do we need the Dependent_Units parameter for this function
      --  (Probably, we do not).
      --  ???  What about adding some code for other Relation values???

      --  First, we convert arguments into Unit_Id lists:

      for I in Compilation_Units'Range loop
         Next_Unit := Get_Unit_Id (Compilation_Units (I));

         if not In_List (Next_Unit, Arg_List, Arg_List_Len) then
            Arg_List_Len := Arg_List_Len + 1;
            Arg_List (Arg_List_Len) := Next_Unit;
         end if;

      end loop;

      for I in Dependent_Units'Range loop
         Next_Unit := Get_Unit_Id (Dependent_Units (I));

         if not In_List (Next_Unit, Dep_List, Dep_List_Len) then
            Dep_List_Len := Dep_List_Len + 1;
            Dep_List (Dep_List_Len) := Next_Unit;
         end if;

      end loop;

      --  Now, collecting all the supporters for Compilation_Units

      for I in 1 .. Arg_List_Len loop

         Next_Support_Elmt :=
            First_Elmt (Unit_Table.Table (Arg_List (I)).Supporters);

         while Present (Next_Support_Elmt) loop
            Next_Support_Unit := Unit (Next_Support_Elmt);
            Add_To_Unit_Id_List (Next_Support_Unit, Result_List);

            Next_Support_Elmt := Next_Elmt (Next_Support_Elmt);

         end loop;

      end loop;

      --  Here we have to take into account Dependent_Units
      --  ???  Is not done at the moment

      --  And here we have to order Result_List to eliminate forward
      --  semantic dependencies

      if Result_List /= null then
         Result := new Compilation_Unit_List'
                        (Get_Comp_Unit_List (Result_List.all, Cont));
         Free (Result_List);
      else
         Result := new Compilation_Unit_List (1 .. 0);
      end if;

   end Set_All_Supporters;

   ------------------
   -- Set_Subunits --
   ------------------

   procedure Set_Subunits (C : Context_Id; U : Unit_Id; Top : Node_Id) is
      Body_Node : Node_Id;
      Stub_Node : Node_Id;
   begin
      Get_Name_String (C, U, Norm_Ada_Name);
      Body_Node := Unit (Top);

      if Nkind (Body_Node) = N_Subunit then
         Body_Node := Proper_Body (Body_Node);
      end if;

      Stub_Node := Get_First_Stub (Body_Node);

      if No (Stub_Node) then
         return;
      end if;

      while Present (Stub_Node) loop
         Process_Stub (C, U, Stub_Node);
         Stub_Node := Get_Next_Stub (Stub_Node);
      end loop;

      Unit_Table.Table (U).Subunits_Computed := True;

   end Set_Subunits;

   --------------------
   -- Set_Supporters --
   --------------------

   procedure Set_Supporters (C : Context_Id; U : Unit_Id; Top : Node_Id) is
   begin
      Set_Withed_Units      (C, U, Top);
      Set_Direct_Dependents (C, U);
   end Set_Supporters;

   ----------------------
   -- Set_Withed_Units --
   ----------------------

   procedure Set_Withed_Units (C : Context_Id; U : Unit_Id; Top : Node_Id)
   is
      With_Clause_Node  : Node_Id;
      Cunit_Node        : Node_Id;
      Cunit_Number      : Unit_Number_Type;
      Current_Supporter : Unit_Id;
      Tmp               : Unit_Id;
      Include_Unit      : Boolean := False;
   begin
      --  the maim control structure - cycle through the with clauses
      --  in the tree
      if No (Context_Items (Top)) then
         return;
      end if;

      With_Clause_Node := First_Non_Pragma (Context_Items (Top));

      while Present (With_Clause_Node) loop
         --  here we simply get the name of the next supporting unit from
         --  the GNAT Units Table (defined in Lib)
         Cunit_Node    := Library_Unit (With_Clause_Node);
         Cunit_Number  := Get_Cunit_Unit_Number (Cunit_Node);
         Get_Decoded_Name_String (Unit_Name (Cunit_Number));

         Set_Norm_Ada_Name_String_With_Check (Cunit_Number, Include_Unit);

         if Include_Unit then

            Current_Supporter := Name_Find (C);

            if A_Name_Buffer (A_Name_Len) = 'b' then
               A_Name_Buffer (A_Name_Len) := 's';
               Tmp := Name_Find (C);

               if Present (Tmp) then
                  --  OPEN PROBLEM: is this the best solution for this problem?
                  --
                  --  Here we are in the potentially hard-to-report-about and
                  --  definitely involving inconsistent unit set situation.
                  --  The last version of U depends on subprogram body at least
                  --  in one of the consistent trees, but the Context contains
                  --  a spec (that is, a library_unit_declaration or a
                  --  library_unit_renaming_declaration) for the same full
                  --  expanded Ada name. The current working decision is
                  --  to set this dependency as if U depends on the spec.
                  --
                  --  Another (crazy!) problem: in one consistent tree
                  --  U depends on the package P (and P does not require a
                  --  body), and in another consistent tree U depends on
                  --  the procedure P which is presented by its body only.
                  --  It may be quite possible, if these trees were created
                  --  with different search paths. Is our decision reasonable
                  --  for this crazy situation :-[ ??!!??

                  Current_Supporter := Tmp;
               end if;

            end if;

            --  and now we store this dependency - we have to use
            --  Add_To_Elmt_List instead of Append_Elmt - some units
            --  may be mentioned several times in the context clause:
            if Implicit_With (With_Clause_Node) then
               Add_To_Elmt_List
                 (Unit => Current_Supporter,
                  List => Unit_Table.Table (U).Implicit_Supporters);
            else
               Add_To_Elmt_List
                 (Unit => Current_Supporter,
                  List => Unit_Table.Table (U).Direct_Supporters);
            end if;

            With_Clause_Node := Next_Non_Pragma (With_Clause_Node);

            while Present (With_Clause_Node) and then
                  Nkind (With_Clause_Node) /= N_With_Clause
            loop
               With_Clause_Node := Next_Non_Pragma (With_Clause_Node);
            end loop;

         end if;

      end loop;
   end Set_Withed_Units;

   -------------------------------------------------------
   -- Dynamic Unit_Id list abstraction (implementation) --
   -------------------------------------------------------

   ----------------------
   --  In_Unit_Id_List --
   ----------------------

   function In_Unit_Id_List
     (U : Unit_Id;
      L : Unit_Id_List_Access)
       return Boolean
   is
   begin

      if L /= null then

         for I in L'Range loop

            if U = L (I) then
               return True;
            end if;

         end loop;

      end if;

      return False;
   end In_Unit_Id_List;

   --------------------------
   --  Add_To_Unit_Id_List --
   --------------------------

   procedure Add_To_Unit_Id_List
     (U : Unit_Id;
      L : in out Unit_Id_List_Access)
   is
   begin

      if not In_Unit_Id_List (U, L) then
         Append_Unit_To_List (U, L);
      end if;

   end Add_To_Unit_Id_List;

   -------------------------
   -- Append_Unit_To_List --
   -------------------------

   procedure Append_Unit_To_List
     (U : Unit_Id;
      L : in out Unit_Id_List_Access)
   is
   begin

      if L = null then
         L := new Unit_Id_List'(1 => U);
      else
         Free (Tmp_Unit_Id_List_Access);
         Tmp_Unit_Id_List_Access := new Unit_Id_List'(L.all & U);
         Free (L);
         L := new Unit_Id_List'(Tmp_Unit_Id_List_Access.all);
      end if;

   end Append_Unit_To_List;

end A4G.Contt.DP;