/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel3.h"

void ATL_drefsymm
(
   const enum ATLAS_SIDE      SIDE,
   const enum ATLAS_UPLO      UPLO,
   const int                  M,
   const int                  N,
   const double               ALPHA,
   const double               * A,
   const int                  LDA,
   const double               * B,
   const int                  LDB,
   const double               BETA,
   double                     * C,
   const int                  LDC
)
{
/*
 * Purpose
 * =======
 *
 * ATL_drefsymm  performs one of the matrix-matrix operations
 *
 *    C := alpha * A * B + beta * C,
 *
 * or
 *
 *    C := alpha * B * A + beta * C,
 *
 * where alpha and beta are scalars,  A is a symmetric matrix and B and
 * C are m by n matrices.
 *
 * Arguments
 * =========
 *
 * SIDE    (input)                       const enum ATLAS_SIDE
 *         On entry,  SIDE  specifies  whether the  symmetric  matrix  A
 *         appears  on  the left or right in the operation as follows:
 *
 *            SIDE = AtlasLeft     C := alpha * A * B + beta * C,
 *
 *            SIDE = AtlasRight    C := alpha * B * A + beta * C.
 *
 *         Unchanged on exit.
 *
 * UPLO    (input)                       const enum ATLAS_UPLO
 *         On entry, UPLO  specifies whether the upper or lower triangu-
 *         lar part of the array A is to be referenced as follows:
 *
 *             UPLO = AtlasUpper   Only the upper triangular part of A
 *                                 is to be referenced.
 *
 *             UPLO = AtlasLower   Only the lower triangular part of A
 *                                 is to be referenced.
 *
 *         Unchanged on exit.
 *
 * M       (input)                       const int
 *         On entry,  M  specifies  the number  of rows of the matrix C.
 *         M  must be at least zero. Unchanged on exit.
 *
 * N       (input)                       const int
 *         On entry, N  specifies the number of columns of the matrix C.
 *         N must be at least zero. Unchanged on exit.
 *
 * ALPHA   (input)                       const double
 *         On entry, ALPHA specifies the scalar alpha.   When  ALPHA  is
 *         supplied  as  zero  then the elements of the matrices A and B
 *         need not be set on input.
 *
 * A       (input)                       const double *
 *         On entry,  A  points  to an array of size equal to or greater
 *         than   LDA * ka * sizeof(   double  ),  where ka  is  m  when
 *         SIDE = AtlasLeft   and is  n  otherwise.  Before  entry  with
 *         SIDE = AtlasLeft, the  m by m  part of the  array A must con-
 *         tain the symmetric matrix, such that when  UPLO = AtlasUpper,
 *         the leading m by m upper triangular part of the array A  must
 *         contain the upper triangular part of the symmetric matrix and
 *         the  strictly lower triangular part of  A  is not referenced,
 *         and when  UPLO = AtlasLower, the  leading m by m lower trian-
 *         gular part of the array A must contain the  lower  triangular
 *         part of the symmetric matrix and the  strictly upper triangu-
 *         lar part of  A  is not referenced.
 *         Before  entry  with  SIDE = AtlasRight,  the  n by n  part of
 *         the  array  A  must contain the  symmetric matrix,  such that
 *         when  UPLO = AtlasUpper, the  leading n by n upper triangular
 *         part of the array  A  must  contain the upper triangular part
 *         of the  symmetric matrix  and the  strictly  lower triangular
 *         part of  A  is not  referenced,  and when  UPLO = AtlasLower,
 *         the leading  n by n  lower  triangular part  of the  array  A
 *         must  contain  the  lower  triangular part  of the  symmetric
 *         matrix and the  strictly  upper triangular part of  A  is not
 *         referenced. Unchanged on exit.
 *
 * LDA     (input)                       const int
 *         On entry, LDA  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDA  must be  at least
 *         MAX( 1, m ) when SIDE = AtlasLeft, and MAX( 1, n ) otherwise.
 *         Unchanged on exit.
 *
 * B       (input)                       const double *
 *         On entry,  B  points  to an array of size equal to or greater
 *         than   LDB * n * sizeof(   double  ).  Before entry, the lea-
 *         ding m by n  part of the array  B  must contain the matrix B.
 *         Unchanged on exit.
 *
 * LDB     (input)                       const int
 *         On entry, LDB  specifies the leading dimension of B as decla-
 *         red  in  the  calling  (sub) program.  LDB  must be  at least
 *         MAX( 1, m ). wise. Unchanged on exit.
 *
 * BETA    (input)                       const double
 *         On entry,  BETA  specifies the scalar  beta.   When  BETA  is
 *         supplied  as  zero  then  the  elements of the matrix C  need
 *         not be set on input. Unchanged on exit.
 *
 * C       (input/output)                double *
 *         On entry,  C  points  to an array of size equal to or greater
 *         than   LDC * n * sizeof(   double  ).  Before entry, the lea-
 *         ding m by n part of the array  C  must contain the matrix  C,
 *         except when beta is zero,  in which case C need not be set on
 *         entry.  On exit, the array C is overwritten by the m by n up-
 *         dated matrix.
 *
 * LDC     (input)                       const int
 *         On entry, LDC  specifies the leading dimension of A as decla-
 *         red  in  the  calling  (sub) program.  LDC  must be  at least
 *         MAX( 1, m ). Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   int                        i, iaj, iajk, iaki, iakj, ibij, ibik, ibkj,
                              icij, ickj, j, jai, jaj, jbj, jbk, jcj, k;
   register double            t0, t1;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( M == 0 ) || ( N == 0 ) ||
       ( ( ALPHA == ATL_dZERO ) && ( BETA  == ATL_dONE  ) ) ) return;

   if(      BETA == ATL_dZERO )
   {
      for( j = 0, jcj = 0; j < N; j++, jcj += LDC )
      {
         for( i = 0, icij = jcj; i < M; i++, icij += 1 )
         {
            C[icij] = ATL_dZERO;
         }
      }
   }
   else if( BETA != ATL_dONE )
   {
      for( j = 0, jcj = 0; j < N; j++, jcj += LDC )
      {
         for( i = 0, icij  = jcj; i < M; i++, icij += 1 )
         {
            C[icij] *= BETA;
         }
      }
   }

   if( ALPHA == ATL_dZERO ) return;

   if( SIDE == AtlasLeft )
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      jbj  = 0,   jcj  = 0;
              j < N; j++, jbj += LDB, jcj += LDC )
         {
            for( i = 0,      jai  = 0,   ibij  = jbj, icij  = jcj;
                 i < M; i++, jai += LDA, ibij += 1,   icij += 1 )
            {
               t0 = ALPHA * B[ibij];
               t1 = ATL_dZERO;
               for( k = 0,      iaki  = jai, ibkj  = jbj, ickj  = jcj;
                    k < i; k++, iaki += 1,   ibkj += 1,   ickj += 1 )
               {
                  C[ickj] += t0   * A[iaki];
                  t1   += B[ibkj] * A[iaki];
               }
               C[icij] += t0 * A[i+jai] + ALPHA * t1;
            }
         }
      }
      else
      {
         for( j = 0,      jbj  = 0,   jcj  = 0;
              j < N; j++, jbj += LDB, jcj += LDC )
         {
            for( i = M-1,  jai  = (M-1)*LDA, ibij  = M-1+jbj, icij  = M-1+jcj;
              i >= 0; i--, jai -= LDA,       ibij -= 1,       icij -= 1 )
            {
               t0 = ALPHA * B[ibij];
               t1 = ATL_dZERO;
               for( k = i+1, iaki  = i+1+jai, ibkj  = i+1+jbj, ickj  = i+1+jcj;
                 k < M; k++, iaki += 1,       ibkj += 1,       ickj += 1 )
               {
                  C[ickj] += t0   * A[iaki];
                  t1   += B[ibkj] * A[iaki];
               }
               C[icij] += t0 * A[i+jai] + ALPHA * t1;
            }
         }
      }
   }
   else
   {
      if( UPLO == AtlasUpper )
      {
         for( j = 0,      iaj  = 0, jaj  = 0,   jbj  = 0,   jcj  = 0;
              j < N; j++, iaj += 1, jaj += LDA, jbj += LDB, jcj += LDC )
         {
            t0 = ALPHA * A[j+jaj];
            for( i = 0,      ibij  = jbj, icij  = jcj;
                 i < M; i++, ibij += 1,   icij += 1 )
            {
               C[icij] += t0 * B[ibij];
            }
            for( k = 0,      iakj  = jaj, jbk  = 0;
                 k < j; k++, iakj += 1,   jbk += LDB )
            {
               t0 = ALPHA * A[iakj];
               for( i = 0,      ibik  = jbk, icij = jcj;
                    i < M; i++, ibik += 1,   icij += 1 )
               {
                  C[icij] += t0 * B[ibik];
               }
            }
            for( k = j+1,    iajk  = iaj+(j+1)*LDA, jbk  = (j+1)*LDB;
                 k < N; k++, iajk += LDA,           jbk += LDB )
            {
               t0 = ALPHA * A[iajk];
               for( i = 0,      ibik  = jbk, icij = jcj;
                    i < M; i++, ibik += 1,   icij += 1 )
               {
                  C[icij] += t0 * B[ibik];
               }
            }
         }
      }
      else
      {
         for( j = 0,      iaj  = 0, jaj  = 0,   jbj  = 0,   jcj  = 0;
              j < N; j++, iaj += 1, jaj += LDA, jbj += LDB, jcj += LDC )
         {
            t0 = ALPHA * A[j+jaj];
            for( i = 0,      ibij  = jbj, icij  = jcj;
                 i < M; i++, ibij += 1,   icij += 1 )
            {
               C[icij] += t0 * B[ibij];
            }
            for( k = 0,      iajk  = iaj, jbk  = 0;
                 k < j; k++, iajk += LDA, jbk += LDB )
            {
               t0 = ALPHA * A[iajk];
               for( i = 0,      ibik  = jbk, icij  = jcj;
                    i < M; i++, ibik += 1,   icij += 1 )
               {
                  C[icij] += t0 * B[ibik];
               }
            }
            for( k = j+1,    iakj  = j+1+jaj, jbk  = (j+1)*LDB;
                 k < N; k++, iakj += 1,       jbk += LDB )
            {
               t0 = ALPHA * A[iakj];
               for( i = 0,      ibik  = jbk, icij  = jcj;
                    i < M; i++, ibik += 1,   icij += 1 )
               {
                  C[icij] += t0 * B[ibik];
               }
            }
         }
      }
   }
/*
 * End of ATL_drefsymm
 */
}
