/* codegen.c -- generate code from the package tree
 * Written by Charles Briscoe-Smith; refer to the file LEGAL for details.
 */

/* Interface definition */

#include <stdio.h>
#include "parsetree.h"

/* Generate a C program equivalent to this java program */
extern void codegen(FILE *, char *);


#ifndef SEEN_codegen_h

#include <string.h>
#include "packagetree.h"
#include "error.h"
#include "stringpool.h"

int nullstring;
int initstring;
int clinitstring;

/* Current output file */
static FILE *o;

static int nexttemp=0;

static void
printtemp(int temp)
{
	if (temp==-1) thiscanthappen;
	fprintf(o, "l%04d", temp);
}

/* Print the C form of a type name */
static void
printname(struct name *n)
{
	if (n->prefix) {
		printname(n->prefix);
		putc('_', o);
	}
	fputs(n->id, o);
}

static void
printprimname(enum typeid p, struct name *n)
{

	switch (p) {
	case Boolean:	fputs("bool", o); break;
	case Byte:	fputs("byte", o); break;
	case Short:	fputs("short", o); break;
	case Int:	fputs("int", o); break;
	case Long:	fputs("long", o); break;
	case Char:	fputs("char", o); break;
	case Float:	fputs("float", o); break;
	case Double:	fputs("double", o); break;
	case Ref:	putc('_', o); printname(n); break;
	case Void: case Null:
			fputs("void", o); break;
	}
}

/* Print C declaration for a Java type.  KIND==0 means print a normal
   declaration, KIND=1 means leave off the final '*' (for use in the
   sizeof() in the argument of malloc()).  */
static void
printtypedecl(struct type *t, int kind)
{
	int dims=t->dims;

	if (t->prim==Ref || dims) {
		fputs("struct ", o);
	}
	if (t->prim!=Void && t->prim!=Null) {
		fputs("Java", o);
	}
	if (t->prim==Ref || dims) {
		putc('i', o);
	}
	while (dims--) {
		putc('a', o);
	}

	printprimname(t->prim, t->ref);

	if ((t->prim==Ref || t->prim==Null || t->dims) && kind!=1
	    && !(t->prim==Ref && t->dims==0
	         && packagetree_findtype(t->ref)->modifiers & ACC_interface))
	{
		putc('*', o);
	} else {
		putc(' ', o);
	}
}

/* Print the C form of a type specifier, as used in the class file format */
static void
printtypespec(struct type *t)
{
	int d=t->dims;

	while (d--) {
		fputs("_3", o);
	}

	switch (t->prim) {
	case Boolean:	putc('Z', o); break;
	case Byte:	putc('B', o); break;
	case Short:	putc('S', o); break;
	case Int:	putc('I', o); break;
	case Long:	putc('J', o); break;
	case Char:	putc('C', o); break;
	case Float:	putc('F', o); break;
	case Double:	putc('D', o); break;
	case Ref:	putc('L', o);
			printname(t->ref);
			fputs("_2", o); break;
	case Void:	putc('V', o); break;
	case Null:	thiscanthappen;
	}
}


/* Pre-declare all the structures we'll need later */
static void
printstructstub(struct typedecl *t)
{
	int d=t->arrdepth;

	while (d>=0) {
		int dd=d--;
		fputs("struct Javai", o);
		while (dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(";\n", o);
	}
}

static void
printprimstub(int maxdepth, const char *name)
{
	int d;

	for (d=maxdepth; d; d--) {
		int dd=d;
		fputs("struct Javai", o);
		while (dd--) putc('a', o);
		fprintf(o, "%s;\n", name);
	}
}

static void
printprimstubs(void)
{
	printprimstub(getprimtypedepth(Boolean), "bool");
	printprimstub(getprimtypedepth(Byte), "byte");
	printprimstub(getprimtypedepth(Short), "short");
	printprimstub(getprimtypedepth(Int), "int");
	printprimstub(getprimtypedepth(Long), "long");
	printprimstub(getprimtypedepth(Char), "char");
	printprimstub(getprimtypedepth(Float), "float");
	printprimstub(getprimtypedepth(Double), "double");
}


/* Generate per-instance (non-static) struct definition */
static void
declareinstancestruct(struct typedecl *t)
{
	struct memberdecl *m;
	int d;

	fputs("struct Javai_", o);
	printname(t->name);
	fputs(" {\n", o);

	if (t->modifiers & ACC_interface) {
		fputs("\tstruct Javai_java_lang_Object *object;\n", o);
	} else if (t->super) {
		/* Superclass instance structure; embedded superobject */
		fputs("\tstruct Javai_", o);
		printname(t->super);
		fputs(" super;\n", o);
	}

	/* Output reference to instance method structure, if there are
	   any relevant methods */
	if (!(t->modifiers & ACC_native)) {
		fputs("\tstruct Javam_", o);
		printname(t->name);
		fprintf(o, " *methods;\n");
	}

	/* Output field members */
	for (m=t->body; m; m=m->rest) {
		if (m->tag==Field
		    && !(m->content.field->modifiers & ACC_static))
		{
			struct vardecl *d;
			for (d=m->content.field->decls; d; d=d->rest) {
				putc('\t', o);
				printtypedecl(m->content.field->type, 0);
				fprintf(o, "f_%s;\n", d->name);
			}
		}
	}

	fputs("};\n\n", o);

	for (d=1; d<=t->arrdepth; d++) {
		int dd;
		fputs("struct Javai", o);
		for (dd=d; dd; dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(" {\n\tstruct Javai_java_lang_Object super;\n"
		      "\tJavaint f_length;\n\tstruct Javai", o);
		for (dd=d-1; dd; dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(" *component[0];\n};\n\n", o);
	}
}

static void
declareprimarrayinstancestruct(int maxdepth, const char *name)
{
	int d;

	for (d=maxdepth; d; d--) {
		int dd;
		fputs("struct Javai", o);
		for (dd=d; dd; dd--) putc('a', o);
		fprintf(o, "%s {\n\tstruct Javai_java_lang_Object super;\n"
		           "\tJavaint f_length;\n\t", name);
		if (d>1) {
			fputs("struct Javai", o);
			for (dd=d-1; dd; dd--) putc('a', o);
			fprintf(o, "%s *", name);
		} else {
			fprintf(o, "Java%s ", name);
		}
		fputs("component[0];\n};\n\n", o);
	}
}

static void
declareprimarrayinstancestructs(void)
{
	declareprimarrayinstancestruct(getprimtypedepth(Boolean), "bool");
	declareprimarrayinstancestruct(getprimtypedepth(Byte), "byte");
	declareprimarrayinstancestruct(getprimtypedepth(Short), "short");
	declareprimarrayinstancestruct(getprimtypedepth(Int), "int");
	declareprimarrayinstancestruct(getprimtypedepth(Long), "long");
	declareprimarrayinstancestruct(getprimtypedepth(Char), "char");
	declareprimarrayinstancestruct(getprimtypedepth(Float), "float");
	declareprimarrayinstancestruct(getprimtypedepth(Double), "double");
}


enum paramprintmode { Spec, Decl, Defn };

static void
printparamlist(struct formparlist *args, enum paramprintmode pmode, int pcomma)
{
	if (args==0) {
		if (!pcomma && pmode!=Spec) {
			fputs("void", o);
		}
		return;
	}
	if (args->rest) printparamlist(args->rest, pmode, pcomma);

	switch (pmode) {
	case Spec:
		if (pcomma || args->rest) putc('_', o);
		printtypespec(args->type);
		break;
	case Decl:
		if (pcomma || args->rest) putc(',', o);
		printtypedecl(args->type, 0);
		break;
	case Defn:
		if (pcomma || args->rest) putc(',', o);
		printtypedecl(args->type, 0);
		fprintf(o, "l_%s", args->name);
		break;
	}
}

/* Call a function for each interface in ILIST or extended by an interface
   in ILIST, but which is not implemented, directly or indirectly,
   by SUPERCLASS.  */

static void
foreachinterfacenotin(struct name *ilist, struct name *superclass,
                      void (*fun)(struct typedecl *, void *), void *data)
{
	for (; ilist; ilist=ilist->rest) {
		if (!implements(superclass, ilist)) {
			struct typedecl *t=packagetree_findtype(ilist);
			(*fun)(t, data);
			foreachinterfacenotin(t->implements, superclass,
			                      fun, data);
		}
	}
}

static void
declareifacestruct(struct typedecl *t, void *data)
{
	fputs("\tstruct Javam_", o);
	printname(t->name);
	fputs(" ii_", o);
	printname(t->name);
	fputs(";\n", o);
}

/* Generate class jump-struct definition (& "semi-static" fields) */
static void
declarejumpstruct(struct typedecl *t)
{
	struct memberdecl *m;
	struct name *iface;

	fputs("struct Javam_", o);
	printname(t->name);
	fprintf(o, " {\n");

	if (t->modifiers & ACC_interface) {
		/* Interface: do nothing */
	} else if (t->super) {
		/* Class other than Object: superclass jump-struct.  */
		fputs("\tstruct Javam_", o);
		printname(t->super);
		fputs(" super;\n", o);
	} else {
		/* java.lang.Object: "semi-static" class reference and
		   interface lookup table.  */
		fputs("\tstruct Javai_java_lang_Class *myclass;\n"
		      "\tvoid **interfaces;\n", o);
	}

	for (m=t->body; m; m=m->rest) {
		if (m->tag==Method
		    && !(m->content.meth->modifiers
		         & (ACC_static|ACC_private|ACC_final))
		    && m->content.meth->overrides==0) {
			putc('\t', o);
			printtypedecl(m->content.meth->type, 0);
			fprintf(o, "(*m_%s__", m->content.meth->sig->name);
			printparamlist(m->content.meth->sig->params, Spec, 0);
			fputs(")(struct JavaStackFrame *,"
			      "struct Javai_java_lang_Object *", o);
			printparamlist(m->content.meth->sig->params, Decl, 1);
			fputs(");\n", o);
		}
	}

	if (!(t->modifiers & ACC_interface)) {
		foreachinterfacenotin(t->implements, t->super,
		                      declareifacestruct, 0);
	}

	for (iface=t->implements; iface; iface=iface->rest) {
		fputs("\tstruct Javam_", o);
		printname(iface);
		fputs(" *i_", o);
		printname(iface);
		fputs(";\n", o);
	}

	fputs("};\n\n", o);
}


/* Generate declarations for constructors, methods and static fields */
static void
declarestatics(struct typedecl *t)
{
	struct memberdecl *m;
	int seenconstructor=0;

	if (t->modifiers & ACC_interface) {
		seenconstructor=1;
	}

	fputs("/****** ", o);
	printname(t->name);
	fputs(" ******/\n",o);

	for (m=t->body; m; m=m->rest) {
		if (m->tag==Field
		    && m->content.field->modifiers & ACC_static) {
			printtypedecl(m->content.field->type, 0);
			fputs("Javasf_", o);
			printname(t->name);
			fprintf(o, "_%s;\n", m->content.field->decls->name);
		} else if (m->tag==Method
		           && !(m->content.meth->modifiers & ACC_abstract)) {
			struct methoddecl *meth=m->content.meth;
			fputs("extern ", o);
			printtypedecl(meth->type, 0);
			/* Was Javasm_, but changed for JNI compat. */
			fputs("Java_", o);
			printname(t->name);
			fprintf(o, "_%s__", meth->sig->name);
			printparamlist(meth->sig->params, Spec, 0);
			fputs("(struct JavaStackFrame *", o);
			if (!(meth->modifiers & ACC_static)) {
				fputs(",struct Javai_java_lang_Object *", o);
				printparamlist(meth->sig->params, Decl, 1);
			} else {
				printparamlist(meth->sig->params, Decl, 1);
			}
			fputs(");\n", o);
		} else if (m->tag==Constructor) {
			struct constrdecl *con=m->content.constr;
			seenconstructor=1;
			fputs("extern void Javac_", o);
			printname(t->name);
			fputs("__", o);
			printparamlist(con->sig->params, Spec, 0);
			fputs("(struct JavaStackFrame *,"
			      "struct Javai_", o);
			printname(t->name);
			fputs(" *, struct Javam_", o);
			printname(t->name);
			fputs(" *", o);
			printparamlist(con->sig->params, Decl, 1);
			fputs(");\n", o);
		}
	}

	/* Default constructor */
	if (!seenconstructor) {
		fputs("extern void Javac_", o);
		printname(t->name);
		fputs("__(struct JavaStackFrame *,"
		      "struct Javai_", o);
		printname(t->name);
		fputs(" *, struct Javam_", o);
		printname(t->name);
		fputs(" *);\n", o);
	}

	/* Class static initialiser */
	fputs("extern void Javasi_", o);
	printname(t->name);
	fputs("(struct JavaStackFrame *);\n\n", o);
}


/* Generate class object definitions */
static void
declareclassobjects(struct typedecl *t)
{
	int d;

	for (d=0; d<=t->arrdepth; d++) {
		int dd;
		fputs("struct Javai_java_lang_Class Javaco", o);
		for (dd=d; dd; dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(";\n", o);
	}
}

static void
declareprimarrayclassobject(int maxdepth, const char *name)
{
	int d;

	for (d=1; d<=maxdepth; d++) {
		int dd;
		fputs("struct Javai_java_lang_Class Javaco", o);
		for (dd=d; dd; dd--) putc('a', o);
		fprintf(o, "%s;\n", name);
		/* fprintf(o, "%s={ { 0 }, 0, 0, 0 };\n", name); */
	}
}

static void
declareprimarrayclassobjects(void)
{
	declareprimarrayclassobject(getprimtypedepth(Boolean), "bool");
	declareprimarrayclassobject(getprimtypedepth(Byte), "byte");
	declareprimarrayclassobject(getprimtypedepth(Short), "short");
	declareprimarrayclassobject(getprimtypedepth(Int), "int");
	declareprimarrayclassobject(getprimtypedepth(Long), "long");
	declareprimarrayclassobject(getprimtypedepth(Char), "char");
	declareprimarrayclassobject(getprimtypedepth(Float), "float");
	declareprimarrayclassobject(getprimtypedepth(Double), "double");
}


static struct name *
findmethclass(struct typedecl *top, struct type *ret, struct methodsig *sig)
{
	struct typedecl *tscan;

	for (tscan=top; tscan; tscan=packagetree_findtype(tscan->super)) {
		struct memberdecl *mscan;
		for (mscan=tscan->body; mscan; mscan=mscan->rest) {
			if (mscan->tag==Method
			    && equaltypes(ret, mscan->content.meth->type)
			    && equalsig(sig, mscan->content.meth->sig))
			{
				if (mscan->content.meth->modifiers
				    & ACC_abstract)
				{
					fprintf(stderr, "Abstract method %s in"
					      " concrete class `", sig->name);
					o=stderr;
					printname(top->name);
					fputs("'!\n", stderr);
					error("stop");
				} else {
					return tscan->name;
				}
			}
		}
	}

	error("Can't find method!");
}

static void printpartjumptable(struct typedecl *, struct typedecl *,
                               struct name *, const char *, int);

static void
printifacestructjumptable(struct typedecl *t, void *top)
{
	fputs("\t{\n", o);
	printpartjumptable(t, top, 0, 0, 0);
	fputs("\t},\n", o);
}

static void
printpartjumptable(struct typedecl *this, struct typedecl *top,
                   struct name *topname, const char *primname, int dims)
{
	struct memberdecl *m;
	struct name *iface;

	if (this->modifiers & ACC_interface) {
		/* Interface: do nothing */
	} else if (this->super) {
		/* Superclass jump-struct here.  */
		fputs("\t{\n", o);
		printpartjumptable(packagetree_findtype(this->super),
		                   top, topname, primname, dims);
		fputs("\t},\n", o);
	} else {
		/* Reached java.lang.Object.  Reference class object.  */
		int dd;
		fputs("\t&Javaco", o);
		for (dd=dims; dd; dd--) putc('a', o);
		if (topname) {
			putc('_', o);
			printname(topname);
		} else {
			fputs(primname, o);
		}
		fputs(",\n\t&Javaif_", o);
		if (dims) {
			fputs("java_lang_Object", o);
		} else {
			printname(topname);
		}
		fputs("[0],\n", o);
	}

	for (m=this->body; m; m=m->rest) {
		if (m->tag==Method
		    && !(m->content.meth->modifiers
		         & (ACC_static|ACC_private|ACC_final))
		    && m->content.meth->overrides==0)
		{
			struct name *n;
			n=findmethclass(top, m->content.meth->type,
			                m->content.meth->sig);
			if (n) {
				fputs("\tJava_", o);
				printname(n);
				fprintf(o, "_%s__", m->content.meth->sig->name);
				printparamlist(m->content.meth->sig->params,
				               Spec, 0);
				fputs(",\n", o);
			} else {
				fputs("\t(void *) 0,\t/* abstract */\n", o);
			}
		}
	}

	if (!(this->modifiers & ACC_interface)) {
		foreachinterfacenotin(this->implements, this->super,
		                      printifacestructjumptable, top);
	}

	for (iface=this->implements; iface; iface=iface->rest) {
		struct typedecl *contain;
		int dd;

		fputs("\t&(Javami", o);
		for (dd=dims; dd; dd--) putc('a', o);
		putc('_', o);
		printname(top->name);
		contain=top;
		while (1) {
			struct typedecl *superclass;
			superclass=packagetree_findtype(contain->super);
			if (superclass==0)
				error("Can't find widest implementing class!");
			if (!implements(superclass->name, iface))
				break;
			fputs(".super", o);
			contain=superclass;
		}
		fputs(".ii_", o);
		printname(iface);
		fputs("),\n", o);
	}
}

static void
interfacetablewiden(struct name *cur, struct name *target)
{
	struct typedecl *t=packagetree_findtype(cur);

	if (implements(t->super, target)) {
		fputs(".super", o);
		interfacetablewiden(t->super, target);
		return;
	}

	fputs(".ii_", o);
	printname(target);
	return;
}

static void
printpartinterfacetable(struct typedecl *t, void *data)
{
	fputs("&Javaco_", o);
	printname(t->name);
	fputs(",\n  &Javami_", o);
	printname(data);
	interfacetablewiden(data, t->name);
	fputs(",\n", o);
}

static void
printinterfacetable(struct typedecl *t)
{
	fputs("void *Javaif_", o);
	printname(t->name);
	fputs("[]={\n", o);
	foreachinterfacenotin(t->implements, ObjectType.ref,
	                      printpartinterfacetable, t->name);
	fputs("0\n};\n\n", o);
}

/* Instantiate class jump-struct */
static void
printjumptable(struct typedecl *t)
{
	struct typedecl *obj=packagetree_findtype(ObjectType.ref);
	int d;

	for (d=0; d<=t->arrdepth; d++) {
		if (d>0 || !(t->modifiers & ACC_abstract)) {
			int dd;
			if (d==0) {
				fputs("extern struct Javam_", o);
				printname(t->name);
				fputs(" Javami_", o);
				printname(t->name);
				fputs(";\n\n", o);
				printinterfacetable(t);
			}
			fputs("struct Javam_", o);
			printname(d ? ObjectType.ref : t->name);
			fputs(" Javami", o);
			for (dd=d; dd; dd--) putc('a', o);
			putc('_', o);
			printname(t->name);
			fprintf(o, "={\n");

			if (d) {
				printpartjumptable(obj, obj, t->name, 0, d);
			} else {
				printpartjumptable(t, t, t->name, 0, 0);
			}

			fputs("};\n\n", o);
		}
	}
}

static void
printprimarrayjumptable(int maxdepth, const char *name)
{
	struct typedecl *t=packagetree_findtype(ObjectType.ref);
	int d;

	for (d=1; d<=maxdepth; d++) {
		int dd;
		fputs("struct Javam_java_lang_Object Javami", o);
		for (dd=d; dd; dd--) putc('a', o);
		fputs(name, o);
		fprintf(o, "={\n");

		printpartjumptable(t, t, 0, name, d);

		fputs("};\n\n", o);
	}
}

static void
printprimarrayjumptables(void)
{
	printprimarrayjumptable(getprimtypedepth(Boolean), "bool");
	printprimarrayjumptable(getprimtypedepth(Byte), "byte");
	printprimarrayjumptable(getprimtypedepth(Short), "short");
	printprimarrayjumptable(getprimtypedepth(Int), "int");
	printprimarrayjumptable(getprimtypedepth(Long), "long");
	printprimarrayjumptable(getprimtypedepth(Char), "char");
	printprimarrayjumptable(getprimtypedepth(Float), "float");
	printprimarrayjumptable(getprimtypedepth(Double), "double");
}


static int
canwiden(struct name *cur, struct name *target)
{
	struct typedecl *t=packagetree_findtype(cur);
	if (t==0) return 0;
	if (equalnames(cur, target)) return 1;
	if (implements(cur, target)) return 1;
	return canwiden(t->super, target);
}

/* This prints the trailing part of a reference widening operation.
   If TARGET is an interface type, the result will give the jump struct
   for the object regarded as an interface, otherwise, a full instance
   struct.  If initial type is an interface type, we must already have
   accessed its jump struct member before calling widen().  */

static void
widen(struct name *cur, struct name *target)
{
	struct typedecl *t=packagetree_findtype(cur);
	struct name *iface;

	if (t==0) {
		fputs("Reference widening to ", stderr);
		o=stderr;
		printname(target);
		fputs(" failed!\n", stderr);
		error("stop");
	}

	if (equalnames(cur, target)) {
		return;
	}

	if (canwiden(t->super, target)) {
		fputs(".super", o);
		widen(t->super, target);
		return;
	}

	for (iface=t->implements; iface; iface=iface->rest) {
		if (canwiden(iface, target)) {
			fputs("->i_", o);
			printname(iface);
			widen(iface, target);
			return;
		}
	}

	error("Reference widening failed!");
}

/* As widen(), but widen from any reference type to type Object. */

static void
fullywiden(struct name *cur)
{
	struct typedecl *t=packagetree_findtype(cur);

	if (t==0) {
		error("Can't perform maximal reference widening!");
	}

	if (t->modifiers & ACC_interface) {
		fputs(".object", o);
		return;
	}

	if (t->super==0) {
		return;
	}

	fputs(".super", o);
	fullywiden(t->super);
}


static int
newtemp(struct type *type, int variable)
{
	int temp=nexttemp++;
	printtypedecl(type, 0);
	if (variable) putc('*', o);
	/* fputs("volatile ", o); */
	printtemp(temp);
	fputs(";\n", o);
	return temp;
}

static void
usetemp(int temp)
{
	fputs("(void) &", o);
	printtemp(temp);
	fputs(";\n", o);
}

static void
codegen_expr_value(struct expr *, int, struct typedecl *, struct memberdecl *);

static void
codegen_expr_var(struct expr *expr, int result, struct typedecl *t,
                 struct memberdecl *m)
{
	int targettemp, temp1, temp2;

	fputs("{\n", o);
	switch (expr->tag) {
	case Name: thiscanthappen;
	case Prim: thiscanthappen;
	case Arrayinit: thiscanthappen;

	case Assign:	case Multass:	case Divass:	case Remass:
	case Addass:	case Subass:	case Lshiftass:	case Rshiftass:
	case Urshiftass: case Andass:	case Xorass:	case Orass:
	case Cond:	case Condor:	case Condand:
	case Or:	case Xor:	case And:
	case Eq:	case Neq:	case Inst:
	case Lt:	case Gt:	case Leq:	case Geq:
	case Lshift:	case Rshift:	case Urshift:
	case Add:	case Sub:	case NullString:
	case Mult:	case Div:	case Rem:
	case Cast:	case Not:	case Comp:
	case Predec:	case Preinc:	case Postdec:	case Postinc:
	case Plus:	case Minus:	case Intlit:	case Longlit:
	case Stringlit:	case Boollit:	case Charlit:
	case Floatlit:	case Doublit:	case Nulllit:
	case This:	case New:	case Meth:
		error("Tried to use a value as a variable!");

	case Primfield:
		if (expr->exp1->tag != Name &&
		    !(expr->integ & ACC_static && expr->exp1->tag==This))
		{
			targettemp=newtemp(expr->exp1->type, 0);
			usetemp(targettemp);
			codegen_expr_value(expr->exp1, targettemp, t, m);
		} else {
			/* Gratuitous initialisation for GCC */
			targettemp=-1;
		}
		printtemp(result);
		fputs("=&(", o);
		if (expr->integ & ACC_static) {
			fputs("Javasf_", o);
			printname(expr->exp1->type->ref);
			fprintf(o, "_%s);\n", expr->str);
			fputs("Javasi_", o);
			printname(expr->exp1->type->ref);
			fputs("(stackframe);\n", o);
		} else {
			printtemp(targettemp);
			fprintf(o, "->f_%s);\n", expr->str);
		}
		break;
	case Super: thiscanthappen;
	case Array:
		temp1=newtemp(expr->exp1->type, 0);
		temp2=newtemp(expr->exp2->type, 0);
		usetemp(temp1); usetemp(temp2);
		codegen_expr_value(expr->exp1, temp1, t, m);
		codegen_expr_value(expr->exp2, temp2, t, m);
		fputs("if (", o);
		printtemp(temp1);
		fputs("==0) {\n"
		      "\tthrown_object=constNullPointerException;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n"
		      "if (", o);
		printtemp(temp2);
		fputs("<0 || ", o);
		printtemp(temp2);
		fputs(">=", o);
		printtemp(temp1);
		fputs("->f_length) {\n"
		      "\tthrown_object=constIndexOutOfBoundsException;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n", o);
		printtemp(result);
		fputs("=&(", o);
		printtemp(temp1);
		fputs("->component[", o);
		printtemp(temp2);
		fputs("]);\n", o);
		break;
	case Localacc:
		printtemp(result);
		fprintf(o, "=&l_%s;\n", expr->str);
		break;
	}
	fputs("}\n", o);
}

static int
countargs(struct expr *args)
{
	if (args==0) return 0;
	return 1+countargs(args->rest);
}

static int
declareargs(struct expr *args, int result)
{
	if (args==0) return result;

	result=declareargs(args->rest, result);

	printtypedecl(args->type, 0);
	/* fputs("volatile ", o); */
	printtemp(result);
	fputs(";\n", o);
	return result+1;
}

static int
useargs(struct expr *args, int result)
{
	if (args==0) return result;

	result=useargs(args->rest, result);

	usetemp(result);
	return result+1;
}

static int
evalargs(struct expr *args, int result, struct typedecl *t,
         struct memberdecl *m)
{
	if (args==0) return result;

	result=evalargs(args->rest, result, t, m);

	codegen_expr_value(args, result, t, m);
	return result+1;
}

static void
printargspecs(struct expr *args)
{
	if (args==0) return;

	if (args->rest) {
		printargspecs(args->rest);
		putc('_', o);
	}
	printtypespec(args->type);
}

static void
printtemplist(int firsttemp, int tempcount)
{
	while (tempcount--) {
		printtemp(firsttemp++);
		if (tempcount) putc(',', o);
	}
}

static int
assign_arrayinit(struct expr *expr, int array, int temp,
                 struct typedecl *t, struct memberdecl *m)
{
	int index;
	if (expr==0) return 0;
	index=assign_arrayinit(expr->rest, array, temp, t, m);

	codegen_expr_value(expr, temp, t, m);
	printtemp(array);
	fprintf(o, "->component[%d]=", index);
	printtemp(temp);
	fputs(";\n", o);
	return index+1;
}

static void
codegen_expr_value(struct expr *expr, int result, struct typedecl *t,
                   struct memberdecl *m)
{
	int argcount;
	int firsttemp;
	int targettemp;
	int temp1, temp2;
	int assign=0;

	fputs("{\n", o);
	switch (expr->tag) {
	case Assign: case Multass: case Divass: case Remass:
	case Addass: case Subass: case Lshiftass: case Rshiftass:
	case Urshiftass: case Andass: case Xorass: case Orass:
		assign=1;
		/* and fall through... */
	case Or: case Xor: case And: case Eq: case Neq:
	case Lt: case Gt: case Leq: case Geq:
	case Lshift: case Rshift: case Urshift:
	case Add: case Sub: case Mult: case Div: case Rem:
		/* In all cases, RH operand will be evaluated. */
		targettemp=newtemp(expr->exp2->type, 0);
		if (assign) {
			if (expr->tag!=Assign) {
				/* In compound assignments, we'll need */
				/* to store away the LHS's value. */
				firsttemp=newtemp(expr->exp1->type, 0);
			} else {
				/* Gratuitous initialisation for GCC */
				firsttemp=-1;
			}
			if (expr->exp1->tag==Array) {
				/* Assigning to an array component. */
				temp1=newtemp(expr->exp1->exp1->type, 0);
				temp2=newtemp(expr->exp1->exp2->type, 0);
				usetemp(targettemp);
				if (firsttemp!=-1) usetemp(firsttemp);
				usetemp(temp1);
				usetemp(temp2);
				codegen_expr_value(expr->exp1->exp1,temp1,t,m);
				codegen_expr_value(expr->exp1->exp2,temp2,t,m);
				if (expr->tag==Assign) {
					/* For plain assignment, evaluate
					   RHS early. */
					codegen_expr_value(expr->exp2,
					                   targettemp, t, m);
				}
				/* Check array subscripting on LHS. */
				fputs("if (", o);
				printtemp(temp1);
				fputs("==0) {\n"
				      "\tthrown_object="
				          "constNullPointerException;\n"
				      "\tlongjmp(*cur_exception_handler, 1);\n"
				      "}\n"
				      "if (", o);
				printtemp(temp2);
				fputs("<0 || ", o);
				printtemp(temp2);
				fputs(">=", o);
				printtemp(temp1);
				fputs("->f_length) {\n"
				      "\tthrown_object"
				          "=constIndexOutOfBoundsException;\n"
				      "\tlongjmp(*cur_exception_handler, 1);\n"
				      "}\n", o);
			} else {
				/* Assigning to ordinary variable. */
				temp1=newtemp(expr->exp1->type, 1);
				usetemp(targettemp);
				if (firsttemp!=-1) usetemp(firsttemp);
				usetemp(temp1);
				codegen_expr_var(expr->exp1, temp1, t, m);
				/* Gratuitous initialisation for GCC */
				temp2=-1;
			}
			if (expr->tag!=Assign) {
				/* Compound assignment; store LHS's value. */
				printtemp(firsttemp);
				putc('=', o);
				if (expr->exp1->tag==Array) {
					printtemp(temp1);
					fputs("->component[", o);
					printtemp(temp2);
					fputs("];\n", o);
				} else {
					putc('*', o);
					printtemp(temp1);
					fputs(";\n", o);
				}
			}
		} else {
			/* Non-assignment operation; evaluate LHS. */
			firsttemp=newtemp(expr->exp1->type, 0);
			usetemp(targettemp);
			usetemp(firsttemp);
			codegen_expr_value(expr->exp1, firsttemp, t, m);
			/* Gratuitous initialisations for GCC */
			temp1=temp2=-1;
		}
		if (expr->tag!=Assign || expr->exp1->tag!=Array) {
			/* Evaluate RHS, unless plain assignment to array
			   component, in which case this was done above. */
			codegen_expr_value(expr->exp2, targettemp, t, m);
		}
		/* Do the operation "firsttemp op targettemp".
		   (Except if simple assignment, just use targettemp.) */
		if (expr->tag==Div || expr->tag==Rem
		    || expr->tag==Divass || expr->tag==Remass)
		{
			fputs("if (", o);
			printtemp(targettemp);
			fputs("==0) {\n"
			      "\tthrown_object="
			          "constArithmeticException;\n"
			      "\tlongjmp(*cur_exception_handler, 1);\n"
			      "}\n", o);
		}
		if (expr->tag==Lshift || expr->tag==Lshiftass
		    || expr->tag==Rshift || expr->tag==Rshiftass
		    || expr->tag==Urshift || expr->tag==Urshiftass)
		{
			printtemp(targettemp);
			if (expr->exp1->type->prim==Long) {
				fputs("&=0x3f;\n", o);
			} else {
				fputs("&=0x1f;\n", o);
			}
		}
		printtemp(result);
		putc('=', o);
		if ((expr->tag==Add || expr->tag==Addass)
		    && expr->type->prim==Ref)
		{
			/* Do string concatenation */
			fputs("Java_java_lang_String_concat__"
			      "Ljava_lang_String_2(stackframe,&", o);
			printtemp(firsttemp);
			fputs("->super,", o);
			printtemp(targettemp);
			fputs(");\n", o);
		} else {
			if (expr->tag==Rshift || expr->tag==Rshiftass)
				putc('(', o);
			if (expr->tag!=Assign)
				printtemp(firsttemp);
			switch (expr->tag) {
				case Or: case Orass:
					putc('|', o); break;
				case Xor: case Xorass:
					putc('^', o); break;
				case And: case Andass:
					putc('&', o); break;
				case Eq: fputs("==", o); break;
				case Neq: fputs("!=", o); break;
				case Lt: putc('<', o); break;
				case Gt: putc('>', o); break;
				case Leq: fputs("<=", o); break;
				case Geq: fputs(">=", o); break;
				case Lshift: case Lshiftass:
					fputs("<<", o); break;
				case Rshift: case Rshiftass:
				case Urshift: case Urshiftass:
					fputs(">>", o); break;
				case Add: case Addass:
					putc('+', o); break;
				case Sub: case Subass:
					putc('-', o); break;
				case Mult: case Multass:
					putc('*', o); break;
				case Div: case Divass:
					putc('/', o); break;
				case Rem: case Remass:
					putc('%', o); break;
				case Assign: break;
				default: thiscanthappen;
			}
			printtemp(targettemp);
			if (expr->tag==Rshift || expr->tag==Rshiftass
			    || expr->tag==Urshift
			    || expr->tag==Urshiftass)
			{
				int bits=expr->type->prim==Long?63:31;
				fprintf(o, "&~(~1<<(%d-", bits);
				printtemp(targettemp);
				fputs("))", o);
				if (expr->tag==Rshift) {
					fputs(")|(", o);
					printtemp(firsttemp);
					fprintf(o, "<0?~1<<(%d-", bits);
					printtemp(targettemp);
					fputs("):0)", o);
				}
			}
			fputs(";\n", o);
		}
		if (expr->tag==Assign && expr->exp1->tag==Array) {
			/* Check if result is assignable to runtime
			   array component. */
		}
		if (assign) {
			if (expr->exp1->tag!=Array) {
				putc('*', o);
			}
			printtemp(temp1);
			if (expr->exp1->tag==Array) {
				fputs("->component[", o);
				printtemp(temp2);
				putc(']', o);
			}
			putc('=', o);
			printtemp(result);
			fputs(";\n", o);
		}
		break;
	case Cond:
		targettemp=newtemp(expr->exp1->type, 0);
		usetemp(targettemp);
		codegen_expr_value(expr->exp1, targettemp, t, m);
		fputs("if (", o);
		printtemp(targettemp);
		fputs(") {\n", o);
		codegen_expr_value(expr->exp2, result, t, m);
		fputs("} else {\n", o);
		codegen_expr_value(expr->exp3, result, t, m);
		fputs("}\n", o);
		break;
	case Condor:
	case Condand:
		codegen_expr_value(expr->exp1, result, t, m);
		fputs("if (", o);
		if (expr->tag==Condor) putc('!', o);
		printtemp(result);
		fputs(") {\n", o);
		codegen_expr_value(expr->exp2, result, t, m);
		fputs("}\n", o);
		break;

	case Inst:
		assign=packagetree_findtype(expr->name)
		                           ->modifiers & ACC_interface;
		targettemp=newtemp(expr->exp1->type, 0);
		if (assign) {
			fputs("void **scan;\n", o);
		} else {
			fputs("struct Javai_java_lang_Class *scan;\n", o);
		}
		usetemp(targettemp);
		codegen_expr_value(expr->exp1, targettemp, t, m);
		fputs("if (", o);
		printtemp(targettemp);
		fputs(") {\n", o);
		if (assign) {
			fputs("scan=", o);
			printtemp(targettemp);
			fputs("->methods->interfaces;\n"
			      "while (*scan && *scan!=&Javaco_", o);
			printname(expr->name);
			fputs(") scan+=2;\n", o);
			printtemp(result);
			fputs("=*scan ? 1 : 0;\n", o);
		} else {
			fputs("scan=", o);
			printtemp(targettemp);
			fputs("->methods->myclass;\n"
			      "while (scan && scan!=&Javaco_", o);
			printname(expr->name);
			fputs(") scan=scan->f_superclass;\n", o);
			printtemp(result);
			fputs("=scan ? 1 : 0;\n", o);
		}
		fputs("} else ", o);
		printtemp(result);
		fputs("=0;\n", o);
		break;

	case Cast:
		if (equaltypes(expr->exp1->type, expr->type)) {
			codegen_expr_value(expr->exp1, result, t, m);
			break;
		}
		targettemp=newtemp(expr->exp1->type, 0);
		usetemp(targettemp);
		codegen_expr_value(expr->exp1, targettemp, t, m);
		if (expr->exp1->type->prim==Null) {
			printtemp(result);
			if (expr->type->dims==0
			    && packagetree_findtype(expr->type->ref)
			                         ->modifiers & ACC_interface)
			{
				fputs(".object", o);
			}
			putc('=', o);
			printtemp(targettemp);
			fputs(";\n", o);
			break;
		}
		if (expr->exp1->type->dims || expr->type->dims) {
			if (equaltypes(expr->type, &ObjectType)) {
				printtemp(result);
				fputs("=&(", o);
				printtemp(targettemp);
				fputs("->super);\n", o);
				break;
			}
			error("Casts involving arrays not yet implemented");
		}
		/* No arrays are involved */
		if (expr->exp1->type->prim!=Ref && expr->type->prim!=Ref) {
			printtemp(result);
			fputs("=(", o);
			printtypedecl(expr->type, 0);
			putc(')',o);
			printtemp(targettemp);
			fputs(";\n", o);
			break;
		}
		if (expr->exp1->type->prim!=Ref || expr->type->prim!=Ref) {
			error("Bad cast!");
		}
		/* Only reference types are involved */
		if (canwiden(expr->exp1->type->ref, expr->type->ref)
		    || equaltypes(expr->type, &ObjectType))
		{
			struct typedecl *tf;
			struct typedecl *tt;
			tf=packagetree_findtype(expr->exp1->type->ref);
			tt=packagetree_findtype(expr->type->ref);
			fputs("if (", o);
			printtemp(targettemp);
			if (tf->modifiers & ACC_interface) fputs(".object", o);
			fputs(") {\n", o);
			printtemp(result);
			if (tt->modifiers & ACC_interface) {
				fputs(".object=", o);
				if (tf->modifiers & ACC_interface) {
					printtemp(targettemp);
					fputs(".object;\n", o);
				} else {
					fputs("&((*", o);
					printtemp(targettemp);
					putc(')', o);
					fullywiden(expr->exp1->type->ref);
					fputs(");\n", o);
				}
				printtemp(result);
				fputs(".methods=", o);
				printtemp(targettemp);
				if (tf->modifiers & ACC_interface) {
					putc('.', o);
				} else {
					fputs("->", o);
				}
				fputs("methods", o);
				widen(expr->exp1->type->ref, expr->type->ref);
				fputs(";\n", o);
			} else {
				putc('=', o);
				if (! (tf->modifiers & ACC_interface)) {
					fputs("&((*", o);
				}
				printtemp(targettemp);
				if (! (tf->modifiers & ACC_interface)) {
					putc(')',o);
				}
				if (equaltypes(expr->type, &ObjectType)) {
					fullywiden(expr->exp1->type->ref);
				} else {
					widen(expr->exp1->type->ref,
					      expr->type->ref);
				}
				if (! (tf->modifiers & ACC_interface)) {
					putc(')',o);
				}
				fputs(";\n",o);
			}
			fputs("} else {\n", o);
			printtemp(result);
			if (tt->modifiers & ACC_interface) fputs(".object", o);
			fputs("=0;\n"
			      "}\n", o);
			break;
		}
		if (equaltypes(expr->exp1->type, &ObjectType)) {
			struct typedecl *t;
			t=packagetree_findtype(expr->type->ref);
			fputs("if (", o);
			printtemp(targettemp);
			fputs("==0) ", o);
			printtemp(result);
			if (t->modifiers & ACC_interface) {
				fputs(".object", o);
			}
			fputs("=0; else {\n", o);
			if (t->modifiers & ACC_interface) {
				fputs("void **scan;\n"
				      "scan=", o);
				printtemp(targettemp);
				fputs("->methods->interfaces;\n"
				      "while (*scan && *scan!=&Javaco_", o);
				printname(expr->type->ref);
				fputs(") scan+=2;\n"
				      "if (*scan==0) {\n\t"
				      "thrown_object=constClassCastException;\n"
				      "\tlongjmp(*cur_exception_handler, 1);\n"
				      "}\n", o);
				printtemp(result);
				fputs(".object=", o);
				printtemp(targettemp);
				fputs(";\n", o);
				printtemp(result);
				fputs(".methods=scan[1];\n}\n", o);
			} else {
				fputs("struct Javai_java_lang_Class *scan;\n",
				      o);
				fputs("scan=(*", o);
				printtemp(targettemp);
				putc(')',o);
				fullywiden(expr->exp1->type->ref);
				fputs(".methods->myclass;\n"
				      "while (scan && scan!=&Javaco_", o);
				printname(expr->type->ref);
				fputs(") scan=scan->f_superclass;\n", o);
				fputs("if (scan==0) {\n\t"
				      "thrown_object=constClassCastException;\n"
				      "\tlongjmp(*cur_exception_handler, 1);\n"
				      "}\n", o);
				printtemp(result);
				fputs("=(struct Javai_", o);
				printname(expr->type->ref);
				fputs("*) (((char *) ", o);
				printtemp(targettemp);
				fputs(")-(((char *) "
				      "&((*((struct Javai_", o);
				printname(expr->type->ref);
				fputs("*) 0))", o);
				widen(expr->type->ref, expr->exp1->type->ref);
				fputs("))-(char*)0));\n"
				      "}\n", o);
			}
			break;
		}
		error("This kind of cast not yet implemented");

	case Not: case Comp: case Plus: case Minus:
		targettemp=newtemp(expr->exp1->type, 0);
		usetemp(targettemp);
		codegen_expr_value(expr->exp1, targettemp, t, m);
		printtemp(result);
		putc('=', o);
		if (expr->tag==Not) putc('!', o);
		if (expr->tag==Comp) putc('~', o);
		if (expr->tag==Minus) putc('-', o);
		printtemp(targettemp);
		fputs(";\n", o);
		break;
	case Predec: case Preinc: case Postdec: case Postinc:
		targettemp=newtemp(expr->exp1->type, 1);
		usetemp(targettemp);
		codegen_expr_var(expr->exp1, targettemp, t, m);
		if (expr->tag==Preinc) {
			fputs("(*", o);
			printtemp(targettemp);
			fputs(")++;\n", o);
		}
		if (expr->tag==Predec) {
			fputs("(*", o);
			printtemp(targettemp);
			fputs(")--;\n", o);
		}
		printtemp(result);
		fputs("=(*", o);
		printtemp(targettemp);
		fputs(");\n", o);
		if (expr->tag==Postinc) {
			fputs("(*", o);
			printtemp(targettemp);
			fputs(")++;\n", o);
		}
		if (expr->tag==Postdec) {
			fputs("(*", o);
			printtemp(targettemp);
			fputs(")--;\n", o);
		}
		break;
	case Name: thiscanthappen;
	case Prim: thiscanthappen;
	case Intlit:	case Longlit:	case Boollit:	case Charlit:
		printtemp(result);
		fprintf(o, "=0x%08X%08XLL;\n",
		           (unsigned int) (expr->integ / 0x100000000ULL),
		           (unsigned int) (expr->integ % 0x100000000ULL));
		break;
	case Stringlit:
		printtemp(result);
		fprintf(o, "=&s%d;\n", (int) expr->integ);
		break;
	case Floatlit:
	case Doublit:
		printtemp(result);
		fprintf(o, "=%s;\n", expr->str);
		break;
	case Nulllit:
		printtemp(result);
		fputs("=(void *) 0;\n", o);
		break;
	case This:
		printtemp(result);
		fputs("=this;\n", o);
		break;
	case New:
		fputs("int len;\n", o);
		if (expr->type->dims) {
			temp1=newtemp(expr->exp1->type, 0);
			usetemp(temp1);
			codegen_expr_value(expr->exp1, temp1, t, m);
			fputs("if (", o);
			printtemp(temp1);
			fputs("<0) {\n"
			      "\tthrown_object"
			         "=constNegativeArraySizeException;\n"
			      "\tlongjmp(*cur_exception_handler, 1);\n"
			      "}\n", o);
		} else {
			temp1=-1;
		}
		fputs("len=sizeof(", o);
		printtypedecl(expr->type, 1);
		fputs(")", o);
		if (expr->type->dims) {
			expr->type->dims--;
			fputs("+sizeof(", o);
			printtypedecl(expr->type, 0);
			fputs(")*", o);
			printtemp(temp1);
			expr->type->dims++;
		}
		fputs(";\n", o);
		printtemp(result);
		fputs("=(void *)malloc(len);\n", o);
		fputs("if (", o);
		printtemp(result);
		fputs("==0) {\n"
		      "\tthrown_object=constOutOfMemoryError;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n", o);
		fputs("memset(", o);
		printtemp(result);
		fputs(", 0, len);\n", o);
		if (expr->type->dims==0) {
			argcount=countargs(expr->exp2);
			firsttemp=nexttemp;
			nexttemp+=argcount;
			fputs("{\n", o);
			declareargs(expr->exp2, firsttemp);
			useargs(expr->exp2, firsttemp);
			evalargs(expr->exp2, firsttemp, t, m);
			fputs("Javac_", o);
			printname(expr->type->ref);
			fputs("__", o);
			printargspecs(expr->exp2);
			fputs("(stackframe,", o);
			printtemp(result);
			fputs(", &Javami_", o);
			printname(expr->type->ref);
			if (expr->exp2) putc(',', o);
			printtemplist(firsttemp, argcount);
			fputs(");\n", o);
			fputs("}\n", o);
		} else {
			int dd;
			printtemp(result);
			fputs("->f_length=", o);
			printtemp(temp1);
			fputs(";\n", o);
			fputs("Javac_java_lang_Object__(stackframe,&", o);
			printtemp(result);
			fputs("->super, &Javami", o);
			for (dd=expr->type->dims; dd; dd--) putc('a', o);
			printprimname(expr->type->prim, expr->type->ref);
			fputs(");\n", o);
		}
		break;
	case Primfield:
		if (expr->exp1->tag != Name &&
		    !(expr->integ & ACC_static && expr->exp1->tag==This))
		{
			targettemp=newtemp(expr->exp1->type, 0);
			usetemp(targettemp);
			codegen_expr_value(expr->exp1, targettemp, t, m);
		} else {
			/* Gratuitous initialisation for GCC */
			targettemp=-1;
		}
		if (expr->integ & ACC_static) {
			fputs("Javasi_", o);
			printname(expr->exp1->type->ref);
			fputs("(stackframe);\n", o);
			printtemp(result);
			fputs("=Javasf_", o);
			printname(expr->exp1->type->ref);
			fprintf(o, "_%s;\n", expr->str);
		} else {
			fputs("if (", o);
			printtemp(targettemp);
			fputs("==0) {\n"
			      "\tthrown_object=constNullPointerException;\n"
			      "\tlongjmp(*cur_exception_handler, 1);\n"
			      "}\n", o);
			printtemp(result);
			putc('=', o);
			printtemp(targettemp);
			fprintf(o, "->f_%s;\n", expr->str);
		}
		break;
	case Super: thiscanthappen;
	case Meth:
		argcount=countargs(expr->exp2);
		firsttemp=nexttemp;
		nexttemp+=argcount;
		declareargs(expr->exp2, firsttemp);
		if (expr->exp1 && expr->exp1->tag != Name) {
			targettemp=newtemp(expr->exp1->type, 0);
			useargs(expr->exp2, firsttemp);
			usetemp(targettemp);
			codegen_expr_value(expr->exp1, targettemp, t, m);
		} else {
			useargs(expr->exp2, firsttemp);
			targettemp=-1;
		}
		evalargs(expr->exp2, firsttemp, t, m);
		if (!(expr->integ & ACC_static)) {
			if (targettemp==-1) {
				/* No target reference */
				thiscanthappen;
			}
			fputs("if (", o);
			printtemp(targettemp);
			if (expr->exp1->type->dims==0
			    && packagetree_findtype(expr->exp1->type->ref)
			                        ->modifiers & ACC_interface)
			{
				fputs(".object", o);
			}
			fputs("==0) {\n"
			      "\tthrown_object=constNullPointerException;\n"
			      "\tlongjmp(*cur_exception_handler, 1);\n"
			      "}\n", o);
		}
		if (result != -1) {
			printtemp(result);
			putc('=', o);
		}
		if (expr->integ & (ACC_static|ACC_final|ACC_private)) {
			fputs("Java_", o);
			printname(expr->exp1->type->ref);
			fprintf(o, "_%s__", expr->str);
			printargspecs(expr->exp2);
		} else {
			fputs("(*", o);
			printtemp(targettemp);
			if (expr->exp1->type->dims==0
			    && packagetree_findtype(expr->exp1->type->ref)
			                        ->modifiers & ACC_interface)
			{
				putc('.', o);
			} else {
				fputs("->", o);
			}
			fprintf(o, "methods->m_%s__", expr->str);
			printargspecs(expr->exp2);
			putc(')', o);
		}
		fputs("(stackframe", o);
		if (!(expr->integ & ACC_static)) {
			if (targettemp==-1) {
				/* No target reference */
				thiscanthappen;
			}
			putc(',', o);
			if (expr->exp1->type->dims==0
			    && packagetree_findtype(expr->exp1->type->ref)
			                        ->modifiers & ACC_interface)
			{
				printtemp(targettemp);
				fputs(".object", o);
			} else {
				fputs("&((*", o);
				printtemp(targettemp);
				fputs(")", o);
				fullywiden(expr->exp1->type->ref);
				putc(')', o);
			}
		}
		if (expr->exp2) putc(',', o);
		printtemplist(firsttemp, argcount);
		fputs(");\n", o);
		break;
	case Array:
		temp1=newtemp(expr->exp1->type, 0);
		temp2=newtemp(expr->exp2->type, 0);
		usetemp(temp1);
		usetemp(temp2);
		codegen_expr_value(expr->exp1, temp1, t, m);
		codegen_expr_value(expr->exp2, temp2, t, m);
		fputs("if (", o);
		printtemp(temp1);
		fputs("==0) {\n"
		      "\tthrown_object=constNullPointerException;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n"
		      "if (", o);
		printtemp(temp2);
		fputs("<0 || ", o);
		printtemp(temp2);
		fputs(">=", o);
		printtemp(temp1);
		fputs("->f_length) {\n"
		      "\tthrown_object=constIndexOutOfBoundsException;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n", o);
		printtemp(result);
		putc('=', o);
		printtemp(temp1);
		fputs("->component[", o);
		printtemp(temp2);
		fputs("];\n", o);
		break;
	case Localacc:
		printtemp(result);
		fprintf(o, "=l_%s;\n", expr->str);
		break;
	case NullString:
		codegen_expr_value(expr->exp1, result, t, m);
		fputs("if (", o);
		printtemp(result);
		fputs("==0) ", o);
		printtemp(result);
		fprintf(o, "=&s%d", nullstring);
		if (equaltypes(expr->type, &ObjectType)) {
			fputs(".super", o);
		}
		fputs(";\n", o);
		break;
	case Arrayinit:
		fputs("int len=sizeof(", o);
		printtypedecl(expr->type, 1);
		fputs(")+sizeof(", o);
		expr->type->dims--;
		printtypedecl(expr->type, 0);
		fprintf(o, ")*%d;\n", (int) expr->integ);
		if (expr->integ) {
			temp1=newtemp(expr->type, 0);
		} else {
			/* Gratuitous initialisation for GCC */
			temp1=-1;
		}
		expr->type->dims++;
		printtemp(result);
		fputs("=(void *)malloc(len);\n"
		      "if (", o);
		printtemp(result);
		fputs("==0) {\n"
		      "\tthrown_object=constOutOfMemoryError;\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "}\n"
		      "memset(", o);
		printtemp(result);
		fputs(", 0, len);\n", o);
		printtemp(result);
		fprintf(o, "->f_length=%d;\n"
		           "Javac_java_lang_Object__(stackframe,&",
		        (int) expr->integ);
		printtemp(result);
		fputs("->super, &Javami", o);
		{ int dd; for (dd=expr->type->dims; dd; dd--) putc('a', o); }
		printprimname(expr->type->prim, expr->type->ref);
		fputs(");\n", o);
		assign_arrayinit(expr->exp1, result, temp1, t, m);
		break;
	}
	fputs("}\n", o);
}

static void
codegen_block(struct stmt *, struct typedecl *, struct memberdecl *);

static void
codegen_catchclause(struct catchclause *catch,
                    struct typedecl *t, struct memberdecl *m)
{
	if (catch==0) return;
	if (catch->rest==0) {
		fputs("\tstruct Javai_java_lang_Class *scan;\n", o);
	} else {
		codegen_catchclause(catch->rest, t, m);
	}

	fputs("scan=thrown_object->methods->myclass;\n"
	      "while (scan && scan!=&Javaco_", o);
	printname(catch->param->type->ref);
	fputs(") scan=scan->f_superclass;\n", o);
	fputs("if (scan) {\n"
	      "\tstate=0;\n"
	      "\tif (setjmp(handler)==0) {\n"
	      "\t\tdo {\n\t\t\t", o);
	printtypedecl(catch->param->type, 0);
	fprintf(o, "l_%s;\n"
	           "\t\t\tl_%s=(struct Javai_", catch->param->name,
	                                        catch->param->name);
	printname(catch->param->type->ref);
	fputs("*) (((char *) thrown_object)-(((char *) &((*((struct Javai_", o);
	printname(catch->param->type->ref);
	fputs("*) 0))", o);
	fullywiden(catch->param->type->ref);
	fputs("))-(char*)0));\n{\n", o);
	codegen_block(catch->block, t, m);
	fputs("}\n\t\t\tstate=1; break;\n"
	      "\t\t} while (state=2, 0);\n"
	      "\t\tif (state==0) state=3;\n"
	      "\t\tif (retflag) { retflag=0; state=4; }\n"
	      "\t\tbreak;\n"
	      "\t}\n"
	      "}\n", o);
}

static void
declare_localvars(struct vardecl *d, struct type *t)
{
	if (d==0) return;
	if (d->rest) declare_localvars(d->rest, t);

	printtypedecl(t, 0);
	fprintf(o, "l_%s;\n", d->name);
}

static void
assign_localvars(struct vardecl *decl, struct type *type,
                 struct typedecl *t, struct memberdecl *m)
{
	if (decl==0) return;
	if (decl->rest) assign_localvars(decl->rest, type, t, m);

	if (decl->init) {
		int temp1;
		fputs("{\n", o);
		temp1=newtemp(type, 0);
		usetemp(temp1);
		codegen_expr_value(decl->init, temp1, t, m);
		fprintf(o, "l_%s=", decl->name);
		printtemp(temp1);
		fputs(";\n}\n", o);
	}
}

static void
declare_block_localvars(struct stmt *stmt,
                        struct typedecl *t, struct memberdecl *m)
{
	while (stmt) {
		if (stmt->tag==Localvar) {
			declare_localvars(stmt->var->decls, stmt->var->type);
		}
		stmt=stmt->rest;
	}
}

static void
codegen_stmt(struct stmt *stmt, struct typedecl *t, struct memberdecl *m)
{
	int temp1;
	struct type *rettype;
	struct stmt *s;

	if (stmt==0) return;
	switch (stmt->tag) {
	case Localvar:
		assign_localvars(stmt->var->decls, stmt->var->type, t, m);
		break;
	case If:
		fputs("{\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("if(", o);
		printtemp(temp1);
		fputs(") {\n", o);
		codegen_block(stmt->stmt1, t, m);
		fputs("} else {\n", o);
		codegen_block(stmt->stmt2, t, m);
		fputs("}\n}\n", o);
		break;
	case While:
		fputs("while(1) {\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("if(!", o);
		printtemp(temp1);
		fputs(") break;\n{\n", o);
		codegen_block(stmt->stmt1, t, m);
		fputs("}\n}\nif (retflag) break;", o);
		break;
	case For:
		fputs("{\n"
		      "int forflag=0;\n", o);
		if (stmt->stmt1) {
			thiscanthappen;
			if (stmt->stmt1->tag==Exprstmt) {
				codegen_block(stmt->stmt1, t, m);
			} else /* => stmt->stmt1->tag==Localvar */ {
				struct fielddecl *f=stmt->stmt1->var;
				declare_localvars(f->decls, f->type);
				assign_localvars(f->decls, f->type, t, m);
			}
		}
		fputs("while(1) {\n", o);
		if (stmt->expr) {
			temp1=newtemp(stmt->expr->type, 0);
			usetemp(temp1);
		} else {
			/* Gratuitous initialisation for GCC */
			temp1=-1;
		}
		fputs("if (forflag) {\n", o);
		codegen_block(stmt->stmt2, t, m);
		fputs("}\n"
		      "forflag=1;\n", o);
		if (stmt->expr) {
			codegen_expr_value(stmt->expr, temp1, t, m);
			fputs("if (!", o);
			printtemp(temp1);
			fputs(") break;\n", o);
		}
		fputs("{\n", o);
		codegen_block(stmt->stmt3, t, m);
		fputs("}\n}\nif (retflag) break;\n}\n", o);
		break;
	case Block:
		fputs("{\n", o);
		codegen_block(stmt->stmt1, t, m);
		fputs("}\n", o);
		break;
	case Empty:
		break;
	case Exprstmt:
		fputs("{\n", o);
		if (stmt->expr->type->prim!=Void) {
			temp1=newtemp(stmt->expr->type, 0);
			usetemp(temp1);
		} else
			temp1=-1;
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("}\n", o);
		break;
	case Switch:
		fputs("{\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		fputs("int switcheroo;", o);
		fputs("int continued=0;\n", o);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("switcheroo=", o);
		printtemp(temp1);
		fputs(";\ndo {\nint switched_on=0;\n", o);
		s=stmt->stmt1;
		declare_block_localvars(s, t, m);
		fputs("if (switched_on) {\n", o);
		while (s) {
			codegen_stmt(s, t, m);
			s=s->rest;
		}
		fputs("}\n"
		      "break;\n"
		      "} while (continued=1, 0);\n"
		      "if (retflag) break;\n"
		      "if (continued) continue;\n", o);
		fputs("}\n", o);
		break;
	case Case:
		fputs("}\n{\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("if (switcheroo==", o);
		printtemp(temp1);
		fputs(") switched_on=1;\n}\nif (switched_on) {\n", o);
		break;
	case Default:
		fputs("}\nswitched_on=1;\nif (switched_on) {\n", o);
		break;
	case Do:
		fputs("{\nint doflag=0;\n"
		      "while(1) {\n"
		      "if (doflag) {\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("if (!", o);
		printtemp(temp1);
		fputs(") break;\n}\ndoflag=1;\n{\n", o);
		codegen_block(stmt->stmt1, t, m);
		fputs("}\n}\nif (retflag) break;\n}\n", o);
		break;
	case Break:
		fputs("break;\n", o);
		if (stmt->rest
		    && stmt->rest->tag != Case && stmt->rest->tag != Default)
		{
			error("Unreachable statement");
		}
		break;
	case Cont:
		fputs("continue;\n", o);
		if (stmt->rest
		    && stmt->rest->tag != Case && stmt->rest->tag != Default)
		{
			error("Unreachable statement");
		}
		break;
	case Return:
		fputs("{\n", o);
		if (stmt->expr) {
			temp1=newtemp(stmt->expr->type, 0);
			usetemp(temp1);
			codegen_expr_value(stmt->expr, temp1, t, m);
			fputs("retval=", o);
			printtemp(temp1);
			fputs(";\n", o);
		}
		fputs("retflag=1;\n"
		      "break;\n}\n", o);
		if (stmt->rest
		    && stmt->rest->tag != Case && stmt->rest->tag != Default)
		{
			error("Unreachable statement");
		}
		break;
	case Throw:
		fputs("{\n", o);
		temp1=newtemp(stmt->expr->type, 0);
		usetemp(temp1);
		codegen_expr_value(stmt->expr, temp1, t, m);
		fputs("thrown_object=&((*", o);
		printtemp(temp1);
		fputs(")", o);
		fullywiden(stmt->expr->type->ref);
		fputs(");\n"
		      "longjmp(*cur_exception_handler, 1);\n}\n", o);
		if (stmt->rest
		    && stmt->rest->tag != Case && stmt->rest->tag != Default)
		{
			error("Unreachable statement");
		}
		break;
	case Try:
		/* state: 0=exception, 1=normal, 2=continue, 3=break,
		          4=return. */
		rettype=m->content.meth->type;
		if (rettype->prim==Void) rettype=0;
		fputs("{\njmp_buf handler, *volatile oldhandler;\n"
		      "struct Javai_java_lang_Object *oldthrown=0;\n"
		      "volatile int oldstate, state=0;\n", o);
		if (rettype) {
			printtypedecl(rettype, 0);
			fputs("volatile oldretval;\n", o);
		}
		fputs("oldhandler=cur_exception_handler;\n"
		      "if (setjmp(handler)==0) {\n"
		      "\tcur_exception_handler=&handler;\n"
		      "\tdo {\n", o);
		codegen_block(stmt->stmt1, t, m);
		fputs("\t\tstate=1; break;\n"
		      "\t} while (state=2, 0);\n"
		      "\tif (state==0) state=3;\n"
		      "\tif (retflag) { retflag=0; state=4; }\n"
		      "}\n"
		      /* After that bit of convoluted jiggery-pokery,
		         state is 0, 1, 2, 3 or 4 as required.  */
		      "if (state==0) {\n", o);
		fputs("do {\n", o);
		codegen_catchclause(stmt->catch, t, m);
		fputs("} while (0);\n"
		      "}\n"
		      "oldstate=state;\n"
		      "oldthrown=thrown_object;\n", o);
		if (rettype) fputs("oldretval=retval;\n", o);
		fputs("state=0;\n"
		      "if (setjmp(handler)==0) {\n"
		      "\tdo {\n", o);
		codegen_block(stmt->stmt2, t, m);
		fputs("\t\tstate=1; break;\n"
		      "\t} while(state=2, 0);\n"
		      "\tif (state==0) state=3;\n"
		      "\tif (retflag) { retflag=0; state=4; }\n"
		      "}\n"
		      "if (state==1) {\n"
		      "\tstate=oldstate;\n"
		      "\tthrown_object=oldthrown;\n", o);
		if (rettype) fputs("\tretval=oldretval;\n", o);
		fputs("}\n"
		      "cur_exception_handler=oldhandler;\n"
		      "if (state==0) {\n"
		      "\tlongjmp(*cur_exception_handler, 1);\n"
		      "} else if (state==2) {\n"
		      "\tcontinue;\n"
		      "} else if (state==3) {\n"
		      "\tbreak;\n"
		      "} else if (state==4) {\n"
		      "\tretflag=1; break;\n"
		      "}\n}\n", o);
		break;
	case NativeStmt: thiscanthappen;
	}
}

static void
codegen_block(struct stmt *stmt, struct typedecl *t, struct memberdecl *m)
{
	declare_block_localvars(stmt, t, m);

	while (stmt) {
		codegen_stmt(stmt, t, m);
		stmt=stmt->rest;
	}
}


static void
staticfieldinit(struct type *type, struct vardecl *d,
                struct typedecl *t, struct memberdecl *m)
{
	int temp1;

	if (d==0) return;
	staticfieldinit(type, d->rest, t, m);

	if (d->init==0) return;

	fputs("\t\t/* from class variable initialiser */\n"
	      "{\n", o);
	temp1=newtemp(type, 0);
	usetemp(temp1);
	codegen_expr_value(d->init, temp1, t, m);
	fputs("\t\tJavasf_", o);
	printname(t->name);
	fprintf(o, "_%s=", d->name);
	printtemp(temp1);
	fputs(";\n"
	      "}\n"
	      "\t\t/* end class variable initialiser */\n", o);
}

static void
staticinit(struct typedecl *t, struct memberdecl *m)
{
	if (m==0) return;
	staticinit(t, m->rest);

	if (m->tag==Field && m->content.field->modifiers & ACC_static) {
		staticfieldinit(m->content.field->type,
		                m->content.field->decls, t, m);
	} else if (m->tag==StaticInit) {
		fputs("\t\tdo { /* from static initialiser */\n", o);
		codegen_block(m->content.init, t, m);
		fputs("\t\t} while (0); /* end static initialiser */\n", o);
	}
}

static void
printnativebody(struct expr *e)
{
	const char *s;
	int i;
	if (e==0) return;
	if (e->rest) printnativebody(e->rest);
	fputs(e->str, o);
	s=e->str;
	i=((unsigned int) ((unsigned char) s[0]))*256
	  + (unsigned int) ((unsigned char) s[1]);
	s+=2;
	while (i-->0) {
		putc(s[1], o);
		s+=2;
	}
	putc('\n', o);
}

static void
printfieldinits(struct type *type, struct vardecl *decl,
                struct typedecl *t, struct memberdecl *m)
{
	int temp;

	if (decl==0) return;
	printfieldinits(type, decl->rest, t, m);
	if (decl->init==0) return;

	fputs("{\n", o);
	temp=newtemp(type, 0);
	usetemp(temp);
	codegen_expr_value(decl->init, temp, t, m);
	fputs("this->f_", o);
	fputs(decl->name, o);
	putc('=', o);
	printtemp(temp);
	fputs(";\n}\n", o);
}

static void
printinstvarinits(struct memberdecl *curm, struct typedecl *t,
                  struct memberdecl *m)
{
	if (curm==0) return;
	printinstvarinits(curm->rest, t, m);
	if (curm->tag==Field && !(curm->content.field->modifiers & ACC_static)){
		printfieldinits(curm->content.field->type,
		                curm->content.field->decls, t, m);
	} else if (curm->tag==InstanceInit) {
		fputs("\t\tdo { /* from instance initialiser */\n", o);
		codegen_block(curm->content.init, t, m);
		fputs("\t\t} while (0); /* end instance initialiser */\n", o);
	}
}

/* Generate definitions for constructors, methods and static fields */
static void
definemethods(struct typedecl *t)
{
	struct memberdecl *m;
	int seenconstructor=0;

	if (t->modifiers & ACC_interface) {
		seenconstructor=1;
	}

	fputs("/****** ", o);
	printname(t->name);
	fputs(" ******/\n\n",o);

	for (m=t->body; m; m=m->rest) {
		if (m->tag==Method
		    && !(m->content.meth->modifiers & ACC_abstract)
		    /* && !(m->content.meth->modifiers & ACC_native) */ )
		{
			struct methoddecl *meth=m->content.meth;
			printtypedecl(meth->type, 0);
			fputs("Java_", o);
			printname(t->name);
			fprintf(o, "_%s__", meth->sig->name);
			printparamlist(meth->sig->params, Spec, 0);
			fputs("(struct JavaStackFrame *upstf", o);
			if (!(meth->modifiers & ACC_static)) {
				fprintf(o, ",struct Javai_java_lang_Object *%s",
				        t->super ? "tobj" : "this");
				printparamlist(meth->sig->params, Defn, 1);
				fputs(")\n{\n", o);
				if (t->super) {
					fputs("\tstruct Javai_", o);
					printname(t->name);
					fputs(" *this=(struct Javai_", o);
					printname(t->name);
					fputs("*) (((char *) tobj)-(((char"
					      " *) &((*((struct Javai_", o);
					printname(t->name);
					fputs("*) 0))", o);
					fullywiden(t->name);
					fputs("))-(char*)0));\n"
					      "\t(void) this;\n", o);
				}
			} else {
				printparamlist(meth->sig->params, Defn, 1);
				fputs(")\n{\n", o);
			}
			fputs("{\n"
			      "struct JavaStackFrame stf, "
			              "*const stackframe=&stf;\n"
			      "int retflag=0;\n", o);
			if (meth->type->prim!=Void) {
				printtypedecl(meth->type, 0);
				fputs("retval;\n"
				      "(void) &retval;\n", o);
			}
			fprintf(o, "stf.class=&s%d;\n"
			           "stf.method=&s%d;\n"
			           "stf.previous=upstf;\n"
			           "(void) &retflag;\n"
			           "(void) stackframe;\n"
			           "Javasi_", t->name->data,
			                stringpool_intern(meth->sig->name));
			printname(t->name);
			fputs("(stackframe);\n", o);
			fputs("do {\n", o);
			if (meth->modifiers & ACC_native) {
				if (meth->body && meth->body->tag==NativeStmt) {
					printnativebody(meth->body->expr);
				}
			} else {
				codegen_block(meth->body, t, m);
			}
			fputs("} while (0);\n", o);
			if (meth->type->prim!=Void) {
				fputs("return retval;\n", o);
			} else {
				fputs("return;\n", o);
			}
			fputs("}\n}\n\n", o);
		} else if (m->tag==Constructor) {
			struct constrdecl *con=m->content.constr;
			int super=!(con->cons && con->cons->super==0);
			seenconstructor=1;
			fputs("void Javac_", o);
			printname(t->name);
			fputs("__", o);
			printparamlist(con->sig->params, Spec, 0);
			fputs("(struct JavaStackFrame *upstf,struct Javai_", o);
			printname(t->name);
			fputs(" *this, struct Javam_", o);
			printname(t->name);
			fputs(" *meths", o);
			printparamlist(con->sig->params, Defn, 1);
			fprintf(o, ")\n{\n"
			           "struct JavaStackFrame stf, "
			                   "*const stackframe=&stf;\n"
			           "int retflag=0;\n"
			           "stf.class=&s%d;\n"
			           "stf.method=&s%d;\n"
			           "stf.previous=upstf;\n"
			           "(void) &retflag;\n"
			           "(void) stackframe;\n"
			           "Javasi_", t->name->data, initstring);
			printname(t->name);
			fputs("(stackframe);\n", o);
			if (!(t->modifiers & ACC_native) && super)
			{
				fputs("\tthis->methods=meths;\n", o);
			}
			if (t->super || con->cons) {
				struct expr *args=con->cons
				                  ? con->cons->arglist : 0;
				int argcount=countargs(args);
				int firsttemp=nexttemp;
				nexttemp+=argcount;
				fputs("{\n", o);
				declareargs(args, firsttemp);
				useargs(args, firsttemp);
				evalargs(args, firsttemp, t, m);
				fputs("Javac_", o);
				printname(super ? t->super : t->name);
				fputs("__", o);
				printargspecs(args);
				fputs("(stackframe, ", o);
				if (super) {
					fputs("&this->super,&meths->super", o);
				} else {
					fputs("this,meths", o);
				}
				if (argcount) putc(',', o);
				printtemplist(firsttemp, argcount);
				fputs(");\n}\n", o);
			}
			if (super) {
				printinstvarinits(t->body, t, m);
			}
			fputs("do {\n", o);
			codegen_block(con->body, t, m);
			fputs("} while (0);\n", o);
			fputs("}\n\n", o);
		}
	}

	/* Default constructor */
	if (!seenconstructor) {
		fputs("void Javac_", o);
		printname(t->name);
		fputs("__(struct JavaStackFrame *upstf, struct Javai_", o);
		printname(t->name);
		fputs(" *this, struct Javam_", o);
		printname(t->name);
		fprintf(o, " *meths)\n{\n"
		           "struct JavaStackFrame stf, "
		                   "*const stackframe=&stf;\n"
		           "int retflag=0;\n"
		           "stf.class=&s%d;\n"
		           "stf.method=&s%d;\n"
		           "stf.previous=upstf;\n"
		           "(void) &retflag;\n"
		           "(void) stackframe;\n"
		           "Javasi_", t->name->data, initstring);
		printname(t->name);
		fputs("(stackframe);\n", o);
		if (!(t->modifiers & ACC_native)) {
			fputs("\tthis->methods=meths;\n", o);
		}
		if (t->super) {
			fputs("\tJavac_", o);
			printname(t->super);
			fputs("__(stackframe, &this->super, &meths->super);\n",
			      o);
		}
		printinstvarinits(t->body, t, m);
		fputs("}\n\n", o);
	}

	/* Class static initialiser */
	fputs("void Javasi_", o);
	printname(t->name);
	fputs("(struct JavaStackFrame *upstf)\n{\n"
	      "\tJavaint *const initState=&Javaco_", o);
	printname(t->name);
	fprintf(o, ".f_initState;\n"
	           "\tjmp_buf handler, *volatile oldhandler;\n"
	           "\tstruct JavaStackFrame stf, "
	                     "*const stackframe=&stf;\n"
	           "int retflag=0;\n"
	           "(void) &retflag;\n"
	           "(void) stackframe;\n"
	           "\tstf.class=&s%d;\n"
	           "\tstf.method=&s%d;\n"
	           "\tstf.previous=upstf;\n"
	           "\tif (*initState==0) return;\n"
	           "\tif (*initState<0) {\n"
	           "\t\tthrown_object=constNoClassDefFoundError;\n"
	           "\t\tfputs(\"Java Error: Erroneous class state\\n\", "
	                                                         "stderr);\n"
	           "\t\tlongjmp(*cur_exception_handler, 1);\n"
	           "\t}\n"
	           "\t*initState=0;\n", t->name->data, clinitstring);
	if (t->super) {
		fputs("\toldhandler=cur_exception_handler;\n"
		      "\tif (setjmp(handler)==0) {\n"
		      "\t\tcur_exception_handler=&handler;\n"
		      "\t\tJavasi_", o);
		printname(t->super);
		fputs("(stackframe);\n"
		      "\t\tcur_exception_handler=oldhandler;\n"
		      "\t} else {\n"
		      "\t\tcur_exception_handler=oldhandler;\n"
		      "\t\t*initState=-1;\n"
		      "\t\tlongjmp(*cur_exception_handler, 1);\n"
		      "\t}\n", o);
	}
	fputs("\toldhandler=cur_exception_handler;\n"
	      "\tif (setjmp(handler)==0) {\n"
	      "\t\tcur_exception_handler=&handler;\n", o);
	staticinit(t, t->body);
	fputs("\t\tcur_exception_handler=oldhandler;\n"
	      "\t} else {\n"
	      "\t\tcur_exception_handler=oldhandler;\n"
	      /* FIXME: If thrown object isn't an Error, replace it with
	         an ExceptionInInitializerError (or OutOfMemoryError) */
	      "\t\t*initState=-1;\n"
	      "\t\tlongjmp(*cur_exception_handler, 1);\n"
	      "\t}\n"
	      "}\n\n", o);
}


static void
printstrings(void)
{
	int highest, scan;

	highest=stringpool_highestindex();
	for (scan=0; scan<=highest; scan++) {
		const char *str=stringpool_fetch(scan);
		int len=((int) (unsigned char) str[0])*256
		        + (int) (unsigned char) str[1];

		fprintf(o, "struct Javai_java_lang_String s%d;\n", scan);
		fprintf(o, "struct Javaiachar sc%d={{&Javamiachar},"
		           " %d, {", scan, len);

		str+=2;
		while (len--) {
			fprintf(o, "0x%02X%02X,",
			           (int) (unsigned char) str[0],
			           (int) (unsigned char) str[1]);
			str+=2;
		}

		fputs("}};\n", o);
	}
}

static void
initstrings(void)
{
	int highest, scan;

	highest=stringpool_highestindex();

	for (scan=0; scan<=highest; scan++) {
		fprintf(o, "Javac_java_lang_String___3C"
		           "(0, &s%d, &Javami_java_lang_String, &sc%d);\n",
		           scan, scan);
	}
}

static void
internstrings(void)
{
	int highest, scan;

	highest=stringpool_highestindex();

	for (scan=0; scan<=highest; scan++) {
		fprintf(o, "Java_java_lang_String_intern__(0, &(s%d.super));\n",
		           scan);
	}
}

static int
countnames(struct name *n)
{
	if (n==0) {
		return 0;
	} else {
		return 1+countnames(n->rest);
	}
}

static int
insertinterfaces(struct name *n, struct name *iface)
{
	int next;

	if (iface==0) return 0;
	next=insertinterfaces(n, iface->rest);

	fputs("Java_java_lang_Class_initInterface_"
	      "_I"
	      "_Ljava_lang_Class_2"
	      "(0, "
	      "&Javaco_", o);
	printname(n);
	fprintf(o, ".super, %d, &Javaco_", next);
	printname(iface);
	fputs(");", o);

	return next+1;
}

static void
initialiseclass(struct typedecl *t)
{
	int d;

	fputs("Javac_java_lang_Class_"
	      "_I"
	      "_Ljava_lang_Class_2"
	      "_Ljava_lang_String_2"
	      "_I"
	      "(0, "
	      "&Javaco_", o);
	printname(t->name);
	fprintf(o, ", &Javami_java_lang_Class, "
	           "%d, ", t->modifiers & ACC_interface ? 1 : 0);
	if (t->super) {
		fputs("&Javaco_", o);
		printname(t->super);
	} else {
		putc('0', o);
	}
	fprintf(o, ", &s%d, %d);\n", t->name->data, countnames(t->implements));
	insertinterfaces(t->name, t->implements);

	for (d=1; d<=t->arrdepth; d++) {
		int dd;
		fputs("Javac_java_lang_Class_"
		      "_I"
		      "_Ljava_lang_Class_2"
		      "_Ljava_lang_String_2"
		      "_I"
		      "(0, "
		      "&Javaco", o);
		for (dd=d; dd; dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(", &Javami_java_lang_Class, 2, ", o);
		fputs("&Javaco", o);
		for (dd=d-1; dd; dd--) putc('a', o);
		putc('_', o);
		printname(t->name);
		fputs(", 0, 0);\n", o);
	}
}

static void
internprimarraynames(void)
{
	if (getprimtypedepth(Byte))	stringpool_intern("[B");
	if (getprimtypedepth(Char))	stringpool_intern("[C");
	if (getprimtypedepth(Double))	stringpool_intern("[D");
	if (getprimtypedepth(Float))	stringpool_intern("[F");
	if (getprimtypedepth(Int))	stringpool_intern("[I");
	if (getprimtypedepth(Long))	stringpool_intern("[J");
	if (getprimtypedepth(Short))	stringpool_intern("[S");
	if (getprimtypedepth(Boolean))	stringpool_intern("[Z");
}

static void
initialiseprimarrayclass(int maxdepth, const char *name, char spec)
{
	int d;
	char n[3]="[x";

	n[1]=spec;

	for (d=1; d<=maxdepth; d++) {
		int dd;
		fputs("Javac_java_lang_Class_"
		      "_I"
		      "_Ljava_lang_Class_2"
		      "_Ljava_lang_String_2"
		      "_I(0, &Javaco", o);
		for (dd=d; dd; dd--) putc('a', o);
		fputs(name, o);
		fputs(", &Javami_java_lang_Class, 2, ", o);
		if (d>1) {
			fputs("&Javaco", o);
			for (dd=d-1; dd; dd--) putc('a', o);
			fputs(name, o);
			fputs(", 0, 0);\n", o);
		} else {
			fprintf(o, "0, &s%d, 0);\n", stringpool_intern(n));
		}
	}
}

static void
initialiseprimarrayclasses(void)
{
	initialiseprimarrayclass(getprimtypedepth(Boolean), "bool", 'Z');
	initialiseprimarrayclass(getprimtypedepth(Byte), "byte", 'B');
	initialiseprimarrayclass(getprimtypedepth(Short), "short", 'S');
	initialiseprimarrayclass(getprimtypedepth(Int), "int", 'I');
	initialiseprimarrayclass(getprimtypedepth(Long), "long", 'J');
	initialiseprimarrayclass(getprimtypedepth(Char), "char", 'C');
	initialiseprimarrayclass(getprimtypedepth(Float), "float", 'F');
	initialiseprimarrayclass(getprimtypedepth(Double), "double", 'D');
}

void
codegen(FILE *outfile, char *startclass)
{
	char *scan;
	struct name *nscan;
	static struct name njava={0,"java",0,0};
	static struct name nlang={&njava,"lang",0,0};
	static struct name cce={&nlang,"ClassCastException",0,0};
	static struct name ae={&nlang,"ArithmeticException",0,&cce};
	static struct name nase={&nlang,"NegativeArraySizeException",0,&ae};
	static struct name ioobe={&nlang,"IndexOutOfBoundsException",0,&nase};
	static struct name nnpe={&nlang,"NullPointerException",0,&ioobe};
	static struct name nncdfe={&nlang,"NoClassDefFoundError",0,&nnpe};
	static struct name constobs={&nlang,"OutOfMemoryError",0,&nncdfe};

	for (scan=startclass; *scan; scan++) {
		if (*scan=='.') {
			*scan='_';
		}
	}

	nullstring=stringpool_intern("null");
	initstring=stringpool_intern("<init>");
	clinitstring=stringpool_intern("<clinit>");
	internprimarraynames();

	o=outfile;

	fputs("#include <stdlib.h>\n"
	      "#include <stdio.h>\n"
	      "#include <setjmp.h>\n"
	      "#include <unistd.h>\n"
	      "#include <string.h>\n"
	      "#include <signal.h>\n"
	      "#include <fcntl.h>\n"
	      "#include <errno.h>\n"
	      "#include <sys/time.h>\n"
	      "#include <sys/wait.h>\n"
	      "#include <sys/types.h>\n"
	      "#include <sys/stat.h>\n"
#ifdef HAVE_LIBGC
	      "\n"
	      "#ifndef BOCK_NO_GC\n"
	      "extern void *GC_malloc(size_t);\n"
	      "#define malloc(n) GC_malloc(n);\n"
	      "#endif /* BOCK_NO_GC */\n"
#endif
	      "\n"
	      "typedef unsigned char Javabool;\n"
	      "typedef signed char Javabyte;\n"
	      "typedef unsigned short int Javachar;\n"
	      "typedef signed short int Javashort;\n"
	      "typedef signed int Javaint;\n"
	      "typedef signed long long int Javalong;\n"
	      "typedef float Javafloat;\n"
	      "typedef double Javadouble;\n"
	      "\n"
	      "struct JavaStackFrame {\n"
	      "\tstruct Javai_java_lang_String *class;\n"
	      "\tstruct Javai_java_lang_String *method;\n"
	      "\tstruct JavaStackFrame *previous;\n"
	      "};\n"
	      "\n/* Structure tags we'll be using later ************************************/\n\n", o);
	printprimstubs();
	packagetree_foralltypessuperfirst(printstructstub);
	fputs("\njmp_buf *cur_exception_handler;\n"
	      "struct Javai_java_lang_Object *thrown_object;\n\n", o);
	for (nscan=&constobs; nscan; nscan=nscan->rest) {
		fprintf(o, "struct Javai_java_lang_Object *const%s;\n",
		        nscan->id);
	}
	fputs("\n/* Class instance structures **********************************************/\n\n", o);
	packagetree_foralltypessuperfirst(declareinstancestruct);
	declareprimarrayinstancestructs();
	fputs("/* Class jump structures **************************************************/\n\n", o);
	packagetree_foralltypessuperfirst(declarejumpstruct);
	fputs("/* Constructors, methods and static class members *************************/\n\n", o);
	packagetree_foralltypessuperfirst(declarestatics);
	fputs("/* Class objects **********************************************************/\n\n", o);
	declareprimarrayclassobjects();
	packagetree_foralltypessuperfirst(declareclassobjects);
	fputs("\n/* Jump tables ************************************************************/\n\n", o);
	packagetree_foralltypessuperfirst(printjumptable);
	printprimarrayjumptables();
	fputs("/* String constants *******************************************************/\n\n", o);
	printstrings();
	fputs("\n/* Constructors and methods ***********************************************/\n\n", o);
	packagetree_foralltypessuperfirst(definemethods);

#if 0
	fputs("/* Fetch native methods */\n#include \"native.c\"\n\n", o);
	fputs("/* Standard native and utility methods ************************************/\n\n"
	      "Javaint Java_java_lang_Object_hashCode__(struct Javai_java_lang_Object *self)\n"
	      "{\n"
	      "\treturn (Javaint) self;\n"
	      "}\n"
	      "\n"
	      "struct Javai_java_lang_Class *Java_java_lang_Object_getClass__(struct Javai_java_lang_Object *self)\n"
	      "{\n"
	      "\treturn self->methods->myclass;\n"
	      "}\n"
	      "\n", o);
#endif

	fputs("/* Program entry point ****************************************************/\n\n"
	      "int main(int ac,char *av[])\n{\n"
	      "\tjmp_buf handler;\n"
	      "\tint i;\n"
	      "\tstruct Javaia_java_lang_String *args;\n", o);
	for (nscan=&constobs; nscan; nscan=nscan->rest) {
		fputs("\tstruct Javai_", o);
		printname(nscan);
		fprintf(o, " c%s;\n", nscan->id);
	}
	fputs("\tif (setjmp(handler)==0) {\n"
	      "\t\tcur_exception_handler=&handler;\n", o);
	packagetree_foralltypessuperfirst(initialiseclass);
	initialiseprimarrayclasses();
	initstrings();
	for (nscan=&constobs; nscan; nscan=nscan->rest) {
		struct typedecl *t=packagetree_findtype(nscan);
		fprintf(o, "\t\tconst%s=&(c%s", nscan->id, nscan->id);
		fullywiden(t->name);
		/* while (t->super) {
			fputs(".super", o);
			t=packagetree_findtype(t->super);
		} */
		fputs(");\n"
		      "\t\tJavac_", o);
		printname(nscan);
		fprintf(o, "__(0, &c%s, &Javami_", nscan->id);
		printname(nscan);
		fputs(");\n", o);
	}
	/* Bad things can happen if code which can throw exceptions is
	   placed before this point in main.  */
	internstrings();
	fputs("\t\targs=malloc(sizeof(struct Javaia_java_lang_String)\n"
	      "\t\t            +sizeof(struct Javai_java_lang_String *)*(ac-1));\n"
	      "\t\targs->super.methods=&Javamia_java_lang_String;\n"
	      "\t\targs->f_length=ac-1;\n"
	      "\t\tfor (i=1; i<ac; i++) {\n"
	      "\t\t\tstruct Javaiachar *array;\n"
	      "\t\t\tint len=strlen(av[i]);\n"
	      "\t\t\tint scan;\n"
	      "\t\t\tarray=malloc(sizeof(struct Javaiachar)\n"
	      "\t\t\t             +sizeof(Javachar)*len);\n"
	      "\t\t\tarray->super.methods=&Javamiachar;\n"
	      "\t\t\tarray->f_length=len;\n"
	      "\t\t\tfor (scan=0; scan<len; scan++) {\n"
	      "\t\t\t\tarray->component[scan]=av[i][scan];\n"
	      "\t\t\t}\n"
	      "\t\t\targs->component[i-1]\n"
	      "\t\t\t      =malloc(sizeof(struct Javai_java_lang_String));\n"
	      "\t\t\tJavac_java_lang_String___3C(0, args->component[i-1],\n"
	      "\t\t\t      &Javami_java_lang_String, array);\n"
	      "\t\t}\n", o);
	fprintf(o, "\t\tJava_%s_main___3Ljava_lang_String_2(0, args);\n"
		   "\t} else if (setjmp(handler)==0) {\n"
		   "\t\tJava_java_lang_ThreadGroup_uncaughtException__Ljava_lang_Object_2(0, thrown_object);\n"
		   "\t}\n"
	           "\treturn 0;\n}\n", startclass);
}

#endif /* SEEN_codegen_h */
