(* camlp4r *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: odyl.ml,v 2.1 1999/03/04 16:34:24 ddr Exp $ *)

(* Load files in core *)

value find_in_path path name =
  if not (Filename.is_implicit name) then
    if Sys.file_exists name then name else raise Not_found
  else
    let rec try_dir =
      fun
      [ [] -> raise Not_found
      | [dir :: rem] ->
          let fullname = Filename.concat dir name in
          if Sys.file_exists fullname then fullname else try_dir rem ]
    in
    try_dir path
;

exception Error of string and string;

value objfiles = ref [];
value nolib = ref False;
value path = ref [];

value loadfile file =
  let fname =
    try find_in_path path.val file with
    [ Not_found -> raise (Error file "file not found in path") ]
  in
  try Dynlink.loadfile fname with
  [ Dynlink.Error e -> raise (Error fname (Dynlink.error_message e)) ]
;

value load objfiles =
  do Dynlink.init (); Dynlink.allow_unsafe_modules True; return
  List.iter loadfile objfiles
;

value apply_load () =
  let i = ref 1 in
  let stop = ref False in
  do while not stop.val && i.val < Array.length Sys.argv do
       let s = Sys.argv.(i.val) in
       if s = "-I" && i.val + 1 < Array.length Sys.argv then
         do path.val := [Sys.argv.(i.val + 1) :: path.val];
            i.val := i.val + 2;
         return ()
       else if s = "-nolib" then
         do nolib.val := True; incr i; return ()
       else if s = "-where" then
         do print_string Odyl_config.standard_library;
            print_newline ();
            flush stdout;
         return exit 0
       else if s = "--" then
         do incr i; stop.val := True; return ()
       else if String.length s > 0 && s.[0] == '-' then
         stop.val := True
       else if
         Filename.check_suffix s ".cmo" ||
         Filename.check_suffix s ".cma" then
         do objfiles.val := [s :: objfiles.val];
            incr i;
         return ()
       else stop.val := True;
     done;
     if nolib.val then ()
     else path.val := path.val @ [Odyl_config.standard_library];
  return
  let objs = List.rev objfiles.val in
  load objs
;

value main () =
  do try apply_load () with
     [ Error fname str ->
         do flush stdout;
            Printf.eprintf "Error while loading \"%s\": " fname;
            Printf.eprintf "%s.\n" str;
            flush stderr;
         return exit 2 ];
     Odyl_main.go.val ();
  return ()
;

Printexc.catch main ();
