(* camlp4r *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: camlp4_top.ml,v 2.1 1999/09/10 15:43:03 ddr Exp $ *)

open Parsetree;
open Lexing;
open Stdpp;

value ast2pt_directive loc =
  fun
  [ MLast.DpNon -> Pdir_none
  | MLast.DpStr s -> Pdir_string s
  | MLast.DpInt i -> Pdir_int (int_of_string i)
  | MLast.DpIde i -> Pdir_ident (Ast2pt.long_id_of_string_list loc i) ]
;

value ast2pt_phrase =
  fun
  [ MLast.PhStr _ si -> Ptop_def (Ast2pt.str_item si [])
  | MLast.PhDir loc d dp -> Ptop_dir d (ast2pt_directive loc dp) ]
;

value rec skip_to_eol cs =
  match Stream.peek cs with
  [ Some '\n' -> ()
  | Some c -> do Stream.junk cs; return skip_to_eol cs
  | _ -> () ]
;

value wrap f lb =
  let cs =
    Stream.from
      (fun _ ->
         do while
              lb.lex_curr_pos >= String.length lb.lex_buffer
              && not lb.lex_eof_reached
            do
              lb.refill_buff lb;
            done;
         return
         if lb.lex_eof_reached then None
         else
           let c = lb.lex_buffer.[lb.lex_curr_pos] in
           do lb.lex_curr_pos := lb.lex_curr_pos + 1; return
           Some c)
  in
  try f cs with
  [ Exc_located _ (Sys.Break as x) -> raise x
  | End_of_file as x -> raise x
  | x ->
      let x =
        match x with
        [ Exc_located loc x ->
            do Toploop.print_location (Ast2pt.mkloc loc); return x
        | x -> x ]
      in
      do skip_to_eol cs;
         Format.open_hovbox 0;
         Pcaml.report_error x;
         Format.close_box ();
         Format.print_newline ();
      return raise Exit ]
;

value toplevel_phrase cs =
  match Grammar.Entry.parse Pcaml.top_phrase cs with
  [ Some phr -> ast2pt_phrase phr
  | None -> raise End_of_file ]
;
value use_file cs =
  let v = Pcaml.input_file.val in
  do Pcaml.input_file.val := Toploop.input_name.val; return
  let restore () = Pcaml.input_file.val := v in
  try
    let r = List.map ast2pt_phrase (Grammar.Entry.parse Pcaml.use_file cs) in
    do restore (); return r
  with e -> do restore (); return raise e
;

Toploop.parse_toplevel_phrase.val :=
  fun lb ->
    do Printf.eprintf "\tCamlp4 Parsing version %s\n\n" Pcaml.version;
       flush stderr;
       Toploop.parse_toplevel_phrase.val := wrap toplevel_phrase;
    return Toploop.parse_toplevel_phrase.val lb
;
Toploop.parse_use_file.val := wrap use_file;

Pcaml.warning.val :=
  fun loc txt -> Toploop.print_warning (Ast2pt.mkloc loc) (Warnings.Other txt)
;
