(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 names.ml                                 *)
(****************************************************************************)

open Std;;
open More_util;;
open Pp;;

(********************************************************************)
(*                             Names                                *)
(********************************************************************)

module StrObj = struct type key = string  let pr_key s = [< 'sTR s >] end;;
module Map =
  struct
    type key = string
    type 'a t = (string,'a) Mhm.t
    let create () = Mhm.create 17
    let add = Mhm.add
    let map = Mhm.map
  end;;

(* Au choix : avec ou sans codage par un entier *)

module StrListObj =
  struct 
    type key = string list
    let pr_sharp () = [< 'sTR "#" >]
    let pr_string s = [< 'sTR s >]
    let pr_key l = [< 'sTR "#"; prlist_with_sep pr_sharp pr_string l >]
end;;
module FmavmMap =
  struct
    type key = string list
    type 'a t = (key,'a) Fmavm.t
    let create () = Fmavm.create (rev_lexico compare_strings,17)
    let add = Fmavm.add
    let map = Fmavm.map
  end;;

type identifier = {atom : string ; index : int};;


type path_kind = CCI | FW | OBJ;;
type section_path = {dirpath : string list ;
                     basename : identifier ;
                     kind : path_kind}
;;

let make_ident sa n = {atom = sa ; index = n};;
let repr_ident {atom=sa; index=n} = (sa,n);;

let make_path pa id k = {dirpath = pa; basename = id; kind = k};;
let repr_path {dirpath=pa; basename=id; kind=k} = (pa,id,k);;

type name = Name of identifier | Anonymous;;

let explode_id {atom=s;index=n} =
  (explode s)@(if n = (-1) then [] else explode(string_of_int n))
;;

let print_id {atom=a;index=n} = match (a,n) with
    ("",-1) -> [< 'sTR"[]" >]
  | ("",n)  -> [< 'sTR"[" ; 'iNT n ; 'sTR"]" >]
  | (s,n)   -> [< 'sTR s ; (if n = (-1) then [< >] else [< 'iNT n >]) >]
;;

let pr_idl idl = prlist_with_sep pr_spc print_id idl
;;


let string_of_id {atom=s;index=n} =
    s ^ (if n = (-1) then "" else (string_of_int n))
;;

let code_of_0 = Char.code '0';;
let code_of_9 = Char.code '9';;

let id_of_string s =
    let slen = String.length s in
    (* n' is the position of the first non nullary digit *)
    let rec numpart n n' =
        if n = 0 then failwith("identifier " ^ s ^ " cannot be split")
        else let c = Char.code(String.get s (n-1)) in
            if c = code_of_0  & n <> slen then numpart (n-1) n' 
            else if code_of_0 <= c & c <= code_of_9 then numpart (n-1) (n-1)
            else n'
    in
    let numstart = numpart slen slen in
    if numstart = slen then {atom=s;index=(-1)} else
        {atom=String.sub s 0 numstart;
         index =int_of_string (String.sub s numstart (slen - numstart))}
;;

let atompart_of_id id = id.atom;;
let index_of_id id = id.index;;


let id_ord id1 id2 =
  let ((s1,n1)) = repr_ident id1 and
      ((s2,n2)) = repr_ident id2 in
  let s_bit = Pervasives.compare s1 s2
  in if s_bit = 0 then n1 - n2 else s_bit
;;

let id_without_number id = id.index = (-1);;

let next_ident_away ({atom=str} as id) avoid = 
 let rec name_rec i =
    let create = if i = (-1) then id else {atom=str;index=i} in
    if List.mem create avoid then name_rec (i+1) else create
 in name_rec (-1)
;;

let lift_ident {atom=str;index=i} = {atom=str;index=i+1};;

let rec next_ident_away_from {atom=str;index=i} avoid = 
 let rec name_rec i =
    let create = {atom=str;index=i} in
    if List.mem create avoid then name_rec (i+1) else create
 in name_rec i 
;;

let next_name_away_with_default default name l = match name with
   Name(str) -> next_ident_away str l
 | Anonymous -> next_ident_away (id_of_string default) l
;;

let next_name_away name l = match name with
   Name(str) -> next_ident_away str l
 | Anonymous -> id_of_string "_"
;;


(* returns lids@[i1..in] where i1..in are new identifiers prefixed id *)
let get_new_ids n id lids  =
let rec get_rec n acc =
 if n=0 then acc 
  else let nid=next_ident_away id (acc@lids)
        in get_rec (n-1) (nid::acc) 
in get_rec n []
;;

let kind_of_path sp = sp.kind;;

let assoc_kind k l =
  try List.assoc k l with Not_found -> error "No such kind"

let string_of_kind = function
    CCI -> "cci" | FW -> "fw" | OBJ -> "obj"
;;

let kind_of_string = function
    "cci" -> CCI | "fw" -> FW | "obj" -> OBJ
  | _ -> invalid_arg "Malformed path-kind"
;;

let coerce_path k {dirpath=p;basename=id} =
  {dirpath=p;basename=id;kind=k};;
let ccisp_of_fwsp = function
  {dirpath=p;basename=id;kind=FW} -> {dirpath=p;basename=id;kind=CCI}
| _ -> anomaly "Names: ccisp_of_fwsp";;
let ccisp_of {dirpath=p;basename=id} =
  {dirpath=p;basename=id;kind=CCI};;
let objsp_of {dirpath=p;basename=id} =
  {dirpath=p;basename=id;kind=OBJ};;
let fwsp_of_ccisp = function
  {dirpath=p;basename=id;kind=CCI} -> {dirpath=p;basename=id;kind=FW}
| _ -> anomaly "Names: fwsp_of_ccisp";;
let fwsp_of {dirpath=p;basename=id} =
  {dirpath=p;basename=id;kind=FW};;

let sp_ord sp1 sp2 =
  let (p1,id1,k) = repr_path sp1 and
      (p2,id2,k') = repr_path sp2 in
  if k = k' then
    let p_bit = Pervasives.compare p1 p2
    in if p_bit = 0 then id_ord id1 id2 else p_bit
  else
    compare_strings (string_of_kind k) (string_of_kind k')
;;

let sp_gt (sp1,sp2) =
  if sp_ord sp1 sp2 > 0 then true else false
;;

let section_path sl s =
  let sl = List.rev sl in
  make_path (List.tl sl) (id_of_string (List.hd sl)) (kind_of_string s)
;;

let path_section loc sp =
  let (sl,bn,pk) = repr_path sp in
    CoqAst.Path(loc,List.rev(string_of_id bn :: sl), string_of_kind pk)
;;

let string_of_path sp =
  let (sl,id,k) = repr_path sp in
  implode(List.flatten(List.map (fun s -> ["#";s])
			 (List.rev (string_of_id id :: sl)))
	  @[".";string_of_kind k])
;;

let string_of_path_mind sp mindid =
  let (sl,id,k) = repr_path sp in
  implode(List.flatten(List.map (fun s -> ["#";s])
			 (List.rev (string_of_id mindid :: sl)))
	  @[".";string_of_kind k])
;;


let path_eoi = Pcoq.eoi_entry Pcoq.Prim.path;;
let path_of_string s =
  try
    (match Pcoq.parse_string path_eoi s with
	CoqAst.Path(_,sl,knd) -> section_path sl knd 
       | _ -> anomaly "path_of_string : absurd result from Pcoq.parse_string")
  with Stream.Error _ | Token.Error _ | Match_failure _ ->
    errorlabstrm "Names.path_of_string" [< 'sTR"Invalid path: "; 'sTR s >]
;;

let basename sp = sp.basename;;
let dirpath sp = sp.dirpath;;

let append_to_path sp str =
  let (sp,id,k) = repr_path sp in
  make_path sp (id_of_string((string_of_id id)^str)) k
;;

let sp_of_wd = function
   (bn::dp) -> make_path dp (id_of_string bn) OBJ
 |  _       -> anomaly "Names: sp_of_wd"
;;


type 'a signature = identifier list * 'a list;;
type 'a db_signature = (name * 'a) list;;
type ('a,'b) env = ENVIRON of 'a signature * 'b db_signature;;

let gLOB hyps = ENVIRON(hyps,[]);;

let ids_of_sign (idl,_) = idl;;
let vals_of_sign (_,vals) = vals;;
let add_sign (id,ty) (idl,tyl) = (id::idl,ty::tyl);;
let sign_it f (idl,tyl) e = List.fold_right2 f idl tyl e;;
let it_sign f e (idl,tyl) = List.fold_left2 f e idl tyl;;
let nil_sign = ([],[]);;
let rev_sign (idl,tyl) = (List.rev idl,List.rev tyl);;
let map_sign_typ f (idl,tyl) = (idl,List.map f tyl);;
let concat_sign (idl1,tyl1) (idl2,tyl2) = (idl1@idl2,tyl1@tyl2);;
let diff_sign   (idl1,tyl1) (idl2,tyl2) = ((subtractq idl1 idl2),(subtractq tyl1 tyl2));;

let dbenv_it f (ENVIRON(_,dbs)) init =
    List.fold_right (fun (na,t) v -> f na t v) dbs init;;
let it_dbenv f init (ENVIRON(_,dbs)) =
    List.fold_left (fun v (na,t) -> f v na t) init dbs;;

let isnull_sign = function 
    ([],[])     -> true
  | (_::_,_::_) -> false
  | _           -> invalid_arg "isnull_sign"
;;

let isnull_rel_env (ENVIRON(_,dbs)) = (dbs = []);;
let uncons_rel_env (ENVIRON(sign,dbs)) = (List.hd dbs,ENVIRON(sign,List.tl dbs));;

let hd_sign = function
    (id::_,ty::_) -> (id,ty)
  | _             -> failwith "hd_sign"
;;

let tl_sign = function
    (_::idl,_::tyl) -> (idl,tyl)
  | _               -> failwith "tl_sign"
;;

let ids_of_env (ENVIRON(sign,dbenv)) =
  let filter (n,_) l = match n with (Name id) -> id::l | Anonymous -> l in
  (ids_of_sign sign)@(List.fold_right filter dbenv [])
;;

type ('b,'a) search_result =
    GLOBNAME of identifier  * 'b
  | RELNAME of int * 'a
;;

let get_globals (ENVIRON(g,_)) = g;;
let get_rels (ENVIRON(_,r)) = r;;

let add_rel (n,x) (ENVIRON(g,r)) = (ENVIRON(g,(n,x)::r));;

let add_glob (id,x) (ENVIRON((dom,rang),r)) = (ENVIRON((id::dom,x::rang),r));;

let lookup_glob id (ENVIRON((dom,rang),_)) = 
 let rec aux = function
     ([],          [])        -> raise Not_found
   | ((id'::id'l), (ty::tyl)) -> if id' = id then (id',ty) else aux (id'l,tyl)
   | _                        -> anomaly "Names: lookup_glob"
 in aux (dom,rang)
;;

let mem_glob id (ENVIRON((dom,_),_)) = List.mem id dom
;;

(* Amok *)

let lookup_sign id (dom,rang) = 
 let rec aux = function
     ([],          [])        -> raise Not_found
   | ((id'::id'l), (ty::tyl)) -> if id' = id then (id',ty) else aux (id'l,tyl)
   | _                        -> anomaly "Names: lookup_sign"
 in aux (dom,rang)
;;

let lookup_rel n (ENVIRON(_,r)) = 
 let rec lookrec n l = match (n,l) with
     (1, ((na,x)::l)) -> (na,x)
   | (n, (_::l))      -> lookrec (n-1) l
   | (_, [])          -> raise Not_found
 in lookrec n r
;;

let rec lookup_rel_id id (ENVIRON(_,r)) = 
 let rec lookrec = function
     (n, ((Anonymous,x)::l)) -> lookrec (n+1,l)
   | (n, ((Name id',x)::l))  -> if id' = id then (n,x) else lookrec (n+1,l)
   | (_, [])                 -> raise Not_found
 in lookrec (1,r)
;;

let lookup_id id env =
  try let (x,y) = lookup_rel_id id env in RELNAME(x,y)
  with Not_found -> let (x,y) = lookup_glob id env in GLOBNAME(x,y)
;;

let map_rel_env f (ENVIRON(g,r)) =
    (ENVIRON(g,List.map (fun (na,x) -> (na,f x)) r));;

let map_var_env f (ENVIRON((dom,rang),r)) =
  (ENVIRON(List.fold_right2 (fun na x (doml,rangl) -> (na::doml,(f x)::rangl))
      	       	       	       	       	       	       	  dom rang ([],[]),r))
;;

let nth_sign (idl,tyl) n = (List.nth idl (n-1), List.nth tyl (n-1));;
let map_sign_graph f (ids,tys) = List.map2 f ids tys;;

let list_of_sign (ids,tys) =
  try List.combine ids tys
  with _ -> anomaly "Corrupted signature"

let make_sign = List.split;;
let do_sign f (ids,tys) = List.iter2 f ids tys;;

let uncons_sign = function
    (id::idl,ty::tyl) -> ((id,ty),(idl,tyl))
  | _ -> anomaly "signatures are being manipulated in a non-abstract way"
;;

let sign_length (idl,tyl) =
  let lenid = List.length idl
  and lenty = List.length tyl in
  if lenid = lenty then lenid
                   else invalid_arg "lookup_sign"
;;

let mem_sign sign id = List.mem id (ids_of_sign sign)

let modify_sign id ty = 
 let rec modrec = function
     [],[] -> invalid_arg "modify_sign"
   | sign  -> let (id',ty') = hd_sign sign in
              if id = id' then
      	        add_sign (id,ty) (tl_sign sign)
              else
      	       	add_sign (id',ty') (modrec (tl_sign sign))
 in modrec
  
;;

let add_envp_rel (na,v1,v2) (env1,env2) =
    (add_rel (na,v1) env1,add_rel (na,v2) env2);;

let exists_sign f = 
 let rec exrec sign =
    if isnull_sign sign then false
    else let ((id,t),tl) = uncons_sign sign
         in f id t or exrec tl
 in exrec
  
;;

exception NotFound;;
let witness_sign f = 
 let rec exrec sign =
    if isnull_sign sign then raise NotFound 
    else let ((id,t),tl) = uncons_sign sign
         in if f id t then (id,t) else exrec tl
 in exrec
  
;;

(* [sign_prefix id sign] returns the signature up to and including id,
   with all later assumptions stripped off.  It is an error to call it
   with a signature not containing id, and that error is generated
   with error. *)

let sign_prefix id sign = 
 let rec prefrec sign =
    if isnull_sign sign then
      error "sign_prefix"
    else
      let ((id',t),sign') = uncons_sign sign in
      if id' = id then sign else prefrec sign'
 in prefrec sign
;;

let add_sign_after whereid (id,t) sign = 
 let rec addrec sign =
    if isnull_sign sign then
      error "add_sign_after"
    else
      let ((id',t'),sign') = uncons_sign sign in
      if id' = whereid then add_sign (id,t) sign
      else add_sign (id',t') (addrec sign')
 in addrec sign
;;

let add_sign_replacing whereid (id,t) sign = 
 let rec addrec sign =
    if isnull_sign sign then
      error "add_replacing_after"
    else
      let ((id',t'),sign') = uncons_sign sign in
      if id' = whereid then add_sign (id,t) sign'
      else add_sign (id',t') (addrec sign')
 in addrec sign
  
;;

(* [prepend_sign Gamma1 Gamma2]
   prepends Gamma1 to the front of Gamma2, given that their namespaces
   are distinct. *)

let prepend_sign gamma1 gamma2 =
  if [] = intersect (ids_of_sign gamma1) (ids_of_sign gamma2) then
    let (ids1,vals1) = gamma1 and
        (ids2,vals2) = gamma2
    in (ids1@ids2,vals1@vals2)
  else
    failwith "prepend_sign"
;;

module Sp =
  struct
    type t = section_path
    let compare = sp_ord
  end;;

module Spset = Set.Make(Sp);;


(* Hash-consing of name objects *)
module Hident = Hashcons.Make(
  struct type t = identifier
         type u = string -> string
         let hash_sub hstr id =
           {atom = hstr id.atom; index = id.index}
         let equal id1 id2 = id1.atom==id2.atom & id1.index = id2.index
         let hash = Hashtbl.hash
  end);;

module Hname = Hashcons.Make(
  struct type t = name
         type u = identifier -> identifier
         let hash_sub hident = function
             Name id -> Name (hident id)
           | n -> n
         let equal n1 n2 =
           match (n1,n2) with
               (Name id1, Name id2) -> id1==id2
             | (Anonymous,Anonymous) -> true
             | _ -> false
         let hash = Hashtbl.hash
  end);;

module Hsp = Hashcons.Make(
  struct type t = section_path
         type u = (identifier -> identifier) * (string -> string)
         let hash_sub (hident,hstr) sp =
           {dirpath = List.map hstr sp.dirpath;
            basename = hident sp.basename;
            kind = sp.kind}
         let equal sp1 sp2 =
           (sp1.basename==sp2.basename) & (sp1.kind=sp2.kind)
           & (List.length sp1.dirpath = List.length sp2.dirpath)
           & List.for_all2 (==) sp1.dirpath sp2.dirpath
         let hash = Hashtbl.hash
  end);;

let hcons_names () =
  let hstring = Hashcons.simple_hcons Hashcons.Hstring.f () in
  let hident = Hashcons.simple_hcons Hident.f hstring in
  let hname = Hashcons.simple_hcons Hname.f hident in
  let hspcci = Hashcons.simple_hcons Hsp.f (hident,hstring) in
  let hspfw = Hashcons.simple_hcons Hsp.f (hident,hstring) in
    (hspcci,hspfw,hname,hident,hstring)
;;


(* $Id: names.ml,v 1.30 1999/08/02 19:26:05 barras Exp $ *)
